// c_real_channel_inline.h: implementation of the c_real_channel class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_REAL_CHANNEL_INCLUDE_H__EB69AC61_2087_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_REAL_CHANNEL_INCLUDE_H__EB69AC61_2087_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_real_channel::c_real_channel(p_mixer mixer, p_module module, signed long channel, signed long volume, signed long panning)
{
	m_mixer = mixer ;
	m_module = module ;
	m_channel = channel ;
	if (panning <= 64)
	{
		m_default_panning = panning - 32 ;
		m_default_muted_flag = false ;
	}
	else
	{
		m_default_panning = 0 ;
		m_default_muted_flag = true ;
	}

	if (volume <= 64)
	{
		m_default_volume = volume ;
	}
	else
	{
		m_default_volume = 64 ;
	}
	init () ;
}

c_real_channel::~c_real_channel()
{
	// Nothing
}


// return value :
// true if channel need frame treatment
// false else
bool c_real_channel::update_line(bool flag, bool just_read)
{
	bool new_note = false ;
	bool note_changed = false ;
	bool new_instrument = false ;
	bool instrument_changed = false ;
	bool new_volume_panning = false ;
	bool new_command = false ;
	bool no_instrument_reset = false ;

	if (just_read)
	{
		if (flag)
		{ /* read byte into mask variable */
			m_mask_variable = m_mixer->get_pattern_data() ;
		}
		if (m_mask_variable & 1)
		{
			m_note = m_mixer->get_pattern_data() ;
		}
		if (m_mask_variable & 2)
		{
			signed long instrument = m_mixer->get_pattern_data() ;

			if (instrument != 0)
			{
				m_instrument = instrument ;
			}
		}
		if (m_mask_variable & 4)
		{
			m_volume_panning = m_mixer->get_pattern_data() ;
		}
		if (m_mask_variable & 8)
		{
			m_command = m_mixer->get_pattern_data() ;
			m_command_value = m_mixer->get_pattern_data() ;
		}

		return false ;
	}

	p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;

	clear_effects() ;

	if (flag)
	{ /* read byte into mask variable */
		m_mask_variable = m_mixer->get_pattern_data() ;
	}
	
	signed long previous_note = m_note ;
	if (m_mask_variable & 1) /* read note (byte value 0->119, 254 = note cut, 255 = note off) */
	{
		signed long note_value = m_mixer->get_pattern_data() ;
		note_changed = (m_note != note_value) ;
		m_note = note_value ;
		new_note = true ;
	}
	else if (m_mask_variable & 16) /* use previous note value */
	{
		new_note = true ;
		note_changed = false ;
	}
	else
	{
		note_changed = false ;
		new_note = false ;
	}

	
	if (m_mask_variable & 2) /* read instrument (byte value 0->128) */
	{
		signed long instrument = m_mixer->get_pattern_data() ;

		if (instrument != 0)
		{
			instrument_changed = (m_instrument != instrument) ;
			m_instrument = instrument ;
			new_instrument = true ;
		}
		else
		{
			instrument_changed = false ;		
			new_instrument = true ;
		}
	}
	else if (m_mask_variable & 32) /* use previous instrument */
	{
		new_instrument = true ;
		instrument_changed = false ;
	}
	else
	{
		new_instrument = false ;
		instrument_changed = false ;
	}
	
	p_instrument pins = m_module->get_instrument(m_instrument) ;

	if (new_instrument && (pins == (p_instrument)NULL))
	{ 
		/* invalid instrument */
		new_note = false ;
		new_instrument = false ;
		if (m_playing)
		{
			m_playing = false ;
			m_mixer->drop_virtual_channel(m_virtual_channel) ;
		}
	}
	
	if (m_mask_variable & 4) /* read volume / panning */
	{
		m_volume_panning = m_mixer->get_pattern_data() ;
		new_volume_panning = true ;
	}
	else if (m_mask_variable & 64) /* use previous volume / panning */
	{
		new_volume_panning = true ;
	}
	else
	{
		new_volume_panning = false ;
	}
	
	
	if (m_mask_variable & 8) /* read command and command value */
	{
		m_command = m_mixer->get_pattern_data() ;
		m_command_value= m_mixer->get_pattern_data() ;
	
		new_command = true ;
	}
	else if (m_mask_variable & 128) /* use previous command and command value */
	{
		new_command = true ;
	}
	else
	{
		new_command = false ;
	}
	
	if ((new_instrument || new_note) && (pins == (p_instrument)NULL) && (m_note < 254))
	{ 
		/* invalid instrument */
		new_note = false ;
		new_instrument = false ;
		if (m_playing)
		{
			/* Drop the channel */
			m_mixer->drop_virtual_channel(m_virtual_channel) ;
			m_playing = false ;
		}
	}
	
	signed long new_note_action = 0 ;
	no_instrument_reset = m_playing && new_command && ((m_command == 7) || (m_command == 12)) ;

	if (m_playing 
		&& (new_instrument || new_note) 
		&& (m_note <= 119) 
		&& !no_instrument_reset)
	{
		signed long sample_value = pins->get_sample((signed long)m_note) ;
		
		bool duplicate_check_result ;

		if ((m_new_note_action == 0) || (m_duplicate_check_type == 0)) /* DCT off */
		{
			duplicate_check_result = false ;
		}
		else if (m_duplicate_check_type == 1)/* DCT = NOTE */
		{
			duplicate_check_result = !note_changed ;
		}
		else if (m_duplicate_check_type == 2)/* DCT = SAMPLE */
		{
			duplicate_check_result = (m_sample == sample_value) ;
		}
		else if (m_duplicate_check_type == 3)/* DCT = INSTRUMENT */
		{
			duplicate_check_result = !instrument_changed ;
		}
		
		if (duplicate_check_result)
		{
			new_note_action = m_duplicate_check_action ;
			if (new_note_action > 0)
			{
				new_note_action++ ;
			}
		}
		else
		{
			new_note_action = m_new_note_action ;
		}
	}


	/* new note handling */
	
	if ((new_instrument || new_note) && !no_instrument_reset)
	{
		if (m_note < 120) /* note_value is a true note */
		{
			if (m_playing)
			{
				switch (new_note_action)
				{
				case 0 : pvc->note_cut() ; break ;
				case 2 : pvc->note_off() ; break ;
				case 3 : pvc->set_fade_on() ; break ;
				default : break ;
				}

				pvc->set_in_background() ;
			}

			if (m_muted)
			{
				m_playing = false ;
			}
			else
			{
				m_virtual_channel = m_mixer->allocate_virtual_channel(m_channel) ;

				if (m_virtual_channel == -1)
				{
					new_note = false ;
					new_instrument = false ;
					m_playing = false ;
				}
				else
				{
					pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
					pvc->set_channel_panning(m_panning) ;
					m_playing = true ;
				}
			}
		}
		else
		{ 
			if (m_playing && (m_note >= 254))
			{
				if (m_note == 254)
				{ 
					/* note cut */
					pvc->note_cut() ;
					m_playing = false ;
					pvc->set_in_background() ;
				} 
				else
				{ 
					/* note off */
					pvc->note_off() ;
				} 

			}
		}
	} /* end of new note handling */
		
	
	/* new note / new instrument common handling */
	if (m_playing && (new_note || new_instrument) && (m_note <= 119) && !no_instrument_reset)
	{
		m_sample = pins->get_sample((signed long)m_note) ;
		m_pitch_slide_note = m_sample_note = pins->get_note((signed long)m_note)   ;

		p_sample psmp = m_module->get_sample(m_sample) ;
		if (psmp == (p_sample)NULL)
		{
			/* bad sample */
			m_playing = false ;
			m_mixer->drop_virtual_channel(m_virtual_channel) ;
			return false ;
		}
		else
		{
			pvc->new_note(m_instrument, m_note) ;
			if (!new_instrument)
			{
				pvc->set_note_volume(m_note_volume) ;
			}
			else
			{
				m_note_volume = pvc->get_note_volume() ;
			}

			m_tremor_time = 0 ;
			
			if (m_mixer->are_new_note_actions_activated())
			{
				m_new_note_action = pins->get_new_note_action() ;
				m_duplicate_check_type = pins->get_duplicate_check_type() ;
				m_duplicate_check_action = pins->get_duplicate_check_action() ;
			}
			else
			{
				m_new_note_action = 0 ;
				m_duplicate_check_type = 0 ;
				m_duplicate_check_action = 0 ;
			}
		}
	}
	/* end of new note or new instrument common handling */

	if (m_playing && new_instrument && no_instrument_reset)
	{
		pvc->reset_volume() ;
	}
	
	if (new_command)
	{
		switch (m_command)
		{
		case 1 : m_mixer->set_speed(m_command_value) ; break ;
		case 2 : m_mixer->jump_to_order(m_command_value) ; break ;
		case 3 : m_mixer->pattern_break(m_command_value) ; break ;
		case 13 : set_channel_volume() ; break ;
		case 14 : channel_volume_slide() ; break ;
		case 16 : panning_slide() ; break ;
		case 19 : 
			{
				switch (m_command_value & 0xF0)
				{
				default : break ;
				case 0x30 : m_vibrato_waveform = m_command_value & 0x03 ; break ;
				case 0x40 : m_tremelo_waveform = m_command_value & 0x03 ; break ;
				case 0x50 : m_panbrello_waveform = m_command_value & 0x03 ; break ;
				case 0x70 : 
					{
						switch (m_command_value)
						{
						default : break ;
						case 0x70 : m_mixer->past_note_cut(m_channel) ; break ;
						case 0x71 : m_mixer->past_note_off(m_channel) ; break ;
						case 0x72 : m_mixer->past_note_fade(m_channel) ; break ;
						case 0x73 : m_new_note_action = 0 ; break ;
						case 0x74 : m_new_note_action = 1 ; break ;
						case 0x75 : m_new_note_action = 2 ; break ;
						case 0x76 : m_new_note_action = 3 ; break ;
						case 0x77 : if (m_playing) pvc->set_volume_envelope(false); break ;
						case 0x78 : if (m_playing) pvc->set_volume_envelope(true); break ;
						}
					}
					break ;
				case 0x80 :
					{
						signed long new_panning = (m_command_value & 0x0F) - 7 ;
						m_panning = (new_panning > 0)?new_panning * 4:(new_panning - 1) * 4;
					}
					break ;
				case 0x90 : 
					{
						if (m_command_value == 0x91)
						{
							m_surround_on = true ;
							if (m_playing)
							{
								pvc->set_surround(m_surround_on);
							}
						}
					}
					break ;
				case 0xB0 : 
					{
						loop(m_command_value & 0x0F) ;
					}
					break ;
				}
			}
			break ;
		case 20 : m_mixer->set_tempo(m_command_value) ; break ;
		case 22 : m_mixer->set_global_volume(m_command_value) ; break ;
		case 23 : m_mixer->global_volume_slide(m_command_value) ; break ;
		case 24 : set_panning() ; break ;
		default : 
			{
				if (m_playing)
				{
					switch (m_command)
					{
					default : break ;
					case 4 : volume_slide(); break ;
					case 5 : pitch_slide_down() ; break ;
					case 6 : pitch_slide_up() ; break ;
					case 7 : 
						{
							pitch_slide_to_note(new_note, instrument_changed, previous_note) ; 
						}
						break ;
					case 8 : 
						{
							if (new_note)
							{
								m_vibrato_position = 0 ;
								m_vibrato_time = 0 ;
							}
							
							vibrato();
						}
						break ;
					case 9 :
						{
							if (new_note)
							{
								m_tremor_time = 0 ;
							}
							tremor();
						}
						break ;
					case 10 : arpeggio() ; break ;
					case 11 : 
						{
							if (new_note)
							{
								m_vibrato_position = 0 ;
								m_vibrato_time = 0 ;
							}
							
							vibrato();
							volume_slide(); 
						}
						break ;
					case 12 : 
						{
							pitch_slide_to_note(new_note, instrument_changed, previous_note) ; 
							volume_slide(); 
						}
						break ;
					case 15 : set_sample_offset() ; break ;
					case 17 : retrigger() ; break ;
					case 18 : 
						{
							if (new_note)
							{
								m_tremelo_position = 0 ;
								m_tremelo_time = 0 ;
							}
							tremelo();
						}
						break ;										
					case 21 : 
						{
							if (new_note)
							{
								m_vibrato_position = 0 ;
								m_vibrato_time = 0 ;
							}
							
							vibrato();
						}
						break ;
					case 25 : 
						{
							if (new_note)
							{
								m_panbrello_position = 0 ;
								m_panbrello_time = 0 ;
							}
							
							panbrello();
						}
						break ;
					}					
				} /* m_playing */				
			}
			break ;
		}
	}
	
	if (new_volume_panning)
	{
		if (m_volume_panning <= 64)
		{
			if (m_playing)
			{
				pvc->set_note_volume(m_volume_panning) ;
				m_note_volume = m_volume_panning ;
			}
		}
		else if ((m_volume_panning >= 128) && (m_volume_panning <= 192))
		{
			m_panning = m_volume_panning - 160 ;
		}
	}
	
	if (m_playing)
	{
		pvc->set_volume (m_volume, m_panning) ;
	}

	return m_playing || m_effect_update ;
}

void c_real_channel::set_channel_volume()
{
	m_volume = m_command_value < 64?m_command_value:64 ;
}

void c_real_channel::channel_volume_slide()
{
	if (m_command_value == 0)
	{
		m_command_value = m_channel_volume_slide  ;
	}
	else
	{
		m_channel_volume_slide = m_command_value ;
	}

	if ((m_command_value & 0x0F) == 0x00) /* volume slide up */
	{
		m_effect_update = true ;
		set_channel_volume_slide_value((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0x00) /* volume slide down */
	{
		m_effect_update = true ;
		set_channel_volume_slide_value (-(m_command_value & 0x0F)) ;
	}
	else if ((m_command_value & 0x0F) == 0x0F) /* fine volume slide up */
	{
		slide_channel_volume((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0xF0) /* fine volume slide down */
	{
		slide_channel_volume(-(m_command_value & 0x0F)) ;
	}
}

void c_real_channel::panning_slide()
{
	if (m_command_value == 0)
	{
		m_command_value = m_panning_slide  ;
	}
	else
	{
		m_panning_slide = m_command_value ;
	}

	if ((m_command_value & 0x0F) == 0x00) /* panning slide left */
	{
		m_effect_update = true ;
		set_panning_slide_value ((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0x00) /* panning slide right */
	{
		m_effect_update = true ;
		set_panning_slide_value (-(m_command_value & 0x0F)) ;
	}
	else if ((m_command_value & 0x0F) == 0x0F) /* fine panning slide left */
	{
		slide_panning ((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0xF0) /* fine panning slide right */
	{
		slide_panning (-(m_command_value & 0x0F)) ;
	}

}

void c_real_channel::set_panning()
{
	double new_panning = m_command_value - 128;
	m_panning = (new_panning < 0)?new_panning / 4:(new_panning + 1) / 4 ;
}

void c_real_channel::pitch_slide_to_note(bool new_note, bool instrument_changed, signed long previous_note)
{
	p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel);

	if ((m_command != 12) && (m_command_value != 0))
	{
		m_pitch_slide = m_command_value ;
	}
		
	p_instrument pins = m_module->get_instrument(m_instrument) ;

/*	if (instrument_changed)
	{
		double delta = m_pitch_slide_note - m_sample_note ;

		m_sample = pins->get_sample(previous_note);
		m_sample_note = pins->get_note(previous_note);
		
		pvc->new_note(m_instrument, previous_note) ;
		
		if (m_mixer->are_new_note_actions_activated())
		{
			m_new_note_action = pins->get_new_note_action() ;
			m_duplicate_check_type = pins->get_duplicate_check_type() ;
			m_duplicate_check_action = pins->get_duplicate_check_action() ;
		}
		else
		{
			m_new_note_action = 0 ;
			m_duplicate_check_type = 0 ;
			m_duplicate_check_action = 0 ;
		}
		m_pitch_slide_note = m_sample_note + delta ; 
	} */
	if (new_note)
	{
		m_pitch_slide_note = m_note ;
		signed long sample = pins->get_sample(m_note) ;

		if (sample != m_sample)
		{
			// not handled for now
		}

		if (m_pitch_slide_note > m_sample_note)
		{
			m_effect_update = true ;
			m_pitch_slide_speed = m_pitch_slide;
		}
		else if (m_pitch_slide_note < m_sample_note)
		{
			m_effect_update = true ;
			m_pitch_slide_speed = -m_pitch_slide;
		}
		else
		{
			m_pitch_slide_speed = 0 ;
		}
	}
	else
	{
		if (m_pitch_slide_speed < 0)
		{
			if (m_pitch_slide_note < m_sample_note)
			{
				m_effect_update = true ;
				m_pitch_slide_speed = -m_pitch_slide;
			}
		}
		else if (m_pitch_slide_speed > 0)
		{
			if (m_pitch_slide_note > m_sample_note)
			{
				m_effect_update = true ;
				m_pitch_slide_speed = m_pitch_slide;
			}
		}
		else
		{
			if (m_pitch_slide_note > m_sample_note)
			{
				m_effect_update = true ;
				m_pitch_slide_speed = m_pitch_slide;
			}
			else if (m_pitch_slide_note < m_sample_note)
			{
				m_effect_update = true ;
				m_pitch_slide_speed = -m_pitch_slide;
			}
			else
			{
				m_pitch_slide_speed = 0 ;
			}
		}
	}
}

void c_real_channel::volume_slide()
{
	if (m_command_value == 0)
	{
		m_command_value = m_volume_slide  ;
	}
	else
	{
		m_volume_slide = m_command_value ;
	}

	if ((m_command_value & 0x0F) == 0x00) /* volume slide up */
	{
		m_effect_update = true ;
		set_volume_slide_value ((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0x00) /* volume slide down */
	{
		m_effect_update = true ;
		set_volume_slide_value (-(m_command_value & 0x0F)) ;
	}
	else if ((m_command_value & 0x0F) == 0x0F) /* fine volume slide up */
	{
		slide_volume((m_command_value & 0xF0) / 16) ;
	}
	else if ((m_command_value & 0xF0) == 0xF0) /* fine volume slide down */
	{
		slide_volume(-(m_command_value & 0x0F)) ;
	}

}

void c_real_channel::arpeggio()
{
	m_effect_update = true ;
	m_arpeggio_on = true ;
	m_arpeggio_counter = 0 ;
	m_arpeggio_offset_1 = (m_command_value & 0xF0) >> 4 ;
	m_arpeggio_offset_2 = m_command_value & 0x0F ;

	/*if ((m_command_value & 0xF0) != 0)
	{
		m_arpeggio_offset_1 = (m_command_value & 0xF0) >> 4 ;
	}
	if ((m_command_value & 0x0F) != 0)
	{
		m_arpeggio_offset_2 = m_command_value & 0x0F ;
	} */
}

void c_real_channel::retrigger()
{
	m_effect_update = true ;
	m_retrigger_on = true ;
	m_retrigger_counter = 0 ; /* ? */
	if ((m_command_value & 0x0F) != 0)
	{
		m_retrigger_delay = m_command_value & 0x0F ;
	}
	switch ((m_command_value & 0xF0) >> 4)
	{
	case 0x0 : m_retrigger_add = 0 ; m_retrigger_mul = 1 ; break ;
	case 0x1 : m_retrigger_add = -1 ; m_retrigger_mul = 1 ; break ;
	case 0x2 : m_retrigger_add = -2 ; m_retrigger_mul = 1 ; break ;
	case 0x3 : m_retrigger_add = -4 ; m_retrigger_mul = 1 ; break ;
	case 0x4 : m_retrigger_add = -8 ; m_retrigger_mul = 1 ; break ;
	case 0x5 : m_retrigger_add = -16 ; m_retrigger_mul = 1 ; break ;
	case 0x6 : m_retrigger_add = 0 ; m_retrigger_mul = 2/3 ; break ;
	case 0x7 : m_retrigger_add = 0 ; m_retrigger_mul = 1/2 ; break ;
	case 0x8 : m_retrigger_add = 0 ; m_retrigger_mul = 1 ; break ;
	case 0x9 : m_retrigger_add = 1 ; m_retrigger_mul = 1 ; break ;
	case 0xA : m_retrigger_add = 2 ; m_retrigger_mul = 1 ; break ;
	case 0xB : m_retrigger_add = 4 ; m_retrigger_mul = 1 ; break ;
	case 0xC : m_retrigger_add = 8 ; m_retrigger_mul = 1 ; break ;
	case 0xD : m_retrigger_add = 16 ; m_retrigger_mul = 1 ; break ;
	case 0xE : m_retrigger_add = 0 ; m_retrigger_mul = 3/2 ; break ;
	case 0xF : m_retrigger_add = 0 ; m_retrigger_mul = 2 ; break ;
	default : m_retrigger_add = 0 ; m_retrigger_mul = 1 ; break ; /* shouldn't be needed :-) */
	}
}

void c_real_channel::vibrato()
{
	m_effect_update = true ;
	m_vibrato_on = true ;
	if (m_command != 11)
	{		
		if ((m_command_value & 0xF0) != 0)
		{
			m_vibrato_speed = (m_command_value & 0xF0) >> 2 ;
		}
		if((m_command_value & 0x0F) != 0)
		{
			if (m_command == 8)
			{
				m_vibrato_depth = (m_command_value & 0x0F) * 4 ;
			}
			else
			{
				m_vibrato_depth = m_command_value & 0x0F ;
			}

			if (m_module->use_old_effects())
			{
				m_vibrato_depth *= 2 ;
			}
		}		
	}
}

void c_real_channel::tremelo()
{
	m_effect_update = true ;
	m_tremelo_on = true ;
	if ((m_command_value & 0xF0) != 0)
	{
		m_tremelo_speed = (m_command_value & 0xF0) >> 2 ; // ???
	}
	if((m_command_value & 0x0F) != 0)
	{
		m_tremelo_depth = m_command_value & 0x0F ;

		if (m_module->use_old_effects())
		{ // ???
			m_tremelo_depth *= 2 ;
		}
	}		
}

void c_real_channel::panbrello()
{
	m_effect_update = true ;
	m_panbrello_on = true ;
	if ((m_command_value & 0xF0) != 0)
	{
		m_panbrello_speed = (m_command_value & 0xF0) >> 4 ; // ???
	}
	if((m_command_value & 0x0F) != 0)
	{
		m_panbrello_depth = m_command_value & 0x0F ;
	}		
}

void c_real_channel::tremor()
{
	m_effect_update = true ;
	m_tremor_on = true ;
	if ((m_command_value & 0xF0) != 0)
	{
		m_tremor_ontime = (m_command_value & 0xF0) >> 4 ; 
		if (m_module->use_old_effects())
		{
			m_tremor_ontime++ ;
		}
//		m_tremor_time = 0 ;
	}
	if((m_command_value & 0x0F) != 0)
	{
		m_tremor_offtime = m_command_value & 0x0F ; 
		if (m_module->use_old_effects())
		{
			m_tremor_offtime++ ;
		}
//		m_tremor_time = 0 ;
	}
}

bool c_real_channel::pitch_slide_down()
{
	if (m_command_value == 0)
	{
		m_command_value = m_pitch_slide ;
	}
	else
	{
		m_pitch_slide = m_command_value ;
	}
			 
	if (m_command_value < 0xE0) /* pitch slide down by command_value */
	{
		m_effect_update = true ;
		set_pitch_slide_value(- m_command_value);
	}
	else if (m_command_value < 0xF0) /* extra fine pitch slide down */
	{
		if (slide_pitch(-((m_command_value & 0xFF) - 0xE0) / 4.0))
		{
			return true ;
		}
	}
	else /* fine pitch slide down */
	{
		if (slide_pitch(-((m_command_value & 0xFF) - 0xF0)))
		{
			return true ;
		}
	}
	return false ;
}

bool c_real_channel::pitch_slide_up()
{
	if (m_command_value == 0)
	{
		m_command_value = m_pitch_slide ;
	}
	else
	{
		m_pitch_slide = m_command_value ;
	}
			 
	if (m_command_value < 0xE0) /* pitch slide up by command_value */
	{
		m_effect_update = true ;
		set_pitch_slide_value(m_command_value);
	}
	else if (m_command_value < 0xF0) /* extra fine pitch slide up */
	{
		if (slide_pitch(((m_command_value & 0xFF) - 0xE0) / 4.0))
		{
			return true ;
		}
	}
	else /* fine pitch slide up */
	{
		if (slide_pitch((m_command_value & 0xFF) - 0xF0))
		{
			return true ;
		}
	}
	return false ;
}

void c_real_channel::set_sample_offset()
{
	p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
	unsigned long offset = (unsigned long)m_command_value ;
	offset *= 256 ;
	pvc->set_sample_offset(offset) ;
}

void c_real_channel::slide_panning(signed long slide_value)
{
	m_panning += slide_value ;

	if (m_panning < -32)
	{
		m_panning = -32 ;
	}
	else if (m_panning > 32)
	{
		m_panning = 32 ;
	}
}

void c_real_channel::slide_volume(signed long slide_value)
{
	if (m_playing)
	{
		m_mixer->get_virtual_channel(m_virtual_channel)->slide_note_volume(slide_value) ;
		m_note_volume = m_mixer->get_virtual_channel(m_virtual_channel)->get_note_volume() ;
	}	
}

void c_real_channel::set_volume_slide_value(signed long slide_value)
{
	m_volume_slide_value = slide_value ;
}

void c_real_channel::slide_channel_volume(signed long slide_value)
{
	m_volume += slide_value ;
	if (m_volume > 64)
	{
		m_volume = 64 ;
	}
	else if (m_volume < 0)
	{
		m_volume = 0 ;
	}	
}

void c_real_channel::set_channel_volume_slide_value(signed long slide_value)
{
	m_channel_volume_slide_value = slide_value ;
}

void c_real_channel::set_panning_slide_value(signed long slide_value)
{
	m_panning_slide_value = slide_value ;
}

void c_real_channel::set_virtual_channel(signed long virtual_channel)
{
	m_virtual_channel = virtual_channel ;
}

void c_real_channel::update_frame(bool frame_0)
{
	if (m_effect_update)
	{
		if (!frame_0)
		{ 
			// Effects that aren't updated on 'row' frames
			if (m_retrigger_on != 0)
			{
				if (apply_retrigger())
				{
					return ;
				}
			}

			if (m_arpeggio_on != 0)
			{
				if (apply_arpeggio())
				{
					return ;
				}
			}

			if (m_channel_volume_slide_value != 0)
			{
				slide_channel_volume(m_channel_volume_slide_value) ;
			}
			
			if (m_volume_slide_value != 0)
			{
				slide_volume(m_volume_slide_value) ;
			}
			
			if (m_panning_slide_value != 0)
			{
				slide_panning(m_panning_slide_value) ;
			}
			
			if (m_playing)
			{
				if (m_pitch_slide_speed != 0)
				{
					bool result = slide_pitch(m_pitch_slide_speed) ;
					if (m_pitch_slide_speed < 0)
					{
						if (result || (m_sample_note < m_pitch_slide_note))
						{
							m_sample_note = m_pitch_slide_note ;
							m_mixer->get_virtual_channel(m_virtual_channel)->set_pitch(m_sample_note) ;
							m_pitch_slide_speed = 0 ;
						}
					}
					else if (m_pitch_slide_speed > 0)
					{
						if (result || (m_sample_note > m_pitch_slide_note))
						{
							m_sample_note = m_pitch_slide_note ;
							m_mixer->get_virtual_channel(m_virtual_channel)->set_pitch(m_sample_note) ;
							m_pitch_slide_speed = 0 ;
						}
					}
				}
				
				if (m_pitch_slide_value != 0)
				{
					if (slide_pitch (m_pitch_slide_value))
					{
						m_mixer->drop_virtual_channel(m_virtual_channel) ;
						m_playing = false ;
					}
				}
			}
		}
	}

	if (m_playing)
	{
		m_mixer->get_virtual_channel(m_virtual_channel)->set_volume (m_volume, m_panning) ;
	}

	if (m_effect_update)
	{
		if (!(frame_0 && m_module->use_old_effects()))
		{
			// Effects that are updated every frames, or every non-row frames
			// depending on the old_effects flag
			if (m_vibrato_on)
			{
				apply_vibrato() ;
			}

			if (m_tremelo_on)
			{
				apply_tremelo() ;
			}
		}

		// Effects that are updated every frames
		if (m_tremor_on)
		{
			apply_tremor() ;
		}

		if (m_panbrello_on)
		{
			apply_panbrello() ;
		}
	}
}

bool c_real_channel::apply_arpeggio()
{ /*new_note(signed long instrument, signed long note, bool volume_envelope_on)*/
	if (m_playing)
	{
		m_arpeggio_counter++;
		if (m_arpeggio_counter > 2)
		{
			m_arpeggio_counter = 0 ;
		}

		p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;

		pvc->note_cut() ;
		pvc->set_in_background() ;
		
		m_virtual_channel = m_mixer->allocate_virtual_channel(m_channel) ;
		
		if (m_virtual_channel == -1)
		{
			m_playing = false ;
			return true ;
		}
		else
		{
			pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->set_channel_panning(m_panning) ;
			m_playing = true ;
		}
		
		if (m_arpeggio_counter & 2)
		{
			pvc->new_note(m_instrument, m_note + m_arpeggio_offset_2) ;				
		}
		else if (m_arpeggio_counter & 1)
		{
			pvc->new_note(m_instrument, m_note + m_arpeggio_offset_1) ;
		}
		else
		{
			pvc->new_note(m_instrument, m_note) ;
		}

		pvc->set_note_volume(m_note_volume) ;
		pvc->set_volume (m_volume, m_panning) ;
	}
	
	return false ;
}

bool c_real_channel::apply_retrigger()
{
	if (m_playing)
	{
		m_retrigger_counter++ ; 
		if (m_retrigger_counter == m_retrigger_delay)
		{
			m_retrigger_counter = 0 ;

			p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->note_cut() ;
			pvc->set_in_background() ;

			m_virtual_channel = m_mixer->allocate_virtual_channel(m_channel) ;
			
			if (m_virtual_channel == -1)
			{
				m_playing = false ;
				return true ;
			}
			else
			{
				pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
				pvc->set_channel_panning(m_panning) ;
				m_playing = true ;
			}

			pvc->new_note(m_instrument, m_note) ;
			pvc->set_note_volume(m_note_volume);
			pvc->slide_note_volume(m_retrigger_add) ;
			pvc->multiply_note_volume(m_retrigger_mul) ;
			pvc->set_volume (m_volume, m_panning) ;
			m_note_volume = pvc->get_note_volume() ;
		}
	}

	return false ;
}

void c_real_channel::apply_tremor()
{
	if (m_playing)
	{
		if (m_tremor_time >= m_tremor_ontime)
		{
			// Tremor offtime
			p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->set_second_volume(0) ; // set the volume to 0 for y frame
		}
		// Increments the tremor frame counter
		m_tremor_time = (m_tremor_time + 1) % (m_tremor_ontime + m_tremor_offtime) ;
	}
}

void c_real_channel::apply_vibrato()
{
	if (m_playing)
	{
		if ((m_vibrato_time == 0) || (m_vibrato_waveform != 3))
		{
			double value = get_waveform_value(m_vibrato_position, m_vibrato_waveform) ;
			p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->slide_second_pitch(value * m_vibrato_depth/256.0, m_module->use_linear_slides()) ;
		}

		m_vibrato_time ++ ;
		if (m_vibrato_time >= m_vibrato_speed)  ; // speed / 4 ???
		{
			m_vibrato_time = 0 ;
		}
		m_vibrato_position = (m_vibrato_position + m_vibrato_speed) & 0xFF ; // <=> % 256
	}
}

void c_real_channel::apply_tremelo()
{
	if (m_playing)
	{
		if ((m_tremelo_time == 0) || (m_tremelo_waveform != 3))
		{
			double value = get_waveform_value(m_tremelo_position, m_tremelo_waveform) ;
			p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->slide_second_volume(value * m_tremelo_depth / 32.0) ;
		}

		m_tremelo_time = (m_tremelo_time + 1) % m_tremelo_speed ; // speed / 4 ???
		m_tremelo_position = (m_tremelo_position + m_tremelo_speed) & 0xFF ; // <=> % 256
	}
}

void c_real_channel::apply_panbrello()
{
	if (m_playing)
	{
		if ((m_panbrello_time == 0) || (m_panbrello_waveform != 3))
		{
			double value = get_waveform_value(m_panbrello_position, m_panbrello_waveform) ;
			p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
			pvc->slide_second_panning(value * m_panbrello_depth / 32.0) ;
		}

		m_panbrello_time = (m_panbrello_time + 1) % m_panbrello_speed ; // speed / 4 ???
		m_panbrello_position = (m_panbrello_position + m_panbrello_speed) & 0xFF ; // <=> % 256
	}
}

bool c_real_channel::slide_pitch(double slide_value)
{
	p_virtual_channel pvc = m_mixer->get_virtual_channel(m_virtual_channel) ;
	if (m_module->use_linear_slides())
	{
		/* Linear slides */
		m_sample_note += slide_value / 16 ;

		if (m_sample_note <= 0)
		{
			return true ;
		}		
	}
	else
	{
		/* Amiga slides */
		double amiga_value = note_to_amiga (m_sample_note, m_sample) ;
		amiga_value -= (slide_value * m_module->get_sample(m_sample)->get_C5_speed() / 8363.42) ;

		if (amiga_value <= 0)
		{
			return true ;
		}

		m_sample_note = amiga_to_note (amiga_value, m_sample) ;
	}

	pvc->set_pitch(m_sample_note) ;

	return false ;
}

double c_real_channel::note_to_amiga(double note, signed long sample)
{
	p_sample psmp = m_module->get_sample(sample) ;

	return (m_amiga_const/note_to_speed(note)) ;  
}

double c_real_channel::amiga_to_note(double amiga_value, signed long sample)
{
	p_sample psmp = m_module->get_sample(m_sample) ;

	return speed_to_note (m_amiga_const/amiga_value) ;
}

double c_real_channel::note_to_speed(double note)
{
	/* note -> speed conversion */
	return pow(2, (note/12) - 5) ;
}

double c_real_channel::speed_to_note(double speed)
{
	/* speed -> note conversion */
	return 12 * (5 + log(speed) / m_log2_const) ;
}

void c_real_channel::set_pitch_slide_value(double slide_value)
{
	m_pitch_slide_value = slide_value ;
}

void c_real_channel::loop(signed long loop_param)
{
	/* loop effect handler : sB0 / sBX */
	if (loop_param == 0)
	{
		if (m_loop_begin != m_mixer->get_current_line())
		{
			m_loop_begin = m_mixer->get_current_line() ;
		}
	}
	else
	{
		if (m_loop_counter < loop_param)
		{
			m_loop_counter ++ ;
			m_mixer->loop_to_line(m_loop_begin) ;
		}
		else
		{
			m_loop_counter = 0 ;
		}
	}
}

void c_real_channel::clear_effects()
{	
	m_effect_update = false ;
	m_channel_volume_slide_value = 0 ;
	m_volume_slide_value = 0 ;
	m_panning_slide_value = 0 ;
	m_pitch_slide_value = 0 ;
	m_pitch_slide_speed = 0 ;
	m_vibrato_on = false ;
	m_tremelo_on = false ;
	m_panbrello_on = false ;
	m_tremor_on = false ;
	m_retrigger_on = false ;
	m_arpeggio_on = false ;
}

void c_real_channel::init()
{
	m_vibrato_waveform = 0 ;
	m_tremelo_waveform = 0 ;
	m_panbrello_waveform = 0 ;
	m_channel_volume_slide = 0 ;
	m_channel_volume_slide_value = 0 ;
	m_command = 0 ;
	m_command_value = 0 ;
	m_duplicate_check_action = 0 ;
	m_duplicate_check_type = 0 ;
	m_effect_update = false ;
	m_instrument = 0 ;
	m_mask_variable = 0 ;
	m_muted = m_default_muted_flag ;
	m_new_note_action = 0 ;
	m_note = 0 ;
	m_note_volume = 0 ;
	m_panning = m_default_panning ;
	m_panning_slide = 0 ;
	m_panning_slide_value = 0 ;
	m_pitch_slide = 0 ;
	m_pitch_slide_note = 0 ;
	m_pitch_slide_speed = 0 ;
	m_pitch_slide_value = 0 ;
	m_playing = false ;
	m_sample = 0 ;
	m_sample_note = 0 ;
	m_virtual_channel = -1 ;
	m_volume = m_default_volume ;	
	m_volume_panning = 0 ;
	m_volume_slide = 0 ;
	m_volume_slide_value = 0 ;
}

void c_real_channel::new_pattern()
{
	/* this function must be called each time a new pattern is started */
	m_loop_begin = -1 ;
	m_loop_counter = 0 ;
}

void c_real_channel::clear_playing()
{
	m_playing = false ;
}

bool c_real_channel::set_muted(bool muted)  
{
	bool previous_muted = m_muted ;
	m_muted = muted ;
	return m_muted ;
}

double c_real_channel::get_waveform_value(signed long index, signed long waveform)
{
	index &= 0xFF ;

	switch (waveform)
	{
	case 0:
		{
			return 64 * sin(index * m_sin_cst);
		}
	break;
	case 1:
		{
			return -((index << 8) / 510.0) + 64.0 ;
		}
	break;
	case 2:
		{
			return (index < 128)?64.0:0.0;
		}
	break;
	case 3:
		{
			const double factor = 64.0 / (RAND_MAX);
			return factor * rand();
		}
	break;
	default:
		{
			return 0 ;
		}
	break;
	}
}

const double c_real_channel::m_amiga_const = 428 ; // amiga constant
const double c_real_channel::m_log2_const = log(2) ;
const double c_real_channel::m_sin_cst = 3.14159265358979323 / 128.0 ;

#endif AFX_C_REAL_CHANNEL_INCLUDE_H__EB69AC61_2087_11D1_B35E_DCE971BF2962__INCLUDED_
