// c_instrument_inline.h: implementation of the c_instrument class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_INSTRUMENT_INLINE_H__2691C104_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_INSTRUMENT_INLINE_H__2691C104_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_instrument::c_instrument(signed long num)
{
	init_variables() ;
	
	for (signed long local_counter = 0 ; local_counter < 120 ; local_counter++)
	{
		m_note_sample_keyboard_table[local_counter] = new c_note_sample(local_counter, num + 1) ;
		
		if ((void *)m_note_sample_keyboard_table[local_counter] == NULL)
		{
			engine_error = "Not enough memory to allocate note/sample conversion table" ;
		}
		
		if ((bool)engine_error)
		{
			return ;
		}
	}
	
	m_volume_envelope = new c_envelope() ;
	if ((void *)m_volume_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate a default envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
	m_panning_envelope = new c_envelope ;
	if ((void *)m_panning_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate a default envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
	m_pitch_envelope = new c_envelope ;
	if ((void *)m_pitch_envelope  == NULL)
	{
		engine_error = "Not enough memory to allocate a default envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
}

c_instrument::c_instrument(FILE *module_file)
{
	init_variables() ;
	
	long starting_position ;
	
	// buffers
	signed char sc ;
	unsigned char uc ;
	unsigned short us ;
	
	starting_position = ftell (module_file) ;
	
	{
		/* Verification of the IT instrument marker 'IMPI' */
		
		char buffer [5] ;
		size_t return_value ;
		
		return_value = fread (buffer, (size_t) 1, (size_t) 4, module_file) ;
		
		if (return_value == (size_t) -1)
		{
			engine_error = "Cannot read the instrument" ;
			return ;
		}
		
		if (strncmp ("IMPI", buffer, 4) != 0)
		{
			engine_error = "Bad instrument marker" ;
			return ;
		}
	}    
	
	/* Now we can assume (at least for now) that we
	have a regular IT instrument */
	
	/* Reading of the instrument parameters */
	
	(void) fread (m_dos_file_name, (size_t) 1, (size_t) 12, module_file) ;
	/* instrument dos file name */
	
	(void) fseek (module_file, 1L, SEEK_CUR) ;
	/* jumping over the 1 byte padding */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_new_note_action = (signed long)uc ;
	/* new note action field */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_duplicate_check_type = (signed long)uc ;
	/* duplicate check type */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_duplicate_check_action = (signed long)uc ;
	/* duplicate check action */
	
	(void) fread (&(us), (size_t) 2, (size_t) 1, module_file) ;
	m_fadeout = (signed long)us ;
	/* fadeout */
	
	(void) fread (&(sc), (size_t) 1, (size_t) 1, module_file) ;
	m_pitch_pan_separation = (signed long)sc ;
	/* pitch-pan separation */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_pitch_pan_center = (signed long)uc ;
	/* pitch-pan center */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_global_volume = (signed long)uc ;
	/* global volume */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_default_panning = (signed long)uc ;
	/* default panning */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_random_volume_variation = (signed long)uc ;
	/* random volume variation */
	
	(void) fread (&(uc), (size_t) 1, (size_t) 1, module_file) ;
	m_random_panning_variation = (signed long)uc ;
	/* random panning variation */
	
	(void) fseek (module_file, 0x20L + starting_position, SEEK_SET) ;
	/* moving the file pointer after Tracker version and no of samples */
	
	(void) fread (m_name, (size_t) 1, (size_t) 26, module_file) ;
	/* Instrument real name */
	
	(void) fseek (module_file, 0x40L + starting_position, SEEK_SET) ;
	/* adjusting file pointer to the beginning of the note-sample / keyboard table */
	
	for (signed long note = 0 ; note <= 119 ; note ++)
	{
		m_note_sample_keyboard_table [note] = new c_note_sample(module_file) ;
		
		if ((void *)m_note_sample_keyboard_table[note] == NULL)
		{
			engine_error = "Not enough memory to allocate note/sample conversion table" ;
		}
		
		if ((bool)engine_error)
		{
			return ;
		}
	}
	
	
	m_volume_envelope = new c_envelope (module_file, false) ;
	if ((void *)m_volume_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
    /* adjusting file pointer on a word boundary */
    (void) fseek (module_file, 1L, SEEK_CUR) ;
	
	m_panning_envelope = new c_envelope (module_file, true) ;
	if ((void *)m_panning_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
    /* adjusting file pointer on a word boundary */
    (void) fseek (module_file, 1L, SEEK_CUR) ;
	
	m_pitch_envelope = new c_envelope (module_file, true) ;
	if ((void *)m_pitch_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
}

c_instrument::c_instrument(istream &is, int format)
{
	/* 'format' is ignored for now */

	init_variables() ;
	
	// buffers
	signed char sc ;
	unsigned char uc ;
	unsigned short us ;
	
	streampos starting_position = is.tellg() ;
	
	{
		/* Verification of the IT instrument marker 'IMPI' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("IMPI", buffer, 4) != 0)
		{
			engine_error = "Bad instrument marker" ;
			return ;
		}
	}    
	
	/* Now we can assume (at least for now) that we
	have a regular IT instrument */
	
	/* Reading of the instrument parameters */
	
	/* instrument dos file name */
	is.read(m_dos_file_name, 12);
	
	/* jumping over the 1 byte padding */
	is.seekg((streamoff)1, ios::cur) ;
	
	/* new note action field */
	is.read((char *)&uc, 1) ;
	m_new_note_action = (signed long)uc ;
	
	/* duplicate check type */
	is.read((char *)&uc, 1) ;
	m_duplicate_check_type = (signed long)uc ;
	
	/* duplicate check action */
	is.read((char *)&uc, 1) ;
	m_duplicate_check_action = (signed long)uc ;
	
	/* fadeout */
	is.read((char *)&us, 2) ;
	m_fadeout = (signed long)us ;
	
	/* pitch-pan separation */
	is.read((char *)&sc, 1) ;
	m_pitch_pan_separation = (signed long)sc ;
	
	/* pitch-pan center */
	is.read((char *)&uc, 1) ;
	m_pitch_pan_center = (signed long)uc ;
	
	/* global volume */
	is.read((char *)&uc, 1) ;
	m_global_volume = (signed long)uc ;
	
	/* default panning */
	is.read((char *)&uc, 1) ;
	m_default_panning = (signed long)uc ;
	
	/* random volume variation */
	is.read((char *)&uc, 1) ;
	m_random_volume_variation = (signed long)uc ;
	
	/* random panning variation */
	is.read((char *)&uc, 1) ;
	m_random_panning_variation = (signed long)uc ;
	
	/* moving the file pointer after Tracker version and no of samples */
	is.seekg(starting_position + (streamoff)0x20) ;
	
	/* Instrument real name */
	is.read(m_name, 26) ;
	
	/* adjusting file pointer to the beginning of the note-sample / keyboard table */
	is.seekg(starting_position + (streamoff)0x40) ;
	
	for (signed long note = 0 ; note <= 119 ; note ++)
	{
		m_note_sample_keyboard_table [note] = new c_note_sample(is) ;
		
		if ((void *)m_note_sample_keyboard_table[note] == NULL)
		{
			engine_error = "Not enough memory to allocate note/sample conversion table" ;
		}
		
		if ((bool)engine_error)
		{
			return ;
		}
	}
	
	
	m_volume_envelope = new c_envelope (is, 0) ;
	if ((void *)m_volume_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
    /* adjusting file pointer on a word boundary */
	is.seekg((streamoff)1, ios::cur) ;
	
	m_panning_envelope = new c_envelope (is, 1) ;
	if ((void *)m_panning_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
	
    /* adjusting file pointer on a word boundary */
	is.seekg((streamoff)1, ios::cur) ;
	
	m_pitch_envelope = new c_envelope (is, 2) ;
	if ((void *)m_pitch_envelope == NULL)
	{
		engine_error = "Not enough memory to allocate an envelope" ;
	}
	if ((bool)engine_error)
	{
		return ;
	}
}

c_instrument::~c_instrument()
{
	for (signed long local_counter = 0 ; local_counter < 120 ; local_counter++)
	{
		if ((void *)m_note_sample_keyboard_table[local_counter] != NULL)
		{
			delete m_note_sample_keyboard_table[local_counter] ;
		}
	}
	
	if ((void *)m_volume_envelope != NULL)
	{
		delete m_volume_envelope ;
	}
	if ((void *)m_panning_envelope != NULL)
	{
		delete m_panning_envelope ;
	}
	if ((void *)m_pitch_envelope != NULL)
	{
		delete m_pitch_envelope ;
	}
}

void c_instrument::init_variables()
{
	m_name[0] = 0 ;
	m_dos_file_name[0] = 0 ;
	m_new_note_action = 0 ;
	m_duplicate_check_type = 0 ;
	m_duplicate_check_action = 0 ;
	m_fadeout = 0 ;
	m_pitch_pan_separation = 0 ;
	m_pitch_pan_center = 60 ;
	m_global_volume = 128 ;
	m_default_panning = 128 ;
	m_random_volume_variation = 0 ;
	m_random_panning_variation = 0 ;
	
	for (signed long local_counter = 0 ; local_counter < 120 ; local_counter++)
	{
		m_note_sample_keyboard_table[local_counter] = (p_note_sample)NULL ;
	}
	
	m_volume_envelope = (p_envelope)NULL ;
	m_panning_envelope = (p_envelope)NULL ;
	m_pitch_envelope = (p_envelope)NULL ;
}


const char *c_instrument::get_name()
{
	return m_name ;
}

const char *c_instrument::get_dos_name()
{
	return m_dos_file_name ;
}

signed long c_instrument::get_new_note_action()
{
	return m_new_note_action ; 
}

signed long c_instrument::get_duplicate_check_type()
{
	return m_duplicate_check_type ;
}

signed long c_instrument::get_duplicate_check_action()
{
	return m_duplicate_check_action ;
}

signed long c_instrument::get_fadeout()
{
	return m_fadeout ;
}

signed long c_instrument::get_pitch_panning_separation()
{
	return m_pitch_pan_separation ;
}

signed long c_instrument::get_pitch_panning_center()
{
	return m_pitch_pan_center ;
}

signed long c_instrument::get_volume()
{
	return m_global_volume ;
}

signed long c_instrument::get_panning()
{
	return m_default_panning ;
}

signed long c_instrument::get_random_volume_variation()
{
	return m_random_volume_variation ;
}

signed long c_instrument::get_random_panning_variation()
{
	return m_random_panning_variation ;
}

p_note_sample c_instrument::get_note_sample_record(signed long note)
{
	if ((note >= 0) && (note <= 119))
	{
		return m_note_sample_keyboard_table[note] ;
	}
	else
	{
		return (p_note_sample)NULL ;
	}
}

signed long c_instrument::get_note(signed long note)
{
	if ((note >= 0) && (note <= 119))
	{
		return m_note_sample_keyboard_table[note]->get_note() ;
	}
	else
	{
		return -1 ;
	}
}

signed long c_instrument::get_sample(signed long note)
{
	if ((note >= 0) && (note <= 119))
	{
		return m_note_sample_keyboard_table[note]->get_sample() ;
	}
	else
	{
		return -1 ;
	}
}

p_envelope c_instrument::get_volume_envelope()
{
	return m_volume_envelope ;
}

p_envelope c_instrument::get_panning_envelope()
{
	return m_panning_envelope ;
}

p_envelope c_instrument::get_pitch_envelope()
{
	return m_pitch_envelope ;
}

#endif // AFX_C_INSTRUMENT_INLINE_H__2691C104_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_
