/* win32drv.c */
/*
     PLAY_ITW.EXE v0.02b : Player for Impulse Tracker modules files
     Copyright (C) 1997  Olivier AUMAGE
     E-mail : Olivier.Aumage@ens-lyon.fr
     Web : http://www.ens-lyon.fr/~oaumage/

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* main header files */
#include <windows.h>
#include <winbase.h>
#include <mmsystem.h>
#include <stdlib.h>
#include <stdio.h>

/* project header files */
#include "win32drv.h"
#include "buffer.h"

static void error (char *error_message) ;
static void CALLBACK WaveProc (HWAVE hWave,
                               UINT uMsg,
                               DWORD dwInstance,
                               DWORD dwParam1,
                               DWORD dwParam2
                              ) ;

static DWORD WINAPI thread_start_routine (LPVOID) ;
static DWORD thread_id ;
static HANDLE thread_handle ;
static SECURITY_ATTRIBUTES security_struct ;
static HWAVEOUT output_device_handler ;
static CRITICAL_SECTION critical_section_object ;

static unsigned short sampling_rate ;
static unsigned short number_of_buffers ;
static WAVEHDR sound_buffers[255] ; /* max number of sound buffers is 255 for now */
static volatile unsigned short number_of_ready_buffers ;
static volatile unsigned short first_buffer ;
static unsigned short last_buffer ;
static unsigned long length_of_a_sound_buffer ; /* in samples (or pair of samples for stereo) */

static idle_priority_class ;
static normal_priority_class ;
static calculation_priority_class ;
static thread_priority ;
static stop_requested ;


static void (*fill_mixing_buffer_function) (void) ;

/* error handler */
static void error (char *error_message)
{
  (void)fprintf (stderr, "\nError : %s\n", error_message) ;
  exit (EXIT_FAILURE) ;
}

DWORD WINAPI thread_start_routine (LPVOID lpthreadparameter)
{
  for (;;)
  {
    if (stop_requested)
    {
       ExitThread(0) ;
    }
    if (number_of_ready_buffers < (number_of_buffers - 1))
    {
      MMRESULT result ;

      last_buffer ++ ;
      last_buffer %= number_of_buffers ;
      mixing_buffer = (signed short*) sound_buffers[last_buffer].lpData ;
      SetPriorityClass (GetCurrentProcess(), calculation_priority_class) ;
      fill_mixing_buffer_function () ;
      SetPriorityClass (GetCurrentProcess(), normal_priority_class) ;
      sound_buffers[last_buffer].dwBufferLength = length_of_a_sound_buffer << 2 ;
      sound_buffers[last_buffer].dwFlags = 0 ;
      sound_buffers[last_buffer].dwLoops = 0 ;
      result = waveOutPrepareHeader (output_device_handler, &sound_buffers[last_buffer], sizeof (WAVEHDR)) ;
      if (result != MMSYSERR_NOERROR)
      {
        printf ("\nwaveOutPrepareHeader : Error\n") ;
        switch (result)
        {
          case MMSYSERR_INVALHANDLE :
            printf ("Invalid handle\n") ;
            break ;
          case MMSYSERR_NODRIVER :
            printf ("No driver present\n") ;
            break ;
          case MMSYSERR_NOMEM :
            printf ("Unable to allocate memory\n") ;
            break ;
          case WAVERR_UNPREPARED :
            printf ("Unprepared block\n") ;
            break ;
          default :
            printf ("Unknown error\n") ;
        }
        waveOutClose (output_device_handler) ;
        exit (1) ;
      }
      sound_buffers[last_buffer].dwLoops = 0 ;
      result = waveOutWrite (output_device_handler, &sound_buffers[last_buffer], sizeof (WAVEHDR)) ;
      if (result != MMSYSERR_NOERROR)
      {
        printf ("\nwaveOutWrite : Error\n") ;
        switch (result)
        {
          case MMSYSERR_INVALHANDLE :
            printf ("Invalid handle\n") ;
            break ;
          case MMSYSERR_NODRIVER :
            printf ("No driver present\n") ;
            break ;
          case MMSYSERR_NOMEM :
            printf ("Unable to allocate memory\n") ;
            break ;
          default :
            printf ("Unknown error\n") ;
        }
        waveOutClose (output_device_handler) ;
        exit (1) ;
      }
      number_of_ready_buffers++ ;
      if (end_of_song)
      {
        printf ("\tEnd of mixing.\015") ;
        ExitThread (0) ;
      }
      else
      {
        printf ("\t%3d (%3d)\015", (int)(number_of_ready_buffers), (int)number_of_buffers) ; fflush (stdout) ;
      }
    }
    else
    {
      SetPriorityClass (GetCurrentProcess(), idle_priority_class) ;
    }
  }
  return (DWORD) 0 ; /* dummy return */
}

void CALLBACK WaveProc (HWAVE hWave,
                        UINT uMsg,
                        DWORD dwInstance,
                        DWORD dwParam1,
                        DWORD dwParam2
                       )
{
  EnterCriticalSection(&critical_section_object) ;
  if (uMsg != WOM_DONE)
  {
    return ;
  }
  waveOutUnprepareHeader (output_device_handler, &sound_buffers[first_buffer], sizeof (WAVEHDR)) ;
  number_of_ready_buffers -- ;
  first_buffer ++ ;
  first_buffer %= number_of_buffers ;

  LeaveCriticalSection(&critical_section_object) ;
  if ((number_of_ready_buffers == 0) && (end_of_song))
  {
    end_of_song = 2 ;
  }
}

void init_buffers (void)
{
  unsigned short buffer_counter ;
  for (buffer_counter = 0 ; buffer_counter < number_of_buffers ; buffer_counter++)
  {
    MMRESULT result ;

    mixing_buffer = (signed short*) sound_buffers[buffer_counter].lpData ;
    SetPriorityClass (GetCurrentProcess(), calculation_priority_class) ;
    fill_mixing_buffer_function () ;
    printf ("\t%d/%d\015", buffer_counter, number_of_buffers) ; fflush (stdout) ;
    SetPriorityClass (GetCurrentProcess(), normal_priority_class) ;
    sound_buffers[buffer_counter].dwBufferLength = length_of_a_sound_buffer << 2 ;
    sound_buffers[buffer_counter].dwFlags = 0 ;
    sound_buffers[buffer_counter].dwLoops = 0 ;
    result = waveOutPrepareHeader (output_device_handler, &sound_buffers[buffer_counter], sizeof (WAVEHDR)) ;
    if (result != MMSYSERR_NOERROR)
    {
      printf ("\nwaveOutPrepareHeader : Error\n") ;
      switch (result)
      {
        case MMSYSERR_INVALHANDLE :
          printf ("Invalid handle\n") ;
          break ;
        case MMSYSERR_NODRIVER :
          printf ("No driver present\n") ;
          break ;
        case MMSYSERR_NOMEM :
          printf ("Unable to allocate memory\n") ;
          break ;
        case WAVERR_UNPREPARED :
          printf ("Unprepared block\n") ;
          break ;
        default :
          printf ("Unknown error\n") ;
      }
      waveOutClose (output_device_handler) ;
      exit (1) ;
    }
    sound_buffers[buffer_counter].dwLoops = 0 ;
    if (end_of_song)
    {
      printf ("\tEnd of mixing.\015") ;
      buffer_counter ++ ;
      break ;
    }
  }
  first_buffer = 0 ;
  number_of_ready_buffers = buffer_counter ;
  last_buffer = buffer_counter - 1 ;
  for (buffer_counter = 0 ; buffer_counter < number_of_ready_buffers ; buffer_counter++)
  {
    MMRESULT result ;
    result = waveOutWrite (output_device_handler, &sound_buffers[buffer_counter], sizeof (WAVEHDR)) ;
    if (result != MMSYSERR_NOERROR)
    {
      printf ("\nwaveOutWrite : Error\n") ;
      switch (result)
      {
        case MMSYSERR_INVALHANDLE :
          printf ("Invalid handle\n") ;
          break ;
        case MMSYSERR_NODRIVER :
          printf ("No driver present\n") ;
          break ;
        case MMSYSERR_NOMEM :
          printf ("Unable to allocate memory\n") ;
          break ;
        default :
          printf ("Unknown error\n") ;
      }
      waveOutClose (output_device_handler) ;
      exit (1) ;
    }
  }
}

void win32drv_start (void)
{
  WAVEFORMATEX wave_format_extended_structure ;
  MMRESULT result ;


  security_struct.nLength = sizeof (SECURITY_ATTRIBUTES) ;
  security_struct.lpSecurityDescriptor = NULL ;
  security_struct.bInheritHandle = TRUE ;
  thread_handle = CreateThread (
                                 &security_struct,
                						0,
                 						thread_start_routine,
                 						0,
                 						CREATE_SUSPENDED,
                 						&thread_id
                               );
  SetThreadPriority(thread_handle, thread_priority) ;

  wave_format_extended_structure.wFormatTag = WAVE_FORMAT_PCM ;
  wave_format_extended_structure.nChannels = 2 ;
  wave_format_extended_structure.nSamplesPerSec = (unsigned long) sampling_rate ;
  wave_format_extended_structure.nAvgBytesPerSec = 4UL * sampling_rate ;
  wave_format_extended_structure.nBlockAlign = 4 ;
  wave_format_extended_structure.wBitsPerSample = 16 ;
  wave_format_extended_structure.cbSize = 0 ;

  result = waveOutOpen (&output_device_handler,
               WAVE_MAPPER,
               &wave_format_extended_structure,
               (unsigned long)WaveProc,
               0,
               CALLBACK_FUNCTION
              );

  if (result != MMSYSERR_NOERROR)
  {
    printf ("\nwaveOutOpen : ") ;
    switch (result)
    {
      case MMSYSERR_ALLOCATED :
        printf ("Ressource already allocated\n") ;
        break ;
      case MMSYSERR_BADDEVICEID :
        printf ("Device Identifier out of range\n") ;
        break ;
      case MMSYSERR_NODRIVER :
        printf ("No driver present\n") ;
        break ;
      case MMSYSERR_NOMEM :
        printf ("Unable to allocate memory\n") ;
        break ;
      case WAVERR_BADFORMAT :
        printf ("Unsupported waveform-audio format\n") ;
        break ;
      case WAVERR_SYNC :
        printf ("Synchronous device") ;
        break ;
      default :
        printf ("Unknown error\n") ;
    }
    exit (1) ;
  }

  init_buffers () ;
  stop_requested = 0 ;
  ResumeThread(thread_handle) ;

}

void win32drv_stop (void)
{
  stop_requested = 1 ;
  waveOutClose (output_device_handler) ;
  if (end_of_song == 0)
  {
    TerminateThread (thread_handle, 0) ;
  }
}


void win32drv_init (unsigned short sp_rate, unsigned char nb_buffers, unsigned char priority_boost, void (*mix_function)(void))
{
  printf ("\tWIN32DRV\n") ;
  sampling_rate = sp_rate ;
  fill_mixing_buffer_function = mix_function ;

  length_of_a_sound_buffer = mixing_buffer_length ;

  number_of_buffers = nb_buffers ;

  {
    unsigned char buffer_counter ;

    for (buffer_counter = 0 ; buffer_counter < number_of_buffers ; buffer_counter++)
    {
      if ((sound_buffers[buffer_counter].lpData = GlobalAlloc(GMEM_FIXED, length_of_a_sound_buffer << 2)) == 0)
      {
        printf ("Not Enough memory\n") ;
        exit (1) ;
      }
    }
  }


  InitializeCriticalSection(&critical_section_object) ;

  if (priority_boost == 2)
  {
    idle_priority_class = NORMAL_PRIORITY_CLASS ;
    normal_priority_class = HIGH_PRIORITY_CLASS ;
    calculation_priority_class = REALTIME_PRIORITY_CLASS ;
    thread_priority = THREAD_PRIORITY_HIGHEST ;
  }
  else if (priority_boost == 1)
  {
    idle_priority_class = IDLE_PRIORITY_CLASS ;
    normal_priority_class = NORMAL_PRIORITY_CLASS ;
    calculation_priority_class = HIGH_PRIORITY_CLASS ;
    thread_priority = THREAD_PRIORITY_NORMAL ;
  }
  else
  {
    idle_priority_class = IDLE_PRIORITY_CLASS ;
    normal_priority_class = NORMAL_PRIORITY_CLASS ;
    calculation_priority_class = NORMAL_PRIORITY_CLASS ;
    thread_priority = THREAD_PRIORITY_NORMAL ;
  }
}

