#include <stdlib.h>
#include <stdio.h>
#include <i86.h>
#include <dos.h>
#include "svga.h"
#include "draw.h"
#include "timer.h"
#include "lzoconf.h"
#include "lzo1c.h"

#define LZA_Pixels 1
#define LZA_Color  2
#define Block_LZO  1
#define Block_Skip 2

struct LZAHead {
                 char signature[6];
                 char version;
                 unsigned short frames;
                 unsigned short width, height;
                 int speed;
               };
struct FrameHead {
                   char frametp;
                   unsigned short blocks;
                 };
struct PixelHead1 {
                    unsigned short skip;
                    int blocksize;
                  };
struct PixelHead2 {
                    unsigned short skip;
                    char blocktp;
                    int blocksize;
                  };

FILE *lzafile;
struct LZAHead head;
struct FrameHead fhead;
struct PixelHead1 phead1;
struct PixelHead2 phead2;
int grmode;
int scrwidth, scrheight;
int fullscr;
lzo_byte *blockdata;
char *virscreen;
int playspeed;
int startedtimer;

void CopyMemArea(char *src, char *dest, unsigned int size);
#pragma aux CopyMemArea = \
"          shr ecx, 1" \
"          jnc shift2" \
"          movsb" \
"shift2:   shr ecx, 1" \
"          jnc copyloop" \
"          movsw" \
"copyloop: rep movsd" \
parm [esi] [edi] [ecx];

void Error(char *e)
{
  if (startedtimer > 0) 
    RestoreTimer();
  TextMode();
  printf ("%s",e);
  exit(EXIT_FAILURE);
}

void InitLZA(char *fn)
{
 char outerr[255];

  lzafile = fopen(fn, "rb");
  strcpy(outerr,"File not found "); strcat(outerr,fn);
  if (lzafile == NULL) 
    Error(outerr);
  fread(&head, sizeof(struct LZAHead), 1, lzafile);
  if ((head.signature[0] != 'L') || (head.signature[1] != 'Z') ||
      (head.signature[2] != 'A') || (head.signature[3] != 'N') ||
      (head.signature[4] != 'I') || (head.signature[5] != 'M'))
    Error("Not a LZANIM file");
  if ((head.version != 1) && (head.version != 2))
    Error("Unknown LZA version. This player only supports LZA version 1 and 2");
}

void SetScreenMode()
{
  if (grmode == 0)
    if ((head.width <= 320) && (head.height <= 200)) {
      grmode = grmode_320x200;
      scrwidth = 320; scrheight = 200;
    } else if ((head.width <= 640) && (head.height <= 400)) {
      grmode = grmode_640x400;
      scrwidth = 640; scrheight = 400;
    } else if ((head.width <= 640) && (head.height <= 480)) {
      grmode = grmode_640x480;
      scrwidth = 640; scrheight = 480;
    } else if ((head.width <= 800) && (head.height <= 600)) {
      grmode = grmode_800x600;
      scrwidth = 800; scrheight = 600;
    } else if ((head.width <=1024) && (head.height <= 768)) {
      grmode = grmode_1024x768;
      scrwidth = 1024; scrheight = 768;
    } else 
      Error("Unsupported resolution. Only up to 1024x768 supported");
  InitGraphics(grmode);
  fullscr = ((scrwidth == head.width) && (scrheight == head.height));
}

int ReadNumber(char *s)
{
  int total;

  total = 0;
  s += 2;
  while (*s != '\0') {
    total = total * 10;
    total = total + *s++ - '0';
  }

  return total;
}

void ReadParameters(int argc, char *argv[])
{ /* Read and parse cmd-line parameters passed to program */
  int a;
  char out[255],curtxt[255];

  if (argc < 2) {
    TextMode();
    puts("LZA animation player V2.01");
    puts("Programming: Brian Jensen / Purple");
    puts("");
    puts("Usage: PLAYLZA <LZA-FILENAME> {options}");
    puts("");
    puts("Options: -S<x> Set delay between frames to x (1-255)");
    puts("         -F<x> Force graphics mode to x:");
    puts("               1 = MCGA 320x200x256");
    puts("               2 = SVGA 640x400x256  (requires VESA 2.0)");
    puts("               3 = SVGA 640x480x256  (requires VESA 2.0)");
    puts("               4 = SVGA 800x600x256  (requires VESA 2.0)");
    puts("               5 = SVGA 1024x768x256 (requires VESA 2.0)");
    exit(EXIT_SUCCESS);
  }
  InitLZA(argv[1]);
  playspeed = head.speed;

  grmode = 0;

  for (a = 2; a < argc; a++) {
    strcpy(curtxt,argv[a]);
    if ((curtxt[0] == '-') && ((curtxt[1] == 'S') || (curtxt[1] == 's'))) {
      playspeed = ReadNumber(curtxt);
      if (playspeed <= 0)
	playspeed = 1;
      if (playspeed > 255) Error("Maximum delay is 255");
    } else if ((curtxt[0] == '-') && ((curtxt[1] == 'F') || (curtxt[1] == 'f'))) {
      if ((curtxt[2] <= '0') || (curtxt[2] > '5'))
	Error("Graphics mode not supported");
      switch (curtxt[2] - '0') {
	case 1: grmode = grmode_320x200; scrwidth = 320; scrheight = 200; break;
	case 2: grmode = grmode_640x400; scrwidth = 640; scrheight = 400; break;
	case 3: grmode = grmode_640x480; scrwidth = 640; scrheight = 480; break;
	case 4: grmode = grmode_800x600; scrwidth = 800; scrheight = 600; break;
	case 5: grmode = grmode_1024x768; scrwidth = 1024; scrheight = 768; break;
      }
      if ((scrwidth < head.width) || (scrheight < head.height))
	Error("Selected resolution too small");
    } else {
      strcpy(out,"Unknown parameter: ");
      strcat(out,curtxt);
      Error(out);
    }
  }
}

void ReadPalette()
{
  char curpal[256][3];
  int t;

  fread(curpal, 768, 1, lzafile);
  for (t = 0; t <= 255; t++)
    setcolor(t,curpal[t][0],curpal[t][1],curpal[t][2]);
}

void ReadBlocks(int blocks)
{
  int      t;
  lzo_byte *winpos;
  int      r;
  lzo_uint outsize;
  int      datasize;
  char     *data;
  int      count;

  if (fullscr)
    winpos = (lzo_byte *) usedmode.linwindow;
  else winpos = (lzo_byte *) virscreen;

  for (t = 1; t <= blocks; t++) {
    if (head.version == 1) {
      fread(&phead1, sizeof(struct PixelHead1), 1, lzafile);
      winpos += head.width * phead1.skip;
      if (phead1.blocksize > 0) {
	fread(blockdata, phead1.blocksize, 1, lzafile);
	r = lzo1c_decompress(blockdata, phead1.blocksize, winpos, &outsize, NULL);
	winpos += outsize;
      }
    } else { /* Version 2 */
      fread(&phead2, sizeof(struct PixelHead2), 1, lzafile);
      winpos += head.width * phead2.skip;
      if (phead2.blocksize > 0) {
	fread(blockdata, phead2.blocksize, 1, lzafile);
	if (phead2.blocktp == Block_LZO) {
	  r = lzo1c_decompress(blockdata, phead2.blocksize, winpos, &outsize, NULL);
	  winpos += outsize;
	} else if (phead2.blocktp == Block_Skip) { /* Skip block */
	  datasize = phead2.blocksize;
	  data = (char *) blockdata;
	  while (datasize > 0) {
	    winpos += *data++;
	    count = *data++; 
	    datasize -= 2 + count;
	    for ( ; count > 0; count--)
	      *winpos++ = *data++;
	  }
	} else
	  Error("Unknown block type.");
      }
    }
  }
}

void ClearVirtual()
{
  char *dest;
  int t;

  dest = virscreen;
  for (t = head.height * head.width; t > 0; t--)
    *dest++ = 0;
}

void CopyVirtual()
{
  char *src, *dest;
  unsigned int x, y;
  unsigned int pushx;

  src = virscreen;
  dest = (char *) usedmode.linwindow;

  if (head.height < scrheight)
    dest += scrwidth * ((scrheight - head.height) / 2);
  pushx = (scrwidth - head.width) / 2;

  dest += pushx;
  for (y = head.height; y > 0; y--) {
    CopyMemArea(src, dest, head.width);
    dest += scrwidth;
    src += head.width;
  }
}

void ReadFrames()
{
  int t;

  for (t = 1; t <= head.frames; t++) {
    fread(&fhead, sizeof(struct FrameHead), 1, lzafile);
    if (fhead.frametp == LZA_Pixels)
      ReadBlocks(fhead.blocks);
    else if (fhead.frametp == LZA_Color)
      ReadPalette();
    else Error("Unknown frame type");
    /* If using virtual screen, copy that to actual screen: */
    if (kbhit()) break;
    if (fhead.frametp == LZA_Pixels)
      Waitfor(playspeed);
    if (!fullscr) CopyVirtual();
    if (kbhit()) break;
    StartTimer();
  }
}

void main(int argc, char *argv[])
{
  startedtimer = 0;
  ReadParameters(argc,argv);
  SetScreenMode();
  if (!fullscr) {
    virscreen = (char *) malloc(head.width * head.height);
    ClearVirtual();
  }
  blockdata = (lzo_byte *) malloc(head.width * head.height * 2);
  if (!usedmode.linear) 
    Error("Linear frame buffer not available. Install a VESA 2.0 driver");
  InitTimer(); startedtimer = 1;
  StartTimer();
  do {
    ReadFrames();
    fseek(lzafile, sizeof(struct LZAHead), SEEK_SET);
  } while (!kbhit());
  free(virscreen);
  free(blockdata);
  TextMode();
  RestoreTimer();
}
