{
  Ejemplo de las rutinas bsicas de 3D:

             - Generacin de un objeto tridimensional
             - Proyeccin a dos dimensiones
             - Rotacin y traslacin
             - Deteccin y eliminacin de caras invisibles
             - Rellenado de polgonos


 Codificado por FAC, para el tutorial de grficos #0
}

program Base3D_2;        { versin 0.2 }

uses Mode13, FMath, Crt;
{              |
               |----- Se incluye la unidad de nmeros de punto fijo
}

const MaxVertices = 8;  { Nmero mximo de vrtices (8 para un cubo) }
      MaxCaras = 6;     { Nmero mximo de caras (6 para un cubo) }
      MaxVerticesCara = 4; { Nmero mximo de vrtices en cada cara }


{ Definicin de tipos }

     { Tipo Vrtice }
type TVertice = record
                x, y, z : Fixed;  { Coordenadas del vrtice }
                x2d, y2d : integer; { Coordenadas de la proyeccin en 2D }
                end;

     { Tipo cara (polgono) }
     TCara = record
             NVertices : byte;  { Nmero de vrtices en el polgono }
             Vertice : array[1..MaxVerticesCara] of integer;
                       { Indices correspondientes a cada vrtice }
             normal : TVertice;  { vector normal al plano de la cara }
             color : byte;       { color de la cara }
             end;

     { Tipo Objeto Tridimensional }
     TObjeto3D = record
                 NVertices : integer;  { Nmero de vrtices }
                 NCaras : integer;     { Nmero de caras }
                 ox, oy, oz : integer; { Coordenadas del origen del objeto }
                 Vertice : array[1..MaxVertices] of TVertice; { Vrtices }
                 Cara : array[1..Maxcaras] of TCara;          { Caras }
                 end;



var Seno, Coseno : array[0..359] of Fixed;  { Tablas trigonomtricas }

    Cubo : TObjeto3D;  { Objeto tridimensional }
    pal : TPalette;    { paleta utilizada }



{ El siguiente procedimiento genera las tablas trigonomtricas en formato
  de punto fijo 16.16 }
procedure GeneraTablas;
var a : integer;
begin
     for a := 0 to 359 do
     begin
          Seno[a] := Real2Fixed(sin(a * Pi / 180.0));
          Coseno[a] := Real2Fixed(cos(a * Pi / 180.0));
     end;
end;


{ Procedimientos de los vrtices }

{ Este procedimiento calcula la proyeccin a 2D de un vrtice }
procedure VerticeCalcula2D(var v : TVertice; Zdist : integer);
var p : Fixed;
begin
     { Zdist es la coordenada Z del origen del objeto tridimensional }
     p := Int2Fixed(1024 - Zdist) - v.z;
     v.x2d := Fixed2Int(FDiv(v.x, p) shl 9) + 160;
     v.y2d := Fixed2Int(FDiv(v.y, -p) shl 9) + 100;
     { shl 9 equivale a multiplicar por 512 (perspectiva) }
end;

{ Este procedimiento rota un vrtice sobre cualquier eje }
procedure VerticeRota(var v : TVertice; ax, ay, az : integer);
var nx, ny, nz : Fixed;
begin
     { Aseguramos que los ngulos sean positivos (entre 0 y 359) }
     if ax < 0 then inc(ax, 360);
     if ay < 0 then inc(ay, 360);
     if az < 0 then inc(az, 360);

     { Rotacin sobre el eje X }
     if ax <> 0 then
     begin
          ny := FMul(v.y, Coseno[ax]) - FMul(v.z, Seno[ax]);
          nz := FMul(v.y, Seno[ax]) + FMul(v.z, Coseno[ax]);
          v.y := ny;
          v.z := nz;
     end;

     { Rotacin sobre el eje Y }
     if ay <> 0 then
     begin
          nx := FMul(v.x, Coseno[ay]) - FMul(v.z, Seno[ay]);
          nz := FMul(v.x, Seno[ay]) + FMul(v.z, Coseno[ay]);
          v.x := nx;
          v.z := nz;
     end;

     { Rotacin sobre el eje Z }
     if az <> 0 then
     begin
          nx := FMul(v.x, Coseno[az]) - FMul(v.y, Seno[az]);
          ny := FMul(v.x, Seno[az]) + FMul(v.y, Coseno[az]);
          v.x := nx;
          v.y := ny;
     end;
end;

{ Este procedimiento normaliza un vector }
procedure VerticeNormaliza(var v : TVertice);
var mag : Fixed; { magnitud }
begin
     mag := FSqrt(FMul(v.x, v.x) + FMul(v.y, v.y) + FMul(v.z, v.z));
     if mag = 0 then exit;
     v.x := FDiv(v.x, mag);
     v.y := FDiv(v.y, mag);
     v.z := FDiv(v.z, mag);
end;


{ Procedimientos de las caras }

{ Este procedimiento resetea las variables principales de una cara }
procedure CaraReinicia(var cara : TCara);
begin
     cara.NVertices := 0;
end;

{ Este procedimiento aade un vrtice (su ndice) a una cara }
{ IMPORTANTE!!! Los vrtices se deben aadir en el sentido OPUESTO
                a las manecillas del reloj.                       }
procedure CaraAgregaVertice(var cara : TCara; vertice : integer);
begin
     { Primero comprobamos si podemos aadir otro vrtice }
     if cara.NVertices = MaxVerticesCara then exit;
     { Si es as, lo aadimos }
     inc(cara.NVertices);
     cara.Vertice[cara.NVertices] := vertice;
end;


function CaraVisible(cara : TCara) : boolean;
begin
     if cara.normal.z >= 0 then CaraVisible := true
                           else CaraVisible := false;
end;


procedure CaraDibuja(obj : TObjeto3D; cara : TCara; where : word);
var Xmin, Xmax : array[0..199] of integer;
    i, x1, y1, x2, y2 : integer;
    Ymin, Ymax, y : integer;

    procedure ScanSide(x1, y1, x2, y2 : integer);
    var y, ydiv, temp : integer;
        x, xinc : Fixed;
    begin
         if y1 > y2 then
         begin
              { nos aseguramos de que (x1, y1) sea el punto SUPERIOR
                y (x2, y2) sea el punto INFERIOR }
              temp := y1; y1 := y2; y2 := temp;
              temp := x1; x1 := x2; x2 := temp;
         end;
         ydiv := y2 - y1;
         if ydiv = 0 then exit;
         xinc := Int2Fixed(x2 - x1) div ydiv; { obtenemos la pendiente }
         x := Int2Fixed(x1);
         for y := y1 to y2 do
         begin
              if Fixed2Int(x) < Xmin[y] then Xmin[y] := Fixed2Int(x);
              if RoundFixed2Int(x) > Xmax[y] then Xmax[y] := RoundFixed2Int(x);
              inc(x, xinc);
         end;
    end;

begin
     { Lo primero es obtener las coordenadas Y mxima y mnima del
       polgono }
     Ymin := 32000;
     Ymax := -32000;
     for i := 1 to cara.NVertices do
     begin
          y := obj.Vertice[cara.Vertice[i]].y2d;
          if y < Ymin then Ymin := y;
          if y > Ymax then Ymax := y;
     end;

     { Si el polgono tiene altura igual a cero, entonces no lo dibujamos }
     if (Ymax - Ymin) = 0 then exit;

     for i := 0 to 199 do       { Inicializamos Xmin y Xmax }
     begin
          Xmin[i] := 32000;
          Xmax[i] := -32000;
     end;

     { Obtenemos las coordenadas del primer vrtice }
     x1 := obj.Vertice[cara.Vertice[1]].x2d;
     y1 := obj.Vertice[cara.Vertice[1]].y2d;

     { y hacemos un ciclo para dibujar todas las aristas }
     for i := 2 to cara.NVertices do
     begin
          { Obtenemos las coordenadas del siguiente vrtice }
          x2 := obj.Vertice[cara.Vertice[i]].x2d;
          y2 := obj.Vertice[cara.Vertice[i]].y2d;
          { "Escaneamos" el lado (x1, y1)-(x2, y2) }
          ScanSide(x1, y1, x2, y2);
          { y continuamos con el siguiente lado }
          x1 := x2;
          y1 := y2;
     end;

     { Obtenemos nuevamente las coordenadas del primer vrtice para
       escanear la lnea entre el primero y el ltimo vrtices }
     x2 := obj.Vertice[cara.Vertice[1]].x2d;
     y2 := obj.Vertice[cara.Vertice[1]].y2d;
     ScanSide(x1, y1, x2, y2);

     { En este momento, ya tenemos la informacin necesaria almacenada
       en Xmin y Xmax. Ahora procedemos a dibujar el polgono }
     for i := Ymin to Ymax do
         HLine(Xmin[i], Xmax[i], i, cara.color, where);
end;



{ Procedimientos de los objetos 3D }

{ Este procedimiento reinicia las variables importantes de un objeto 3D }
procedure Objeto3DReinicia(var obj : TObjeto3D);
begin
     obj.NVertices := 0;
     obj.NCaras := 0;
     obj.ox := 0;
     obj.oy := 0;
     obj.oz := 0;
end;


{ Este procedimiento agrega un vrtice al objeto tridimensional }
{ (no importa el orden, pero hay que recordar el ndice de cada vrtice
   a la hora de especificar los polgonos) }
procedure Objeto3DAgregaVertice(var obj : TObjeto3D; nx, ny, nz : integer);
begin
     { Comprobamos si se le puede aadir otro vrtice al objeto }
     if obj.NVertices = MaxVertices then exit;
     { Si se puede, lo agregamos }
     inc(obj.NVertices);
     obj.Vertice[obj.NVertices].x := Int2Fixed(nx);
     obj.Vertice[obj.NVertices].y := Int2Fixed(ny);
     obj.Vertice[obj.NVertices].z := Int2Fixed(nz);
end;


{ Este procedimiento aade una cara (o polgono) al objeto }
procedure Objeto3DAgregaCara(var obj : TObjeto3D; cara : TCara);
begin
     { Comprobamos que se pueda aadir otra cara }
     if obj.NCaras = MaxCaras then exit;
     { Si es as, la aadimos }
     inc(obj.NCaras);
     obj.Cara[obj.NCaras] := cara;
end;

{ Este procedimiento rota un objeto mediante la rotacin de sus vrtices }
procedure Objeto3DRota(var obj : TObjeto3D; ax, ay, az : integer);
var i : integer;
begin
     { Simplemente rotamos todos los vrtices }
     for i := 1 to obj.NVertices do
         VerticeRota(obj.Vertice[i], ax, ay, az);

     { Y todas las normales }
     for i := 1 to obj.NCaras do
         VerticeRota(obj.Cara[i].normal, ax, ay, az);
end;

{ Este procedimiento mueve el origen del objeto (y por lo tanto, el objeto)
  al punto (nx, ny, nz) con respecto al centro de la pantalla }
procedure Objeto3DMueve(var obj : TObjeto3D; nx, ny, nz : integer);
begin
     obj.ox := nx;
     obj.oy := ny;
     obj.oz := nz;
end;

{ Este procedimiento calcula la proyeccin bidimensional de sus vrtices
  y los traslada respecto al origen del objeto }
procedure Objeto3DCalcula2D(var obj : TObjeto3D);
var i : integer;
begin
     { Calcula la proyeccin 2D de cada vrtice y lo desplaza con respecto
       al origen del objeto }
     for i := 1 to obj.NVertices do
     begin
          VerticeCalcula2D(obj.Vertice[i], obj.oz);
          inc(obj.Vertice[i].x2d, obj.ox);
          inc(obj.Vertice[i].y2d, obj.oy);
     end;
end;

{ Solo es necesario llamar una vez a este procedimiento }
procedure Objeto3DCalculaNormales(var obj : TObjeto3D);
var ax, ay, az, bx, by, bz, nx, ny, nz, mag : Fixed;
    i : integer;
begin
     with obj do
     for i := 1 to NCaras do
     begin
       { Calculamos los dos vectores en el plano de la cara (A y B) }
       ax := Vertice[Cara[i].Vertice[1]].x - Vertice[Cara[i].Vertice[2]].x;
       ay := Vertice[Cara[i].Vertice[1]].y - Vertice[Cara[i].Vertice[2]].y;
       az := Vertice[Cara[i].Vertice[1]].z - Vertice[Cara[i].Vertice[2]].z;
       bx := Vertice[Cara[i].Vertice[3]].x - Vertice[Cara[i].Vertice[2]].x;
       by := Vertice[Cara[i].Vertice[3]].y - Vertice[Cara[i].Vertice[2]].y;
       bz := Vertice[Cara[i].Vertice[3]].z - Vertice[Cara[i].Vertice[2]].z;

       { Obtenemos el producto cruz de A x B }
       Cara[i].normal.x := FMul(by, az) - FMul(ay, bz);
       Cara[i].normal.y := FMul(ax, bz) - FMul(bx, az);
       Cara[i].normal.z := FMul(bx, ay) - FMul(by, ax);

       { Normalizamos el vector perpendicular (A x B) }
       VerticeNormaliza(Cara[i].normal);
     end;
end;




{ Procedimientos para dibujar objetos 3D }
{ IMPORTANTE!!! No se hace ninguna comprobacin acerca de si los objetos
                quedan dentro de los lmites de la pantalla (0, 0, 319, 199),
                por lo tanto se pueden producir resultados inesperados si
                no se controla el movimiento de los objetos }

{ Este procedimiento simplemente dibuja un punto en cada vrtice del objeto }
procedure Objeto3DDibujaVertices(obj : TObjeto3D; color : byte; where : word);
var i : integer;
begin
     Objeto3DCalcula2D(obj); { Primero calculamos la proyeccin en 2D }
     for i := 1 to obj.NVertices do
         PutPixel(obj.Vertice[i].x2d, obj.Vertice[i].y2d, color, where);
end;


{ Este procedimiento dibuja un "modelo de alambre" del objeto, es decir,
  dibuja nicamente las aristas }
procedure Objeto3DDibujaMalla(obj : TObjeto3D; color : byte; where : word);
var x1, y1, x2, y2 : Fixed;
    i, j : integer;
begin
     Objeto3DCalcula2D(obj);  { Calculamos la proyeccin en 2D }
     for j := 1 to obj.NCaras do { Dibujamos todas las caras }
     begin
          if CaraVisible(obj.cara[j]) then
          begin
               { Obtenemos las coordenadas del primer vrtice }
               x1 := obj.Vertice[obj.Cara[j].Vertice[1]].x2d;
               y1 := obj.Vertice[obj.Cara[j].Vertice[1]].y2d;

               { y hacemos un ciclo para dibujar todas las aristas }
               for i := 2 to obj.cara[j].NVertices do
               begin
                    { Obtenemos las coordenadas del siguiente vrtice }
                    x2 := obj.Vertice[obj.Cara[j].Vertice[i]].x2d;
                    y2 := obj.Vertice[obj.Cara[j].Vertice[i]].y2d;
                    { Dibujamos una lnea entre los dos vrtices }
                    Line(x1, y1, x2, y2, color, where);
                    x1 := x2;
                    y1 := y2;
               end;

               { Obtenemos nuevamente las coordenadas del primer vrtice para
                  dibujar una lnea entre el primero y el ltimo vrtices }
               x2 := obj.Vertice[obj.Cara[j].Vertice[1]].x2d;
               y2 := obj.Vertice[obj.Cara[j].Vertice[1]].y2d;
               Line(x1, y1, x2, y2, color, where);
          end;
     end;
end;


procedure Objeto3DDibujaPlano(obj : TObjeto3D; where : word);
var i : integer;
begin
     Objeto3DCalcula2D(obj);    { calcula la proyeccin a 2D del objeto }
     for i := 1 to obj.NCaras do
         if CaraVisible(obj.cara[i]) then CaraDibuja(obj, obj.cara[i], where);
end;


{ Este procedimiento genera un cubo con centro en el origen }
{ Se puede alterar este procedimiento (y solamente este) para
  generar otros objetos diferentes (pirmides, letras, etc...) }
procedure GeneraCubo;
const lado = 80;

var cara : TCara;

{  Nuestro cubo est formado de la siguiente manera:

                                 3       4
                                 o-------o
                                /|      /|
                               / |     / |
                           1  /  |  2 /  |
                             o---|---o   |
                             |   o---|---o
                             |  / 5  |  / 6
                             | /     | /
                             |/      |/
                             o-------o
                             7       8

}
begin
     { Agregamos los 8 vrtices del cubo }
     Objeto3DReinicia(Cubo);
     Objeto3DAgregaVertice(Cubo, -lado, lado, lado);
     Objeto3DAgregaVertice(Cubo, lado, lado, lado);
     Objeto3DAgregaVertice(Cubo, -lado, lado, -lado);
     Objeto3DAgregaVertice(Cubo, lado, lado, -lado);
     Objeto3DAgregaVertice(Cubo, -lado, -lado, -lado);
     Objeto3DAgregaVertice(Cubo, lado, -lado, -lado);
     Objeto3DAgregaVertice(Cubo, -lado, -lado, lado);
     Objeto3DAgregaVertice(Cubo, lado, -lado, lado);

     { Ahora agregamos las 6 caras, construyndolas una por una }
     CaraReinicia(cara);
     CaraAgregaVertice(cara, 2);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 7);
     CaraAgregaVertice(cara, 8);
     cara.color := 255;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 2);
     CaraAgregaVertice(cara, 8);
     CaraAgregaVertice(cara, 6);
     CaraAgregaVertice(cara, 4);
     cara.color := 254;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 4);
     CaraAgregaVertice(cara, 6);
     CaraAgregaVertice(cara, 5);
     CaraAgregaVertice(cara, 3);
     cara.color := 253;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 3);
     CaraAgregaVertice(cara, 5);
     CaraAgregaVertice(cara, 7);
     CaraAgregaVertice(cara, 1);
     cara.color := 251;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 2);
     CaraAgregaVertice(cara, 4);
     CaraAgregaVertice(cara, 3);
     cara.color := 250;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 6);
     CaraAgregaVertice(cara, 8);
     CaraAgregaVertice(cara, 7);
     CaraAgregaVertice(cara, 5);
     cara.color := 249;
     Objeto3DAgregaCara(Cubo, cara);
end;


{ Como ejemplo, aqu se muestra OTRO procedimiento GeneraCubo que
  genera un objeto diferente (una pirmide), aunque el nombre del
  objeto sigue siendo "cubo" :) }

{
                                    1
                                    o
                                   /|\
                                  / |  \
                                 o--|----o
                                / 5 |   / 4
                               /    |  /
                              /     | /
                             o-------o
                             2       3
}
procedure GeneraPiramide;
const lado = 80;
var cara : TCara;
begin
     Objeto3DReinicia(Cubo);
     Objeto3DAgregaVertice(Cubo, 0, lado, 0);
     Objeto3DAgregaVertice(Cubo, -lado, -lado, lado);
     Objeto3DAgregaVertice(Cubo, lado, -lado, lado);
     Objeto3DAgregaVertice(Cubo, lado, -lado, -lado);
     Objeto3DAgregaVertice(Cubo, -lado, -lado, -lado);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 2);
     CaraAgregaVertice(cara, 3);
     cara.color := 255;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 3);
     CaraAgregaVertice(cara, 4);
     cara.color := 254;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 4);
     CaraAgregaVertice(cara, 5);
     cara.color := 253;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 1);
     CaraAgregaVertice(cara, 5);
     CaraAgregaVertice(cara, 2);
     cara.color := 252;
     Objeto3DAgregaCara(Cubo, cara);

     CaraReinicia(cara);
     CaraAgregaVertice(cara, 5);
     CaraAgregaVertice(cara, 4);
     CaraAgregaVertice(cara, 3);
     CaraAgregaVertice(cara, 2);
     cara.color := 251;
     Objeto3DAgregaCara(Cubo, cara);
end;


{ Procedimiento que realiza la demostracin }
procedure Demo;
var ax, ay, az : integer;  { Angulos de rotacin sobre cada eje }
    VirScr, FondoScr : PTVirtual;  { Usaremos 2 pantallas virtuales }
    VirSeg, FondoSeg : word;

    TablaZ : array[0..255] of integer;  { tabla con las posiciones en Z }
    i, j, tablapos : byte;       { posicin actual en la tabla }


begin
     { Iniciamos las pantallas virtuales y cargamos la imagen de fondo }
     SetupVirtual(VirScr, VirSeg);
     SetupVirtual(FondoScr, FondoSeg);
     LoadPCX('fondo.pcx', FondoSeg, 320, 200, 0, 0, pal);

     { Generamos la tabla de posiciones en Z usando un periodo del seno }
     for i := 0 to 255 do
         TablaZ[i] := round(300 * sin(2 * Pi * i / 256));

     { Hacemos un degradado de amarillo en los ltimos 16 colores de la paleta }
     for i := 0 to 15 do
     begin
          pal[240 + i][0] := i * 4 + 3;    { igual cantidad de rojo...  }
          pal[240 + i][1] := i * 4 + 3;    { ... y de verde             }
          pal[240 + i][2] := 0;              { pero nada de azul!         }
     end;

     ax := 30;
     ay := -30; { Angulos iniciales de rotacin }
     az := 0;

     SetMode13;  { Entramos al modo grfico }
     FadeOut(0); { Y hacemos negra toda la paleta }
     CopyScreen(FondoSeg, VGA); { Copiamos la imagen de fondo a VGA }
     FadeTo(pal, 5); { Y la iluminamos }

     { Preparamos el objeto tridimensional }
     Objeto3DCalculaNormales(Cubo);
     tablapos := 0;

     { Primera parte: Dibujo de malla con eliminacin de partes ocultas }
     while not keypressed do  { mientras no se presione una tecla... }
     begin
          Objeto3DMueve(Cubo, 0, 0, TablaZ[tablapos]); { movemos el objeto
                                                         en el eje Z }
          CopyScreen(FondoSeg, VirSeg); { Copia el fondo a la pantalla virtual }
          Objeto3DDibujaMalla(Cubo, 255, VirSeg); { Dibuja el objeto }
          VRetrace;       { Espera al retrazado vertical (solo por si...) }
          CopyScreen(VirSeg, VGA);  { y muestra todo en VGA }

          { aumentamos el ndice de TablaZ para mover el objeto }
          inc(tablapos, 4);

          { Rota el objeto y actualiza los ngulos de rotacin }
          Objeto3DRota(Cubo, ax div 10, ay div 10, az div 10);
          if ax < 50 then inc(ax) else ax := -50;
          if ay < 50 then inc(ay) else ay := -50;
          if az > -50 then dec(az) else az := 50;
     end;

     readkey;  { lee la tecla pulsada y la desecha }

     { Segunda parte: Polgonos rellenos y eliminacin de partes ocultas }
     while not keypressed do  { mientras no se presione una tecla... }
     begin
          Objeto3DMueve(Cubo, 0, 0, TablaZ[tablapos]); { movemos el objeto
                                                         en el eje Z }
          CopyScreen(FondoSeg, VirSeg); { Copia el fondo a la pantalla virtual }
          Objeto3DDibujaPlano(Cubo, VirSeg); { Dibuja el objeto }
          VRetrace;       { Espera al retrazado vertical (solo por si...) }
          CopyScreen(VirSeg, VGA);  { y muestra todo en VGA }

          { aumentamos el ndice de TablaZ para mover el objeto }
          inc(tablapos, 4);

          { Rota el objeto y actualiza los ngulos de rotacin }
          Objeto3DRota(Cubo, ax div 10, ay div 10, az div 10);
          if ax < 50 then inc(ax) else ax := -50;
          if ay < 50 then inc(ay) else ay := -50;
          if az > -50 then dec(az) else az := 50;
     end;

     readkey;  { lee la tecla pulsada y la desecha }

     { Outro (Hace un fadeout en la imagen del fondo) }
     for j := 1 to 64 do { El fadeout toma 64 pasos }
     begin
          Objeto3DMueve(Cubo, 0, 0, TablaZ[tablapos]);
          CopyScreen(FondoSeg, VirSeg); { copiamos el fondo a la p.v. }
          Objeto3DDibujaPlano(Cubo, VirSeg);
          VRetrace; { esperamos al retrazado (solo en caso necesario) }
          CopyScreen(VirSeg, VGA); { y mostramos en VGA }
          SetPalette(pal); { y actualizamos la paleta }
          inc(tablapos, 4);
          Objeto3DRota(Cubo, ax div 8, ay div 8, az div 8);
          if ax < 50 then inc(ax) else ax := -50;
          if ay < 50 then inc(ay) else ay := -50;
          if az > -50 then dec(az) else az := 50;

          { Hacemos el fadeout de los colores 1 a 8 y 40 a 185 }
          { La imagen de fondo usa los colores 1 a 185, pero los
            colores del 9 al 39 son usados por el logotipo de "FAC",
            por lo tanto si no hacemos fade out en estos colores,
            el logotipo permanecer en la pantalla }
          for i := 1 to 8 do
          begin
               if pal[i][0] > 0 then dec(pal[i][0]);
               if pal[i][1] > 0 then dec(pal[i][1]);
               if pal[i][2] > 0 then dec(pal[i][2]);
          end;
          for i := 40 to 185 do
          begin
               if pal[i][0] > 0 then dec(pal[i][0]);
               if pal[i][1] > 0 then dec(pal[i][1]);
               if pal[i][2] > 0 then dec(pal[i][2]);
          end;
     end;

     { Hacemos el fade out del logotipo }
     for j := 1 to 64 do
     begin
          Objeto3DMueve(Cubo, 0, 0, TablaZ[tablapos]);
          CopyScreen(FondoSeg, VirSeg); { copiamos el fondo a la p. v. }
          Objeto3DDibujaPlano(Cubo, VirSeg);
          VRetrace; { Sincronizamos con la pantalla }
          CopyScreen(VirSeg, VGA); { y mostramos todo en VGA }
          SetPalette(pal); { Actualizamos la paleta }
          inc(tablapos, 4);
          Objeto3DRota(Cubo, ax div 8, ay div 8, az div 8);
          if ax < 50 then inc(ax) else ax := -50;
          if ay < 50 then inc(ay) else ay := -50;
          if az > -50 then dec(az) else az := 50;

          { Hacemos fadeout en los colores del logotipo }
          for i := 9 to 39 do
          begin
               if pal[i][0] > 0 then dec(pal[i][0]);
               if pal[i][1] > 0 then dec(pal[i][1]);
               if pal[i][2] > 0 then dec(pal[i][2]);
          end;
     end;

     { Hacemos fadeout en el objeto tridimensional }
     for j := 1 to 64 do
     begin
          Objeto3DMueve(Cubo, 0, 0, TablaZ[tablapos]);
          ClearScreen(0, VirSeg); { Borramos la pantalla virtual }
                                  { (el fondo ya no se necesita) }
          Objeto3DDibujaPlano(Cubo, VirSeg);
          VRetrace; { Esperamos el retrazado vertical }
          CopyScreen(VirSeg, VGA); { copiamos a VGA }
          SetPalette(pal); { Actualizamos la paleta }
          inc(tablapos, 4);
          Objeto3DRota(Cubo, ax div 8, ay div 8, az div 8);
          if ax < 50 then inc(ax) else ax := -50;
          if ay < 50 then inc(ay) else ay := -50;
          if az > -50 then dec(az) else az := 50;

          { hacemos fadeout en los colores del cubo }
          for i := 240 to 255 do
          begin
               if pal[i][0] > 0 then dec(pal[i][0]);
               if pal[i][1] > 0 then dec(pal[i][1]);
               if pal[i][2] > 0 then dec(pal[i][2]);
          end;
     end;

     { Liberamos la memoria de las pantallas virtuales }
     ShutDownVirtual(VirScr);
     ShutDownVirtual(FondoScr);
end;


{ Secuencia principal }
begin
     clrscr;
     writeln;
     writeln('Demostracin de las rutinas bsicas de 3D.');
     writeln('Ahora con eliminacin de caras ocultas y polgonos slidos!');
     writeln;
     writeln('Presiona una tecla...');
     readkey;

     GeneraTablas;
     GeneraCubo;  { Se puede cambiar por GeneraPiramide }
     Demo;
     SetTextMode;
end.
