
unit Objeto3D;

interface

uses Vertices;

const MaxVertices = 8;  { Nmero mximo de vrtices (8 para un cubo) }
      MaxCaras = 6;     { Nmero mximo de caras (6 para un cubo) }
      MaxVerticesCara = 4; { Nmero mximo de vrtices en cada cara }


{ Variables globales usadas para el sombreado }
var rango, colorinicial : byte;
    FuenteLuz : TVertice;


{ Tipo cara (polgono) }
type TCara = record
             NVertices : byte;  { Nmero de vrtices en el polgono }
             Vertice : array[1..MaxVerticesCara] of integer;
                       { Indices correspondientes a cada vrtice }
             normal : TVertice;  { vector normal al plano de la cara }
             color : byte;       { color de la cara }
             end;


{ Tipo Objeto Tridimensional }
type TObjeto3D = record
                 NVertices : integer;  { Nmero de vrtices }
                 NCaras : integer;     { Nmero de caras }
                 ox, oy, oz : integer; { Coordenadas del origen del objeto }
                 Vertice : array[1..MaxVertices] of TVertice; { Vrtices }
                 Cara : array[1..Maxcaras] of TCara;          { Caras }
                 end;


{ procedimientos para las caras }
procedure CaraReinicia(var cara : TCara);
procedure CaraAgregaVertice(var cara : TCara; vertice : integer);
{function CaraVisible(cara : TCara) : boolean; }
function CaraVisible(cara : TCara; objeto : TObjeto3D) : boolean;
procedure CaraDibuja(obj : TObjeto3D; cara : TCara; where : word);


{ procedimientos para los objetos }
procedure Objeto3DReinicia(var obj : TObjeto3D);
procedure Objeto3DAgregaVertice(var obj : TObjeto3D; nx, ny, nz : integer);
procedure Objeto3DAgregaCara(var obj : TObjeto3D; cara : TCara);
procedure Objeto3DRota(var obj : TObjeto3D; ax, ay, az : integer);
procedure Objeto3DMueve(var obj : TObjeto3D; nx, ny, nz : integer);
procedure Objeto3DCalcula2D(var obj : TObjeto3D);
procedure Objeto3DCalculaNormales(var obj : TObjeto3D);
procedure Objeto3DDibujaVertices(obj : TObjeto3D; color : byte; where : word);
procedure Objeto3DDibujaMalla(obj : TObjeto3D; color : byte; where : word);
procedure Objeto3DDibujaPlano(obj : TObjeto3D; where : word);
procedure Objeto3DDibujaSombreadoPlano(obj : TObjeto3D; where : word);


implementation

uses Mode13;

{ Procedimientos de las caras }

{ Este procedimiento resetea las variables principales de una cara }
procedure CaraReinicia(var cara : TCara);
begin
     cara.NVertices := 0;
end;

{ Este procedimiento aade un vrtice (su ndice) a una cara }
{ IMPORTANTE!!! Los vrtices se deben aadir en el sentido OPUESTO
                a las manecillas del reloj.                       }
procedure CaraAgregaVertice(var cara : TCara; vertice : integer);
begin
     { Primero comprobamos si podemos aadir otro vrtice }
     if cara.NVertices = MaxVerticesCara then exit;
     { Si es as, lo aadimos }
     inc(cara.NVertices);
     cara.Vertice[cara.NVertices] := vertice;
end;

{
function CaraVisible(cara : TCara; objeto : TObjeto3D) : boolean;
begin
     if cara.normal.z >= 0.0 then CaraVisible := true
                             else CaraVisible := false;
end;
}

{ Algoritmo mejorado para detectar si una cara es visible o no:
  Funciona de la misma forma, es decir, comprobamos si el vector
  normal al plano de la cara est dirigido hacia el observador
  (normal.z >= 0), pero lo hacemos despus de hacer la proyeccin a 2D }

function CaraVisible(cara : TCara; objeto : TObjeto3D) : boolean;
var ax, ay, bx, by : integer;
begin
     if cara.NVertices < 3 then exit;
     ax := objeto.Vertice[cara.Vertice[1]].x2d -
           objeto.Vertice[cara.Vertice[2]].x2d;
     ay := objeto.Vertice[cara.Vertice[1]].y2d -
           objeto.Vertice[cara.Vertice[2]].y2d;
     bx := objeto.Vertice[cara.Vertice[3]].x2d -
           objeto.Vertice[cara.Vertice[2]].x2d;
     by := objeto.Vertice[cara.Vertice[3]].y2d -
           objeto.Vertice[cara.Vertice[2]].y2d;
     CaraVisible := (bx * ay) > (by * ax);
end;


procedure CaraDibuja(obj : TObjeto3D; cara : TCara; where : word);
var Xmin, Xmax : array[0..199] of integer;
    i, x1, y1, x2, y2 : integer;
    Ymin, Ymax, y : integer;

    procedure ScanSide(x1, y1, x2, y2 : integer);
    var y, ydiv, temp : integer;
        x, xinc : double;
    begin
         if y1 > y2 then
         begin
              { nos aseguramos de que (x1, y1) sea el punto SUPERIOR
                y (x2, y2) sea el punto INFERIOR }
              temp := y1; y1 := y2; y2 := temp;
              temp := x1; x1 := x2; x2 := temp;
         end;
         ydiv := y2 - y1;
         if ydiv = 0 then exit;
         xinc := (x2 - x1) / ydiv;
         x := x1;
         for y := y1 to y2 do
         begin
              if trunc(x) < Xmin[y] then Xmin[y] := trunc(x);
              if round(x) > Xmax[y] then Xmax[y] := round(x);
              x := x + xinc;
         end;
    end;

begin
     { Lo primero es obtener las coordenadas Y mxima y mnima del
       polgono }
     Ymin := 32000;
     Ymax := -32000;
     for i := 1 to cara.NVertices do
     begin
          y := obj.Vertice[cara.Vertice[i]].y2d;
          if y < Ymin then Ymin := y;
          if y > Ymax then Ymax := y;
     end;

     { Si el polgono tiene altura igual a cero, entonces no lo dibujamos }
     if (Ymax - Ymin) = 0 then exit;

     for i := 0 to 199 do       { Inicializamos Xmin y Xmax }
     begin
          Xmin[i] := 32000;
          Xmax[i] := -32000;
     end;

     { Obtenemos las coordenadas del primer vrtice }
     x1 := obj.Vertice[cara.Vertice[1]].x2d;
     y1 := obj.Vertice[cara.Vertice[1]].y2d;

     { y hacemos un ciclo para dibujar todas las aristas }
     for i := 2 to cara.NVertices do
     begin
          { Obtenemos las coordenadas del siguiente vrtice }
          x2 := obj.Vertice[cara.Vertice[i]].x2d;
          y2 := obj.Vertice[cara.Vertice[i]].y2d;
          { "Escaneamos" el lado (x1, y1)-(x2, y2) }
          ScanSide(x1, y1, x2, y2);
          { y continuamos con el siguiente lado }
          x1 := x2;
          y1 := y2;
     end;

     { Obtenemos nuevamente las coordenadas del primer vrtice para
       escanear la lnea entre el primero y el ltimo vrtices }
     x2 := obj.Vertice[cara.Vertice[1]].x2d;
     y2 := obj.Vertice[cara.Vertice[1]].y2d;
     ScanSide(x1, y1, x2, y2);

     { En este momento, ya tenemos la informacin necesaria almacenada
       en Xmin y Xmax. Ahora procedemos a dibujar el polgono }
     for i := Ymin to Ymax do
         HLine(Xmin[i], Xmax[i], i, cara.color, where);
end;


{ Procedimientos de los objetos 3D }

{ Este procedimiento reinicia las variables importantes de un objeto 3D }
procedure Objeto3DReinicia(var obj : TObjeto3D);
begin
     obj.NVertices := 0;
     obj.NCaras := 0;
     obj.ox := 0;
     obj.oy := 0;
     obj.oz := 0;
end;


{ Este procedimiento agrega un vrtice al objeto tridimensional }
{ (no importa el orden, pero hay que recordar el ndice de cada vrtice
   a la hora de especificar los polgonos) }
procedure Objeto3DAgregaVertice(var obj : TObjeto3D; nx, ny, nz : integer);
begin
     { Comprobamos si se le puede aadir otro vrtice al objeto }
     if obj.NVertices = MaxVertices then exit;
     { Si se puede, lo agregamos }
     inc(obj.NVertices);
     obj.Vertice[obj.NVertices].x := nx;
     obj.Vertice[obj.NVertices].y := ny;
     obj.Vertice[obj.NVertices].z := nz;
end;


{ Este procedimiento aade una cara (o polgono) al objeto }
procedure Objeto3DAgregaCara(var obj : TObjeto3D; cara : TCara);
begin
     { Comprobamos que se pueda aadir otra cara }
     if obj.NCaras = MaxCaras then exit;
     { Si es as, la aadimos }
     inc(obj.NCaras);
     obj.Cara[obj.NCaras] := cara;
end;

{ Este procedimiento rota un objeto mediante la rotacin de sus vrtices }
procedure Objeto3DRota(var obj : TObjeto3D; ax, ay, az : integer);
var i : integer;
begin
     { Simplemente rotamos todos los vrtices }
     for i := 1 to obj.NVertices do
         VerticeRota(obj.Vertice[i], ax, ay, az);

     { Y todas las normales }
     for i := 1 to obj.NCaras do
         VerticeRota(obj.Cara[i].normal, ax, ay, az);
end;

{ Este procedimiento mueve el origen del objeto (y por lo tanto, el objeto)
  al punto (nx, ny, nz) con respecto al centro de la pantalla }
procedure Objeto3DMueve(var obj : TObjeto3D; nx, ny, nz : integer);
begin
     obj.ox := nx;
     obj.oy := ny;
     obj.oz := nz;
end;

{ Este procedimiento calcula la proyeccin bidimensional de sus vrtices
  y los traslada respecto al origen del objeto }
procedure Objeto3DCalcula2D(var obj : TObjeto3D);
var i : integer;
begin
     { Calcula la proyeccin 2D de cada vrtice y lo desplaza con respecto
       al origen del objeto }
     for i := 1 to obj.NVertices do
     begin
          VerticeCalcula2D(obj.Vertice[i], obj.oz);
          inc(obj.Vertice[i].x2d, obj.ox);
          inc(obj.Vertice[i].y2d, obj.oy);
     end;
end;

{ Solo es necesario llamar una vez a este procedimiento }
procedure Objeto3DCalculaNormales(var obj : TObjeto3D);
var ax, ay, az, bx, by, bz, nx, ny, nz, mag : double;
    i : integer;
begin
     with obj do
     for i := 1 to NCaras do
     begin
       { Calculamos los dos vectores en el plano de la cara (A y B) }
       ax := Vertice[Cara[i].Vertice[1]].x - Vertice[Cara[i].Vertice[2]].x;
       ay := Vertice[Cara[i].Vertice[1]].y - Vertice[Cara[i].Vertice[2]].y;
       az := Vertice[Cara[i].Vertice[1]].z - Vertice[Cara[i].Vertice[2]].z;
       bx := Vertice[Cara[i].Vertice[3]].x - Vertice[Cara[i].Vertice[2]].x;
       by := Vertice[Cara[i].Vertice[3]].y - Vertice[Cara[i].Vertice[2]].y;
       bz := Vertice[Cara[i].Vertice[3]].z - Vertice[Cara[i].Vertice[2]].z;

       { Obtenemos el producto cruz de A x B }
       Cara[i].normal.x := by * az - ay * bz;
       Cara[i].normal.y := ax * bz - bx * az;
       Cara[i].normal.z := bx * ay - by * ax;

       { Normalizamos el vector perpendicular (A x B) }
       VerticeNormaliza(Cara[i].normal);
     end;
end;



{ Procedimientos para dibujar objetos 3D }
{ IMPORTANTE!!! No se hace ninguna comprobacin acerca de si los objetos
                quedan dentro de los lmites de la pantalla (0, 0, 319, 199),
                por lo tanto se pueden producir resultados inesperados si
                no se controla el movimiento de los objetos }

{ Este procedimiento simplemente dibuja un punto en cada vrtice del objeto }
procedure Objeto3DDibujaVertices(obj : TObjeto3D; color : byte; where : word);
var i : integer;
begin
     Objeto3DCalcula2D(obj); { Primero calculamos la proyeccin en 2D }
     for i := 1 to obj.NVertices do
         PutPixel(obj.Vertice[i].x2d, obj.Vertice[i].y2d, color, where);
end;


{ Este procedimiento dibuja un "modelo de alambre" del objeto, es decir,
  dibuja nicamente las aristas }
procedure Objeto3DDibujaMalla(obj : TObjeto3D; color : byte; where : word);
var x1, y1, x2, y2 : integer;
    i, j : integer;
begin
     Objeto3DCalcula2D(obj);  { Calculamos la proyeccin en 2D }
     for j := 1 to obj.NCaras do { Dibujamos todas las caras }
     begin
          if CaraVisible(obj.cara[j], obj) then
          begin
               { Obtenemos las coordenadas del primer vrtice }
               x1 := obj.Vertice[obj.Cara[j].Vertice[1]].x2d;
               y1 := obj.Vertice[obj.Cara[j].Vertice[1]].y2d;

               { y hacemos un ciclo para dibujar todas las aristas }
               for i := 2 to obj.cara[j].NVertices do
               begin
                    { Obtenemos las coordenadas del siguiente vrtice }
                    x2 := obj.Vertice[obj.Cara[j].Vertice[i]].x2d;
                    y2 := obj.Vertice[obj.Cara[j].Vertice[i]].y2d;
                    { Dibujamos una lnea entre los dos vrtices }
                    Line(x1, y1, x2, y2, color, where);
                    x1 := x2;
                    y1 := y2;
               end;

               { Obtenemos nuevamente las coordenadas del primer vrtice para
                  dibujar una lnea entre el primero y el ltimo vrtices }
               x2 := obj.Vertice[obj.Cara[j].Vertice[1]].x2d;
               y2 := obj.Vertice[obj.Cara[j].Vertice[1]].y2d;
               Line(x1, y1, x2, y2, color, where);
          end;
     end;
end;


procedure Objeto3DDibujaPlano(obj : TObjeto3D; where : word);
var i : integer;
begin
     Objeto3DCalcula2D(obj);    { calcula la proyeccin a 2D del objeto }
     for i := 1 to obj.NCaras do
         if CaraVisible(obj.cara[i], obj) then
            CaraDibuja(obj, obj.cara[i], where);
end;


{ Procedimiento que dibuja un objeto usando sombreado plano }
procedure Objeto3DDibujaSombreadoPlano(obj : TObjeto3D; where : word);
var i : integer;
    cosang : double;
begin
     Objeto3DCalcula2D(obj);    { calcula la proyeccin a 2D del objeto }
     for i := 1 to obj.NCaras do
         if CaraVisible(obj.cara[i], obj) then
         begin
              with obj.cara[i] do
              begin
                   cosang := normal.x * FuenteLuz.x +
                             normal.y * FuenteLuz.y +
                             normal.z * FuenteLuz.z;
                   color := trunc((cosang + 1) * rango / 2) + colorinicial;
              end;
              CaraDibuja(obj, obj.cara[i], where);
         end;
end;

end.