/*      mpoll.c
 *
 * Functions for polling MIDAS Sound System in a thread
 *
 * $Id: mpoll.c,v 1.1 1996/08/06 20:36:37 pekangas Exp $
 *
 * Copyright 1996 Petteri Kangaslampi and Jarno Paananen
 *
 * This file is part of the MIDAS Sound System, and may only be
 * used, modified and distributed under the terms of the MIDAS
 * Sound System license, LICENSE.TXT. By continuing to use,
 * modify or distribute this file you indicate that you have
 * read the license and understand and accept it fully.
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include "midas.h"

#ifdef __WC32__
#include <process.h>
#endif

RCSID(const char *mpoll_rcsid = "$Id: mpoll.c,v 1.1 1996/08/06 20:36:37 pekangas Exp $";)




/****************************************************************************\
*
* Function:     void PollMIDAS(void)
*
* Description:  Polls MIDAS Sound System
*
\****************************************************************************/

void PollMIDAS(void)
{
    int         error;
    int         callMP;

    if ( !midasSDInit )
        return;

    if ( (error = midasSD->StartPlay()) != OK )
        midasError(error);
    do
    {
        if ( (error = midasSD->Play(&callMP)) != OK )
            midasError(error);
        if ( callMP )
        {
            if ( midasGMPInit )
            {
                if ( (error = gmpPlay()) != OK )
                    midasError(error);
            }
        }
    } while ( callMP && (midasSD->tempoPoll == 0) );
}



static volatile int stopPolling = 0;
static volatile unsigned pollSleep = 50;

#ifdef __WC32__
static void PollerThread(void *dummy)
#else
static int PollerThread(void *dummy)
#endif
{
    dummy = dummy;

    /* We'd better make the player thread's priority still above normal: */
    SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_ABOVE_NORMAL);

    while ( stopPolling == 0 )
    {
        PollMIDAS();
        Sleep(pollSleep);
    }

    stopPolling = 0;

#ifdef __WC32__
    _endthread();
#else
    ExitThread(0);
    return 0;
#endif
}



/****************************************************************************\
*
* Function:     void StartPlayThread(unsigned pollPeriod)
*
* Description:  Starts polling MIDAS in a thread
*
* Input:        unsigned pollPeriod     polling period (delay between two
*                                       polling loops) in milliseconds
*
\****************************************************************************/

void StartPlayThread(unsigned pollPeriod)
{
#ifdef __WC32__
    int         pollThread;
#else
    HANDLE      pollThread;
    DWORD       pollThreadID;
#endif

    pollSleep = pollPeriod;

    /* Start polling MIDAS in a thread: */
#ifdef __WC32__
    pollThread = _beginthread(PollerThread, NULL, 4096, NULL);
    if ( pollThread == -1 )
        midasErrorExit("StartPlayThread: Couldn't create player thread!");
#else
    pollThread = CreateThread(NULL, 4096, (LPTHREAD_START_ROUTINE)
        PollerThread, NULL, 0, &pollThreadID);
    if ( pollThread == NULL )
        midasErrorExit("StartPlayThread: Couldn't create player thread!");
#endif
}




/****************************************************************************\
*
* Function:     void StopPlayThread(void)
*
* Description:  Stops polling MIDAS in a thread
*
\****************************************************************************/

void StopPlayThread(void)
{
    /* Ugly but works */

    stopPolling = 1;
    while ( stopPolling )
        Sleep(pollSleep/2);
}


/*
 * $Log: mpoll.c,v $
 * Revision 1.1  1996/08/06 20:36:37  pekangas
 * Initial revision
 *
*/