
;ͻ;
; RAW/XMS initialization and system functions.                              ;
;ͼ;

; selector values
Code32Sel_EQU       EQU     (08h)
Data32Sel_EQU       EQU     (10h)
Flat32Sel_EQU       EQU     (18h)
KernelSel_EQU       EQU     (20h)
Stack16Sel_EQU      EQU     (28h)


;*****************************************************************************;
Kernel    SEGMENT
          ASSUME CS:Kernel, DS:Code32

;Ŀ;
; XMS system high memory initialization                                     ;
;;
XMS_System:
  ; Get driver address
  mov ax,4310h
  int 2Fh
  mov m w XMS_Driver  ,bx
  mov m w XMS_Driver+2,es

  ; Enable A20
  mov ah,05h
  call m XMS_Driver

  ; Query free memory
  mov ah,08h
  xor bl,bl
  call m XMS_Driver     ; returns AX with largest block (Kb)
  movzx eax,ax
  mov dx,ax             ; save for allocate
  shl eax,10
  mov m Heap_Size,eax

  ; Allocate largest block
  mov ah,09h
  call m XMS_Driver
  mov m XMS_Handle,dx

  ; Lock Block
  mov ah,0Ch
  call m XMS_Driver     ; DX:BX contains linear address of allocated block

  ; Adjust high memory address to point to XMS Block
  mov m w Heap_Base  ,bx
  mov m w Heap_Base+2,dx

  jmp RAWXMS_Init


;Ŀ;
; Raw system high memory initialization                                     ;
;;
Raw_System:
  ; Fix-up high memory deallocation address
  mov d CS:DeallocHimemAddr,90909090h

  ; Get Amount of available high memory
  mov ah,88h
  int 15h
  movzx eax,ax ; Amount returned is in KB
  shl eax,10   ; EAX now contains memory in bytes

  ; store value in HighMemory_Avail
  mov m Heap_Size,eax


;Ŀ;
; Raw system A20 Enable                                                     ;
;;
Raw_EnableA20:
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  call EmptySP

  ; Send command to write output port
  mov al,0D1h
  out 64h,al
  call EmptySP

  ; Send command to enable the A20 Gate
  mov al,0DFh
  out 60h,al
  call EmptySP
  jmp RAWXMS_Init

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Empty the 8042 SP Buffer
EmptySP:
  xor cx,cx           ; CX=65536 for timeout value
EmptySPLoop:
  in al,64h           ; Read the status port, find
  and al,2h           ;   out if the buffer is full
  loopnz EmptySPLoop  ; Loop until buffer empty
  jnz A20Error        ;   exit on failure
  ret

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
A20Error:    ; Exit with error message and error code 0FFh
  mov dx,m o RAWXMS_GenErrorMSG
  jmp ErrorExit


;Ŀ;
; Setup the PIC to relocate the hardware interrupts.                        ;
;;
Remap_PIC:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Save PIC masks
  in al,021h
  push ax
  DELAY
  in al,0A1h
  push ax
  DELAY

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Send the initialization sequence to both the 8259 PICs
  mov al,11h
  out 020h,al
  DELAY
  out 0A0h,al
  DELAY

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Set the new interrupt vector bases
  mov al,PIC1_VECTOR
  out 021h,al
  DELAY
  mov al,PIC2_VECTOR
  out 0A1h,al
  DELAY

  mov al,4
  out 021h,al
  DELAY
  mov al,2
  out 0A1h,al
  DELAY

  mov al,1h
  out 021h,al
  DELAY
  out 0A1h,al
  DELAY

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Restore PIC masks
  pop ax
  out 0A1h,AL
  DELAY
  pop ax
  out 021h,AL
  DELAY

  ret


;Ŀ;
; Real Mode Interrupt 21h ISR                                               ;
;;
Int21h:
  cmp ah,4Ch
  jz Exit16

  ; Chain to original interrupt
  db 0EAh
Old_Int21h dd 0


;Ŀ;
; Setup the Real Mode interrupt table                                       ;
;;
Setup_rm_Int:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  push w 0
  pop es

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Save Int 21h
  mov eax,ES:[21h*4]
  mov CS:Old_Int21h,eax

  ; Direct Int 21h to the ISR
  mov w ES:[21h*4],o Int21h
  mov ES:[21h*4+2],cs

  ret


;Ŀ;
; Setup Interrupt Descriptor Table                                          ;
;;
Build_IDT:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Test IDT upon entry
  sidt m r0_IDT
  mov dx,m o RAWXMS_GenErrorMSG
  cmp m r0_IDTaddr,0                  ; Table located at 00000000h
  jnz ErrorExit

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Make address of IDTs linear
  mov eax,IDTSeg
  mov es,ax                           ; ES points to IDTSeg
  shl eax,4
  add m pm_IDTaddr,eax
  add m rm_IDTaddr,eax


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Create pmIDT table
  ; 256 entries pointing to pmISR
  mov cx,256
  mov di,o pmIDT
  mov edx,o pmISR
  add edx,eax                         ; eax is linear address of IDTSeg
  sub edx,m Code32addr                ; make edx relative to Code32
  Make_pmIDT_Table:
    mov ax,dx
    stosw
    mov ax,Code32Sel_EQU
    stosw
    mov ax,8E00h
    stosw
    shld eax,edx,16
    stosw
    add edx,10
    loop Make_pmIDT_Table

  ; Setup guaranteed Real Mode interrupt call
  mov di,o pmIDT + rm_INTCALL_VECTOR*8
  mov ax,o rm_Interrupt
  stosw
  mov ax,KernelSel_EQU
  stosw
  mov ax,8E00h
  stosw
  xor ax,ax
  stosw

  ; Create pmISR table
  mov di,o pmISR
  xor dx,dx
  Make_pmISR_Table:

    ; push intnum
    mov ax,06866h
    stosw
    mov ax,dx
    stosw

    ; jmp KernelSel_EQU:rm_IntCall
    mov ax,0EA66h
    stosw
    mov ax,o rm_IntCall
    stosw
    mov ax,KernelSel_EQU
    stosw

    inc dl                      ; next interrupt
    jnz Make_pmISR_Table


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Create rmIDT table
  ; 256 entries pointing to rmISR
  mov cx,256
  mov di,o rmIDT
  mov dx,o rmISR
  Make_rmIDT_Table:
    mov ax,dx
    stosw
    mov ax,IDTSeg
    stosw
    add dx,8
    loop Make_rmIDT_Table

  ; Create rmISR table
  mov di,o rmISR
  xor dx,dx
  Make_rmISR_Table:

    ; push intnum
    mov ax,068h
    stosb
    mov ax,dx
    stosw

    ; jmp KernelSeg:rm_IntRedirect
    mov ax,0EAh
    stosb
    mov ax,o rm_IntRedirect
    stosw
    mov ax,m KernelSeg
    stosw

    inc dl                      ; next interrupt
    jnz Make_rmISR_Table

  ret


;Ŀ;
; Guaranteed Real Mode interrupt call back                                  ;
;;
rm_Interrupt:
  push m w rm_Int
;  jmp rm_IntCall


;Ŀ;
; Real Mode interrupt call back                                             ;
; Stack: (bottom) Int EIP CS EFLAGS (top)                                   ;
;;
rm_IntCall:
  ; Save segment registers
  push ds es fs gs

  ; Load DS with Code32 segment
  push w Data32Sel_EQU
  pop ds

  ; Save EAX and flags
  push eax
  pushfd

  ; Load 64Kb limit stack selector
  push w Stack16Sel_EQU
  pop ss

  ; Load Real Mode IDT
  lidt m rm_IDT

  ; Switch to Real Mode
  mov eax,cr0
  and al,NOT 1
  mov cr0,eax

  ; Load CS:IP
  db 0EAh
  dw $+4,Kernel

  ; Calculate real mode SS:SP for stack pointer
  mov eax,m d rm_SS
  mov ss,ax
  shl eax,4
  add esp,m Code32addr
  sub esp,eax

  ; Modify code for interrupt redirection
  mov al,[esp+16]
  mov CS:rm_Interrupt_Index,al
  jmp $+2

  ; Load segment registers from virtual registers
  mov es,m rm_ES
  mov ds,m rm_DS

  ; Restore EAX and flags
  popfd
  pop eax

  ; Process Interrupt
  db 0CDh
  rm_Interrupt_Index db ?

  cli                       ; disable interrupts

  ; Save EAX and flags
  push eax
  pushfd

  ; Place DS onto the Stack
  push ds

  ; Load DS with Code32 segment
  push w Code32
  pop ds

  ; Save segment registers in virtual registers
  pop m rm_DS               ; Remove DS from the Stack
  mov m rm_ES,es

  ; Calculate address of stack relative to Code32
  xor eax,eax
  mov ax,ss
  mov m rm_SS,ax            ; save Real Mode SS
  shl eax,4
  add esp,eax
  sub esp,m Code32addr

  ; Load Protected Mode GDT
  lgdt m pm_GDT

  ; Load Protected Mode IDT
  lidt m pm_IDT

  ; Switch to Protected Mode
  mov eax,cr0
  or al,1
  mov cr0,eax

  ; Load CS:IP
  db 0EAh
  dw $+04, KernelSel_EQU

  ; Load Protected Mode Stack
  push w Data32Sel_EQU
  pop ss

  ; Restore eax and flags
  popfd
  pop eax

  ; Restore segment registers
  pop gs fs es ds

  ; Remove interrupt number from stack (without affecting flags)
  mov [esp],ax
  pop ax

  ; Enable interupts
  sti

  ; CS:EIP ret and remove the flags from stack
  db 066h,0CAh,004h,000h,000h,000h


;Ŀ;
; Real Mode interrupt redirection                                           ;
;;
rm_IntRedirect:

  ; Save registers
  pushfd
  push ds ebx

  ; Load pointer to instruction stream before entry
  lds bx,[esp+12]

  ; Check if interrupt was invoked due to an INT opcode
  cmp b [bx-2],0CDh
  jnz pm_IntCall

  ; Make sure that the interrupt invoked was specified by the INT instruction
  movzx ebx,b [bx-1]
  cmp bl,[esp+10]
  jnz pm_IntCall

  ; Modify code for interrupt redirection
  push w 0
  pop ds
  mov ebx,[ebx*4]
  mov CS:rm_Redirect_addr,ebx
  jmp $+2

  ; Restore registers
  pop ebx ds
  popfd

  ; Remove interrupt number from stack (without affecting flags)
  mov [esp],ax
  pop ax

  ; Process interrupt
  db 0EAh
  rm_Redirect_addr dd ?


;Ŀ;
; Protected Mode interrupt call back                                        ;
; Entry is from rm_IntRedirect                                              ;
; Stack: (bottom) FLAGS CS IP Int EFLAGS DS EBX (top)                       ;
;;
pm_IntCall:
  ; Restore EBX
  pop ebx

  ; Save other registers
  push es fs gs eax

  ; Load DS with Code32 Segment
  push w Code32
  pop ds

  ; Modify code for interrupt instruction
  ; (IPQ loaded below)
  mov al,[esp+16]
  mov CS:pm_Interrupt_Index,al

  ; Calculate SS:ESP relative to Code32
  xor eax,eax
  mov ax,ss
  mov m rm_SS,ax            ; save Real Mode SS
  shl eax,4
  add esp,eax
  sub esp,m Code32addr

  ; Load Protected Mode GDT
  lgdt m pm_GDT

  ; Load Protected Mode IDT
  lidt m pm_IDT

  ; Switch to Protected Mode
  mov eax,cr0
  or al,1
  mov cr0,eax

  ; Load CS:IP and IPQ
  db 0EAh
  dw $+4,KernelSel_EQU

  ; Load segment registers with valid selectors
  mov ax,Data32Sel_EQU
  mov ds,ax
  mov es,ax
  mov fs,ax
  mov gs,ax
  mov ss,ax

  ; Process interrupt
  db 0CDh
  pm_Interrupt_Index db ?

  cli                       ; disable interrupts

  ; Load 64Kb limit Stack selector
  push w Stack16Sel_EQU
  pop ss

  ; Load Real Mode IDT
  lidt m rm_IDT

  ; Switch to Real Mode
  mov eax,cr0
  and al,NOT 1
  mov cr0,eax

  ; Load CS:IP
  db 0EAh
  dw $+4,Kernel

  ; Calculate real mode SS:SP for stack pointer
  mov eax,m d rm_SS
  mov ss,ax
  shl eax,4
  add esp,m Code32addr
  sub esp,eax

  ; Restore registers
  pop eax gs fs es ds
  popfd

  ; Remove interrupt number from stack (without affecting flags)
  mov [esp],ax
  pop ax

  ; Return from interrupt
  iret


;Ŀ;
; RAW/XMS protected mode initialization                                     ;
;;
RAWXMS_Init:

  ; Remap PIC
  call Remap_PIC

  ; Setup Real Mode interrupt table
  call Setup_rm_Int

  ; Build IDT
  call Build_IDT

  ; Load GDT
  lgdt m pm_GDT

  ; Load IDT
  lidt m pm_IDT

  ; Enter Protected Mode
  mov eax,cr0
  or al,1
  mov cr0,eax

  ; Jump to Code32 segment
  db 66h, 0EAh
  dd o Enter32
  dw Code32Sel_EQU


;Ŀ;
; Unlock and De-Allocate entended memory (XMS)                              ;
;;
Free_XMS:

  ; Unlock block
  mov dx,m XMS_Handle
  mov ah,0Dh
  call m XMS_Driver

  ; Deallocate block
  mov dx,m XMS_Handle
  mov ah,0Ah
  call m XMS_Driver

  ret


;Ŀ;
; Exit Program and Return to OS                                             ;
;;
Exit16:
  ; Set DS to Code32
  push w Code32
  pop ds

  ; Restore IDT
  lidt m r0_IDT

  ; Enable Interrupts
  sti

  ; Deallocate High Memory
DeallocHimemAddr LABEL
  call Free_XMS
  nop

  ; Restore Real Mode interrupts
  push w 0
  pop es
  mov eax,CS:Old_Int21h
  mov ES:[21h*4],eax

  ; Send EOI
  mov al,20h
  out 020h,al
  out 0A0h,al

  ; Read extra scan codes
  in al,60h

  ; Set PIT to 18.2Hz
  mov al,36h
  out 43h,al
  DELAY
  xor al,al
  out 40h,al
  DELAY
  out 40h,al
  DELAY

  ; Call DOS Terminate with Exit 0
  mov ax,4C00h
  int 21h

;*****************************************************************************;
Kernel    ENDS


;*****************************************************************************;
Code32    SEGMENT
          ASSUME CS:Code32, DS:Code32, SS:Code32


;Ŀ;
; Error messages                                                            ;
;;
RAWXMS_GenErrorMSG db 'Could not go to protected mode.',0Ah,24h


;Ŀ;
; Interrupt Descriptor Table                                                ;
;;
; Reference to Interrupt Descriptor Table upon entry
r0_IDT            LABEL   FWORD
r0_IDTsize        dw      256*4h
r0_IDTaddr        dd      0

; Reference to Real Mode Interrupt Descriptor Table
rm_IDT            LABEL   FWORD
                  dw      256*4h
rm_IDTaddr        dd      o rmIDT

; Reference to Protected Mode Interrupt Descriptor Table
pm_IDT            LABEL   FWORD
                  dw      256*8h
pm_IDTaddr        dd      o pmIDT


;Ŀ;
; XMS variables                                                             ;
;;
; XMS information
XMS_Driver        dd      0
XMS_Handle        dw      0


;Ŀ;
; Stack switching registers                                                 ;
;;

rm_SS             dw      0, 0  ; Real mode SS

v_ESP             dd      0     ; virtual ESP


;Ŀ;
; Get protected mode interrupt descriptor                                   ;
; Entry:  BL - Interrupt vector                                             ;
; Exit :  AX:EDX - Selector:offset of interrupt vector                      ;
;;
RAWXMS_Get_pm_Int:
  push edi
  movzx edi,bl
  shl edi,3
  add edi,m pm_IDTaddr
  sub edi,m Code32addr
  mov dx,w [edi+6]
  shl edx,16
  mov dx,w [edi]
  mov ax,w [edi+2]
  pop edi
  ret


;Ŀ;
; Set protected mode interrupt descriptor                                   ;
; Entry:  BL - Interrupt Vector                                             ;
;         AX:EDX - Selector:offset of interrupt vector                      ;
;;
RAWXMS_Set_pm_Int:
  pushad
  movzx edi,bl
  shl edi,3
  add edi,m pm_IDTaddr
  sub edi,m Code32addr
  cli
  mov w [edi],dx
  shr edx,16
  mov w [edi+6],dx
  mov w [edi+2],ax
  sti
  popad
  ret


;*****************************************************************************;
Code32    ENDS


;ͻ;
; Interrupt Descriptor Table segment                                        ;
;ͼ;
IDTSeg    SEGMENT

pmIDT     db 256*8    dup (?)
pmISR     db 256*10   dup (?)

rmIDT     db 256*4    dup (?)
rmISR     db 256*8    dup (?)

IDTSeg    ENDS
;ͻ;
; Interrupt Descriptor Table segment ends                                   ;
;ͼ;
