/* lzo_test.c -- test driver for the LZO library

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <assert.h>
#if defined(__DJGPP__) || defined(__BORLANDC__)
#  include <dir.h>
#endif
#if defined(__DJGPP__) || defined(__linux__)
#  include <unistd.h>
#endif

#include <lzoconf.h>

#include <time.h>
#if defined(UCLOCKS_PER_SEC)
#  undef clock
#  undef clock_t
#  undef CLOCKS_PER_SEC
#  define clock()			uclock()
#  define clock_t			uclock_t
#  define CLOCKS_PER_SEC	UCLOCKS_PER_SEC
#endif


/*************************************************************************
//
**************************************************************************/

#if defined(ONLY_LZO1C_1)
#define HAVE_LZO1C_H
#else
#define HAVE_LZO1_H
#define HAVE_LZO1A_H
#define HAVE_LZO1B_H
#define HAVE_LZO1C_H
#define HAVE_LZO1D_H
#define HAVE_LZO1F_H
#define HAVE_LZO1X_H
#endif

#if defined(MFX) && !defined(ONLY_LZO1C_1)
#define HAVE_LZRW_H
#define HAVE_ZLIB_H
#endif

#if 0
#define HAVE_LZV_H
#define HAVE_LZRW1O_H
#define HAVE_PRED_H
#endif



/*************************************************************************
// define all methods
**************************************************************************/

enum {
	M_LZO1B_1  =     1,
	M_LZO1B_2, M_LZO1B_3, M_LZO1B_4, M_LZO1B_5,
	M_LZO1B_6, M_LZO1B_7, M_LZO1B_8, M_LZO1B_9,

	M_LZO1C_1  =    11,
	M_LZO1C_2, M_LZO1C_3, M_LZO1C_4, M_LZO1C_5,
	M_LZO1C_6, M_LZO1C_7, M_LZO1C_8, M_LZO1C_9,

	M_LZO1     =    21,
	M_LZO1A    =    31,

	M_LZO1B_99 =   901,
	M_LZO1C_99 =   911,
	M_LZO1C_999=   912,
	M_LZO1_99  =   921,
	M_LZO1A_99 =   931,

	M_LZO1D_999=   942,
	M_LZO1E_1  =    51,
	M_LZO1F_1  =    61,
	M_LZO1F_999=   962,
	M_LZO1X_1  =    71,
	M_LZO1X_999=   972,
	M_LZO4     =   401,

	M_LZV      =  1001,
	M_PRED     =  1002,

	M_LZRW1A   =  1010,
	M_LZRW1O   =  1011,
	M_LZRW2    =  1020,
	M_LZRW3    =  1030,
	M_LZRW3A_2 =  1032,
	M_LZRW3A_4 =  1034,
	M_LZRW3A_8 =  1038, M_LZRW3A = M_LZRW3A_8,

	M_ZLIB_8_1 =  1101,
	M_ZLIB_8_2, M_ZLIB_8_3, M_ZLIB_8_4, M_ZLIB_8_5,
	M_ZLIB_8_6, M_ZLIB_8_7, M_ZLIB_8_8, M_ZLIB_8_9,

	M_MEMSET   =   998,
	M_MEMCPY   =   999,

	M_UNUSED
};



#if defined(__LZO_MSDOS16)
#define M_LZO1_99		(-1)
#define M_LZO1A_99		(-1)
#define M_LZO1B_99		(-1)
#define M_LZO1C_99		(-1)
#define M_LZO1C_999		(-1)
#define M_LZO1D_999		(-1)
#define M_LZO1F_999		(-1)
#define M_LZO1X_999		(-1)

#undef HAVE_LZO1_H
#undef HAVE_LZO1A_H
#if !defined(ONLY_LZO1C_1)
#undef HAVE_LZO1C_H
#endif
#undef HAVE_LZO1D_H

#undef HAVE_LZV_H
#undef HAVE_LZRW_H
#undef HAVE_ZLIB_H
#endif



/* LZO algorithms */
#if defined(HAVE_LZO1_H)
#  include <lzo1.h>
#endif
#if defined(HAVE_LZO1A_H)
#  include <lzo1a.h>
#endif
#if defined(HAVE_LZO1B_H)
#  include <lzo1b.h>
#endif
#if defined(HAVE_LZO1C_H)
#  include <lzo1c.h>
#endif
#if defined(HAVE_LZO1D_H)
#  include <lzo1d.h>
#endif
#if defined(HAVE_LZO1E_H)
#  include <lzo1e.h>
#endif
#if defined(HAVE_LZO1F_H)
#  include <lzo1f.h>
#endif
#if defined(HAVE_LZO1X_H)
#  include <lzo1x.h>
#endif

/* other real-time compressors */
#if defined(HAVE_LZRW_H)
#  include <lzrw.h>
#endif
#if defined(HAVE_LZRW1O_H)
#  include "contrib/lzrw1o/lzrw1o.h"
#endif
#if defined(HAVE_LZV_H)
#  include "contrib/lzv/lzv.h"
#endif
#if defined(HAVE_PRED_H)
#  include <pred.h>
#endif

/* other compressors */
#if defined(HAVE_ZLIB_H)
#  include <zlib.h>
#endif


/*************************************************************************
//
**************************************************************************/

#if 1
   /* always use the fast decompressors */
#  define lzo1b_decompress_x	lzo1b_decompress
#  define lzo1c_decompress_x	lzo1c_decompress
#  define lzo1d_decompress_x	lzo1d_decompress
#  define lzo1f_decompress_x	lzo1f_decompress
#  define lzo1x_decompress_x	lzo1x_decompress
#endif


#if defined(HAVE_LZO1X_H)
int method = M_LZO1X_1;
#elif defined(HAVE_LZO1B_H)
int method = M_LZO1B_1;
#else
int method = M_LZO1C_1;
#endif

/* set these to 1 to measure the speed impact of a checksum */
int compute_adler32 = 0;
int compute_crc32 = 0;


/*************************************************************************
//
**************************************************************************/

#if (UINT_MAX >= 0xffffffffL)		/* 32 bit or more */

#define BLOCK_SIZE		(256*1024l)
#define MAX_BLOCK_SIZE	(4*256*1024l)
#define DATA_LEN		MAX_BLOCK_SIZE
#define WORK_LEN		(512*1024l)

#else								/* 16 bit compiler, oh my god */

#if 0
#define BLOCK_SIZE		(256*1024l)
#define MAX_BLOCK_SIZE	(256*1024l)
#define DATA_LEN		(128*1024l)
#else
#define BLOCK_SIZE		32000u
#define MAX_BLOCK_SIZE	32000u
#define DATA_LEN		(60*1024l)
#endif
#define WORK_LEN		(64*1024l)

#endif


static lzo_uint block_size = BLOCK_SIZE;

static lzo_byte _block1[MAX_BLOCK_SIZE * 9L / 8 + 1024 + 256];
static lzo_byte _block2[MAX_BLOCK_SIZE * 9L / 8 + 1024 + 256];
static lzo_byte _data[DATA_LEN + 256];
static lzo_byte _wrkmem[WORK_LEN + 256];

/* align memory blocks (cache issues) */
static lzo_byte *block1 = NULL;
static lzo_byte *block2 = NULL;
static lzo_byte *data = NULL;
static lzo_byte *wrkmem = NULL;

static void align_mem(void)
{
	block1 = (lzo_byte *) LZO_ALIGN(_block1,256);
	block2 = (lzo_byte *) LZO_ALIGN(_block2,256);
	data   = (lzo_byte *) LZO_ALIGN(_data,256);
	wrkmem = (lzo_byte *) LZO_ALIGN(_wrkmem,256);
}


/***********************************************************************
// read from a file
************************************************************************/

lzo_uint my_fread(FILE *f, lzo_voidp s, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return fread(s,1,len,f);
#else
	/* seems to work with len > 64k with Borland C 4.0 */
	lzo_byte *p = (lzo_byte *) s;
	lzo_uint l = 0, k;

	while (l < len)
	{
		k = len - l > 32768u ? 32768u : len - l;
		k = fread(p,1,(unsigned)k,f);
		if (k <= 0)
			break;
		p += k;
		l += k;
	}
	return l;
#endif
}


/*************************************************************************
// compress a block
**************************************************************************/

static int
do_compress(const lzo_byte *in, lzo_byte *out,
                  lzo_uint in_len, lzo_uint *out_len)
{
	int r = -1;
	unsigned long o_len = ULONG_MAX;

	if (method < 0)
		;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1)
		r = lzo1_compress(in,in_len,out,out_len,wrkmem);
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1_99)
		r = lzo1_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A)
		r = lzo1a_compress(in,in_len,out,out_len,wrkmem);
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1A_99)
		r = lzo1a_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
	{
		int clevel = method - M_LZO1B_1 + 1;
		r = lzo1b_compress(in,in_len,out,out_len,wrkmem,clevel);
	}
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1B_99)
		r = lzo1b_99_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1C_H)
#if defined(ONLY_LZO1C_1)
	else if (method == M_LZO1C_1)
		r = lzo1c_1_compress(in,in_len,out,out_len,wrkmem);
#else
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
	{
		int clevel = method - M_LZO1C_1 + 1;
		r = lzo1c_compress(in,in_len,out,out_len,wrkmem,clevel);
	}
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1C_99)
		r = lzo1c_99_compress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1C_999)
		r = lzo1c_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#endif
#if defined(HAVE_LZO1D_H)
	else if (method == M_LZO1D_999)
		r = lzo1d_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1E_H)
	else if (method == M_LZO1E)
		r = lzo1e_1_compress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1F_H)
	else if (method == M_LZO1F_1)
		r = lzo1f_1_compress(in,in_len,out,out_len,wrkmem);
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1F_999)
		r = lzo1f_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1X_H)
	else if (method == M_LZO1X_1)
		r = lzo1x_1_compress(in,in_len,out,out_len,wrkmem);
#if (UINT_MAX >= 0xffffffffL)
	else if (method == M_LZO1X_999)
		r = lzo1x_999_compress(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZRW_H)
	else if (method == M_LZRW1A)
		r = lzrw1a_compress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW2)
		r = lzrw2_compress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3)
		r = lzrw3_compress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_2)
		r = lzrw3a2_compress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_4)
		r = lzrw3a4_compress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_8)
		r = lzrw3a8_compress(wrkmem,in,in_len,out,&o_len);
#endif
#if defined(HAVE_LZRW1O_H)
	else if (method == M_LZRW1O)
		r = lzrw1o_compress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZV_H)
	else if (method == M_LZV)
		r = lzv_compress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_PRED_H)
	else if (method == M_PRED)
	{
		ipred(wrkmem);
		r = wpred(in,out,in_len,out_len);
	}
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
	{
		/* use the undocumented feature to suppress the zlib header */
		z_stream stream;
		int err;
		int level = method - M_ZLIB_8_1 + 1;

		stream.next_in = (lzo_byte *) in;		/* UNCONST */
		stream.avail_in = in_len;
		stream.next_out = out;
		stream.avail_out = *out_len;

		stream.zalloc = (alloc_func)0;
		stream.zfree = (free_func)0;
		*out_len = 0;
#if 0
		err = deflateInit(&stream, level);
#else
		err = deflateInit2(&stream, level, Z_DEFLATED, -MAX_WBITS,
		                   MAX_MEM_LEVEL > 8 ? 8 : MAX_MEM_LEVEL,
		                   Z_DEFAULT_STRATEGY);
#endif
		if (err == Z_OK)
		{
			err = deflate(&stream, Z_FINISH);
			if (err != Z_STREAM_END)
			{
				deflateEnd(&stream);
				err = err == Z_OK ? Z_BUF_ERROR : err;
			}
			else
			{
				*out_len = (lzo_uint) stream.total_out;
				err = deflateEnd(&stream);
			}
		}
		r = err == Z_OK ? 0 : -1;
	}
#endif
	else if (method == M_MEMCPY)
	{
		lzo_memcpy(out,in,in_len);
		*out_len = in_len;
		r = 0;
	}

	if (o_len != ULONG_MAX)
		*out_len = (lzo_uint) o_len;

	if (r == 0 && compute_adler32)
	{
		lzo_uint adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, in, in_len);
	}
#if defined(HAVE_ZLIB_H)
	if (r == 0 && compute_crc32)
	{
		uLong crc;
		crc = crc32(0L, Z_NULL, 0);
		crc = crc32(crc, in, in_len);
	}
#endif

	return r;
}


/*************************************************************************
// decompress a block
**************************************************************************/

static int 
do_decompress(const lzo_byte *in, lzo_byte *out,
              lzo_uint in_len, lzo_uint *out_len)
{
	int r = -1;
	unsigned long o_len = ULONG_MAX;

	if (method < 0)
		;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1 || method == M_LZO1_99)
		r = lzo1_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A || method == M_LZO1A_99)
		r = lzo1a_decompress(in,in_len,out,out_len,NULL);
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
		r = lzo1b_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1B_99)
		r = lzo1b_decompress_x(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1C_H)
#if defined(ONLY_LZO1C_1)
	else if (method == M_LZO1C_1)
		r = lzo1c_decompress(in,in_len,out,out_len,wrkmem);
#else
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
		r = lzo1c_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1C_99 || method == M_LZO1C_999)
		r = lzo1c_decompress_x(in,in_len,out,out_len,wrkmem);
#endif
#endif
#if defined(HAVE_LZO1D_H)
	else if (method == M_LZO1D_999)
		r = lzo1d_decompress_x(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1E_H)
	else if (method == M_LZO1E)
		r = lzo1e_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1F_H)
	else if (method == M_LZO1F_1)
		r = lzo1f_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1F_999)
		r = lzo1f_decompress_x(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZO1X_H)
	else if (method == M_LZO1X_1)
		r = lzo1x_decompress(in,in_len,out,out_len,wrkmem);
	else if (method == M_LZO1X_999)
		r = lzo1x_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZRW_H)
	else if (method == M_LZRW1A)
		r = lzrw1a_decompress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW2)
		r = lzrw2_decompress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3)
		r = lzrw3_decompress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_2)
		r = lzrw3a2_decompress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_4)
		r = lzrw3a4_decompress(wrkmem,in,in_len,out,&o_len);
	else if (method == M_LZRW3A_8)
		r = lzrw3a8_decompress(wrkmem,in,in_len,out,&o_len);
#endif
#if defined(HAVE_LZRW1O_H)
	else if (method == M_LZRW1O)
		r = lzrw1o_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_LZV_H)
	else if (method == M_LZV)
		r = lzv_decompress(in,in_len,out,out_len,wrkmem);
#endif
#if defined(HAVE_PRED_H)
	else if (method == M_PRED)
	{
		ipred(wrkmem);
		*out_len = rpred(in,out,in_len,*out_len);
		r = 0;
	}
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
	{
		/* use the undocumented feature to suppress the zlib header */
		z_stream stream;
		int err;

		stream.next_in = (lzo_byte *) in;		/* UNCONST */
		stream.avail_in = in_len;
		stream.next_out = out;
		stream.avail_out = *out_len;

		stream.zalloc = (alloc_func)0;
		stream.zfree = (free_func)0;
		*out_len = 0;
#if 0
		err = inflateInit(&stream);
#else
		err = inflateInit2(&stream, -MAX_WBITS);
#endif
		if (err == Z_OK)
		{
			err = inflate(&stream, Z_FINISH);
			if (err == Z_OK)		/* sometimes returns Z_OK !!! */
			{
				*out_len = (lzo_uint) stream.total_out;
				err = inflateEnd(&stream);
			}
			else if (err == Z_STREAM_END)
			{
				*out_len = (lzo_uint) stream.total_out;
				err = inflateEnd(&stream);
			}
			else
				inflateEnd(&stream);
		}
		r = err == Z_OK ? 0 : -1;
	}
#endif
	else if (method == M_MEMCPY)
	{
		lzo_memcpy(out,in,in_len);
		*out_len = in_len;
		r = 0;
	}

	if (o_len != ULONG_MAX)
		*out_len = (lzo_uint) o_len;

	if (r == 0 && compute_adler32)
	{
		lzo_uint adler;
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, out, *out_len);
	}
#if defined(HAVE_ZLIB_H)
	if (r == 0 && compute_crc32)
	{
		uLong crc;
		crc = crc32(0L, Z_NULL, 0);
		crc = crc32(crc, out, *out_len);
	}
#endif

	return r;
}


/*************************************************************************
// method info
**************************************************************************/

#define FPRINTF		(f == NULL) ? ((void) 0) : (void) fprintf

static int info(FILE *f)
{
	if (method < 0)
		return 0;
#if defined(HAVE_LZO1_H)
	else if (method == M_LZO1)
	{
		int rbits, clevel;
		lzo1_info(&rbits,&clevel);
		FPRINTF(f,"LZO1-%d",clevel);
	}
	else if (method == M_LZO1_99)
		FPRINTF(f,"LZO1-99");
#endif
#if defined(HAVE_LZO1A_H)
	else if (method == M_LZO1A)
	{
		int rbits, clevel;
		lzo1a_info(&rbits,&clevel);
		FPRINTF(f,"LZO1A-%d",clevel);
	}
	else if (method == M_LZO1A_99)
		FPRINTF(f,"LZO1A-99");
#endif
#if defined(HAVE_LZO1B_H)
	else if (method >= M_LZO1B_1 && method <= M_LZO1B_9)
	{
		int clevel = method - M_LZO1B_1 + 1;
		FPRINTF(f,"LZO1B-%d",clevel);
	}
	else if (method == M_LZO1B_99)
		FPRINTF(f,"LZO1B-99");
#endif
#if defined(HAVE_LZO1C_H)
#if defined(ONLY_LZO1C_1)
	else if (method == M_LZO1C_1)
		FPRINTF(f,"LZO1C-1");
#else
	else if (method >= M_LZO1C_1 && method <= M_LZO1C_9)
	{
		int clevel = method - M_LZO1C_1 + 1;
		FPRINTF(f,"LZO1C-%d",clevel);
	}
	else if (method == M_LZO1C_99)
		FPRINTF(f,"LZO1C-99");
	else if (method == M_LZO1C_999)
		FPRINTF(f,"LZO1C-999");
#endif
#endif
#if defined(HAVE_LZO1D_H)
	else if (method == M_LZO1D_999)
		FPRINTF(f,"LZO1D-999");
#endif
#if defined(HAVE_LZO1E_H)
	else if (method == M_LZO1E)
		FPRINTF(f,"LZO1E-1");
#endif
#if defined(HAVE_LZO1F_H)
	else if (method == M_LZO1F_1)
		FPRINTF(f,"LZO1F-1");
	else if (method == M_LZO1F_999)
		FPRINTF(f,"LZO1F-999");
#endif
#if defined(HAVE_LZO1X_H)
	else if (method == M_LZO1X_1)
		FPRINTF(f,"LZO1X-1");
	else if (method == M_LZO1X_999)
		FPRINTF(f,"LZO1X-999");
#endif
#if defined(HAVE_LZRW_H)
	else if (method == M_LZRW1A)
		FPRINTF(f,"LZRW1-A");
	else if (method == M_LZRW2)
		FPRINTF(f,"LZRW2");
	else if (method == M_LZRW3)
		FPRINTF(f,"LZRW3");
	else if (method == M_LZRW3A_2)
		FPRINTF(f,"LZRW3-A(2)");
	else if (method == M_LZRW3A_4)
		FPRINTF(f,"LZRW3-A(4)");
	else if (method == M_LZRW3A_8)
		FPRINTF(f,"LZRW3-A(8)");
#endif
#if defined(HAVE_LZRW1O_H)
	else if (method == M_LZRW1O)
		FPRINTF(f,"LZRW1-O");
#endif
#if defined(HAVE_LZV_H)
	else if (method == M_LZV)
		FPRINTF(f,"LZV");
#endif
#if defined(HAVE_PRED_H)
	else if (method == M_PRED)
		FPRINTF(f,"PRED");
#endif
#if defined(HAVE_ZLIB_H)
	else if (method >= M_ZLIB_8_1 && method <= M_ZLIB_8_9)
		FPRINTF(f,"zlib-%d/%d", Z_DEFLATED, method - M_ZLIB_8_1 + 1);
#endif
	else if (method == M_MEMCPY)
		FPRINTF(f,"memcpy()");
	else if (f == NULL)
		return 0;
	else
	{
		FPRINTF(f,"invalid method %d !\n",method);
		exit(1);
	}

	return 1;
}

#undef FPRINTF


/*************************************************************************
// compress and decompress a file
**************************************************************************/

static int do_file(const char *name, int c_loops, int d_loops, lzo_uint *ad)
{
	FILE *f;
	int i, r = 0;
	lzo_uint l = 0, t_len = 0, bytes = 0;
	int blocks = 0;
	clock_t t_time = 0, c_time = 0, d_time = 0;
	clock_t t_start, c_start, d_start;
	double secs, kbs;
	double perc, bits, c_kbs, d_kbs;
	lzo_uint adler;
	int loops = 1;
	int try_to_compress_0_bytes = 0;

	if (c_loops < 1)  c_loops = 1;
	if (d_loops < 1)  d_loops = 1;


	/* read the whole file */
	f = fopen(name,"rb");
	if (f == NULL)
	{
		fprintf(stderr,"%s: ",name);
		perror("fopen");
		return 0;
	}
	l = my_fread(f,data,DATA_LEN);
	if (fclose(f) != 0)
	{
		fprintf(stderr,"%s: ",name);
		perror("fclose");
		return -1;
	}

	adler = lzo_adler32(0, NULL, 0);
	adler = lzo_adler32(adler, data, l);
	if (ad)
		*ad = adler;

	printf("File %s: %lu bytes   adler32 = 0x%08lx\n", 
		name, (long) l, (long) adler);

	printf("  compressing %lu bytes (%d/%d/%d loops, %lu block-size)\n",
		(long) l, loops, c_loops, d_loops, (long) block_size);
	printf("  "); info(stdout); printf("\n");
	fflush(stdout);

	/* compress the file */

	t_start = clock();
	for (i = 0; i < loops; i++)
	{
		lzo_uint ll, c_len, c_len_max, d_len;
		lzo_byte *d = data;

		ll = l;
		t_len = 0;
		blocks = 0;
		if (ll > 0 || try_to_compress_0_bytes) do
		{
			lzo_byte *dd = d;
			lzo_byte *b1 = block1;
			lzo_byte *b2 = block2;
			const lzo_uint bl = ll > block_size ? block_size : ll;
			int j;

			blocks++;

#if defined(__BOUNDS_CHECKING_ON)
			/* malloc a block of the exact block size to detect
			 * any overun.
			 */
			dd = malloc(bl);
			if (dd == NULL)
				return -1;
			memcpy(dd,d,bl);
			b2 = dd;
#endif

		/* compress a block */
			c_len = c_len_max = 0;
			c_start = clock();
			for (j = r = 0; r == 0 && j < c_loops; j++)
			{
				c_len = sizeof(_block1) - 256;
				r = do_compress(dd,b1,bl,&c_len);
				if (r == 0 && c_len > c_len_max)
					c_len_max = c_len;
			}
			c_time += clock() - c_start;
			if (r != 0)
			{
				printf("  compression failed in block %d (%d) (%lu %lu)\n",
					blocks, r, (long)c_len, (long)bl);
				return r;
			}

		/* decompress the block and verify */
			lzo_memset(b2,0,bl);
			d_start = clock();
			for (j = r = 0; r == 0 && j < d_loops; j++)
			{
				d_len = bl;
				r = do_decompress(b1,b2,c_len,&d_len);
			}
			d_time += clock() - d_start;
			if (r != 0)
			{
				printf("  decompression failed in block %d (%d) "
					"(%lu %lu %lu)\n", blocks, r,
					(long)c_len, (long)d_len, (long)bl);
				return r;
			}
			if (d_len != bl)
			{
				printf("  decompression size error in block %d (%lu %lu %lu)\n",
					blocks, (long)c_len, (long)d_len, (long)bl);
				return -1;
			}
			if (lzo_memcmp(d,b2,bl) != 0)
			{
				lzo_uint x = 0;
				while (x < bl && d[x] == b2[x])
					x++;
				printf("  decompression data error in block %d at offset "
					"%lu (%lu %lu)\n", blocks, (long)x, (long)c_len,
					(long)d_len);
				return -1;
			}

#if defined(__BOUNDS_CHECKING_ON)
			if (dd != d)
				free(dd);
#endif
			d += bl;
			ll -= bl;
			t_len += c_len_max;

#if 0
			/* LZRW: add 8 bytes overhead per block to get
			 * the same values as published by Ross Williams
			 */
			if (method >= M_LZRW1A && method <= M_LZRW3A_8)
				t_len += 8;
#endif
		}
		while (ll > 0);
	}
	t_time += clock() - t_start;


	/* print some statistics */

	perc = (l > 0) ? t_len * 100.0 / l : 0;
	bits = perc * 0.08;
	printf("  compressed into %ld bytes, %.2f%%   %.3f\n",
		(long) t_len, perc, bits);

	secs = t_time / (double)(CLOCKS_PER_SEC);
	bytes = (l * c_loops + l * d_loops) * loops;
	kbs = (secs > 0.001) ? bytes / secs / 1000 : 0;
	printf("%-15s %5d: ","overall",loops);
	printf("%10lu bytes, %8.2f secs, %8.2f kB/sec\n", (long)bytes, secs, kbs);

	secs = c_time / (double)(CLOCKS_PER_SEC);
	bytes = l * c_loops * loops;
	c_kbs = (secs > 0.001) ? bytes / secs / 1000 : 0;
	printf("%-15s %5d: ","compress",c_loops);
	printf("%10lu bytes, %8.2f secs, %8.2f kB/sec\n", (long)bytes, secs, c_kbs);

	secs = d_time / (double)(CLOCKS_PER_SEC);
	bytes = l * d_loops * loops;
	d_kbs = (secs > 0.001) ? bytes / secs / 1000 : 0;
	printf("%-15s %5d: ","decompress",d_loops);
	printf("%10lu bytes, %8.2f secs, %8.2f kB/sec\n", (long)bytes, secs, d_kbs);


	/* create a line for util/table.pl */

#if defined(__DJGPP__) || defined(__EMX__) || defined(__linux__) || \
    defined(__BORLANDC__) || defined(__WATCOMC__)
	{
#if defined(__DJGPP__) || defined(__BORLANDC__)
		char n[256], ext[80]; fnsplit(name,NULL,NULL,n,ext); strcat(n,ext);
#elif defined(__EMX__) || defined(__WATCOMC__)
		char n[256], ext[80]; _splitpath(name,NULL,NULL,n,ext); strcat(n,ext);
#elif defined(__linux__)
		const char *n = basename(name);
#endif
		printf("\n"); info(stdout);
		printf(" | %-14s %7ld %4d %7ld %6.1f %5.2f %8.2f %8.2f |\n",
			n, (long) l, blocks, (long) t_len, perc, bits, c_kbs, d_kbs);
		printf("\n");
	}
#endif

	return 0;
}


/*************************************************************************
// Calgary Corpus test suite driver
**************************************************************************/

static struct 
{
	const char *name;
	int loops;
	lzo_uint adler;
}
corpus[] =
{
	{ "bib",       7,  0x4bd09e98L },  
	{ "book1",     1,  0xd4d3613eL },
	{ "book2",     1,  0x6fe14cc3L },
	{ "geo",       6,  0xf3cc5be0L },
	{ "news",      2,  0x2ed405b8L },
	{ "obj1",     25,  0x3887dd2cL },
	{ "obj2",      5,  0xf89407c4L },
	{ "paper1",   13,  0xfe65ce62L },
	{ "paper2",   10,  0x1238b7c2L },
	{ "pic",       4,  0xf61a5702L },
	{ "progc",    20,  0x4c00ba45L },
	{ "progl",    20,  0x4cba738eL },
	{ "progp",    20,  0x7495b92bL },
	{ "trans",    10,  0x52a2cec8L }
};


static int do_corpus(const char *path, int c_loops, int d_loops)
{
	int i, n;
	char name[256];

	if (path == NULL)
		return -1;

	strcpy(name,path);
	n = strlen(name);
	if (n > 0 && name[n-1] != '/' && name[n-1] != '\\')
	{
		strcat(name,"/");
		n++;
	}

	for (i = 0; i < (int) (sizeof(corpus)/sizeof(*(corpus))); i++)
	{
		lzo_uint adler;
		int c = c_loops * corpus[i].loops;
		int d = d_loops * corpus[i].loops;
		int r;

		strcpy(name+n,corpus[i].name);
		r = do_file(name, c, d, &adler);
		if (r != 0)
			return r;
		if (adler != corpus[i].adler)
			return -1;
	}
	return 0;
}


/*************************************************************************
// 
**************************************************************************/

static void usage(const char *name, int show_methods)
{
	FILE *f;
	int i;

	f = stdout;

	fprintf(f,"Usage: %s [-m#] [-b#] [-n#] files...\n", name);
	fprintf(f,"\n");
	fprintf(f,"Options:\n");
	fprintf(f,"  -m#     compression method\n");
	fprintf(f,"  -b#     set input block size (default %ld)\n",
		(long) block_size);
	fprintf(f,"  -n#     number of compression/decompression runs\n");
	fprintf(f,"  -c#     number of compression runs\n");
	fprintf(f,"  -d#     number of decompression runs\n");

	if (show_methods)
	{
		fprintf(f,"\n");
		fprintf(f,"The following compression methods are available:\n");
		fprintf(f,"\n");

		for (i = 0; i < 2000; i++)
		{
			method = i;
			if (info(NULL))
			{
				char n[16];
				sprintf(n,"-m%d",i);
				fprintf(f,"  %-6s  ",n);
				info(f);
				fprintf(f,"\n");
			}
		}
	}
	else
	{
		fprintf(f,"\n");
		fprintf(f,"Type '%s -m' to list all available methods.\n", name);
	}

	exit(1);
}


/*************************************************************************
//
**************************************************************************/

int main(int argc, char *argv[])
{
	int i = 1;
	int c_loops = 0;
	int d_loops = 0;
	int r = LZO_E_OK;
	const char *corpus_path = NULL;

#if defined(__EMX__)
	_response(&argc,&argv);
	_wildcard(&argc,&argv);
#endif

	align_mem();

	printf("\nLZO real-time data compression library (v%s, %s).\n",
			LZO_VERSION_STRING, LZO_VERSION_DATE);
	printf("Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer\n\n");

	if (lzo_init() != LZO_E_OK)
	{
		printf("lzo_init() failed !!!\n");
		return 1;
	}

	if (argc < 2)
		usage(argv[0],0);
	
	if (i < argc && isdigit(argv[i][0]) && argv[i][0] > '0')
		method = atoi(&argv[i++][0]);

	for ( ; i < argc && argv[i][0] == '-'; i++)
	{
		char *p = argv[i];

		if (isdigit(p[1]))
			method = atoi(p+1);
		else if (strncmp(p,"-m",2) == 0)
		{
			if (!isdigit(p[2]))
				usage(argv[0],1);
			method = atoi(p+2);
		}
		else if (strcmp(p,"-a") == 0)
			compute_adler32 = 1;
		else if (strncmp(p,"-b",2) == 0 && isdigit(p[2]))
			block_size = atoi(p+2);
		else if (strncmp(p,"-c",2) == 0 && isdigit(p[2]))
			c_loops = atoi(p+2);
		else if (strncmp(p,"-d",2) == 0 && isdigit(p[2]))
			d_loops = atoi(p+2);
		else if (strncmp(p,"-n",2) == 0 && isdigit(p[2]))
			c_loops = atoi(p+2), d_loops = c_loops;
		else if (strncmp(p,"-s",2) == 0)
		{
			if (i + 1 < argc)
			{
				if (isdigit(p[2]))
					c_loops = atoi(p+2), d_loops = c_loops;
				corpus_path = argv[++i];
			}
		}
		else
			usage(argv[0],0);
	}

	if (block_size < 16)
		block_size = 16;
	if (block_size > MAX_BLOCK_SIZE)
		block_size = MAX_BLOCK_SIZE;


	info(stdout); printf(" algorithm\n\n");

	if (corpus_path != NULL)
		r = do_corpus(corpus_path,c_loops,d_loops);
	else
	{
		for ( ; i < argc && r == LZO_E_OK; i++)
			r = do_file(argv[i],c_loops,d_loops,NULL);
	}

	return r == LZO_E_OK ? 0 : 1;
}

/*
vi:ts=4
*/

