// These packing routines are based on the LZHUF.C program by
// Haruyasu Yoshizaki. Copyright (C) 1995 Christian Worm.

/* KONFIGURATION
   Dette program kan bruges til at danne en statisk huffman tabel som
   hurtigt kan bruges i et program via tabel opslag.

   Man skal ved kreationen af selve tabellen angive to ting.
   Dels skal man i MAXBITS skrive hvor mange bits en kodet vrdi hjst kan
   best af (max 16). Den anden "vrdi" er lidt tricky. Forklaring flger:

   Level                                  Freq
     0               A               0
                   /   \
     1           B       C           1
               /   \
     2        D      E               1
                  /   \
     3           F      G            2

   I dette tr vil MAXBITS vre 3, da det er den maksimale vrdi vi kan
   f. Det skal bemrkes at dette tr indeholder 4 levels - alts "linier".
   For hvert level skal man angive hvor mange frie noder der skal vre
   (C,D,F og G er frie node). Det er den vrdi der i hjre side er angivet
   under freq. Det er lige prcist disse vrdier der ogs skal angives

   Bemrk:
   freq[0]/2^0 + freq[1]/2^1 + ... + freq[MAXBITS]/2^(MAXBITS) skal vre = 1.

   Konfigurationen af ovenstende vil alts vre:

   MAXBITS 3
   char freq[MAXBITS+1]={0,1,1,2};

   Bemrk at vrdien FREQ_SUM skal sttes til summen af alle freq's
   elementer - i dette til flde

   FREQ_SUM 4

   I de tabeller der kreeres i PACK_OUTFILE/UNPACK_OUTFILE kan flgende
   angives for layout:

   NUMS_PR_LINE angiver hvor numre der skal st pr linie
   NUM_WITH     angiver hvor mange karaktere hvert nummer skal best af
*/

/* BRUG AF TABELLER
   Der bliver kreeret 4 tabeller:

   huff_pack_code[];

   Man angiver som parameter et tal i intervallet [0;FREQ_SUM]
   som man nsker komprimeret. Det er de mindst betydende bits
   der udgr den pakkede kode. Antallet af bits i koden sls
   op i:

   huff_pack_len[];

   Bemrk at de laveste lngder vil st i starten - man vil alts
   opnr kompression hvis det fortrinsvis er lave numre man giver
   som input (antallet af levels man skal g frem i freq fr summen
   er = eller overstiger input giver antallet af bits der skal bruges,
   hvis man trkker en fra).

   Til udpakning findes

   huff_unpack_code[];

   Man skal lse et tal bestende af MAXBITS svarende til den strst
   mulige kode og give det som index i denne tabel. Man fr s det
   tal ud som der koden angiver. Tabellen

   huff_unpack_len[];

   Angiver hvor mange bits den pgldende kode optog (det samme tal som
   man fik i huff_pack_len)
*/

//------Oplysninger om tabellen---------
#define MAXBITS   8
char freq[MAXBITS+1]={ 0,0,0,1,3,8,12,24,16};
#define FREQ_SUM 64 // Alle freq's elementer lagt sammen

//--------Indholds-mssigt-layout-------
#define CHANGE_BITS
// Hvis denne er defineret vil elementerne huff_pack_code komme i
// faldene orden - ellers i stigende

#ifdef CHANGE_BITS
//  #define UNUSED_TO_1
#endif
// Angiver at de om de bits i huff_pack_code der ikke bliver brugt
// (fordi de ikke er s lange som der er reserveret plads til) skal
// initeres til 1. Ellers bliver de initeret til 0

// ---------Egentligt-layout------------
#define PACK_OUTFILE   "PACKTABL.HPP"
#define UNPACK_OUTFILE "UNPTABL.HPP"
#define NUMS_PR_LINE         8
#define NUM_WITH             2
#define TYPE    "unsigned char"
// Sidste angiver typen p elementerne som man vil have skrevet til
// tabel filen. Normalt vil man vler "unsigned char" hvis
// MAXBITS<=8 og ellers unsigned.

//----------------------------------------------------------------------------
#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <string.h>
#include <ctype.h>
//----------------------------------------------------------------------------
// Bruger beskrivelse for makehuff:
// Nr put_tabl kaldes skrives de to pakke tabeller via put_pack.
// Bemrk at dette sker i random rkke flge.

/* For hvert level findes (nsten) disse data strukturer

  unsigned len  :  Angiver hvor mange noder der er p det pgldende level
                   Bemrk at de frste freq[level] noder slutter i det
                   pgldende level. Denne variabel er for hvert level
  unsigned* left:  Indeholder for indx>=freq[level] indxs p de steder
                   som det underligende level arver til venstre.
                   Findes ikke for sidste level (intet arves herfra)
  unsigned* right: se left

*/

class makehuff {
  unsigned*left[MAXBITS];
  unsigned*right[MAXBITS];
  unsigned len[MAXBITS+1]; // Antallet af elementer i et givent level

  public:
  makehuff();
  ~makehuff();

  virtual void put_pack(unsigned code, int len)=0;
  // Angiver at den givne kode har fet den givne lngde

  void put_tabl(int node=0, int level=0, unsigned curvalue=0);
  // Parametre m ikke ndres - skriver pakke-tret via put_pack
  // i udefineret rkkeflge!
};

makehuff::makehuff() {
  // Denne stter left/right/len klar til brug

  int nodes_on_cur_level=1;
  // En node p det frste level. Variablen indeholder antallet af noder
  // p det level der behandles i flgende lkke
  int sum=0; // En check sum p antallet af slutnoder
  for(int a=0; a<=MAXBITS; a++) { // G ned gemmen alle levels
    sum+=freq[a];

    len[a]=nodes_on_cur_level;

    // Find ud af hvor mange noder der er p nste level:
    nodes_on_cur_level-=freq[a];// S mange kan ikke tages med (er brugt)
    nodes_on_cur_level*=2;      // P nste level fordobles antallet af resten

    // Der skal vre et positivt antal:
    if(nodes_on_cur_level<0) {
      puts("Fejl: Huffman tabelen kan ikke laves pga manglende kombinationsmuligheder");
      exit(1);
    }

    // Hvis der er flere levels tilbage men ikke nogen ledige noder
    // p dette level:
    if((a!=MAXBITS) && (!nodes_on_cur_level)) {
      printf("Fejl: Ikke flere noder i level %u\n",a+1);
      exit(1);
    }
  }

  // Hvis der er lse ender (=ubrugte noder) i det sidste level
  if(nodes_on_cur_level) {
    printf("Advarsel: Huffman tabelen indeholde %u ubrugte node(r) i sidste level\n",nodes_on_cur_level/2);
    // divider med 2 da vi egentlig er get ned p nste (fiktive) level
  }

  if(sum!=FREQ_SUM) {
    printf("Fejl: FREQ_SUM burde indeholde %u og ikke %u\n",sum,FREQ_SUM);
    exit(1);
  }

  // Nu er len sat. Nu gr vi fra frste level og ned, og sammenslutter
  // alle de noder der arves fra i level ned til noder der findes i level+1
  for(a=0; a<MAXBITS; a++) {
     left[a]=new unsigned[len[a]];  // Alloker plads
     right[a]=new unsigned[len[a]];

    for(int s=freq[a],d=0; s<len[a]; s++,d+=2) {
      left[a][s]=d; right[a][s]=d+1;
    }

    if(d!=len[a+1]) abort(); // Intern fejl
    //^ Det antal noder der nedarves fra skal vre = antallet af noder
    // p nste level *2 (bemrk d+=2 i ovenstende).
  }
}

makehuff::~makehuff() {
  for(int a=0; a<MAXBITS; a++) {
    delete left[a]; delete right[a]; // dealloker lager
  }
}

// rekursiv funktioner der kaldes engang for hver bit:
void makehuff::put_tabl(int node, int level, unsigned curvalue) {
  if(node<freq[level]) { // Noden slutter her nr node nr er under freq[level]
    // curvalue indeholder fra bit 0 og level bits frem den kode der er.
    put_pack(curvalue>>(16-level),level);
  } else {
    curvalue>>=1;
    #ifdef CHANGE_BITS
      put_tabl(right[level][node],level+1,curvalue);
      curvalue|=0x8000;
      put_tabl(left[level][node],level+1,curvalue);
    #else
      put_tabl(left[level][node],level+1,curvalue);
      curvalue|=0x8000;
      put_tabl(right[level][node],level+1,curvalue);
    #endif
  }
}

//----------------------------------------------------------------------------
// collecthuf sorterer og opsamler informationer fra makehuff og kalkulerer
// unpack tabellerne. Nr chuff er kaldt vil tabellerne huff_pack og
// huff_unpack indeholde de frdige tabeller.

struct pack {
  unsigned code,len;
};

struct collecthuff:private makehuff {
  int a;
  collecthuff() : a(0) {}

  pack huff_pack[FREQ_SUM];
  pack huff_unpack[1<<MAXBITS];

  virtual void put_pack(unsigned code, int len) {
    huff_pack[a].code=code;
    huff_pack[a].len=len;
    a++;
  }

  void chuff();
};

int sort_huff(const void* a, const void* b) {
  // Sorter pakke tabellerne primrt efter lngde sdan at vrdierne
  // med de laveste lngder kommer frst. Sorter sekunderrt sdan
  // at numrene kommer i stigende/faldende orden alt efter om
  // CHANGE_BITS er defineret

  int lena=((pack*) a)->len, lenb=((pack*) b)->len;
  if(lena!=lenb) return lena>lenb ? 1 : -1;

  #ifdef CHANGE_BITS
    return ((pack*) a)->code<((pack*) b)->code ? 1 : -1;
  #else
    return ((pack*) a)->code>((pack*) b)->code ? 1 : -1;
  #endif
}

void collecthuff::chuff() {
  put_tabl();
  // huff_pack er nu lavet - sorter den:
  qsort(huff_pack,FREQ_SUM,sizeof(pack),sort_huff);

  // nu skal unpack tabellerne kalkuleres.

  for(int s=0; s<FREQ_SUM; s++) {
    int len=huff_pack[s].len;
    unsigned code=huff_pack[s].code;

    // Indst vrdier ved alle dem der har len af code
    // som de nederste bits:
    for(int indx=code; indx<(1<<MAXBITS); indx+=(1<<len)) {
      huff_unpack[indx].code=s;
      huff_unpack[indx].len=len;
    }

    #ifdef UNUSED_TO_1
      // Brugeren nsker at de bits der ikke bliver brugt til noget stttes:
      { for(int a=huff_pack[s].len; a<MAXBITS; a++)
    huff_pack[s].code|=1<<a; }
    #endif
  }
}

//----------------------------------------------------------------------------
// De flgende simple rutiner skrver blot de vrdier som collecthuff
// har kalkuleret.

char pack_code_text[]=  TYPE" huff_pack_code[]= {";
char pack_len_text[]=   TYPE" huff_pack_len[]= {";

char unpack_code_text[]=TYPE" huff_unpack_code[]= {";
char unpack_len_text[]= TYPE" huff_unpack_len[]= {";

char tabel_end_text[]=  "};\n\n";
char tabel_new_line[]=  "\n  ";

class writehuff:collecthuff {
  // Nedenstende skriver en tabel:
  void write_tabel(FILE* f,      // Fil der skal skrives til
                   char* sttext, // Teksten der skal startes med
                   pack* tabel,  // Pointer til frste elemenet i tabellen
                   int elements, // Antallet af elementer i tabellen
                   char code);   // Er TRUE hvis det er code elementet der
                                 // skal skrives, ellers er det len.

  public:
  void whuff();
};

void writehuff::write_tabel(FILE* f, char* sttext, pack* tabel, int elements, char code) {
  fprintf(f,sttext); // Skriv indledningen

  for(int a=0; a<elements; a++) {
    if(!(a%NUMS_PR_LINE)) // Start p en ny linie
      fprintf(f,tabel_new_line);

    // Skriv aktuelle linie:
    fprintf(f,"0x%*.*X",NUM_WITH,NUM_WITH,code ? tabel[a].code : tabel[a].len);

    if(a!=elements-1) fprintf(f,", "); // Skriv komme ved andre end sidste element
  }
  fprintf(f,tabel_end_text); // Skriv afslutningen
}

void writehuff::whuff() {
  chuff();

  FILE* f=fopen(PACK_OUTFILE,"w");
  write_tabel(f,pack_code_text,huff_pack,FREQ_SUM,1);
  write_tabel(f,pack_len_text,huff_pack,FREQ_SUM,0);
  fclose(f);

  f=fopen(UNPACK_OUTFILE,"w");
  write_tabel(f,unpack_code_text,huff_unpack,(1<<MAXBITS),1);
  write_tabel(f,unpack_len_text,huff_unpack,(1<<MAXBITS),0);
  fclose(f);
}

//----------------------------------------------------------------------------

main() {
  writehuff wh;
  printf("Skriver...\n");
  wh.whuff();
  return 0;
}
