#include <dos.h>
#include <malloc.h>

#include "fmod.h"

// PROTOTYPES FOR OLD INTERRUPT
#ifdef __TURBOC__
void interrupt ( *oldmodhandler)(__CPPARGS);
#else
void (interrupt far *oldmodhandler)(void);
#endif

// PATTERN AND SONG DATA VARIABLES
Song FM;                 // the song header information
char *patbuff[256];      // pattern data pointers
SampleHDR *inst[99];     // Instrument pointers
Note *current;           // current - a global 'note'

// HARDWARE SPECIFIC VARIABLES
UWORD divisor;           // mixing rate or gus divisor

// PLAY TIME VARIABLES
UBYTE speed, bpm;        // speed & bpm, figure it out.
UBYTE patdelay;          // # of notes to delay pattern for in EEx
 BYTE row, globalvol=64; // current row being played, master volume
UBYTE defpan=0x30;       // default pan value, L=defpan, R=0xFF-defpan
 WORD ord=0;             // current order being played

// INTERFACE VARIABLES
UDWORD fclock;           // counter for interface clock, for interface
UBYTE pause=0;           // pause flag
UDWORD filelen;          // size of file, for the interface
char *SongMSG;                   // Song Message for formats like MTM etc.

// CHANNEL SPECIFIC VARIABLES
	// general variables
	UBYTE    mute[32];   // toggle whether to mute channel or not
	 BYTE  volume[32];   // current volume of each channel
	 WORD    freq[32];   // amiga frequency of each channel
	 WORD  panval[32];   // pan values for each channel
	 BYTE lastins[32];   // instrument # for each channel (to remember)
	UBYTE lastnot[32];   // last note set in channel (to remember)
	UWORD lastper[32];   // last period set in channel (to remember)
	UBYTE restart[32];   // flag whether to play sample or not
	UDWORD   start[32];      // where to start in sample

	// effect variables
	UWORD soffset[32];   // amount to sample offset by
	UBYTE lastpor[32];   // last porta up or down value (XM + S3M)
	UBYTE lastvsl[32];   // last volume slide value (XM + S3M)

	UBYTE retrigx[32];   // last retrig volume slide used (XM + S3M)
	UBYTE retrigy[32];   // last retrig tick count used (XM + S3M)

	WORD    porto[32];   // note to port to value (signed word)
	UBYTE  portsp[32];   // porta speed

	BYTE   vibpos[32];   // vibrato position
	UBYTE  vibspe[32];   // vibrato speed
	UBYTE  vibdep[32];   // vibrato depth

	BYTE  trempos[32];   // tremolo position
	UBYTE tremspe[32];   // tremolo speed
	UBYTE tremdep[32];   // tremolo depth

 UBYTE patlooprow[32];
  UBYTE patloopno[32];   // pattern loop variables for effect  E6x

	UBYTE  tremor[32];   // tremor position (XM + S3M)
   UBYTE tremparm[32];   // tremor parameters

	UBYTE wavecon[32];   // waveform type for vibrato and tremolo (4bits each)
	UBYTE geffect[32];   // effect played at the time (for interface)

// SINE TABLE FOR TREMOLO AND VIBRATO (from protracker so 100% compatible)
UBYTE sintab[32] = {
	   0, 24, 49, 74, 97,120,141,161,
	 180,197,212,224,235,244,250,253,
	 255,253,250,244,235,224,212,197,
	 180,161,141,120, 97, 74, 49, 24
};

// AMIGA TYPE PERIOD TABLE, FOR 11 OCTAVES (octave 9 and 10 are only for
// .MOD's that use these sort of periods when loading, i.e dope.mod)
UWORD periodtab[134] = {
  27392,25856,24384,23040,21696,20480,19328,18240,17216,16256,15360,14496,//0
  13696,12928,12192,11520,10848,10240, 9664, 9120, 8608, 8128, 7680, 7248,//1
   6848, 6464, 6096, 5760, 5424, 5120, 4832, 4560, 4304, 4064, 3840, 3624,//2
   3424, 3232, 3048, 2880, 2712, 2560, 2416, 2280, 2152, 2032, 1920, 1812,//3
   1712, 1616, 1524, 1440, 1356, 1280, 1208, 1140, 1076, 1016,  960,  906,//4
	856,  808,  762,  720,  678,  640,  604,  570,  538,  508,  480,  453,//5
	428,  404,  381,  360,  339,  320,  302,  285,  269,  254,  240,  226,//6
	214,  202,  190,  180,  170,  160,  151,  143,  135,  127,  120,  113,//7
	107,  101,   95,   90,   85,   80,   75,   71,   67,   63,   60,   56,//8
	 53,   50,   47,   45,   42,   40,   37,   35,   33,   31,   30,   28,//9
	 26,   25,   23,   22,   21,   20,   18,   17,   16,   15,   15,   14,//10
	  0,    0       // <- these last 2 are for no note (132) and keyoff (133)
};


/****************************************************************************
 *    Name : GUSfreq                                                        *
 * Purpose : To convert an amiga period to a GUS frequency                  *
 *  Passed : UWORD period - the amiga/st3 period to convert                 *
 * Returns : UWORD fc -the GUS frequency to use for setting voice pitch/rate*
 *  Locals : UWORD fc -the GUS frequency to use for setting voice pitch/rate*
 *           UDWORD speed_khz - the speed in kHz.                           *
 *   Notes : Formulas from the GUS SDK                                      *
 ****************************************************************************/
UWORD GUSfreq(UWORD period) {
	register UWORD fc;
	register UDWORD speed_khz;

	speed_khz= 14317056L / period;

	fc = (((speed_khz<<9L)+(divisor>>1L)) / divisor) << 1;

	return fc;
}


/****************************************************************************
 *    Name : SetTimerSpeed                                                  *
 * Purpose : To change the rate of the system timer.  If the bpm is 0 then  *
 *           we are resetting the timer.                                    *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UWORD Speed - a temporary variable holding the value of the    *
 *           calculated rate using divisor / (bpm * 2 / 5)                  *
 * WARNING : Now I use a doubled bpm rate, i am actually setting it to twice*
 *           the speed you would expect it too.. I double the bpm, becuase  *
 *           you get a division overflow at 47bpm, so now it is doubled the *
 *           bpm can go down to 24!  (ie, now 125bpm=100hz, not 50hz)       *
 ****************************************************************************/
void SetTimerSpeed() {
	register UWORD Speed;
	if (bpm == 0) Speed = 0;
	else Speed = 5965900L/(bpm<<2);         // less calculation this way

	outp(0x43,0x36);
	outp(0x40,Speed&0xff);
	outp(0x40,Speed>>8);
}


int FinetoHz(UBYTE ft) {
	switch(ft) {
	case  0 : return 8363;
	case  1 : return 8413;
	case  2 : return 8463;
	case  3 : return 8529;
	case  4 : return 8581;
	case  5 : return 8651;
	case  6 : return 8723;
	case  7 : return 8757;
	case  8 : return 7895;
	case  9 : return 7941;
	case 10 : return 7985;
	case 11 : return 8046;
	case 12 : return 8107;
	case 13 : return 8169;
	case 14 : return 8232;
	case 15 : return 8280;
	default : return 8363;
	};
}


/****************************************************************************
 *    Name : dotremor                                                       *
 * Purpose : to carry out an s3m type tremor command                        *
 *  Passed : byte track - the track number to tremor                        *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void dotremor(UBYTE track) {
	tremor[track] %= (tremparm[track] >> 4) + (tremparm[track] & 0xF);

	if (tremor[track] < (tremparm[track] >> 4))
							GUSSetVolume(track, volume[track]*globalvol/64);
	else GUSSetVolume(track, 0);
	tremor[track]++;
}


/****************************************************************************
 *    Name : doporta                                                        *
 * Purpose : to carry out a tone portamento to a certain note               *
 *  Passed : byte track - the track number to do the porta too              *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 ****************************************************************************/
void doporta(UBYTE track) {
	// slide pitch down if it needs too.
	if (freq[track] < porto[track]) {
		freq[track] += (portsp[track] << 2);
		if (freq[track] > porto[track]) freq[track]=porto[track];
	}

	// slide pitch up if it needs too.
	if (freq[track] > porto[track]) {
		freq[track] -= (portsp[track] << 2);
		if (freq[track] < porto[track]) freq[track]=porto[track];
	 }

	 //
	 // if (glissando[track]) {
	 //}
	 //
	 GUSSetFreq(track, GUSfreq(freq[track]));
}


/****************************************************************************
 *    Name : dovibrato                                                      *
 * Purpose : to carry out a vibrato at a certain depth and speed            *
 *  Passed : UBYTE track - the track number to do the vibrato too           *
 * Returns : -                                                              *
 *  Locals : UWORD vib - size of delta to add or subtract from period       *
 *           UBYTE temp - absolute positon of vibrato position counter,     *
 *                        from 0 - 31                                       *
 *   Notes : AND'ing temp with 31 removes the sign bit giving the abs value *
 ****************************************************************************/
void dovibrato(UBYTE track) {
	register UWORD delta;
	register UBYTE temp;

	temp = (vibpos[track] & 31);

	switch(wavecon[track]&3){
		case 0: delta = sintab[temp];             // sine
				break;
		case 1: temp <<= 3;                     // ramp down
				if (vibpos[track]<0) temp=255-temp;
				delta=temp;
				break;
		case 2: delta = 255;                      // square
				break;
		case 3: delta = sintab[temp];             // random
				break;
	};

	delta *=vibdep[track];
	delta >>=7;
	delta <<=2;   // we use 4*periods so make vibrato 4 times bigger

	if (vibpos[track] >= 0) GUSSetFreq(track, GUSfreq(freq[track]+delta));
	else                    GUSSetFreq(track, GUSfreq(freq[track]-delta));

	vibpos[track]+= vibspe[track] ;
	if (vibpos[track] > 31) vibpos[track] -=64;
}

/****************************************************************************
 *    Name : dofinevibrato                                                  *
 * Purpose : to carry out a fine vibrato at a certain depth and speed       *
 *  Passed : UBYTE track - the track number to do the vibrato too           *
 * Returns : -                                                              *
 *  Locals : UWORD delta - size of delta to add or subtract from period     *
 *           UBYTE temp - absolute positon of fine vibrato position counter,*
 *                        from 0 - 31                                       *
 ****************************************************************************/
void dofinevibrato(UBYTE track) {
	register UWORD delta;
	register UBYTE temp;

	temp = (vibpos[track] & 31);

	switch(wavecon[track]&3){
		case 0: delta = sintab[temp];             // sine
				break;
		case 1: temp <<= 3;                     // ramp down
				if (vibpos[track]<0) temp=255-temp;
				delta=temp;
				break;
		case 2: delta = 255;                      // square
				break;
		case 3: delta = sintab[temp];             // random
				break;
	};

	delta *=vibdep[track];
	delta >>=7;

	if (vibpos[track] >= 0) GUSSetFreq(track, GUSfreq(freq[track]+delta));
	else                     GUSSetFreq(track, GUSfreq(freq[track]-delta));

	vibpos[track]+= vibspe[track] ;
	if (vibpos[track] > 31) vibpos[track] -=64;
}


/****************************************************************************
 *    Name : dotremolo                                                      *
 * Purpose : to carry out a tremolo at a certain depth and speed            *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UWORD delta - size of delta to add or subtract from volume     *
 *           UBYTE temp - absolute positon of tremolo position counter,     *
 *                        from 0 - 31                                       *
 ****************************************************************************/
void dotremolo(UBYTE track) {
	register UWORD delta;
	register UBYTE temp;

	temp = (trempos[track] & 31);

	switch((wavecon[track]>>4)&3){
		case 0: delta = sintab[temp];             // sine
				break;
		case 1: temp <<= 3;                     // ramp down
				if(vibpos[track]<0) temp=255-temp;
				delta=temp;
				break;
		case 2: delta = 255;                      // square
				break;
		case 3: delta = sintab[temp];             // random (just use sine)
				break;
	};

	delta *= tremdep[track];
	delta >>= 6;

	if (trempos[track] >= 0) {
		if (volume[track]+delta > 64) delta = 64-volume[track];
		GUSSetVolume(track, (volume[track]+delta)*globalvol/64);
	}
	else {
		if ((WORD)(volume[track]-delta) < 0) delta = volume[track];
		GUSSetVolume(track, (volume[track]-delta)*globalvol/64);
	}

	trempos[track]+= tremspe[track];
	if (trempos[track] > 31) trempos[track] -=64;
}


/****************************************************************************
 *    Name : doS3Mvolslide                                                  *
 * Purpose : to carry out a S3M type volume slide                           *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : -                                                              *
 *   Notes : D0x=slide down,      Dx0=slide up,                             *
 *               DFx=fine slide down, DxF=fine slide up                         *
 ****************************************************************************/
void doS3Mvolslide(UBYTE track) {
	if (!(lastvsl[track]&0xF)) volume[track]+=(lastvsl[track] >> 4);
	if (!(lastvsl[track]>>4)) volume[track]-=(lastvsl[track] & 0xF);
	if (volume[track] > 64) volume[track]=64;
	if (volume[track] < 0) volume[track]=0;
	GUSSetVolume(track, volume[track]*globalvol/64);
}


/****************************************************************************
 *    Name : UpdateNote                                                     *
 * Purpose : To update tick 0 of a row being played, and also play the      *
 *           voices, set their volumes, frequencies, balance etc            *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UBYTE track - the # of the track being processed in the row    *
 *           UBYTE eparmx - temp variable which holds the current x         *
 *                          parameter of an effect                          *
 *           UBYTE eparmy - temp variable which holds the current y         *
 *                          parameter of an effect                          *
 *           UBYTE breakflag - a flag to say if a pattern break has happened*
 *                          so it doesnt bother to increment order again.   *
 *           UBYTE jumpflag - a flag to say if a pattern jump has happened  *
 *                          so pattern break doesnt increment order.        *
 *           UBYTE lastvoice - an optimization variable to hold the last    *
 *                          voice processed, so we dont have to loop through*
 *                          all FM.numchannels, only the few that are used. *
 ****************************************************************************/
void UpdateNote() {
	register UBYTE track, eparmx, eparmy, breakflag=0,jumpflag=0,lastvoice=0;

	// Point our note pointer to the correct pattern buffer, and to the
	// correct offset in this buffer indicated by row and number of channels
	current = (Note *)( patbuff[FM.order[ord]] + (5*row*FM.channels) );

	// Loop through each channel in the row until we have finished
	for (track=0; track<FM.channels; track++) {
		eparmx = current -> eparm >> 4;                 // get effect param x
		eparmy = current -> eparm & 0xF;                // get effect param y

		// These next 3 lines just store the effect # for the interface
		geffect[track] = current -> effect;
		if (geffect[track] == 0xE) geffect[track]+=13+eparmx;
		if (!(current->effect) && (current->eparm)) geffect[track]=43;

		// if an INSTRUMENT NUMBER is given

		if (current->number) {
			lastins[track] = current->number-1; // remember the Instrument #

			// dont get the volume if delay note, set it when the delay note
			// actually happens (and if it doesnt, then it will be correct)
			if (!(current -> effect == 0xE && eparmx == 0xD))
				volume[track] = inst[lastins[track]]->volume;
		}

		// if a NOTE is given

		if (current->note < NONOTE) {
			// get period according to relative note, c2spd and finetune
			lastper[track] = 8363L * periodtab[current->note] / inst[lastins[track]]->middlec;
			lastnot[track] = current->note;         // now remember the note
			restart[track] = 1;                     // retrigger sample
			tremor[track] = 0;                      // retrigger tremor count
			start[track] = inst[lastins[track]]->offset;// store sample start
			lastvoice = track;                      // last voiced used

			// retrigger tremolo and vibrato waveforms

			if ((wavecon[track]&0xF) < 4) vibpos[track]=0;
			if ((wavecon[track]>>4) < 4) trempos[track]=0;

			// frequency only changes if there are no portamento effects

			if (current->effect != 0x3 && current->effect != 0x5 &&
				current->effect != 0x16) freq[track] = lastper[track];
		}
		else restart[track]=0;                  // else dont retrigger a sample

		// process volume byte
		if (current->volume <=0x40) volume[track] = current->volume;
		// process keyoff note
		if (current->note == KEYOFF) volume[track] =0;

		// TICK 0 EFFECTS

		switch (current -> effect) {
			case 0x0: break;                // dont waste my time in here!!!
			case 0x1: break;                // not processed on tick 0
			case 0x2: break;                // not processed on tick 0

			case 0xC: volume[track] = current -> eparm;
					  if (volume[track] > 64) volume[track] = 64;
					  break;

			case 0x3: if (current->eparm) portsp[track]=current->eparm;

			case 0x5: porto[track] = lastper[track];
					  restart[track]=0;
					  break;

			case 0x4: if (eparmx) vibspe[track] = eparmx;
					  if (eparmy) vibdep[track] = eparmy;
			case 0x6: break;                // not processed on tick 0

			case 0x7: if (eparmx) tremspe[track] = eparmx;
					  if (eparmy) tremdep[track] = eparmy;
					  break;

			case 0x8: panval[track] = current -> eparm;
					  GUSSetBalance(track, panval[track]);
					  break;

			case 0x9: if (current->eparm) soffset[track]=current->eparm << 8;
					  if (soffset[track] > inst[lastins[track]]->length)
						  soffset[track] = inst[lastins[track]]->length;
					  start[track] += soffset[track];
					  break;

			case 0xA: break;                // not processed on tick 0

			// --- 00 B00 : --- 00 D63 , should put us at ord=0, row=63
			case 0xB: ord = current->eparm;
					  row = -1;
					  if (ord >= FM.songlength) ord=0;
					  jumpflag = 1;
					  break;

			case 0xD: row = (eparmx*10) + eparmy -1;
					  if (row > 63) row = -1;
					  if (!breakflag && !jumpflag) ord++;
					  if (ord >= FM.songlength) ord=0;
					  breakflag = 1;
					  break;

			case 0xF: if (current->eparm < 0x20) speed = current->eparm;
					  else { bpm = current->eparm; SetTimerSpeed(); }
					  break;

			case 0xE: switch (eparmx) {
					  case 0x1: freq[track] -= (eparmy << 2);
								break;
					  case 0x2: freq[track] += (eparmy << 2);
								break;
					  case 0x3: break;
					  case 0x4: wavecon[track] &= 0xF0;
								wavecon[track] |= eparmy;
								break;
					  case 0x5: inst[lastins[track]]->middlec=FinetoHz(eparmy);
								break;
					  case 0x6: if (eparmy == 0) patlooprow[track] = row;
								else {
									if (!patloopno[track]) patloopno[track]=eparmy;
									else patloopno[track]--;
									if (patloopno[track]) row = patlooprow[track]-1;
								}
								break;
					  case 0x7: wavecon[track] &= 0xF;
								wavecon[track] |= (eparmy<<4);
								break;
					  case 0x8: panval[track] = (eparmy << 4);
								GUSSetBalance(track, panval[track]);
								break;
					  case 0x9: break;
					  case 0xA: volume[track] += eparmy;
								if (volume[track] > 64) volume[track]=64;
								break;
					  case 0xB: volume[track] -= eparmy;
								if (volume[track] < 0)  volume[track]=0;
								break;
					  case 0xC: break;  // not processed on tick 0
					  case 0xD: restart[track] = 0;
								goto nofreq;
					  case 0xE: patdelay = eparmy;
								break;
					  };
					  break;

			// S3M SPECIFIC EFFECTS

			case 0x10: if (current->eparm) speed = current -> eparm;
					   break;

			// Dxy - Volume slide, fine vol  DFx = slide down, DxF = slide up
			case 0x11: if (current->eparm) lastvsl[track]=current->eparm;
					   // DFF is classed as a slide up so it gets priority
					   if ((lastvsl[track]&0xF)==0xF) volume[track]+=(lastvsl[track] >> 4);
					   else if ((lastvsl[track]>>4)==0xF) volume[track]-=(lastvsl[track] & 0xF);
					   // perform an extra slide if using old fast vol slides!
					   if (FM.flags == 1) {
						   if (!(lastvsl[track]&0xF)) volume[track]+=(lastvsl[track] >> 4);
						   if (!(lastvsl[track]>>4)) volume[track]-=(lastvsl[track] & 0xF);
					   }
					  if (volume[track] > 64) volume[track]=64;
					  if (volume[track] < 0) volume[track]=0;
					  break;

			// Exy - porta down
			case 0x12: if (current->eparm) lastpor[track]=current->eparm;
					   if ((lastpor[track]>>4)==0xF) freq[track]+=((lastpor[track] & 0xF) << 2);
					   if ((lastpor[track]>>4)==0xE) freq[track]+=(lastpor[track] & 0xF);
					   break;

			// Fxy - porta up
			case 0x13: if (current->eparm) lastpor[track]=current->eparm;
					   if ((lastpor[track]>>4)==0xF) freq[track]-=((lastpor[track] & 0xF) << 2);
					   if ((lastpor[track]>>4)==0xE) freq[track]-=(lastpor[track] & 0xF);
					   break;

			// Ixy - Tremor
			case 0x14: if (current->eparm) {
							tremparm[track]=((eparmx<<4)+1)+(eparmy+1);
					   }
					   dotremor(track);
					   break;
			// Kxy - Porta+VolSlide
			// Hxy - Vibrato+VolSlide
			case 0x16: porto[track] = lastper[track];
					   restart[track]=0;
			case 0x15: if (current->eparm) lastvsl[track]=current->eparm;
					   goto nofreq;
			// Qxy - Retrig
			case 0x17: if (current->eparm) {
							retrigx[track] = eparmx;
							retrigy[track] = eparmy;
					   }
					   break;

			// Uxy - Fine Vibrato
			case 0x18: if (eparmx) vibspe[track] = eparmx;
					   if (eparmy) vibdep[track] = eparmy;
					   break;

			// Vxy - Global Volume
			case 0x19: globalvol = current->eparm;
					   if (globalvol > 64) globalvol=64;
					   break;

			// SAx - Stereo Control
			case 0x1A: if (eparmy > 7) eparmy-=8;
					   else eparmy+=8;
					   panval[track] = (eparmy << 4);
					   GUSSetBalance(track, panval[track]);
					   break;
		};

		if (current->effect !=7) GUSSetVolume(track, volume[track]*globalvol/64);
		if (freq[track]) GUSSetFreq(track, GUSfreq(freq[track]));
nofreq:
		current ++;
	}

	// now play the samples
	for (track=0; track <= lastvoice; track++) {
		if (restart[track])     {
			GUSPlayVoice(track, inst[lastins[track]]->mode, start[track],
			   inst[lastins[track]]->offset+inst[lastins[track]]->loopstart,
			   inst[lastins[track]]->offset+inst[lastins[track]]->length);
		}
	}       // There is only 1 playvoice here (not 1 for looping and 1 for non
}               // looping like you might have expected), becuase each sample has a
		// MODE byte, which corresponds exactly to the GUS's mode values.


/****************************************************************************
 *    Name : UpdateEffect                                                   *
 * Purpose : To update any tick based effects after tick 0                  *
 *  Passed : UBYTE tick - the actual tick number.                            *
 * Returns : -                                                              *
 *  Locals : UBYTE track - the number of the channel we are in               *
 *           UBYTE effect - a temp variable to get the effect number wanted  *
 *           UBYTE eparmx - a term variable to get the effect parameter x    *
 *           UBYTE eparmy - a term variable to get the effect parameter y    *
 *    Note : To see explanations of effects check out the fmoddoc document  *
 ****************************************************************************/
void UpdateEffect(UBYTE tick) {
	UBYTE track, effect, eparmx, eparmy;

	// rewind back 1 row (as we just incremented the row in modhandler() 
	// function)
	current -= FM.channels;

	for (track=0; track<FM.channels; track++) {
		if (!freq[track]) goto skip;        // no freq?, so dont do anything!

		effect = current -> effect;         // grab the effect number
		eparmx = current -> eparm >> 4;     // grab the effect parameter x
		eparmy = current -> eparm & 0xF;    // grab the effect parameter y

		switch(effect) {
		case 0x0: if (current -> eparm > 0) switch (tick % 3) {
					case 0: GUSSetFreq(track, GUSfreq(freq[track]));
							break;
					case 1: GUSSetFreq(track, GUSfreq(periodtab[lastnot[track]+eparmx]));
							break;
					case 2: GUSSetFreq(track, GUSfreq(periodtab[lastnot[track]+eparmy]));
							break;
				  };
				  break;

		case 0x1: freq[track]-= (current -> eparm << 2); // subtract freq
				  GUSSetFreq(track, GUSfreq(freq[track]));
				  if (freq[track] < 56) freq[track]=56;  // stop at B#8
				  break;

		case 0x2: freq[track]+= (current -> eparm << 2);
				  GUSSetFreq(track, GUSfreq(freq[track]));
				  break;

		case 0x3: doporta(track);
				  break;

		case 0x4: dovibrato(track);
				  break;

		case 0x5: doporta(track);
				  volume[track] += eparmx - eparmy;
				  if (volume[track] < 0)  volume[track] = 0;
				  if (volume[track] > 64) volume[track] = 64;
				  GUSSetVolume(track, volume[track]*globalvol/64);
				  break;

		case 0x6: dovibrato(track);
				  volume[track] += eparmx - eparmy;
				  if (volume[track] < 0)  volume[track] = 0;
				  if (volume[track] > 64) volume[track] = 64;
				  GUSSetVolume(track, volume[track]*globalvol/64);
				  break;

		case 0x7: dotremolo(track);
				  break;

		case 0xA: volume[track] += eparmx - eparmy;
				  if (volume[track] < 0)  volume[track] = 0;
				  if (volume[track] > 64) volume[track] = 64;
				  GUSSetVolume(track, volume[track]*globalvol/64);
				  break;

			  // extended PT effects
		case 0xE: switch(eparmx) {
				  // Note Cut - 0 volume after x ticks
				  case 0xC: if (tick==eparmy) {
								volume[track] = 0;
								GUSSetVolume(track, volume[track]);
							}
							break;
				  // Retrig Note - retrigger every x ticks
				  case 0x9: if (!eparmy) break; // divide by 0 bugfix
							if (!(tick % eparmy)) GUSPlayVoice(track,
								inst[lastins[track]]->mode,
								inst[lastins[track]]->offset,
								inst[lastins[track]]->offset+
								inst[lastins[track]]->loopstart,
								inst[lastins[track]]->offset+
								inst[lastins[track]]->length);
							break;
				  // Note Delay - wait x ticks then play - quite more complex
				  case 0xD: if (tick==eparmy) {             // than it seems.
								if (current->number) volume[track] = inst[lastins[track]]->volume;
								if (current->volume <=0x40) volume[track] = current->volume;
								GUSSetFreq(track, GUSfreq(freq[track]));
								GUSSetVolume(track, volume[track]*globalvol/64);
								GUSPlayVoice(track,
									inst[lastins[track]]->mode,
									inst[lastins[track]]->offset,
									inst[lastins[track]]->offset+
									inst[lastins[track]]->loopstart,
									inst[lastins[track]]->offset+
									inst[lastins[track]]->length);
							}
							break;
			  };
			  break;
					  // S3M effect D - volume slide
		case 0x11: doS3Mvolslide(track);
				   break;
				   // S3M effect E - porta down
		case 0x12: if (lastpor[track]<0xE0) freq[track]+=(lastpor[track] << 2);
				   GUSSetFreq(track, GUSfreq(freq[track]));
				   break;
				   // S3M effect F - porta up
		case 0x13: if (lastpor[track]<0xE0) freq[track]-=(lastpor[track] << 2);
				   GUSSetFreq(track, GUSfreq(freq[track]));
				   break;
				   // S3M effect I - tremor
		case 0x14: dotremor(track);
				   break;
				   // S3M effect K - Vibrato + Volume slide
		case 0x15: dovibrato(track);
				   doS3Mvolslide(track);
				   break;
				   // S3M effect L - Porta + Volume slide
		case 0x16: doporta(track);
				   doS3Mvolslide(track);
				   break;
				   // S3M effect Retrig + Volume slide
		case 0x17: if (!retrigy[track]) break;
				   if (!(tick % retrigy[track])) {
					   if (retrigx[track]) {
						   switch (retrigx[track]) {
								 case 1: volume[track]--;
										 break;
								 case 2: volume[track]-=2;
										 break;
								 case 3: volume[track]-=4;
										 break;
								 case 4: volume[track]-=8;
										 break;
								 case 5: volume[track]-=16;
										 break;
								 case 6: volume[track]*=2/3;
										 break;
								 case 7: volume[track]>>=1;
										 break;
								 case 8: // ?
										 break;
								 case 9: volume[track]++;
										 break;
								 case 0xA: volume[track]=+2;
										 break;
								 case 0xB: volume[track]+=4;
										 break;
								 case 0xC: volume[track]+=8;
										 break;
								 case 0xD: volume[track]+=16;
										 break;
								 case 0xE: volume[track]*=3/2;
										 break;
								 case 0xF: volume[track]<<=1;
										 break;
						   };
						   if (volume[track] > 64) volume[track]=64;
						   if (volume[track] < 0) volume[track]=0;
						   GUSSetVolume(track, volume[track]*globalvol/64);
					   }
					   GUSPlayVoice(track, inst[lastins[track]]->mode,
										   inst[lastins[track]]->offset,
										   inst[lastins[track]]->offset+
										   inst[lastins[track]]->loopstart,
										   inst[lastins[track]]->offset+
										   inst[lastins[track]]->length);
				   }
				   break;
				   // S3M effect fine vibrato
		case 0x18: dofinevibrato(track);
				   break;

		};
skip:
		current ++;
	}
}


/****************************************************************************
 *    Name : modhandler                                                     *
 * Purpose : To process the clock tick at the given rate, and carry out the *
 *           main body of the player routine                                *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UBYTE tick - the tick counter variable to keep track of the    *
 *                       number of ticks passed per note                    *
 *   Notes : tick is set to 0 at the start to let it tick a few times and   *
 *           let the interface catch up :)                                  *
 *           Note also : This section uses 2*tick so it -seems- the system  *
 *                       timer can make it to a lower bpm than is physically*
 *                       possible                                           *
 ****************************************************************************/
void interrupt modhandler(__CPPARGS) {
	static UWORD tick=0;

	if (!pause) {                                                                   // only go if not paused
		tick++;
		if (tick >= (speed<<1)) {
			tick = 0;                              // reset the tick counter!
			fclock += (1000000L/(bpm * 2 / 5))*speed; // increment the clock
			if (row == FM.patlen[FM.order[ord]]) { // if end of pattn (64)
				ord++;                             // so increment the order
				if (ord >= FM.songlength) ord=FM.restart;   // goto start
				row = 0;                           // start at top of pattn
			}
			if (!patdelay) {                       // if there is no pattern delay
				UpdateNote();                      // Update and play the note
				row++;                             // increment the row
			}
			else patdelay --;                      // else decrement pat delay
		}
		else if (!(tick%2)) UpdateEffect(tick>>1); // Else update the effects
	}
	oldmodhandler();
}


/****************************************************************************
 *    Name : PlayMOD                                                        *
 * Purpose : To hook on the interrupt handler to the system timer and play  *
 *           the song.                                                      *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UBYTE count - counter variable                                 *
 ****************************************************************************/
void PlayMOD() {
	UBYTE count;

	for (count=0; count<FM.channels; count++) {
	GUSSetVolume(count, 0);                     // set volume 0
	GUSSetFreq(count, 1712);                    // middle C will do
	GUSSetBalance(count, panval[count]);        // set up panning vals
	GUSPlayVoice(count, 0,0,0,0);               // point voice to 0
	GUSStopVoice(count);                        // stop the voice

		 mute[count]=0;
	   volume[count]=0;         // clear out variables so
		 freq[count]=0;         // we dont start with any
	  lastins[count]=0;         // weird values
	  lastnot[count]=0;
	  lastper[count]=0;
	  restart[count]=0;
	  soffset[count]=0;
	  lastpor[count]=0;
	  lastvsl[count]=0;
		porto[count]=0;
	   portsp[count]=0;
	  retrigx[count]=0;
	  retrigy[count]=0;
	   vibpos[count]=0;
	   vibspe[count]=0;
	   vibdep[count]=0;
	  trempos[count]=0;
	  tremspe[count]=0;
	  tremdep[count]=0;
   patlooprow[count]=0;
	patloopno[count]=0;
	  wavecon[count]=0;
	   tremor[count]=0;
	  geffect[count]=0;
	}
	row = 0;
	ord = 0;
	patdelay=0;                         // notes to delay pattern for EEx
	fclock=0;                           // start clock at start.

	oldmodhandler = _dos_getvect(0x8);  // backup the old timer's handler
	_dos_setvect(0x8, modhandler);      // now set our new one on the timer

	SetTimerSpeed();                    // set timer to right speed

	FM.playing = 1;                                         // Song is now playing
}


/****************************************************************************
 *    Name : StopMOD                                                        *
 * Purpose : To stop the song and unhook the interrupt handler              *
 *  Passed : -                                                              *
 * Returns : -                                                              *
 *  Locals : UBYTE count - counter to count through channels                *
 ****************************************************************************/
void StopMOD() {
	UBYTE count=0;

	// reset the old timer function
	_dos_setvect(0x8,oldmodhandler);

	// bpm=0 (SetTimerSpeed's code to RESET the timer)
	bpm=0;
	SetTimerSpeed();

	// Stop any GUS voices
	for (count=0; count<FM.channels; count++) GUSStopVoice(count);
	// free the pattern buffers from memory
	for (count=0; count<=FM.numpats; count++) free(patbuff[count]);
	// free the instrument buffers from memory
	for (count=0; count<FM.numinsts; count++) free(inst[count]);

	free(SongMSG);

	FM.playing = 0;                 // song is not playing any more
}
