

INCLUDE "SINTABLE.INC"    ; sinus table for effects

TICKSPEED_LOWLIMIT    EQU   3    ; lowest tickspeed that can be played

IFDEF MUSIC_OPTIMISE_SPEED
  ; Player Poll Frequenz 100-50 Hz (av. 80 Hz), play quality low
  Speed_ClickShifter_Table  DW  7 DUP (4,7,12,23,40,67,100)
ELSE
  ; Player Poll Frequenz 128-64 Hz (av. 102 Hz), play quality average
  Speed_ClickShifter_Table  DW  7 DUP (4,8,16,31,58,104,172)
ENDIF

Click_Shifter             DB  1    ; effect click shifter

;ͻ
; GLOBAL Music Commands       
;ͼ

;Ŀ
; PROC Set_BPMSpeed;                            INTERNAL 
;
; Sets BPM Speed
; INPUT: ax = BPMSpeed
; USES: none
PROC Set_BPMSpeed NEAR

               push   bx cx dx
               or     ax,ax       ; Check if valid BPM speed
               jz error_Set_BPMSpeed

               mov    [ds:BPM],ax
               mov    bl,[ds:Beat]
               shr    bl,4        ; Ignore Beats Per Measure
               or     bl,bl
               jz error_Set_BPMSpeed ; Check if valid ticks
               mul    bl
               mov    bx,15       ; 60 / 4 = 1/4 hz steps
               xor    dx,dx
               div    bx          ; Get Tick Speed
               or     ah,ah
               jz NO_BPMSpeed_Overflow
               mov    ax,100h     ; Set max. Tick Speed
No_BPMSpeed_Overflow:

; set speed as tickspeed
               mov    [ds:speed],ax ; Set as Tick Speed
               jmp set_tickspeed_lowlevel  ; jump to set_tickspeed (ax = tickspeed)

error_Set_BPMSpeed:
               mov    [ds:BPM],0   ; disable BPM speed
               pop    dx cx bx
               RET
ENDP Set_BPMSpeed

;Ŀ
; PROC Set_TickSpeed;                           INTERNAL 
;
; Sets Tick Speed
; INPUT: ax = TickSpeed, ds must be on correct segment !
; USES: none
PROC Set_TickSpeed NEAR

               push   bx cx dx
               mov    [ds:speed],ax
               mov    [ds:BPM],0   ; disable BPM speed

; entry point for set_BPMspeed (ax = tickspeed)
set_tickspeed_lowlevel:

               cmp    ax,TICKSPEED_LOWLIMIT
               ja @@above_lowLimit
               mov    ax,TICKSPEED_LOWLIMIT
@@above_lowLimit:

;  calc clicks that can be played with current tickspeed
; if Tickspeed<speed_clickshifer_table[x] -> clickshifter = 7 - x

               xor    bx,bx
               mov    cl,7   ; max click shifter = 7
@@calc_clickshifter:
               cmp    ax,[ds:Speed_clickshifter_table+bx]
               jb @@clickshifter
               add    bx,2
               dec    cl
               jnz @@calc_clickshifter
@@clickshifter:
               mov    [ds:click_Shifter],cl  ; setup internal click shifter
               mov    bx,ax
               shl    ax,cl
               add    bx,ax  ; 1 Note Click + (2^cl) effect clicks
                             ; bx = Speed in 1/4 Hz
               mov    al,1
               shl    al,cl
               mov    [ds:clicks],al
               inc    al
               mov    [ds:clickcounter],al

               call   [ds:SDevice_Timer_speed] ; bx = speed in 1/4 hz

               pop    dx cx bx
               RET
ENDP Set_TickSpeed


;Ŀ
; PROC INT8_Timer_Speed;                        INTERNAL 
;
; Standard Frequenz Setup INT8
; INPUT: bx = Frame Speed in 1/4 Hz
; USES: none
PROC INT8_Timer_speed NEAR
;--- Calc the INT 8 Timer Frequenz
               mov    dx,72  ; dx:ax = 1193180*4
               mov    ax,54128
               div    bx
               mov    bx,ax
               mov    al,36h           ; Set new timer interrupt 8
               out    43h,al           ; Frequenz
               mov    ax,bx
               out    40h,al
               mov    al,ah
               out    40h,al

               RET
ENDP INT8_Timer_speed


;ͻ
; GLOBAL TRACK EVENTS         
;ͼ

;Ŀ
;  GTRACK Event 1  SETSPEED Ticks per Second  
;
Gtrack_Set_Tickspeed:
               xor    ah,ah ; ax = effect data
               inc    ax    ; range 1-256 !
               cmp    [extern_speed],0
               jne @@exit
               call Set_TickSpeed
@@exit:
               jmp No_Gtrack_Entry

;=== GTRACK Event 1 END

;Ŀ
;  GTRACK Event 2  SETSPEED Beats per Minute  
;
Gtrack_Set_BPMspeed:
               xor    ah,ah ; ax = effect data = BPM speed
               cmp    [extern_speed],0
               jne @@exit
               call Set_BPMSpeed
@@exit:
               jmp No_Gtrack_Entry  ; Set the new tick speed

;=== GTRACK Event 2 END


;Ŀ
;  GTRACK Event 3  SET BEAT                   
;
Gtrack_Set_Beat:
                                    ; al = effect data
               mov    [ds:Beat],al
               mov    ax,[ds:BPM]

               call Set_BPMSpeed

               jmp No_Gtrack_Entry

;=== GTRACK Event 3 END

;Ŀ
;  GTRACK Event 4  TICKDELAY                  
;
Gtrack_TickDelay:
                                    ; al = effect data
               push   cx
               xor    ah,ah
               mov    cl,[ds:Click_shifter]
               shl    ax,cl
               shr    ax,4
               mov    [ds:TickDelay],ax
               pop    cx
               jmp No_Gtrack_Entry

;Ŀ
;  GTRACK Event 5  SET EXTERNAL FLAG          
;
Gtrack_Set_Flag:
                                    ; al = effect data
               mov    [ds:Music_Flag],al
               jmp No_Gtrack_Entry

;Ŀ
;  GTRACK Event 6  SLIDE SPEED UP             
;
Gtrack_Slidespeed_Up:

               xor    ah,ah ; ax = effect data

               cmp    [ds:BPM],0
               je @@Tickspeed_UP

               add    ax,[ds:BPM]
               cmp    ax,255
               jb  @@BPMSpeed_up_ok
               mov    ax,255   ; maximum speed
@@BPMSpeed_up_ok:
               call Set_BPMSpeed
               jmp No_Gtrack_Entry

@@Tickspeed_UP:
               add    ax,[ds:Speed]
               cmp    ax,255
               jb @@TickSpeed_up_ok
               mov    ax,255   ; maximum speed
@@TickSpeed_up_ok:
               call Set_TickSpeed
               jmp No_Gtrack_Entry

;=== GTRACK Event 6 END

;Ŀ
;  GTRACK Event 7  SLIDE SPEED DOWN           
;
Gtrack_Slidespeed_Down:

               xor    ah,ah ; ax = effect data
               cmp    [ds:BPM],0
               je @@Tickspeed_Down

               sub    ax,[ds:BPM]
               neg    ax       ; [speed] - ax
               cmp    ax,0
               jg @@BPMSpeed_Down_ok
               mov    ax,1     ; minimum speed
@@BPMSpeed_Down_ok:
               call Set_BPMSpeed
               jmp No_Gtrack_Entry

@@Tickspeed_Down:
               sub    ax,[ds:Speed]
               neg    ax       ; [speed] - ax
               cmp    ax,0
               jg @@TickSpeed_Down_ok
               mov    ax,1     ; minimum speed
@@TickSpeed_Down_ok:
               call Set_TickSpeed
               jmp No_Gtrack_Entry

;=== GTRACK Event 7 END


;ͻ
; INSTRUMENT TRACK EFFECTS    
;ͼ
Instrument_effect_Set     EQU   No_Instrument_Effect

;Ŀ
; Set Instrument Event 1    Stop Sample                
;
Set_Stop_Sample:
               or     [ds:di+channel.status],08h    ; Set Instr. update bit
               mov    [ds:di+channel.Instr_Effect],0
                ; Instrument Update 0 --- Stop Sample
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 2    Stop Sample Loop           
;
Set_Stop_Sample_Loop:
               or     [ds:di+channel.status],08h    ; Set Instr. update bit
               mov    [ds:di+channel.Instr_Effect],1
                ; Instrument Update 1 --- Stop Sample Loop
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 3    Override Sample Volume     
;
Override_Sample_Volume:    ; + Restart Sample
               mov    al,[ds:di+Channel.status]
               test   al,04h
               jz @@Only_restart
; if sample in sample slot, restore orginal volume stored in Instr_data1
               mov   ax,[ds:di+channel.Instr_data1]
               mov   [ds:di+channel.Volume],al
@@Only_restart:
               or     [ds:di+channel.status],74h
                ; Set Instr. Trigger & Instr Chg & Note & Volume & Balance bit
                ; Because of a voice exchange !!!
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 4    Sample Delay               
;
Set_Sample_Delay:
               mov    bl,ah  ; bl = Effect data
               shr    bl,4   ; bl = Parameter (range 0-15)
               inc    bl
               mov    ax,255 ; should be 256 but...
               div    bl
               xor    ah,ah
               inc    ax     ; to adjust drop out with paramenter 15
               shl    ax,4
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Instr_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Instr_data2],ax
               mov    ax,offset Sample_Delay
               mov    [ds:di+Channel.Instr_Adr],ax
               and    [ds:di+Channel.status],0fbh  ; clear sample trigger bit
               jmp Instrument_effect_Set

;Ŀ
; SAMPLE DELAY        Click Routine              
;
Sample_Delay:
               mov    ax,[ds:si+Channel.Instr_data2]
               add    ax,[ds:si+Channel.Instr_data1]
               or     ah,ah
               jz @@Dont_trigger
;--- Trigger the Sample
               or     [ds:si+Channel.status],74h
                ; Set Instr. Trigger & Instr Chg & Note & Volume & Balance bit
                ; Because of a Voice Exchange !!!
               xor    ax,ax
               mov    [ds:si+Channel.Instr_data1],ax ; stop counter

@@Dont_trigger:
               mov    [ds:si+Channel.Instr_data2],ax
               jmp Instrument_Effect_Done

;Ŀ
; Set Instrument Event 5    Tremolo Sample             
;
Tremolo_Sample:
               mov    bl,ah  ; bl = effect data
               shr    bl,4   ; bl = parameter (range 0-15)
               inc    bl
               mov    ax,255 ; should be 256 but...
               div    bl
               xor    ah,ah
               inc    ax     ; to adjust drop out with paramenter 15
               shl    ax,4
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Instr_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Instr_data2],ax
               mov    ax,offset Sample_Tremolo
               mov    [ds:di+Channel.Instr_Adr],ax
               jmp Instrument_effect_Set

;Ŀ
; SAMPLE TREMOLO      Click Routine              
;
Sample_Tremolo:
               mov    ax,[ds:si+Channel.Instr_data2]
               add    ax,[ds:si+Channel.Instr_data1]
               or     ah,ah
               jz @@Dont_Retrigger
;--- Retrigger the Sample
               or     [ds:si+Channel.status],74h
                ; set Instr. Trigger & Instr Chg & Note & Volume & Balance bit
                ; because of a Voice Exchange !!!
               xor    ah,ah
@@Dont_Retrigger:
               mov    [ds:si+Channel.Instr_data2],ax
               jmp Instrument_Effect_Done

;Ŀ
; Set Instrument Event 6    Set Sample Offset 0-64k           
;
Set_Sample_Off0:
               mov    al,ah      ; ah = Effect data
               xor    al,al      ; ax = Sample offset *256
               xor    bx,bx      ; 0 * 64k
               mov    [ds:di+channel.Instr_data1],ax ; save offset low
               mov    [ds:di+channel.Instr_data2],bx ; save offset high
               or     [ds:di+channel.status],07ch
                ; Set Instr. update & Instr Chg & Note & Volume & Balance bit
                ; Force a voice exchange, to reduce WT-Klicks !
               mov    [ds:di+channel.Instr_Effect],3
                ; Instrument Update 3 --- Set Sample Offset
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 7    Set Sample Offset 64-128k         
;
Set_Sample_Off64:
               mov    al,ah      ; ah = Effect data
               xor    al,al      ; ax = Sample offset *256
               mov    bx,1       ; 1 * 64k
               mov    [ds:di+channel.Instr_data1],ax ; save offset low
               mov    [ds:di+channel.Instr_data2],bx ; save offset high
               or     [ds:di+channel.status],07ch
                ; Set Instr. Update & Instr Chg & Note & Volume & Balance bit
                ; Force a voice exchange, to reduce WT-Klicks !
               mov    [ds:di+channel.Instr_Effect],3
                ; Instrument Update 3 --- Set Sample Offset
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 8    Set Sample Offset 128-192k        
;
Set_Sample_Off128:
               mov    al,ah      ; ah = Effect data
               xor    al,al      ; ax = Sample offset *256
               mov    bx,2       ; 2 * 64k
               mov    [ds:di+channel.Instr_data1],ax ; save offset low
               mov    [ds:di+channel.Instr_data2],bx ; save offset high
               or     [ds:di+channel.status],07ch
                ; Set Instr. Update & Instr Chg & Note & Volume & Balance bit
                ; Force a voice exchange, to reduce WT-Klicks !
               mov    [ds:di+channel.Instr_Effect],3
                ; Instrument Update 3 --- Set Sample Offset
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 9    Set Sample Offset 192-256k        
;
Set_Sample_Off192:
               mov    al,ah      ; ah = Effect data
               xor    al,al      ; ax = Sample offset *256
               mov    bx,3       ; 3 * 64k
               mov    [ds:di+channel.Instr_data1],ax ; save offset low
               mov    [ds:di+channel.Instr_data2],bx ; save offset high
               or     [ds:di+channel.status],07ch
                ; Set Instr. Update & Instr Chg & Note & Volume & Balance bit
                ; Force a voice exchange, to reduce WT-Klicks !
               mov    [ds:di+channel.Instr_Effect],3
                ; Instrument Update 3 --- Set Sample Offset
               jmp Instrument_effect_Set

;Ŀ
; Set Instrument Event 10   Invert Instrument Play direction  
;
Invert_Sample_direction:
;               mov    al,ah      ; No Effect data
;               xor    ah,ah
;               mov    [ds:di+channel.Instr_data1],ax ; save direction
               or     [ds:di+channel.status],08h     ; Set Instr. Update bit
               mov    [ds:di+channel.Instr_Effect],2
                ; Instrument Update 2 --- Invert Play direction
               jmp Instrument_effect_Set


;ͻ
; NOTE TRACK EFFECTS          
;ͼ
Note_effect_Set     EQU   No_Note_Effect

;Ŀ
; Set Note Event 1    Note Finetune              
;
Note_Finetune:
               mov    al,ah
               cbw          ; ax = Signed Finettune
               add    [ds:di+Channel.Note],ax    ; Add Fine Tune value
               or     [ds:di+Channel.Status],10h ; Set Note Change bit
               jmp Note_effect_Set

;Ŀ
; Set Note Event 2    Set_Note_Delay             
;
Set_Note_Delay:
               mov    al,[ds:di+Channel.status]
               test   al,10h  ; Check if note was updated this Tick
               jz @@No_effect
               mov    bl,ah  ; bl = Effect data
               shr    bl,4   ; bl = Parameter (range 0-15)
               inc    bl
               mov    ax,255 ; should be 256 but...
               div    bl
               xor    ah,ah
               inc    ax     ; to adjust drop out with paramenter 15
               shl    ax,4
               mov    cl,[Click_shifter]
               shr    ax,cl
               xchg   [ds:di+Channel.Note_data1],ax
               xchg   [ds:di+Channel.Note],ax      ; Swap new note with old Note
               mov    [ds:di+Channel.Note_data3],ax ; Store new note
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    ax,offset Note_Delay
               mov    [ds:di+Channel.Note_Adr],ax
               and    [ds:di+Channel.status],0efh  ; Clear Note Change bit
@@No_effect:
               jmp Note_effect_Set

;Ŀ
; NOTE DELAY          Click Routine              
;
Note_Delay:
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               or     ah,ah
               jz @@Dont_Update_Note
;--- Set new Note
               mov    ax,[ds:si+Channel.Note_data3]
               mov    [ds:si+Channel.Note],ax       ; Set stored note
               or     [ds:si+Channel.status],10h    ; Note update
               xor    ax,ax
               mov    [ds:si+Channel.Note_data1],ax ; Stop Counter

@@Dont_Update_Note:
               mov    [ds:si+Channel.Note_data2],ax
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 3    Arpeggio                   
;
Set_NoteArpeggio:
               mov    al,ah
               xor    ah,ah ; ax = Effect data
               mov    [ds:di+Channel.Note_data1],ax
               mov    ax,[ds:di+Channel.Note]
               mov    [ds:di+Channel.Note_data2],ax
               mov    ax,offset NoteArpeggio
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; ARPEGGIO            Click Routine              
;
NoteArpeggio:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               xor    ax,ax
               mov    al,[ds:clickcounter]
               dec    al
               mov    bl,3
               div    bl
               cmp    ah,1
               jz     NoteArpeggio1
               cmp    ah,2
               jz     NoteArpeggio2
NoteArpeggio0:
               mov    ax,[ds:si+Channel.Note_data2]
               mov    [ds:si+channel.Note],ax
               jmp Note_Effect_Done
NoteArpeggio1:
               mov    ax,[ds:si+Channel.Note_data1]
               and    al,0f0h    ; Get upper nibble+set to half note value
               add    ax,[ds:si+Channel.Note_data2]
               mov    [ds:si+channel.Note],ax
               jmp Note_Effect_Done
NoteArpeggio2:
               mov    ax,[ds:si+Channel.Note_data1]
               and    al,0fh    ; Get lower nibble
               shl    ax,4      ; Set to half note value
               add    ax,[ds:si+Channel.Note_data2]
               mov    [ds:si+channel.Note],ax
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 4    Portamento Up              
;
Set_PortamentoUP:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    [ds:di+Channel.Note_data3],highest_Note ; highest possible Note
               mov    ax,offset NotePortamentoUP
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE PORTAMENTO UP  Click Routine              
;
NotePortamentoUP:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               xor    bx,bx
               mov    bl,ah
               add    [ds:si+Channel.Note],bx
               mov    bx,[ds:si+Channel.Note_data3] ; Portamento border
               cmp    [ds:si+Channel.Note],bx
               jae Note_PortamentoUP_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Note_data2],ax
               jmp Note_Effect_Done
Note_PortamentoUP_Overflow:
               mov    [ds:si+Channel.Note],bx  ; Set to border note
               xor    ax,ax
               mov    [ds:si+Channel.Note_data1],ax ; Stop portamento
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 5    Portamento Down            
;
Set_PortamentoDN:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    [ds:di+channel.Note_data3],1 ; lowest possible note
               mov    ax,offset NotePortamentoDN
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE PORTAMENTO DN  Click Routine              
;
NotePortamentoDN:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               xor    bx,bx
               mov    bl,ah
               sub    [ds:si+Channel.Note],bx
               jc Note_PortamentoDN_Overflow
               mov    bx,[ds:si+Channel.Note_data3] ; Portamento border
               cmp    [ds:si+Channel.Note],bx
               jbe Note_PortamentoDN_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Note_data2],ax
               jmp Note_Effect_Done
Note_PortamentoDN_Overflow:
               mov    [ds:si+Channel.Note],bx ; Set to Border Note
               xor    ax,ax
               mov    [ds:si+Channel.Note_data1],ax ; Disable Portamento
               mov    [ds:si+Channel.Note_data2],ax ; Disable Portamento
               jmp Note_Effect_Done


;Ŀ
; Set Note Event 6    Portamento to Note         
;
Set_PortamentoNOTE:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    al,[ds:di+channel.NoteBuffer]
               shl    ax,4
               mov    [ds:di+channel.Note_data3],ax ; Portamento Border
               cmp    ax,[ds:di+channel.Note]
               jb PortNoteDown
               mov    ax,offset NotePortamentoUP
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set
PortNoteDown:
               mov    ax,offset NotePortamentoDN
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; Set Note Event 7    Scratch to Note            
;
Set_Scratch_to_Note:
               cmp    ah,107
               jb @@note_ok  ; Check high limit 107 (Midi Note B-8)
               mov    ah,107
@@note_ok:     xor    al,al
               inc    ah     ; ah = Scratch to Note (DMF Note 1-108)
               mov    bx,[ds:di+Channel.Note]
               shr    bx,4
               cmp    ah,bl
               jb ScratchNoteDown
               sub    ah,bl
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    ax,offset Scratch_to_NoteUP
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

ScratchNoteDown:
               xchg   ah,bl       ; swap note and destination note
               sub    ah,bl
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    ax,offset Scratch_to_NoteDN
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; SCRATCH TO NOTE DN  Click Routine              
;
Scratch_to_NoteDN:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               xor    bx,bx
               mov    bh,ah
               xor    ah,ah
               mov    [ds:si+Channel.Note_data2],ax
               shr    bx,4
               mov    ax,[ds:si+Channel.Note]
               sub    ax,bx      ; No Border Check, because Scratch to Note limits the values
               and    ax,0fff0h  ; Only Halftone Steps
               mov    [ds:si+Channel.Note],ax
               jmp Note_Effect_Done

;Ŀ
; SCRATCH TO NOTE UP  Click Routine              
;
Scratch_to_NoteUP:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               xor    bx,bx
               mov    bh,ah
               xor    ah,ah
               mov    [ds:si+Channel.Note_data2],ax
               shr    bx,4
               mov    ax,[ds:si+Channel.Note]
               add    ax,bx      ; No Border Check, because Scratch to Note limits the values
               and    ax,0fff0h  ; Only Halftone Steps
               mov    [ds:si+Channel.Note],ax
               jmp Note_Effect_Done


;Ŀ
; Set Note Event 8    Vibrato SIN                
;
Set_VibratoSIN:
               mov    bl,cl   ; cl = 0 = Restore Vibrato else restart Vibrato
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    al,ah   ; al = Effect data
               and    al,0fh
               inc    al       ; Amplitude 0 prevent (range 1-16)
               shl    al,1     ; Amplitude * 2 = 2 Halftone steps Up/Down
               or     bl,bl
               jne @@restart_VibratoSIN
               mov    bx,[ds:di+channel.Note_data3]
               mov    ah,bh         ; Copy Base Note
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
; Dont Restart Counter
;               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
               jmp @@restore_VibratoSIN

@@restart_VibratoSIN:
               mov    bx,[ds:di+channel.Note]
               shr    bx,4
               mov    ah,bl
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
@@restore_VibratoSIN:
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl

               mov    [ds:di+Channel.Note_data1],ax  ; Adder

               mov    ax,offset NoteVibratoSIN
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE VIBRATO SIN    Click Routine              
;
NoteVibratoSIN:
;--- Calc new Position
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1] ; add adder to counter
               mov    [ds:si+Channel.Note_data2],ax
               mov    bl,ah
               and    bx,7fh  ; address the Sindata Table (bit 0-6)
               mov    al,[ds:Sintable_128+bx]  ; al = sin faktor
               shl    ah,1    ; signbit in Carry
               jc NoteSIN_negative

;--- Positive SIN faktor
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
               mul    bl
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,ah
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               add    bx,ax      ; Base Note + Vibrato Position

               mov    [ds:si+Channel.Note],bx  ; Set new note value
               jmp Note_Effect_Done

;--- Negativ SIN faktor
NoteSIN_negative:
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
               mul    bl
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,ah
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               sub    bx,ax      ; Base Note - Vibrato Position
               jc Note_VibratoSin_Overflow
                ; jumped over lowest Note so leave old Note

               mov    [ds:si+Channel.Note],bx  ; Set new Note Value
               jmp Note_Effect_Done

Note_VibratoSin_Overflow:
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 9    Vibrato SLIDE              
;
Set_VibratoSLIDE:
               mov    bl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    al,ah   ; al = Effect data
               and    al,0fh
               inc    al       ; Amplitude 0 prevent (range 1-16)
               shl    al,1     ; Amplitude * 2 = 2 Halftone steps Up/Down
               or     bl,bl
               jne @@restart_VibratoSLIDE
               mov    bx,[ds:di+channel.Note_data3]
               mov    ah,bh         ; Copy Base Note
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
; Dont Restart Counter
;               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
               jmp @@restore_VibratoSLIDE

@@restart_VibratoSLIDE:
               mov    bx,[ds:di+channel.Note]
               shr    bx,4
               mov    ah,bl
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
@@restore_VibratoSLIDE:
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl

               mov    [ds:di+Channel.Note_data1],ax  ; Adder

               mov    ax,offset NoteVibratoSLIDE
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE VIBRATO SLIDE  Click Routine              
;
NoteVibratoSLIDE:
;--- Calc new Position
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Note_data2],ax
               mov    al,ah
;               and    bx,7fh  ; address the Sindata Table (bit 0-6)
;               mov    al,[ds:Sintable_128+bx]  ; al = Sin Faktor
               shl    al,1    ; signbit in Carry
               jc NoteSLIDE_negative

;--- Positive SLIDE faktor
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
               mul    bl
; Amplitude * Slide faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,ah
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               add    bx,ax      ; Base Note + Vibrato Position

               mov    [ds:si+Channel.Note],bx  ; Set new Note Value
               jmp Note_Effect_Done

;--- Negativ SLIDE faktor
NoteSLIDE_negative:
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
               mul    bl
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,ah
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               sub    bx,ax      ; Base Note - Vibrato Position
               jc Note_VibratoSLIDE_Overflow
                ; jumped over lowest Note so leave old Note

               mov    [ds:si+Channel.Note],bx  ; Set new Note Value
               jmp Note_Effect_Done

Note_VibratoSLIDE_Overflow:
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 10   Vibrato SQUARE             
;
Set_VibratoSQUARE:
               mov    bl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    al,ah   ; al = Effect data
               and    al,0fh
               inc    al       ; Amplitude 0 prevent (range 1-16)
               shl    al,1     ; Amplitude * 2 = 2 Halftone steps Up/Down
               or     bl,bl
               jne @@restart_VibratoSQUARE
               mov    bx,[ds:di+channel.Note_data3]
               mov    ah,bh         ; Copy Base Note
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
; Dont Restart Counter
;               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
               jmp @@restore_VibratoSQUARE

@@restart_VibratoSQUARE:
               mov    bx,[ds:di+channel.Note]
               shr    bx,4
               mov    ah,bl
               mov    [ds:di+channel.Note_data3],ax
                ; al = Amplitude, ah = Base Note
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax  ; Restart Counter
@@restore_VibratoSQUARE:
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl

               mov    [ds:di+Channel.Note_data1],ax  ; Adder

               mov    ax,offset NoteVibratoSQUARE
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE VIBRATO SQUARE Click Routine              
;
NoteVibratoSQUARE:
;--- Calc new Position
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Note_data2],ax
               shl    ah,1    ; signbit in Carry
               jc NoteSQUARE_negative

;--- Positive SLIDE faktor
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
; Amplitude * Slide faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,bl
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               add    bx,ax      ; Base Note + Vibrato Position

               mov    [ds:si+Channel.Note],bx  ; Set new Note Value
               jmp Note_Effect_Done

;--- Negativ SLIDE faktor
NoteSQUARE_negative:
               mov    bx,[ds:si+Channel.Note_data3]  ; BL = Amplitude, BH = BaseNote
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)
               mov    al,bl
               xor    ah,ah      ; ax = Note Adder

               mov    bl,bh
               xor    bh,bh
               shl    bx,4       ; Shift to Finetune Note, BX = Base Note

               sub    bx,ax      ; Base Note - Vibrato Position
               jc Note_VibratoSQUARE_Overflow
                ; jumped over lowest Note so leave old Note

               mov    [ds:si+Channel.Note],bx  ; Set new Note Value
               jmp Note_Effect_Done

Note_VibratoSQUARE_Overflow:
               jmp Note_Effect_Done

;Ŀ
; Set Note Event 11   Note Tremolo               
;
Set_NoteTremolo:
               or     cl,cl   ; cl = 0 = Restore Tremolo else Restart Tremolo
               jne @@restart_NoteTremolo
;--- Calc Speed1
               mov    cl,[Click_shifter]
               xor    bx,bx
               mov    bl,ah   ; bx = effect data
               and    bl,0fh  ; low Nibble = Speed 1
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; (Speed1 /16)+1
               mov    al,bl   ; al = Speed1
;--- Calc Speed2
               mov    bl,ah   ; bx = effect data
               shr    bl,4    ; high Nibble = Speed 2
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; (Speed2 /16)+1
               mov    ah,bl   ; al = Speed2
               mov    [ds:di+channel.Note_data2],ax ; al = Speed1 ah = Speed2
;--- Dont reset Start Note
;--- Dont Restart Counter, Tremolo Position
               jmp @@restore_NoteTremolo

@@restart_NoteTremolo:
;--- Calc Speed1
               mov    cl,[Click_shifter]
               xor    bx,bx
               mov    bl,ah   ; bx = effect data
               and    bl,0fh  ; low Nibble = Speed 1
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; Speed1 /16
               mov    al,bl   ; al = Speed1
;--- Calc Speed2
               mov    bl,ah   ; bx = effect data
               shr    bl,4    ; high Nibble = Speed 2
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; Speed2 /16
               mov    ah,bl   ; al = Speed2
               mov    [ds:di+channel.Note_data2],ax ; al = Speed1 ah = Speed2
               mov    ax,[ds:di+channel.Note]
               mov    [ds:di+channel.Note_data1],ax
                ; ax = Orginal Note
               xor    ax,ax
               mov    [ds:di+channel.Note_data3],ax ; Note Counter
                ; ah = Note Position al = Tremolo Position

@@restore_NoteTremolo:

               mov    ax,offset NoteTremolo
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE TREMOLO        Click Routine              
;
NoteTremolo:
               or     [ds:si+Channel.Status],10h ; Set Note Change bit
               mov    ax,[ds:si+Channel.Note_data3]
               mov    bl,al
               shr    bl,1    ; Tremolo Position
               jnc Note_Tremolo_muted
;--- Tremolo Postion Start Note
               mov    bx,[ds:si+Channel.Note_data2]
               inc    ah
               cmp    ah,bl
               jb @@keep_tremolo_StartNote
               xor    ah,ah  ; reset counter
               inc    al     ; toggle tremolo Position
@@keep_tremolo_StartNote:
               mov    [ds:si+Channel.Note_data3],ax ; save new Position
               mov    ax,[ds:si+Channel.Note_data1] ; Start Note
               mov    [ds:si+Channel.Note],ax ; Set Start Note
               jmp Note_Effect_Done

;--- Tremolo Postion muted
Note_Tremolo_muted:
               mov    bx,[ds:si+Channel.Note_data2]
               inc    ah
               cmp    ah,bh
               jb @@keep_tremolo_muted
               xor    ah,ah  ; reset counter
               inc    al     ; toggle tremolo Position
@@keep_tremolo_muted:
               mov    [ds:si+Channel.Note_data3],ax ; save new Position
               mov    [ds:si+Channel.Note],0 ; mute Note
               jmp Note_Effect_Done

;=== Note Effect 11 END

;Ŀ
; Set Note Event 12   Set_Note_Cut               
;
Set_Note_Cut:
               mov    bl,ah  ; bl = Effect data
               shr    bl,4   ; bl = Parameter (range 0-15)
               inc    bl
               mov    ax,255 ; should be 256 but...
               div    bl
               xor    ah,ah
               inc    ax     ; to adjust drop out with paramenter 15
               shl    ax,4
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Note_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Note_data2],ax
               mov    ax,offset Note_Cut
               mov    [ds:di+Channel.Note_Adr],ax
               jmp Note_effect_Set

;Ŀ
; NOTE CUT            Click Routine              
;
Note_Cut:
               mov    ax,[ds:si+Channel.Note_data2]
               add    ax,[ds:si+Channel.Note_data1]
               or     ah,ah
               jz @@Dont_Cut_Note
;--- Set Note Off
               xor    ax,ax
               mov    [ds:si+Channel.Note],ax       ; Set Note Off
               or     [ds:si+Channel.status],10h    ; Note Update
               mov    [ds:si+Channel.Note_data1],ax ; Stop Counter

@@Dont_Cut_Note:
               mov    [ds:si+Channel.Note_data2],ax
               jmp Note_Effect_Done



;ͻ
; VOLUME TRACK EFFECTS        
;ͼ
Volume_effect_Set     EQU   No_Volume_Effect

;Ŀ
; Set Volume Event 1  Volume Slide Up            
;
Set_VolumeSlideUP:
               xor    al,al ; ax = effect data
               mov    cl,[ds:Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Volume_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Volume_data2],ax
               mov    ax,offset VolumeSlideUP
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; VOLUME SLIDE UP     Click Routine              
;
VolumeSlideUP:
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1]
               add    [ds:si+Channel.Volume],ah
               jc Volume_SlideUP_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Volume_data2],ax
               jmp Volume_Effect_Done
Volume_SlideUP_Overflow:
               mov    ax,0ffffh
               mov    [ds:si+Channel.Volume_data1],ax
               mov    [ds:si+Channel.Volume],255
               jmp Volume_Effect_Done

;=== Volume Effect 1 END

;Ŀ
; Set Volume Event 2  Volume Slide Down          
;
Set_VolumeSlideDN:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Volume_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Volume_data2],ax
               mov    ax,offset VolumeSlideDN
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; VOLUME SLIDE DN     Click Routine              
;
VolumeSlideDN:
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1]
               sub    [ds:si+Channel.Volume],ah
               jc Volume_SlideDN_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Volume_data2],ax
               jmp Volume_Effect_Done
Volume_SlideDN_Overflow:
               xor    ax,ax
               mov    [ds:si+Channel.Volume_data1],ax
               mov    [ds:si+Channel.Volume],1
               jmp Volume_Effect_Done

;=== Volume Effect 2 END

;Ŀ
; Set Volume Event 3  Volume Tremolo             
;
Set_VolumeTremolo:
               or     cl,cl   ; cl = 0 = Restore Tremolo else Restart Tremolo
               jne @@restart_VolumeTremolo
;--- Calc Speed1
               mov    cl,[Click_shifter]
               xor    bx,bx
               mov    bl,ah   ; bx = effect data
               and    bl,0fh  ; low Nibble = Speed 1
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; (Speed1 /16)+1
               mov    al,bl   ; al = Speed1
;--- Calc Speed2
               mov    bl,ah   ; bx = effect data
               shr    bl,4    ; high Nibble = Speed 2
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; (Speed2 /16)+1
               mov    ah,bl   ; al = Speed2
               mov    [ds:di+channel.Volume_data2],ax ; al = Speed1 ah = Speed2
;--- Dont reset Start Volume
;--- Dont Restart Counter, Tremolo Position
               jmp @@restore_VolumeTremolo

@@restart_VolumeTremolo:
;--- Calc Speed1
               mov    cl,[Click_shifter]
               xor    bx,bx
               mov    bl,ah   ; bx = effect data
               and    bl,0fh  ; low Nibble = Speed 1
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; Speed1 /16
               mov    al,bl   ; al = Speed1
;--- Calc Speed2
               mov    bl,ah   ; bx = effect data
               shr    bl,4    ; high Nibble = Speed 2
               inc    bl      ; speed+1 (range 1-16)
               shl    bx,cl
               shr    bx,4    ; Speed2 /16
               mov    ah,bl   ; al = Speed2
               mov    [ds:di+channel.Volume_data2],ax ; al = Speed1 ah = Speed2
               xor    ax,ax
               mov    al,[ds:di+channel.Volume]
               mov    [ds:di+channel.Volume_data1],ax
                ; al = Orginal Volume
               xor    ax,ax
               mov    [ds:di+channel.Volume_data3],ax ; Volume Counter
                ; ah = Volume Position al = Tremolo Position

@@restore_VolumeTremolo:

               mov    ax,offset VolumeTremolo
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; VOLUME TREMOLO      Click Routine              
;
VolumeTremolo:
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data3]
               mov    bl,al
               shr    bl,1    ; Tremolo Position
               jnc Volume_Tremolo_muted
;--- Tremolo Postion Start Volume
               mov    bx,[ds:si+Channel.Volume_data2]
               inc    ah
               cmp    ah,bl
               jb @@keep_tremolo_StartVolume
               xor    ah,ah  ; reset counter
               inc    al     ; toggle tremolo Position
@@keep_tremolo_StartVolume:
               mov    [ds:si+Channel.Volume_data3],ax ; save new Position
               mov    ax,[ds:si+Channel.Volume_data1] ; Start Volume
               mov    [ds:si+Channel.Volume],al ; Set Start Volume
               jmp Volume_Effect_Done

;--- Tremolo Postion muted
Volume_Tremolo_muted:
               mov    bx,[ds:si+Channel.Volume_data2]
               inc    ah
               cmp    ah,bh
               jb @@keep_tremolo_muted
               xor    ah,ah  ; reset counter
               inc    al     ; toggle tremolo Position
@@keep_tremolo_muted:
               mov    [ds:si+Channel.Volume_data3],ax ; save new Position
               mov    [ds:si+Channel.Volume],1 ; mute Volume
               jmp Volume_Effect_Done

;=== Volume Effect 3 END

;Ŀ
; Set Volume Event 4  Volume Vibrato SIN         
;
Set_VolVibratoSIN:
               or     cl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               jne @@restart_VolVibratoSIN
;--- Calc Amplitude
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    bl,ah   ; bl = ah = Effect data
               and    bl,0fh  ; bl = Amplitude
               inc    bl      ; Amplitude range 1-16
               mov    ax,[ds:di+channel.Volume_data3]
               add    ah,al   ; Calc Start Volume
               mov    al,ah
               xor    ah,ah   ; ax = Start Volume
               mov    bh,al   ; bh = Start Volume
               mul    bl      ; (Start Volume*Amplitude)/32
               shr    ax,5    ; al = Amplitude
               mov    ah,bh   ; ah = Start Volume
               sub    ah,al   ; Set new Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
;--- Dont Restart Counter
               jmp @@restore_VolVibratoSIN

@@restart_VolVibratoSIN:

               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

;--- Calc Amplitude
               mov    bl,ah   ; cl = ah = Effect data
               and    bl,0fh  ; cl = Amplitude
               inc    bl      ; Amplitude range 1-16
               xor    ax,ax
               mov    al,[ds:di+channel.Volume]
               mul    bl      ; (Start Volume * Amplitude)/32
               shr    ax,5    ; al = Amplitude

               mov    ah,[ds:di+channel.Volume]
               sub    ah,al    ; Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
               mov    ax,4000h  ; Set Current Position 1/4 Phase (highest Amplitude)
               mov    [ds:di+Channel.Volume_data2],ax  ; reset Counter
@@restore_VolVibratoSIN:
;--- Calc Speed
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl
               shl    ax,1   ; double Speed ! (MAY CORRUPT ... CHECK !)

               mov    [ds:di+Channel.Volume_data1],ax  ; Adder

               mov    ax,offset VolumeVibratoSIN
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; VOLUME VIBRATO SIN  Click Routine              
;
VolumeVibratoSIN:
;--- Calc new Position
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Volume_data2],ax
               mov    bl,ah
               and    bx,7fh  ; address the Sindata Table (bit 0-6)
               mov    al,[ds:Sintable_128+bx]  ; al = Sin Faktor
               shl    ah,1    ; signbit in Carry
               jc VolumeSIN_negative

;--- Positive SIN faktor
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               mul    bl         ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)

               add    bh,ah      ; Base Volume + Vibrato Position

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

;--- Negativ SIN faktor
VolumeSIN_negative:
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               mul    bl        ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)

               sub    bh,ah      ; Base Volume - Vibrato Position
               jc Volume_VibratoSin_Overflow
                ; jumped over lowest Volume so leave old Volume
                ; I think not needed, but....

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

Volume_VibratoSin_Overflow:
               jmp Volume_Effect_Done

;=== Volume Effect 4 END

;Ŀ
; Set Volume Event 5  Volume Vibrato SLIDE       
;
Set_VolVibratoSLIDE:
               or     cl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               jne @@restart_VolVibratoSLIDE
;--- Calc Amplitude
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    bl,ah   ; bl = ah = Effect data
               and    bl,0fh  ; bl = Amplitude
               inc    bl      ; Amplitude range 1-16
               mov    ax,[ds:di+channel.Volume_data3]
               add    ah,al   ; Calc Start Volume
               mov    al,ah
               xor    ah,ah   ; ax = Start Volume
               mov    bh,al   ; bh = Start Volume
               mul    bl      ; (Start Volume*Amplitude)/32
               shr    ax,5    ; al = Amplitude
               mov    ah,bh   ; ah = Start Volume
               sub    ah,al   ; Set new Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
;--- Dont Restart Counter
               jmp @@restore_VolVibratoSLIDE

@@restart_VolVibratoSLIDE:

               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

;--- Calc Amplitude
               mov    bl,ah   ; cl = ah = Effect data
               and    bl,0fh  ; cl = Amplitude
               inc    bl      ; Amplitude range 1-16
               xor    ax,ax
               mov    al,[ds:di+channel.Volume]
               mul    bl      ; (Start Volume * Amplitude)/32
               shr    ax,5    ; al = Amplitude

               mov    ah,[ds:di+channel.Volume]
               sub    ah,al    ; Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
               mov    ax,4000h  ; Set Current Position 1/4 Phase (highest Amplitude)
               mov    [ds:di+Channel.Volume_data2],ax  ; reset Counter
@@restore_VolVibratoSLIDE:
;--- Calc Speed
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl
               shl    ax,1   ; double Speed ! (MAY CORRUPT ... CHECK !)

               mov    [ds:di+Channel.Volume_data1],ax  ; Adder

               mov    ax,offset VolumeVibratoSLIDE
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
;VOLUME VIBRATO SLIDE Click Routine              
;
VolumeVibratoSLIDE:
;--- Calc new Position
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Volume_data2],ax
               mov    al,ah
               shl    al,1     ; signbit in Carry
               jc VolumeSLIDE_negative

;--- Positive SLIDE faktor
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               mul    bl         ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)

               add    bh,ah      ; Base Volume + Vibrato Position

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

;--- Negativ SLIDE faktor
VolumeSLIDE_negative:
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               mul    bl        ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)

               sub    bh,ah      ; Base Volume - Vibrato Position
               jc Volume_VibratoSLIDE_Overflow
                ; jumped over lowest Volume so leave old Volume
                ; I think not needed, but....

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

Volume_VibratoSLIDE_Overflow:
               jmp Volume_Effect_Done

;=== Volume Effect 5 END

;Ŀ
; Set Volume Event 6  Volume Vibrato SQUARE      
;
Set_VolVibratoSQUARE:
               or     cl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               jne @@restart_VolVibratoSQUARE
;--- Calc Amplitude
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    bl,ah   ; bl = ah = Effect data
               and    bl,0fh  ; bl = Amplitude
               inc    bl      ; Amplitude range 1-16
               mov    ax,[ds:di+channel.Volume_data3]
               add    ah,al   ; Calc Start Volume
               mov    al,ah
               xor    ah,ah   ; ax = Start Volume
               mov    bh,al   ; bh = Start Volume
               mul    bl      ; (Start Volume*Amplitude)/32
               shr    ax,5    ; al = Amplitude
               mov    ah,bh   ; ah = Start Volume
               sub    ah,al   ; Set new Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
;--- Dont Restart Counter
               jmp @@restore_VolVibratoSQUARE

@@restart_VolVibratoSQUARE:

               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

;--- Calc Amplitude
               mov    bl,ah   ; cl = ah = Effect data
               and    bl,0fh  ; cl = Amplitude
               inc    bl      ; Amplitude range 1-16
               xor    ax,ax
               mov    al,[ds:di+channel.Volume]
               mul    bl      ; (Start Volume * Amplitude)/32
               shr    ax,5    ; al = Amplitude

               mov    ah,[ds:di+channel.Volume]
               sub    ah,al    ; Base Volume (Start Volume-Amplitude)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Volume
               mov    ax,4000h  ; Set Current Position 1/4 Phase (highest Amplitude)
               mov    [ds:di+Channel.Volume_data2],ax  ; reset Counter
@@restore_VolVibratoSQUARE:
;--- Calc Speed
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares ;-)
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl
               shl    ax,1   ; double Speed ! (MAY CORRUPT ... CHECK !)

               mov    [ds:di+Channel.Volume_data1],ax  ; Adder

               mov    ax,offset VolumeVibratoSQUARE
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
;VOLUME VIBRATO SQUARE Click Routine              
;
VolumeVibratoSQUARE:
;--- Calc new Position
               or     [ds:si+Channel.Status],20h ; Set Volume Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Volume_data2],ax
               mov    al,ah
               shl    al,1     ; signbit in Carry
               jc VolumeSQUARE_negative

;--- Positive SQUARE faktor
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               add    bh,bl      ; Base Volume + Vibrato Position

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

;--- Negativ SQUARE faktor
VolumeSQUARE_negative:
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseVolume
               sub    bh,bl      ; Base Volume - Vibrato Position
               jc Volume_VibratoSQUARE_Overflow
                ; jumped over lowest Volume so leave old Volume
                ; I think not needed, but....

               mov    [ds:si+Channel.Volume],bh  ; Set new Volume Value
               jmp Volume_Effect_Done

Volume_VibratoSQUARE_Overflow:
               jmp Volume_Effect_Done

;=== Volume Effect 6 END

;Ŀ
; Set Volume Event 7  Set Balance                
;
Set_VolumeBalance:
               cmp    [ds:extern_Balance],0
               jne @@exit_Set_VolumeBalance
               mov    [ds:di+Channel.Balance],ah ; Set Balance
               or     [ds:di+Channel.Status],40h ; Set Balance Change bit
@@exit_Set_VolumeBalance:
               jmp Volume_effect_Set

;=== Volume Effect 7 END

;Ŀ
; Set Volume Event 8  Slide Balance Left         
;
Slide_Balance_Left:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Volume_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Volume_data2],ax
               mov    ax,offset BalanceSlideLeft
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; BALANCE SLIDE LEFT  Click Routine              
;
BalanceSlideLeft:
               or     [ds:si+Channel.Status],40h ; Set Balance Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1]
               sub    [ds:si+Channel.Balance],ah
               jc Balance_Slide_Left_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Volume_data2],ax
               jmp Volume_Effect_Done
Balance_Slide_Left_Overflow:
               xor    ax,ax
               mov    [ds:si+Channel.Volume_data1],ax
               mov    [ds:si+Channel.Balance],0
               jmp Volume_Effect_Done

;=== Volume Effect 8 END

;Ŀ
; Set Volume Event 9  Slide Balance Right        
;
Slide_Balance_Right:
               xor    al,al ; ax = effect data
               mov    cl,[Click_shifter]
               shr    ax,cl
               mov    [ds:di+Channel.Volume_data1],ax
               xor    ax,ax
               mov    [ds:di+Channel.Volume_data2],ax
               mov    ax,offset BalanceSlideRight
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
; BALANCE SLIDE RIGHT Click Routine              
;
BalanceSlideRight:
               or     [ds:si+Channel.Status],40h ; Set Balance Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1]
               add    [ds:si+Channel.Balance],ah
               jc Balance_Slide_Right_Overflow
               xor    ah,ah
               mov    [ds:si+Channel.Volume_data2],ax
               jmp Volume_Effect_Done
Balance_Slide_Right_Overflow:
               mov    ax,0ffffh
               mov    [ds:si+Channel.Volume_data1],ax
               mov    [ds:si+Channel.Balance],255
               jmp Volume_Effect_Done

;=== Volume Effect 9 END

;Ŀ
; Set Volume Event 10 Balance Vibrato SIN        
;
Set_BalanceVibratoSIN:
               or     cl,cl   ; cl = 0 = Restore Vibrato else Restart Vibrato
               jne @@restart_BalanceVibratoSIN
;--- Calc Amplitude
               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

               mov    bl,ah   ; bl = ah = Effect data
               and    bl,0fh  ; Amplitude
               inc    bl      ; Amplitude range 1-16
               shl    bl,3    ; Amplitude*8 (range 8-128)

               mov    ax,[ds:di+channel.Volume_data3]
               mov    al,bl   ; set new amplitude (restore Base Balance)
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = Base Balance
;--- Dont Restart Counter
               jmp @@restore_BalanceVibratoSIN

@@restart_BalanceVibratoSIN:

               mov    cl,ah   ; ch is used outside this routine !!!
               shr    cl,4    ; Mask Out Speed, cx = Speed
               inc    cl      ; Speed 0 prevent (range 1-16)

;--- Calc Amplitude
               mov    al,ah   ; al = ah = Effect data
               and    al,0fh  ; Amplitude
               inc    al      ; Amplitude range 1-16
               shl    al,3    ; Amplitude*8 (range 8-128)
               mov    ah,[ds:di+channel.Balance]
               mov    [ds:di+channel.Volume_data3],ax
                ; al = Amplitude, ah = base balance
               xor    ax,ax  ; Set Current Phase Position (0 Point)
               mov    [ds:di+Channel.Volume_data2],ax  ; reset counter

@@restore_BalanceVibratoSIN:
;--- Calc Speed
               mov    ax,255
               div    cl    ; Calc Vibrato Speed Adder (255/Speed)/Effect Clicks
                            ; Formel not 100% correct 256/Speed but F* who cares
               mov    ah,al
               mov    cl,[Click_shifter]
               shr    ax,cl

               mov    [ds:di+Channel.Volume_data1],ax  ; Adder

               mov    ax,offset BalanceVibratoSIN
               mov    [ds:di+Channel.Volume_Adr],ax
               jmp Volume_effect_Set

;Ŀ
;BALANCE VIBRATO SIN  Click Routine              
;
BalanceVibratoSIN:
;--- Calc new Position
               or     [ds:si+Channel.Status],40h ; Set Balance Change bit
               mov    ax,[ds:si+Channel.Volume_data2]
               add    ax,[ds:si+Channel.Volume_data1] ; Add Adder to Counter
               mov    [ds:si+Channel.Volume_data2],ax
               mov    bl,ah
               and    bx,7fh  ; address the Sindata Table (bit 0-6)
               mov    al,[ds:Sintable_128+bx]  ; al = Sin Faktor
               shl    ah,1    ; signbit in Carry
               jc BalanceSIN_negative

;--- Positive SIN faktor
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = Base Balance
               mul    bl         ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. sin faktor should be 256 !)

               add    bh,ah      ; Base Balance + Vibrato position
               jnc Balance_VibratoSIN_OK_right  ; check if balance in limits
               mov    bh,255      ; Balance right overflow

Balance_VibratoSIN_OK_right:
               mov    [ds:si+Channel.Balance],bh  ; Set new balance value
               jmp Volume_Effect_Done

;--- Negativ SIN faktor
BalanceSIN_negative:
               mov    bx,[ds:si+Channel.Volume_data3]  ; BL = Amplitude, BH = BaseBalance
               mul    bl        ; ah = Vibrato Amplitude
; Amplitude * Sin faktor (not 100% max. Sin faktor should be 256 !)

               sub    bh,ah      ; Base Balance - Vibrato Position
               jnc Balance_VibratoSIN_OK_left  ; check if balance in limits
               xor    bh,bh      ; Balance left overflow

Balance_VibratoSIN_OK_left:

               mov    [ds:si+Channel.Balance],bh  ; Set new balance value
               jmp Volume_Effect_Done

;=== Volume Effect 10 END


;--- You have reached the end...........
