//
// GIPC version 1.12c (C) 1996 by Gravis, (P) 1996 by ChAr
//
// ChAr's changes:
//   changed default layer attenuation to 15
//   effect1 = 1 with depth 0x28
//   if all waves in layer have same ballance->set it to iwLayer.pan
//   at end of patch is written 8bytes not 00 but last byte/word from
//     sample
//

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <direct.h>
#include "types.hpp"

// Decoded using TD

struct patchdata {
  char wave_name[7];
  byte fractions;
  long wave_size;
  long start_loop;
  long end_loop;
  word sample_rate;
  long low_frequency;
  long high_frequency;
  long root_frequency;
  int tune;
  byte balance;
  byte envelope_rate[6];
  byte envelope_offset[6];
  byte tremolo_sweep;
  byte tremolo_rate;
  byte tremolo_depth;
  byte vibrato_sweep;
  byte vibrato_rate;
  byte vibrato_depth;
  char modes;
  int scale_frequency;
  word scale_factor;
  char reserved[0x24];
};

struct layerdata {
  char layer_duplicate;
  char layer;
  long layer_size;
  char samples;
  char reserved[0x28];
};

struct instrumentdata {
  word instrument;
  char instrument_name[0x10];
  long instrument_size;
  char layers;
  char reserved[0x28];
};

struct patchheader {
  char header[0x0C];
  char gravis_id[0x0A];
  char description[0x3C];
  byte instruments;
  char voices;
  char channels;
  word wave_forms;
  word master_volume;
  dword data_size;
  char reserved[0x24];
};


// InterWave data patch file defs.

struct iw_disk_id {
  int major_id;
  int minor_id;
};

union ID {
  iw_disk_id id;
  void __far *p;
};

struct iw_program {
  ID id;
  ID version;
};

struct iw_envp_header {
  char num_envelopes;
  char flags;
  char mode;
  char index_type;
};

struct iw_envp {
  ID id;
  iw_envp_header h;
};

struct iw_envp_record {
  int nattack;
  int nrelease;
  word sustain_offset;
  word sustain_rate;
  word release_rate;
  byte hirange;
  byte pad;
};

struct iw_wave {
  ID id;
  dword size;
  dword start;
  dword loopstart;
  dword loopend;
  dword m_start;
  dword sample_ratio;
  char attenuation;
  char low_note;
  char high_note;
  byte format;
  byte m_format;
  ID data_id;
};

struct LFO {
  word freq;
  int depth;
  int sweep;
  byte shape;
  byte delay;
};

struct iw_layer {
  ID id;
  char nwaves;
  char flags;
  char high_range;
  char low_range;
  char pan;
  char pan_freq_scale;
  LFO tremolo;
  LFO vibrato;
  char velocity_mode;
  char attenuation;
  int freq_scale;
  char freq_center;
  char layer_event;
  ID penv;
  ID venv;
  iw_wave __far *waves;
};

struct iw_patch {
  ID id;
  int nlayers;
  char layer_mode;
  char exclusion_mode;
  int exclusion_group;
  char effect1;
  char effect1_depth;
  char effect2;
  char effect2_depth;
  byte bank;
  byte program;
  iw_layer __far *layers;
};


static layerdata *gusLayer[10];
static patchdata *gusWave[100];

static iw_layer *iwLayer[10];
static iw_envp *iwEnvp[10];
static iw_wave *iwWave[100];
static iw_envp_record *iwEnvpRecord[100];

static long waveStart;
static word *varPortPtr;
static byte *waveData;
static int dataId;

static FILE *iniHandle;
static FILE *datHandle;
static FILE *fffHandle;

// Helper functions

byte FrequencyToMIDI(long freq) {
  int n;
  double value;
  byte noteNum;

  value = freq/1000.0;
  if (value > 440.0) {
    n=0;
    while (value > 453.0) {
      n++;
      value /= 1.059463094L;
    }
    noteNum=0x45+n;
  } else {
    n=0;
    while (value < 428.0) {
      n++;
      value *= 1.059463094L;
    }
    noteNum=0x45-n;
  }
  if ((n < 0) || (n > 0x45)) {
    return 0;
  }
  return noteNum;
}


int ProcessLFOInfo(patchdata *gusWave, iw_layer *iwLayer) {
  iwLayer->tremolo.freq=gusWave->tremolo_rate*3;
  iwLayer->tremolo.depth=gusWave->tremolo_depth/19;
  iwLayer->tremolo.sweep=0;
  iwLayer->tremolo.shape=0;
  iwLayer->tremolo.delay=0;

  iwLayer->vibrato.freq=gusWave->vibrato_rate*3;
  iwLayer->vibrato.depth=gusWave->vibrato_depth*7/4;
  if (iwLayer->vibrato.depth > 0xFF) {
    iwLayer->vibrato.depth = 0xFF;
  }
  iwLayer->vibrato.sweep=gusWave->vibrato_sweep/2 + 10;
  iwLayer->vibrato.shape=0;
  iwLayer->vibrato.delay=0;
  return 1;
}


int ProcessEnvelopeInfo(patchdata *gusWave, iw_layer *iwLayer,
    iw_wave *iwWave, iw_envp_record *iwEnvpRec, word *localVarPtr) {
  int m;

  iwEnvpRec->nattack=3;
  iwEnvpRec->nrelease=3;

  if ((gusWave->modes & 0x20)!=0) { //Sustain On
    iwEnvpRec->sustain_rate=gusWave->envelope_rate[2];
  } else {
    iwEnvpRec->sustain_rate=0x8A;
  }

  iwEnvpRec->sustain_offset=gusWave->envelope_offset[2];
  iwEnvpRec->hirange=iwWave->high_note;
  iwEnvpRec->pad=0; //ChAr's fix-up for uninitialized data

  iwEnvpRec->release_rate=gusWave->envelope_rate[3];

  if ((gusWave->envelope_rate[0] == gusWave->envelope_rate[1]) &&
      (gusWave->envelope_rate[0] == gusWave->envelope_rate[2]) &&
      (gusWave->envelope_rate[0] == gusWave->envelope_rate[3]) &&
      (gusWave->envelope_rate[0] == gusWave->envelope_rate[4]) &&
      (gusWave->envelope_rate[0] == gusWave->envelope_rate[5])) {
    iwEnvpRec->nrelease=0;
    iwEnvpRec->nattack=1;
    iwEnvpRec->sustain_rate=0xCF;
    iwEnvpRec->sustain_offset=0xFD;
    iwEnvpRec->release_rate=0xCF;
    localVarPtr[0] = 0xFF;
    localVarPtr[1] = 0x7F;
    return 1;
  }

  for (m=0; m<6; m++) {
    if (m>0) {
      if (localVarPtr[m*2]==localVarPtr[(m-1)*2]) {
        localVarPtr[m*2] = localVarPtr[(m-1)*2]-1;
        if (localVarPtr[m*2] > 0xFF) {
          localVarPtr[m*2]=0;
        }
      } else {
        localVarPtr[m*2] = gusWave->envelope_offset[m];
      }
    } else {
      if (gusWave->envelope_offset[m] == iwEnvpRec->sustain_offset) {
        localVarPtr[m*2] = gusWave->envelope_offset[m]-1;
        if (localVarPtr[m*2] > 0xFF) {
          localVarPtr[m*2] = 0;
        }
      } else {
        localVarPtr[m*2] = gusWave->envelope_offset[m];
      }
    }
    localVarPtr[m*2+1] = gusWave->envelope_rate[m];

    if ((m==2) && (gusWave->envelope_offset[m]<=0xC8)) {
      iwEnvpRec->nrelease=0;
      break;
    }
  }
  return 1;
}


int ProcessWaveInfo(patchdata *gusWave, iw_wave *iwWave, int patchNum) {
  if ((gusWave->modes & 0x01)!=0) { //16bit wave
    iwWave->size = gusWave->wave_size / 2;
  } else {
    iwWave->size = gusWave->wave_size;
  }
  iwWave->start = waveStart;
  iwWave->m_start = 0L;
  iwWave->loopstart = gusWave->start_loop;
  if ((gusWave->modes & 0x01)!=0) { //16bit wave
    iwWave->loopstart >>= 1;
  }
  iwWave->loopstart <<= 4;
  iwWave->loopstart += gusWave->fractions & 0x0F;
  iwWave->loopend = gusWave->end_loop;
  if ((gusWave->modes & 0x01)!=0) { //16bit wave
    iwWave->loopend >>=1;
  }
  iwWave->loopend <<=4;
  iwWave->loopend += gusWave->fractions >> 4;
  iwWave->sample_ratio = (dword)(45158.4L / gusWave->sample_rate * gusWave->root_frequency);
  iwWave->attenuation=0;
  if (patchNum>127) {
    iwWave->low_note=0;
    iwWave->high_note=127;
  } else {
    iwWave->low_note=FrequencyToMIDI(gusWave->low_frequency);
    iwWave->high_note=FrequencyToMIDI(gusWave->high_frequency);
  }
  iwWave->format=0;
  if ((gusWave->modes & 0x01)==0) {
    iwWave->format|=0x01; // 8bit wave
  }
  if ((gusWave->modes & 0x02)==0) {
    iwWave->format|=0x02; // Signed wave
  }
  if ((gusWave->modes & 0x10)==0) {
    iwWave->format|=0x04; // loop forward
  }
  if ((gusWave->modes & 0x04)!=0) {
    iwWave->format|=0x08; // loop enabled
  }
  if ((gusWave->modes & 0x08)!=0) {
    iwWave->format|=0x10; // bi-direct loop
  }
  iwWave->m_format=0;
  iwWave->data_id.id.major_id=dataId;
  iwWave->data_id.id.minor_id=0;
  waveStart+= 8L+gusWave->wave_size;
  return 1;
}


int ConvertPatch(char *gusPatStr, FILE *fffHandle, FILE *datHandle, int bankNum,
    int patchNum, int exclude) {
  long size;
  iw_envp_record *tmpEnvpPtr;
  iw_patch iwPatch;
  iw_program iwProgram;
  instrumentdata gusInst;
  patchheader gusPatch;
  int m, hirange, result, n, tempWaveNum;
  int waveNum, layerNum;
  byte lastBalance;
  int setupPan;
  word lastWaveSample;

  FILE *patHandle;

  waveNum=0;

  if ((patHandle=fopen(gusPatStr, "rb")) == 0) {
    printf("Could not open %-12s. This patch will not be converted.\n",
      gusPatStr);
    return -1;
  }

  if ((bankNum < 0) || (bankNum > 255)) {
    printf("Bank number is out of range (0-255):%d\n", bankNum);
    fclose(patHandle);
    return -1;
  }

  if ((patchNum < 0) || (patchNum > 255)) {
    printf("Patch number is out of range (0-255):%d\n", patchNum);
    fclose(patHandle);
    return -1;
  }

  // Read GUS-patch into memory

  if (fread(&gusPatch, sizeof gusPatch, 1, patHandle) == 0) {
    printf("Error reading header of %-12s.\n", gusPatStr);
    fclose(patHandle);
    return -1;
  }

  if (strstr(gusPatch.header, "GF1PATCH110") == 0) {
    printf("Outdated patch version %s.\n", gusPatch.header);
    fclose(patHandle);
    return -1;
  }

  if (fread(&gusInst, sizeof gusInst, 1, patHandle) == 0) {
    printf("Error reading instrument header of %-12s.\n", gusPatStr);
    fclose(patHandle);
    return -1;
  }

  iwPatch.bank = bankNum;
  iwPatch.program = patchNum;

  if (gusPatch.instruments > 1) {
    printf("This patch (%s) contains more than one instrument.\n", gusPatStr);
    fclose(patHandle);
    return -1;
  }

  iwProgram.id.id.minor_id=bankNum+0xAA00;
  iwProgram.id.id.major_id=patchNum+0xAA00;
  iwProgram.version.id.minor_id=0;
  iwProgram.version.id.major_id=1;

  iwPatch.nlayers=gusInst.layers;
  iwPatch.id.id.minor_id=bankNum+0x100;
  iwPatch.id.id.major_id=patchNum+0x100;

  if ((patchNum > 127) && (gusInst.layers==1)) {
    iwPatch.layer_mode = 0;
  } else {
    iwPatch.layer_mode = 1;
  }

  if (exclude != 0) {
    iwPatch.exclusion_mode = 1;
    iwPatch.exclusion_group = exclude;
  } else {
    iwPatch.exclusion_mode = 0;
    iwPatch.exclusion_group = 0;
  }

  iwPatch.effect1 = 0x01;
//  iwPatch.effect1_depth = 0;
  iwPatch.effect1_depth = 0x28;
  iwPatch.effect2 = 0x02;
  iwPatch.effect2_depth = 0;

  iwPatch.layers = NULL;

  for (layerNum=0; layerNum < gusInst.layers; layerNum++) {
    if (fread(gusLayer[layerNum], sizeof **gusLayer, 1, patHandle)==0) {
      printf("Error reading layer header from %-12s.\n", gusPatStr);
      fclose(patHandle);
      return 0;
    }
    iwLayer[layerNum]->id.id.major_id=iwPatch.id.id.major_id;
    iwLayer[layerNum]->id.id.minor_id=gusLayer[layerNum]->layer;
    iwLayer[layerNum]->nwaves=gusLayer[layerNum]->samples;
    iwLayer[layerNum]->venv.id.major_id=iwPatch.id.id.major_id;
    iwLayer[layerNum]->venv.id.minor_id=gusLayer[layerNum]->layer;
    iwLayer[layerNum]->penv.id.major_id=0;
    iwLayer[layerNum]->penv.id.minor_id=0;
    iwLayer[layerNum]->flags=0;
    iwLayer[layerNum]->pan_freq_scale=0;

    if (patchNum>=128) {
      iwLayer[layerNum]->high_range=0;
    } else {
      iwLayer[layerNum]->high_range=255;
    }
    iwLayer[layerNum]->low_range=0;

    if (patchNum>=128) {
      iwEnvp[layerNum]->h.index_type=0;
      iwEnvp[layerNum]->h.mode=1;
    } else {
      iwEnvp[layerNum]->h.index_type=2;
      iwEnvp[layerNum]->h.mode=2;
    }

    iwLayer[layerNum]->pan=0x40;
    iwLayer[layerNum]->velocity_mode=1;

//    iwLayer[layerNum]->attenuation=0;
    iwLayer[layerNum]->attenuation=15; // chtelo by to selektivne...

    iwLayer[layerNum]->freq_scale=0x400;
    iwLayer[layerNum]->freq_center=0x40;
    iwLayer[layerNum]->layer_event=1;
    iwLayer[layerNum]->waves=NULL;

    iwEnvp[layerNum]->id.id.major_id=iwPatch.id.id.major_id;
    iwEnvp[layerNum]->id.id.minor_id=gusLayer[layerNum]->layer;
    iwEnvp[layerNum]->h.num_envelopes=gusLayer[layerNum]->samples;
    iwEnvp[layerNum]->h.flags=0;


    for (n=0; n < gusLayer[layerNum]->samples; n++, waveNum++) {
      if (fread(gusWave[waveNum], sizeof **gusWave, 1, patHandle)==0) {
        printf("Error reading wave header from %-12s.\n", gusPatStr);
        fclose(patHandle);
        return 0;
      }

      iwWave[waveNum]->id.id.major_id=iwLayer[layerNum]->id.id.minor_id+0x200;
      iwWave[waveNum]->id.id.minor_id=n;

      if (gusWave[waveNum]->scale_factor==0) {
        iwLayer[layerNum]->freq_scale=0;
        iwLayer[layerNum]->freq_center=FrequencyToMIDI(gusWave[waveNum]->root_frequency);
      }
      ProcessWaveInfo(gusWave[waveNum], iwWave[waveNum], patchNum);
      ProcessEnvelopeInfo(gusWave[waveNum], iwLayer[layerNum], iwWave[waveNum],
        iwEnvpRecord[waveNum], varPortPtr+waveNum*12);

      if (n==0) {
        ProcessLFOInfo(gusWave[waveNum], iwLayer[layerNum]);
      }

      // now test ballance infos
      if (n == 0) {
        lastBalance=gusWave[waveNum]->balance;
        setupPan=1; // can setup panning for whole layer
      } else if (lastBalance != gusWave[waveNum]->balance) {
        setupPan=0; // balance is not equal for whole layer, don't setup pan
      }

      size=0L;
      while (gusWave[waveNum]->wave_size-1024L > size) {
        if ((m=fread(waveData, 1, 1024, patHandle))==0) {
          printf("Error reading wave data from %-12s.\n", gusPatStr);
          fclose(patHandle);
          return 0;
        }
        fwrite(waveData, 1, m, datHandle);
        size+=1024L;
      }
      if ((m=fread(waveData, 1, (int)(gusWave[waveNum]->wave_size - size), patHandle)
          )==0) {
        printf("Error reading wave data from %-12s.\n", gusPatStr);
        fclose(patHandle);
        return 0;
      }
      fwrite(waveData, 1, m, datHandle);

      // changed, write 8 bytes from last byte/word of sample
      if ((gusWave[waveNum]->modes & 0x01)!=0) { //16bit wave data
        lastWaveSample = waveData[m-2] + ((word)waveData[m-1] << 8);
      } else {
        lastWaveSample = waveData[m-1] + ((word)waveData[m-1] << 8);
      }
      fwrite(&lastWaveSample, 2, 2, datHandle);
      fwrite(&lastWaveSample, 2, 2, datHandle);
    }

    // change default pan for layer?
    if ((setupPan == 1) && (lastBalance < 16)) {
      iwLayer[layerNum]->pan = lastBalance*127/15; //0..15 -> 0..127 range
    }

    for (n=waveNum-1; waveNum-1-gusLayer[layerNum]->samples<n; n--) {
      result=0;
      hirange=0;
      for (m=n; waveNum-1-gusLayer[layerNum]->samples<m; m--) {
        if (iwEnvpRecord[m]->hirange<hirange) continue;
        hirange=iwEnvpRecord[m]->hirange;
        result=m;
      }
      tmpEnvpPtr=iwEnvpRecord[n];
      iwEnvpRecord[n] = iwEnvpRecord[result];
      iwEnvpRecord[result] = tmpEnvpPtr;
    }
  }

  waveNum=0;
  for (layerNum=0; layerNum<gusInst.layers; layerNum++) {
    fprintf(fffHandle, "ENVP");

    size=gusLayer[layerNum]->samples*(sizeof **iwEnvpRecord) + sizeof **iwEnvp;
    for (tempWaveNum=waveNum; tempWaveNum < gusLayer[layerNum]->samples+waveNum; tempWaveNum++) {
      size += (iwEnvpRecord[tempWaveNum]->nattack+iwEnvpRecord[tempWaveNum]->nrelease)*4;
    }
    fwrite(&size, 4, 1, fffHandle);

    fwrite(iwEnvp[layerNum], sizeof **iwEnvp, 1, fffHandle);

    for (n=0; n<gusLayer[layerNum]->samples; n++, waveNum++) {
      fwrite(&(iwEnvpRecord[waveNum]->nattack), 2, 1, fffHandle); //2=sizeof int
      fwrite(&(iwEnvpRecord[waveNum]->nrelease), 2, 1, fffHandle); //2=sizeof int
      fwrite(&(iwEnvpRecord[waveNum]->sustain_offset), 2, 1, fffHandle); //2=int
      fwrite(&(iwEnvpRecord[waveNum]->sustain_rate), 2, 1, fffHandle); //2=int
      fwrite(&(iwEnvpRecord[waveNum]->release_rate), 2, 1, fffHandle); //2=int
      fwrite(&(iwEnvpRecord[waveNum]->hirange), 1, 1, fffHandle); //1=byte
      fwrite(&(iwEnvpRecord[waveNum]->pad), 1, 1, fffHandle); //1=byte
      if (iwEnvpRecord[waveNum]->nattack!=0) {
        fwrite(varPortPtr+waveNum*12, 2, iwEnvpRecord[waveNum]->nattack*2, fffHandle);
      }
      if (iwEnvpRecord[waveNum]->nrelease!=0) {
        fwrite(varPortPtr+waveNum*12+6, 2, iwEnvpRecord[waveNum]->nrelease*2, fffHandle);
      }
    }
  }

  size = waveNum*(8+sizeof **iwWave) + layerNum*(8+sizeof **iwLayer) +
      sizeof iwProgram + sizeof iwPatch + 8;
  fprintf(fffHandle, "PROG");
  fwrite(&size, 4, 1, fffHandle);
  fwrite(&iwProgram, sizeof iwProgram, 1, fffHandle); //sizeof=8

  fprintf(fffHandle, "PTCH");
  size = sizeof iwPatch; //0x14
  fwrite(&size, 4, 1, fffHandle);
  fwrite(&iwPatch, sizeof iwPatch, 1, fffHandle);

  waveNum=0;
  for (layerNum=0; layerNum<gusInst.layers; layerNum++) {
    fprintf(fffHandle, "LAYR");
    size=sizeof **iwLayer; //0x2C
    fwrite(&size, 4, 1, fffHandle);
    fwrite(iwLayer[layerNum], sizeof **iwLayer, 1, fffHandle);

    for (n=0; n<gusLayer[layerNum]->samples; n++, waveNum++) {
      fprintf(fffHandle, "WAVE");
      size=sizeof **iwWave; //0x25
      fwrite(&size, 4, 1, fffHandle);
      fwrite(iwWave[waveNum], sizeof **iwWave, 1, fffHandle);
    }
  }

  fclose(patHandle);
  return 1;
}


int main(int argc, char *argv[]) {
  iw_disk_id id;
  long size;
  char patchPathStr[0x50];
  char datName[0x50];
  char iniStr[0x50];
  char patStr[0x50];
  char tempStr[0x50];
  char cwdStr[0x50];
  char *strPtr;
  char fffName[0x50];
  char destPath[0x50];
  char ch;
  int excludeNum;
  int result;
  int pFlag;
  int bankNum;
  int n;

  bankNum=0; pFlag=0; excludeNum=0;

  printf("GIPC: GUS interwave patch converter\n");
  printf("Version 1.12c (with attenuation for EFX)\n");
  printf("Copyright (C) 1996 by Gravis, (P) 1996 by ChAr\n");

  getcwd(cwdStr, 0x50);
  strcpy(destPath, cwdStr);
  if (destPath[strlen(destPath+1)] != '\\') {
    strcat(destPath, "\\");
  }

  strcpy(fffName, "default.fff");
  strcpy(datName, "default.dat");
  strcpy(iniStr, "ultrasnd.ini");

  waveStart=0x40L;

  // Parse CMD line
  for (n=1; n<argc; n++) {
    ch = *(argv[n]);
    strPtr = argv[n]+2;

    switch (ch) {
      case 'd': {
        strcpy(destPath, strPtr);
        if (chdir(destPath) != 0) {
          printf("Error checking dest. directory, aborting conversion.\n");
          return -1;
        }
        if (destPath[strlen(destPath)] != '\\') {
          strcat(destPath, "\\");
        }
        break;
      }
      case 'i': {
        if (strlen(strPtr) <= 13) {
          if ((strstr(strPtr, ".ini")!=0) || (strstr(strPtr, ".INI")!=0)) {
            strcpy(iniStr, strPtr);
            break;
          }
        }
        printf(".ini file name can't be used.\n");
        return -1;
      }
      case 'n': {
        if (strlen(strPtr) <= 13) {
          if ((strstr(strPtr, ".fff")!=0) || (strstr(strPtr, ".FFF")!=0)) {
            strcpy(fffName, strPtr);
            strnset(datName, 0, 0x50);
            strncpy(datName, fffName, strlen(fffName)-4);
            strcat(datName, ".dat");
            break;
          }
        }
        printf("InterWave patch name can't be used.\n");
        return -1;
      }
      default: {
        printf("USAGE: gipc [d:path] [i:name.ini] [n:name.fff]\n");
        printf("  d - path where IW patch will be built\n");
        printf("  i - get patch info from name.ini instead ULTRASND.INI\n");
        printf("  n - name of IW patch file (.FFF)\n");
        return -1;
      }
    }
  }

  // Memory allocation code

  for (n=0; n<10; n++) {
    if ((gusLayer[n] = (layerdata *)malloc(sizeof **gusLayer))==NULL) {
      printf("Couldn't alloc memory for layers, aborting.\n");
      return -1;
    }
    if ((iwLayer[n] = (iw_layer *)malloc(sizeof **iwLayer))==NULL) {
      printf("Couldn't alloc memory for layers, aborting.\n");
      return -1;
    }
    if ((iwEnvp[n] = (iw_envp *)malloc(sizeof **iwEnvp))==NULL) {
      printf("Couldn't alloc memory for layers, aborting.\n");
      return -1;
    }
  }

  for (n=0; n<100; n++) {
    if ((gusWave[n] = (patchdata *)malloc(sizeof **gusWave))==NULL) {
      printf("Couldn't alloc memory for waves, aborting.\n");
      return -1;
    }
    if ((iwWave[n] = (iw_wave *)malloc(sizeof **iwWave))==NULL) {
      printf("Couldn't alloc memory for waves, aborting.\n");
      return -1;
    }
    if ((iwEnvpRecord[n] = (iw_envp_record *)malloc(sizeof **iwEnvpRecord))==NULL) {
      printf("Couldn't alloc memory for waves, aborting.\n");
      return -1;
    }
  }

  if ((varPortPtr = (word *)malloc(1200)) == NULL) {
    printf("Couldn't alloc memory for port buf, aborting.\n");
    return -1;
  }

  if ((waveData = (byte *)malloc(1024)) == NULL) {
    printf("Couldn't alloc memory for wave buf, aborting.\n");
    return -1;
  }


  // Files open/create code

  strcpy(tempStr, destPath); strcat(tempStr, fffName);
  if ((fffHandle = fopen(tempStr, "wb")) == NULL) {
    printf("Couldn't create %s, aborting.\n", tempStr);
    return -1;
  }

  strcpy(tempStr, destPath); strcat(tempStr, datName);
  if ((datHandle = fopen(tempStr, "wb")) == NULL) {
    printf("Couldn't create %s, aborting.\n", tempStr);
    return -1;
  }

  fprintf(datHandle, " **** GIPC: created from Gravis UltraSound patch wave"
      " data **** "); // len of $ = 32 bytes

  size=0L;
  fprintf(fffHandle, "FFFF");
  fwrite(&size, 4, 1, fffHandle);

  fprintf(fffHandle, "CPRT");
  size=0x2EL; // size of comment in bytes including 0 at end
  fwrite(&size, 4, 1, fffHandle);
  fprintf(fffHandle, "Created by GIPC V1.12c frm UltraSound patches.");

  size=strlen(datName)+5;
  fprintf(fffHandle, "DATA");
  fwrite(&size, 4, 1, fffHandle);

  dataId = bankNum+0xFF00;

  id.major_id=dataId;
  id.minor_id=0;
  fwrite(&id, sizeof id, 1, fffHandle); //=4
  fprintf(fffHandle, "%s", datName);

  ch=0; fwrite(&ch, 1, 1, fffHandle);

  if ((iniHandle = fopen(iniStr, "rt")) == NULL) {
    printf("Couldn't open %s, aborting.\n", iniStr);
    return -1;
  }


  if ((fgets(iniStr, 0x50, iniHandle) == 0)) {
    printf("Error reading ULTRASND.INI, aborting.\n");
    fcloseall();
    return -1;
  }

  pFlag = -1; // No Bank to convert

  do {
    if (strstr(iniStr, "#exclude:") != 0) {
      if (sscanf(iniStr, "#exclude:%d", &excludeNum)!=1) {
        excludeNum=0;
      }
      continue;
    }

    if (strstr(iniStr, "#")!=0) continue; // REM

    if (strstr(iniStr, "[Melodic Bank")!=0) {
      pFlag=0;
      if (sscanf(iniStr, "[Melodic Bank %d", &bankNum)!=1) {
        bankNum=0;
      }
      continue;
    }
    if (strstr(iniStr, "[Drum Bank")!=0) {
      pFlag=1;
      if (sscanf(iniStr, "[Drum Bank %d", &bankNum)!=1) {
        bankNum=0;
      }
      continue;
    }
    if (strstr(iniStr, "PatchDir=")!=0) {
      if (sscanf(iniStr, "PatchDir=%s", &patchPathStr)!=1) {
        strcpy(patchPathStr, ""); // ????
      }
      continue;
    }
    // tady je trochu jina logika v originalnim programu...
    if (strstr(iniStr, "Patches]")!=0) {
      pFlag=-1;
      continue;
    }
    if (strstr(iniStr, "BankName")!=0) continue;

    if ((pFlag>-1) && (sscanf(iniStr, "%d=%s", &n, &tempStr)==2)) {
      strcat(tempStr, ".pat");
      strcpy(patStr, patchPathStr);
      strcat(patStr, tempStr);
      if (pFlag != 0) n+= 128; // Drums
      result=ConvertPatch(patStr, fffHandle, datHandle, bankNum, n, excludeNum);
      excludeNum=0;
      if (result != 0) {
        printf("Converted succesfully:%s\n", patStr);
      }
    }
  } while (fgets(iniStr, 0x50, iniHandle) != 0);

  fclose(iniHandle); // probably close()

  size = ftell(fffHandle);

  if (size > 8L) {
    size -= 8L;
    fseek(fffHandle, 4L, SEEK_SET);
    fwrite(&size, 4, 1, fffHandle);
  } else {
    printf("Error computing file size, file is not usable.\n");
  }
  fcloseall();

  // Free allocated memory buffers

  for (n=0; n<10; n++) {
    free(gusLayer[n]);
    free(iwLayer[n]);
    free(iwEnvp[n]);
  }

  for (n=0; n<100; n++) {
    free(gusWave[n]);
    free(iwWave[n]);
    free(iwEnvpRecord[n]);
  }

  free(waveData);
  return 0;
}