//=he3d=file============================================================================================
//	Helium3d system	(c) 2000 acid.z51 
//------------------------------------------------------------------------------------------------------
//
//	name:	he3d_quaternion.cpp
//  desc:	implementation file for CQuaternion class
//
//	history		
//		+ 26.12.2000				first & final version by acid
//
//	todo
//		+ 25.12.2000				nothing :)
//======================================================================================================
#include "math3d.h"
#include <math.h>

he3d_CQuaternion::he3d_CQuaternion( FLOAT _w, FLOAT _x, FLOAT _y, FLOAT _z ) : w(_w), x(_x), y(_y), z(_z)
{
}

he3d_CQuaternion::he3d_CQuaternion( FLOAT _w, const he3d_CVector& vec ) : w(_w), x(vec.x), y(vec.y), z(vec.z)
{	
}

he3d_CQuaternion::he3d_CQuaternion( const D3DXQUATERNION& quat ) : w(quat.w), x(quat.x), y(quat.y), z(quat.z)
{
}

he3d_CQuaternion::he3d_CQuaternion( const he3d_CQuaternion& quat ) : w(quat.w), x(quat.x), y(quat.y), z(quat.z)
{
}

he3d_CQuaternion& he3d_CQuaternion::operator=( const he3d_CQuaternion& quat )
{
	if( &quat == this )
		return *this;

	w = quat.w;
	x = quat.x;
	y = quat.y;
	z = quat.z;

	return *this;
}

he3d_CQuaternion& he3d_CQuaternion::operator*=( FLOAT a )
{
	w *= a;
	x *= a;
	y *= a;
	z *= a;

	return *this;	
}

he3d_CQuaternion& he3d_CQuaternion::operator/=( FLOAT a )
{
	w /= a;
	x /= a;
	y /= a;
	z /= a;

	return *this;	
}

he3d_CQuaternion& he3d_CQuaternion::operator^=( FLOAT a )
{
	FLOAT				ang = a*acosf( w );
	FLOAT				len = sqrtf( x*x + y*y + z*z );	
	FLOAT				_sin = sinf( ang );

	len = _sin/len;

	w = cosf( ang );
	x *= len;
	y *= len;
	z *= len;

	return *this;
}

he3d_CQuaternion& he3d_CQuaternion::operator+=( const he3d_CQuaternion& quat )
{
	w += quat.w;
	x += quat.x;
	y += quat.y;
	z += quat.z;

	return *this;
}

he3d_CQuaternion& he3d_CQuaternion::operator-=( const he3d_CQuaternion& quat )
{
	w -= quat.w;
	x -= quat.x;
	y -= quat.y;
	z -= quat.z;

	return *this;
}

he3d_CQuaternion he3d_CQuaternion::operator+( const he3d_CQuaternion& quat ) const
{
	he3d_CQuaternion	out;

	out.w = w + quat.w;
	out.x = x + quat.x;
	out.y = y + quat.y;
	out.z = z + quat.z;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator-( const he3d_CQuaternion& quat ) const
{
	he3d_CQuaternion	out;

	out.w = w - quat.w;
	out.x = x - quat.x;
	out.y = y - quat.y;
	out.z = z - quat.z;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator*( const he3d_CQuaternion& q ) const
{
	he3d_CQuaternion	out;

	out.w = w*q.w - x*q.x - y*q.y - z*q.z;
    out.x = w*q.x + x*q.w + y*q.z - z*q.y;
    out.y = w*q.y + y*q.w + z*q.x - x*q.z;
    out.z = w*q.z + z*q.w + x*q.y - y*q.x;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator*( FLOAT a ) const
{
	he3d_CQuaternion	out;

	out.w = w*a;
	out.x = x*a;
	out.y = y*a;
	out.z = z*a;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator/( FLOAT a ) const
{
	he3d_CQuaternion	out;

	out.w = w/a;
	out.x = x/a;
	out.y = y/a;
	out.z = z/a;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator^( FLOAT a ) const
{
	he3d_CQuaternion	out;
	FLOAT				ang = a*acosf( w );
	FLOAT				len = sqrtf( x*x + y*y + z*z );	
	FLOAT				_sin = sinf( ang );

	len = _sin/len;

	out.w = cosf( ang );
	out.x = x*len;
	out.y = y*len;
	out.z = z*len;

	return out;
}

he3d_CQuaternion he3d_CQuaternion::operator-() const
{
	return he3d_CQuaternion( -w, -x, -y, -z );
}

he3d_CQuaternion he3d_CQuaternion::operator+() const
{
	return he3d_CQuaternion( *this );
}

he3d_CQuaternion operator*( FLOAT a, const he3d_CQuaternion& quat )
{
	return he3d_CQuaternion( a*quat.w, a*quat.x, a*quat.y, a*quat.z );
}

FLOAT Length( const he3d_CQuaternion& quat )
{
	return sqrtf( quat.w*quat.w + quat.x*quat.x + quat.y*quat.y + quat.z* quat.z );
}

he3d_CQuaternion Normalize( const he3d_CQuaternion& quat )
{
	he3d_CQuaternion	out;
	FLOAT				len = sqrtf( quat.w*quat.w + quat.x*quat.x + quat.y*quat.y + quat.z* quat.z );

	len = 1/len;

	out.w = quat.w*len;
	out.x = quat.x*len;
	out.y = quat.y*len;
	out.z = quat.z*len;

	return out;
}

he3d_CQuaternion Inverse( const he3d_CQuaternion& quat )
{
	FLOAT				len = sqrtf( quat.w*quat.w + quat.x*quat.x + quat.y*quat.y + quat.z* quat.z );

	len = 1/len;

	return he3d_CQuaternion( quat.w*len, -quat.x*len, -quat.y*len, -quat.z*len );
}

he3d_CQuaternion UnaryInverse( const he3d_CQuaternion& quat )
{
	return he3d_CQuaternion( quat.w, -quat.x, -quat.y, -quat.z );
}

he3d_CQuaternion Conjunction( const he3d_CQuaternion& quat )
{
	return he3d_CQuaternion( quat.w, -quat.x, -quat.y, -quat.z );
}

he3d_CQuaternion ln( const he3d_CQuaternion& quat )
{
	he3d_CQuaternion	out;
	FLOAT				ang = acosf( quat.w );
	FLOAT				_sin = sinf( ang );

	out.w = 0;

	if( ISZERO( _sin ) )
		_sin = 1.0f;
	else		
		_sin = ang/_sin;

	out.x = quat.x*_sin;
	out.y = quat.y*_sin;
	out.z = quat.z*_sin;

	return out;
}

he3d_CQuaternion exp( const he3d_CQuaternion& quat )
{
	he3d_CQuaternion	out;
	FLOAT				ang = sqrtf( quat.x*quat.x + quat.y*quat.y + quat.z*quat.z );		

	out.w = cosf( ang );

	if( ISZERO( ang ) )
		ang = 1;
	else
		ang = sinf( ang )/ang;

	out.x = quat.x*ang;
	out.y = quat.y*ang;
	out.z = quat.z*ang;

	return out;
}

he3d_CQuaternion CrossProd( const he3d_CQuaternion& p, const he3d_CQuaternion& q )
{
	he3d_CQuaternion	out;

	out.w = p.w*q.w - p.x*q.x - p.y*q.y - p.z*q.z;
    out.x = p.w*q.x + p.x*q.w + p.y*q.z - p.z*q.y;
    out.y = p.w*q.y + p.y*q.w + p.z*q.x - p.x*q.z;
    out.z = p.w*q.z + p.z*q.w + p.x*q.y - p.y*q.x;

	return out;
}

FLOAT DotProd( const he3d_CQuaternion& p, const he3d_CQuaternion& q )
{
	return p.w*q.w + p.x*q.x + p.y*q.y + p.z*q.z;
}

he3d_CQuaternion SLERP( FLOAT t, const he3d_CQuaternion& p, const he3d_CQuaternion& q )
{
	FLOAT			ang;
	FLOAT			_sin;
	FLOAT			dot;
	FLOAT			t1, t2;
	
	dot = DotProd( p, q );

	if( dot>1.0f )
		dot = 1.0f;
	
	ang = acosf( dot );
	_sin = sinf( ang );

	if ( ISZERO( _sin ) )
	{
		t1 = 1-t;
		t2 = t;
	}
	else
	{
		_sin = 1.0f/_sin;
		t1 = sinf( ang*( 1-t ) )*_sin;
		t2 = sinf( ang*t )*_sin;
	}

	return t1*p + t2*q;
}

he3d_CQuaternion SQUAD( FLOAT t, const he3d_CQuaternion& p, const he3d_CQuaternion& a, 
								 const he3d_CQuaternion& b, const he3d_CQuaternion& q )
{
	return SLERP(  2*t*(1-t), SLERP( t, p, q ), SLERP( t, a, b ) );
}

he3d_CQuaternion FromAxisAngle( const he3d_CVector& axis, FLOAT angle )
{
	FLOAT				ang = 0.5f*angle;
	FLOAT				_sin = sinf( ang );
	he3d_CQuaternion	out;

	out.w = cosf( ang );
	out.x = _sin*axis.x;
	out.y = _sin*axis.y;
	out.z = _sin*axis.z;

	return out;
}

he3d_CQuaternion FromAxisAngle( FLOAT x, FLOAT y, FLOAT z, FLOAT angle )
{
	FLOAT				ang = 0.5f*angle;
	FLOAT				_sin = sinf( ang );
	he3d_CQuaternion	out;

	out.w = cosf( ang );
	out.x = _sin*x;
	out.y = _sin*y;
	out.z = _sin*z;

	return out;

}