//////////////////////////////////////////////////////////////////////////////
//  FLI and FLC loading routines                                            //
//                                                                          //
//  Note(s):                                                                //
//                                                                          //
//  Revisions:                                                              //
//    28/12/1997, FLI initial version                                      //
//    29/12/1997, FLC initial version                                      //
//                                                                          //
//  Copyright (c) 1997 by Matjaz Trtnik aka maLi/MaLixa                     //
//  Email: mtrtnik@bigfoot.com                                              //
//  Web: http://www2.arnes.si/~ssdmalic/mali/                               //
//                                                                          //
//  The author takes no responsibility, if something in this document or    //
//  the accompanying classes causes any kind of data loss or damage to      //
//  your hardware.                                                          //
//                                                                          //
//  You can use this product strictly for *NON* commercial programs.        //
//  If you want to use it for commercial programs please contact author.    //
//                                                                          //
//  You are not permitted to distribute, sell or use any part of            //
//  this source for your software without special permision of author.      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <mem.h>
#include <i86.h>
#include <conio.h>
#include <assert.h>
#include <malloc.h>

#define _DEBUG_
//#define _TEXT_

#define FLI_COLOR256        4L   // 256-level color palette info    | FLC
#define FLI_SS2             7L   // Word-oriented delta compression | FLC
#define FLI_COLOR          11L   // 64-level color palette info     | FLI
#define FLI_LC             12L   // Byte-oriented delta compression | FLI
#define FLI_BLACK          13L   // Entire frame is color index 0   | FLI/FLC
#define FLI_BRUN           15L   // Byte run length compression     | FLI/FLC
#define FLI_COPY           16L   // No compression                  | FLI/FLC
#define FLI_PSTAMP         18L   // Postage stamp sized image       | FLC
#define FLI_NUMCOLORS     256L   // Number of colors
#define FLI_MAGIC       0xAF11
#define FLC_MAGIC       0xAF12

#pragma pack(1)
typedef struct _FLIFILEHEADER {  // Offset   Size
    long    Size;                //      0      4
    short   Magic;               //      4      2
    short   Frames;              //      6      2
    short   Width;               //      8      2
    short   Height;              //     10      2
    short   BitsPerPixel;        //     12      2
    short   Flags;               //     14      2
    short   Speed;               //     16      2
    long    Next;                //     18      4
    long    Frit;                //     22      4
    char    Expand[102];         //     26    102
} FLIFILEHEADER;                 // Total size: 128 bytes
#pragma pack()

#pragma pack(1)
typedef struct _FLCFILEHEADER {
    long    Size;                //      0      4
    short   Magic;               //      4      2
    short   Frames;              //      6      2
    short   Width;               //      8      2
    short   Height;              //     10      2
    short   BitsPerPixel;        //     12      2
    short   Flags;               //     14      2
    long    Speed;               //     16      2
    short   Reserved1;           //     20      2
    long    Created;             //     22      4
    long    Creator;             //     26      4
    long    Updated;             //     30      4
    long    Updater;             //     34      4
    short   AspectX;             //     38      2
    short   AspectY;             //     40      2
    char    Reserved2[38];       //     42     38
    long    OfsFrame1;           //     80      4
    long    OfsFrame2;           //     84      4
    char    Reserved3[40];       //     88     40
} FLCFILEHEADER;                 // Total size: 128 bytes
#pragma pack();


#pragma pack(1)
typedef struct _FLIFLCHEADER {
    long    Size;                //      0      4
    short   Magic;               //      4      2
    short   Frames;              //      6      2
    short   Width;               //      8      2
    short   Height;              //     10      2
    short   BitsPerPixel;        //     12      2
    long    Speed;               //     14      4
    long    OfsFrame1;           //     18      4
    long    OfsFrame2;           //     22      4
    char    Reserved[6];         //     26      6
} FLIFLCHEADER;                  // Total size: 32 bytes
#pragma pack(1);

#pragma pack(1)
typedef struct _FLIFRAMEHEADER { // Offset   Size
    long    Size;                //      0      4
    short   Magic;               //      4      2
    short   Chunks;              //      6      2
    char    Expand[8];           //      8      8
} FLIFRAMEHEADER;                // Total size: 16 bytes
#pragma pack()

#pragma pack(1)
typedef struct _FLICHUNKHEADER { // Offset   Size
    long    Size;                //      0      4
    short   Type;                //      4      2
} FLICHUNKHEADER;                // Total size: 6 bytes
#pragma pack()

#pragma pack(1)
typedef struct _RGB {            // Offset   Size
    char    rgbRed;              //      0      1
    char    rgbGreen;            //      1      1
    char    rgbBlue;             //      2      1
} RGB;                           // Total size: 3 bytes
#pragma pack()

FLIFILEHEADER   flih;
FLCFILEHEADER   flch;
FLIFLCHEADER    ffh;
FLIFRAMEHEADER  frameh;
FLICHUNKHEADER  chunkh;
RGB             flicolmap[FLI_NUMCOLORS];
char           *buffer;

void SetColor(char num, char r, char g, char b)
{
  outp(0x3c8, num);
  outp(0x3c9, r);
  outp(0x3c9, g);
  outp(0x3c9, b);
}

///////////////////
//  COLOR CHUNK  //
///////////////////
long ReadFLIColor(FILE *f, short type)
{
  long    read;
  short   packets, i, j, idx;
  char    skip, change;

  fread(&packets, sizeof(short), 1, f);
  // FLICHUNKHEADER was already read
  read = sizeof(short) + sizeof(FLICHUNKHEADER);

  idx = 0;
  for (i = 0; i < packets; i++) {
    fread(&skip, sizeof(char), 1, f);
    read += sizeof(char);
    // Skip requested colors
    idx += skip;

    fread(&change, sizeof(char), 1, f);
    read += sizeof(char);

    // 0 is interpreted as all 256 colors
    if (change == 0)
      for (idx = 0; idx < FLI_NUMCOLORS; idx++) {
        fread(&flicolmap[idx].rgbRed, sizeof(char), 1, f);
        fread(&flicolmap[idx].rgbGreen, sizeof(char), 1, f);
        fread(&flicolmap[idx].rgbBlue, sizeof(char), 1, f);
        read += sizeof(RGB);
      }
    else
      // Read only requested colors
      for (j = 0; j < change; j++) {
        fread(&flicolmap[idx].rgbRed, sizeof(char), 1, f);
        fread(&flicolmap[idx].rgbGreen, sizeof(char), 1, f);
        fread(&flicolmap[idx].rgbBlue, sizeof(char), 1, f);
        read += sizeof(RGB);
        idx++;
      }
  }

  // Compressed data must be zero-padded to end on a 16-bit boundary
  if ((read % 2) == 1) {
    fseek(f, 1, SEEK_CUR);
    read++;
  }

  #ifndef _TEXT_

  if (type == FLI_COLOR256)
    for (i = 0; i < FLI_NUMCOLORS; i++) {
      flicolmap[i].rgbRed >>= 2;
      flicolmap[i].rgbGreen >>= 2;
      flicolmap[i].rgbBlue >>= 2;
    }

  #endif

  return read;
}

/////////////////
//  DELTA FLC  //
/////////////////
long ReadFLISS2(FILE *f)
{
  long          read, pos = 0;
  short         lines, l, j, i;
  short         b, color;
  char          skip;
  signed char   change;

  // Number of lines that should change
  fread(&lines, sizeof(short), 1, f);
  read = sizeof(FLICHUNKHEADER) + sizeof(short);

  l = 0;
  while (lines > 0) {

    pos = l*ffh.Width;

    fread(&b, sizeof(short), 1, f);
    read += sizeof(short);

    // Number of packets following
    if ((b & 0xC000) == 0x0000) {
      b &= 0x3FFF;   // Number of packets in low 14 bits

      for (j = 0; j < b; j++) {

        // Skip unchanged pixels
        fread(&skip, sizeof(char), 1, f);
        read += sizeof(char);
        pos += skip;

        // Pixels to change
        fread(&change, sizeof(char), 1, f);
        read += sizeof(char);
        if (change  > 0) {
          for (i = 0; i < change; i++) {
            fread(&color, sizeof(short), 1, f);
            read += sizeof(short);
            buffer[pos++] = (color & 0x00FF);
            buffer[pos++] = ((color >> 8) & 0x00FF);
          }
        }
        else {
          change = -change;
          fread(&color, sizeof(short), 1, f);
          read += sizeof(short);

          for (i = 0; i < change; i++) {
            buffer[pos++] = (color & 0x00FF);
            buffer[pos++] = ((color >> 8) & 0x00FF);
          }
        }
      }
      lines--;
      l++;
    }
    else
    // Number of lines that we should skip
    if ((b & 0xC000) == 0xC000)
      l -= b;
    else
    // Color of last pixel in row
      buffer[pos++] = b & 0x00FF;
  }

  // Compressed data must be zero-padded to end on a 32-bit boundary
  if ((read % 4) == 1) {
    fseek(f, 2, SEEK_CUR);
    read++;
  }

  return read;
}

/////////////////////////
//  LNECOMPRESS CHUNK  //
/////////////////////////
long ReadFLILineCompress(FILE *f)
{
  char          skip, color, packets;
  signed char   change;
  short         line, sline;
  short         i, j, k;
  long          read, pos;

  read = sizeof(FLICHUNKHEADER);

  // Skip line lines
  fread(&sline, sizeof(short), 1, f);
  read += sizeof(short);

  // Number of lines that will be changed
  fread(&line, sizeof(short), 1, f);
  read += sizeof(short);

  for (i = 0; i < line; i++) {
    pos = (sline+i)*ffh.Width;

    // Number of packets in this line
    fread(&packets, sizeof(char), 1, f);
    read += sizeof(char);

    for (k = 0; k < packets; k++) {

      // Number of pixels in this line which will stay the same
      fread(&skip, sizeof(char), 1, f);
      read += sizeof(char);
      pos += skip;

      // Number of pixels that will change
      fread(&change, sizeof(char), 1, f);
      read += sizeof(char);

      if (change > 0) {
        for (j = 0; j < change; j++) {
          fread(&color, sizeof(char), 1, f);
          read += sizeof(char);
          buffer[pos++] = color;
        }
      }
      else {
        change = -change;
        fread(&color, sizeof(char), 1, f);
        read += sizeof(char);
        for (j = 0; j < change; j++)
          buffer[pos++] = color;
      }
    }
  }

  // Compressed data must be zero-padded to end on a 16-bit boundary
  if ((read % 2) == 1) {
    fseek(f, 1, SEEK_CUR);
    read++;
  }

  return read;
}

///////////////////
//  BLACK CHUNK  //
///////////////////
long ReadFLIBlack(FILE *f)
{
  long  read;

  read = sizeof(FLICHUNKHEADER);

  memset(buffer, 0, ffh.Width*ffh.Height);

  return read;
}

//////////////////
//  BRUN CHUNK  //
//////////////////
long  ReadFLIBitwizeRUN(FILE *f)
{
  long         read, pos;
  char         packets, color;
  short        k, j, line;
  signed char  size;

  read = sizeof(FLICHUNKHEADER);
  pos = 0;
  line = 0;

  while (line++ < ffh.Height) {

    fread(&packets, sizeof(char), 1, f);
    read += sizeof(char);

    k = 0;
    while (k < ffh.Width) {

      fread(&size, sizeof(char), 1, f);
      read += sizeof(char);

      if (size > 0) {
        fread(&color, sizeof(char), 1, f);
        read += sizeof(char);
        k += size;
        for (j = 0; j < size; j++)
          buffer[pos++] = color;
      }
      else {
        size = -size;
        read += (size*sizeof(char));
        k += (size*sizeof(char));
        for (j = 0; j < size; j++) {
          fread(&color, sizeof(char), 1, f);
          buffer[pos++] = color;
        }
      }
    }
  }

  // Compressed data must be zero-padded to end on a 16-bit boundary
  if ((read % 2) == 1) {
    fseek(f, 1, SEEK_CUR);
    read++;
  }

  return read;
}

//////////////////
//  COPY CHUNK  //
//////////////////
long ReadFLICopy(FILE *f)
{
  long  read;
  long  size = ffh.Width*ffh.Height;

  read = sizeof(FLICHUNKHEADER);
  //printf("\n%d %d",read, size);
  //exit(1);

  fread(buffer, size, 1, f);
  read += size;

  // Compressed data must be zero-padded to end on a 16-bit boundary
  //if ((read % 2) == 1) {
  //  fseek(f, 1, SEEK_CUR);
  //  read++;
  //}

  return read;
}


void LoadFLI(char *filename)
{
  FILE          *f;
  short          frames, chunks, magic;
  long           read, size, i;

  f = fopen(filename, "r+b");
  #ifdef _DEBUG_
  assert(f != NULL);
  #endif

  fseek(f, 4, SEEK_SET);
  fread(&magic, sizeof(short), 1, f);
  fseek(f, 0, SEEK_SET);

  if ((magic & 0xFFFF) == FLI_MAGIC) {
    // Read FLI header
    fread(&flih, sizeof(FLIFILEHEADER), 1, f);
    ffh.Size = flih.Size;
    ffh.Magic = flih.Magic;
    ffh.Frames = flih.Frames;
    ffh.Width = flih.Width;
    ffh.Height = flih.Height;
    ffh.BitsPerPixel = flih.BitsPerPixel;
    ffh.Speed = flih.Speed*14;
    ffh.OfsFrame1 = 0;
    ffh.OfsFrame2 = 0;
  }
  else
  if ((magic & 0xFFFF) == FLC_MAGIC) {
    // Read FLC header
    fread(&flch, sizeof(FLCFILEHEADER), 1, f);
    ffh.Size = flch.Size;
    ffh.Magic = flch.Magic;
    ffh.Frames = flch.Frames;
    ffh.Width = flch.Width;
    ffh.Height = flch.Height;
    ffh.BitsPerPixel = flch.BitsPerPixel;
    ffh.Speed = flch.Speed;
    ffh.OfsFrame1 = flch.OfsFrame1;
    ffh.OfsFrame2 = flch.OfsFrame2;
    // Skip data PREFIX CHUNK (data about animation)
    fseek(f, ffh.OfsFrame1, SEEK_SET);
  }
  else
    #ifdef _DEBUG_
    assert(0);
    #endif

  #ifdef _TEXT_
  printf("\nFILE HEADER");
  printf("\nFilename    : \"%s\"",filename);
  printf("\nSize        : %d",ffh.Size);
  printf("\nMagic       : %X",ffh.Magic & 0xFFFF);
  printf("\nFrames      : %d",ffh.Frames);
  printf("\nWidth       : %d",ffh.Width);
  printf("\nHeight      : %d",ffh.Height);
  printf("\nBitsPerPixel: %d",ffh.BitsPerPixel);
  printf("\nSpeed       : %d",ffh.Speed);
  printf("\nOfsFrame1   : %d",ffh.OfsFrame1);
  printf("\nOfsFrame2   : %d",ffh.OfsFrame2);
  printf("\n");
  getch();
  #endif

  //buffer = (char *)calloc(ffh.Width*ffh.Height, sizeof(char));
  //#ifdef _DEBUG_
  //assert(buffer != NULL);
  //#endif

  for (frames = 0; frames < ffh.Frames && !kbhit(); frames++) {

    // Read FLI frame header
    fread(&frameh, sizeof(FLIFRAMEHEADER), 1, f);

    #ifdef _TEXT_
    printf("\nFRAME HEADER");
    printf("\nSize  : %d",frameh.Size);
    printf("\nMagic : %X",frameh.Magic & 0xFFFF);
    assert(frameh.Magic != 0xFFFF);
    printf("\nChunks: %d",frameh.Chunks);
    printf("\n");
    //getch();
    //fseek(f, frameh.Size-sizeof(FLIFRAMEHEADER), SEEK_CUR);
    #endif

    size = 0;
    for (chunks = 0; chunks < frameh.Chunks; chunks++) {

      // Read FLI chunk header
      fread(&chunkh, sizeof(FLICHUNKHEADER), 1, f);

      #ifdef _TEXT_
      printf("\nCHUNK HEADER");
      printf("\n  Size: %d",chunkh.Size);
      printf("\n  Type: %d",chunkh.Type & 0xFFFF);
      printf("\n");
      //getch();
      #endif

      switch (chunkh.Type) {

        // FLI Color chunk 256
        case FLI_COLOR256:
          read = ReadFLIColor(f, FLI_COLOR256);
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI Delta FLC
        case FLI_SS2:
          read = ReadFLISS2(f);
          fseek(f, chunkh.Size-read, SEEK_CUR);
          #ifdef _TEXT_
          printf("\nFLI_SS2 ==> Actually read: %d\n",read);
          #endif
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI Color chunk 64
        case FLI_COLOR:
          read = ReadFLIColor(f, FLI_COLOR);
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI LineCompress chunk
        case FLI_LC:
          read = ReadFLILineCompress(f);
          #ifdef _TEXT_
          printf("\nFLI_LC ==> Actually read: %d\n",read);
          #endif
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI Black chunk
        case FLI_BLACK:
          read = ReadFLIBlack(f);
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI Bitwize run-length compression
        case FLI_BRUN:
          read = ReadFLIBitwizeRUN(f);
          #ifdef _TEXT_
          printf("\nFLI_BRUN ==> Actually read: %d\n",read);
          #endif
          #ifdef _DEBUG_
          assert(read == chunkh.Size);
          #endif
          break;

        // FLI Copy chunk
        case FLI_COPY:
          read = ReadFLICopy(f);
          fseek(f, chunkh.Size-read, SEEK_CUR);
          #ifdef _TEXT_
          printf("\nFLI_COPY ==> Actually read: %d\n",read);
          #endif
          #ifdef _DEBUG_
          //assert(read == chunkh.Size);
          #endif
          break;

        default:
          fseek(f, chunkh.Size-sizeof(FLICHUNKHEADER), SEEK_CUR);
      }
      size += chunkh.Size;
    }
    fseek(f, frameh.Size-size-16, SEEK_CUR);

    #ifndef _TEXT_
    delay(ffh.Speed);

    for (i = 0; i < FLI_NUMCOLORS; i++)
      SetColor(i, flicolmap[i].rgbRed, flicolmap[i].rgbGreen, flicolmap[i].rgbBlue);

    memcpy((char *)0xa0000, buffer, 64000);
    #endif

  }
  fclose(f);
}

main(long argc, char **argv)
{
  short mode;

  printf("MDG FLI/FLC Player. \nCopyright (c) 1997 by Matjaz Trtnik.");
  if (argc > 1) {

    #ifndef _TEXT_

    __asm {
      mov   eax,13h
      int   10h
    }
    buffer = (char *)malloc(64000);
    #endif
    LoadFLI(argv[1]);
    #ifndef _TEXT_
    __asm {
      mov   eax,3h
      int   10h
    }
    #endif
  }
  else {
    printf("\n\nUsage: fli.exe <file.fl?>.");

  }

  return 1;

}