unit PCXCDU4;

interface

type

   TRedBook = record
      case Byte of
         0: (RBL : Longint);                    { Red Book Longint }
         1: (Frm : Byte;                        { Frame }
             Sec : Byte;                        { Second }
             Min : Byte;                        { Minute }
             Nul : Byte);                       { Nul }
   end;

   TAudioDisk = record
      LoTrack : Byte;
      HiTrack : Byte;
      LeadOut : TRedBook;
   end;

   TAudioTrack = record
      TrackNo    : Byte;
      TrackStart : TRedBook;
      TrackInfo  : Byte;
      TrackEnd   : TRedBook;
   end;

var
   CDR_Status     : Word;                             { Status }
   CD_Letters     : Byte;                { # of CD-ROM devices }
   CD_StartLetter : Byte;                { # of 1st CDR device }
   CD_Current     : Word;                { # of current device }
   CD_Drives      : array [1..26] of Byte;    { CD-ROM devices }

{--------------------------------------------------------------}

procedure CDR_AudioDiskInfo (var AudioDisk: TAudioDisk);
procedure CDR_AudioTrackInfo (Tno: Byte; var AudioTrack: TAudioTrack);
procedure CDR_PlayAudio (pStartSec, pPlayLen: Longint);
function  RB2HSG (RB: TRedBook) : Longint;

{--------------------------------------------------------------}

implementation

uses DOS;

(**************************************************************)
(* CDX_GetLetterNumber
(*
(* Get Number of CD-ROM drive letters
(*
(* Input:
(*    AX = $1500
(* Output:
(*    BL = Number of CD-ROM drive letters
(*    CL = Starting drive letter of CD-ROM dirve (A=0, B=1, ...)
(**************************************************************)

procedure CDX_GetLetterNumber;

var
   r : Registers;

begin
   r.AX:=$1500;
   Intr($2f,r);
   CD_Letters:=r.BL;
   CD_StartLetter:=r.CL;
   CD_Current:=CD_StartLetter;
end;

(**************************************************************)
(* CDX_GetCDROMDriveLetters
(*
(* Get Number of CD-ROM drive letters
(*
(* Input:
(*    AX    = $150D
(*    ES:BX = Device list buffer
(**************************************************************)

procedure CDX_GetCDROMDriveLetters (var CDBuf);

var
   r : Registers;

begin
   r.AX:=$150D;
   r.ES:=Seg(CDBuf);
   r.BX:=Ofs(CDBuf);
   Intr($2f,r);
end;

(**************************************************************)
(* CDX_SendRequest
(*
(* Send Device Request
(*
(* Input:
(*    AX    = $1510
(*    ES:BX = Reqest header address
(*    CX    = CD-ROM drive letter
(**************************************************************)

procedure CDX_SendRequest (var ReqHdr);

var
   r : Registers;

begin
   r.AX:=$1510;
   r.ES:=Seg(ReqHdr);
   r.BX:=Ofs(ReqHdr);
   r.CX:=CD_Current;
   Intr($2f,r);
end;

{==============================================================}

type

   TReqHdr = record                      { Request Header Type }
      HdrLen  : Byte;                 { Header Length in Bytes }
      SubUnit : Byte;              { SubUnit for minor devices }
      CmdCode : Byte;                           { Command Code }
      Status  : Word;                                 { Status }
      Resrvd  : array [1..8] of Byte;               { Reserved }
   end;

{==============================================================}

(**************************************************************)
(* CDR_IOCTL_Input
(*
(* Read (IOCTL Input)
(*
(* Input:
(*    pXferAddr = Address of request to transfer
(*    pXferLen  = Length of request
(**************************************************************)

procedure CDR_IOCTL_Input (pXferAddr: Pointer; pXferLen: Word);

type
   TIOCTLI = record
      ReqHdr   : TReqHdr;                     { Request Header }
      Media    : Byte;                      { Media descriptor }
      XferAddr : Pointer;                   { Transfer Address }
      XferLen  : Word;              { Transfer Length in Bytes }
      StartSec : Word;                       { Starting Sector }
      VolID    : Longint;            { Ptr to Requested Vol ID }
   end;

var
   IOCTLI : TIOCTLI;

begin
   with IOCTLI do
   begin
      ReqHdr.HdrLen:=SizeOf(TIOCTLI);
      ReqHdr.SubUnit:=0;
      ReqHdr.CmdCode:=3;
      ReqHdr.Status:=0;
      Media:=0;
      XferAddr:=pXferAddr;
      XferLen:=pXferLen;
      StartSec:=0;
      VolID:=0;
   end;
   CDX_SendRequest(IOCTLI);
   CDR_Status:=IOCTLI.ReqHdr.Status;
end;

(**************************************************************)
(* CDR_AudioDiskInfo                IOCLT Input based functions
(*
(* Return first and last track # and lead-out Red Book address
(*
(* Input:
(*    Audiodisk: Variable to store disk info
(* Output:
(*    Audiodisk: Disk info
(**************************************************************)

procedure CDR_AudioDiskInfo (var AudioDisk: TAudioDisk);

type
   TDiskInfo = record
      CBC       : Byte;                   { Control Block Code }
      AudioDisk : TAudioDisk;                { Audio Disk Info }
   end;

var
   DiskInfo : TDiskInfo;

begin
   DiskInfo.CBC:=10;
   CDR_IOCTL_Input(@DiskInfo,SizeOf(DiskInfo));
   AudioDisk:=DiskInfo.AudioDisk;
end;

(**************************************************************)
(* CDR_AudioTrackInfo               IOCLT Input based functions
(*
(* Return starting point (RB) and Track info of a given track
(*
(* Input:
(*    Tno        = Track number
(*    AudioTrack = Variable to store track info
(* Output:
(*    AudioTrack = Track info
(**************************************************************)

procedure CDR_AudioTrackInfo (Tno: Byte; var AudioTrack: TAudioTrack);

type
   TTnoInfo = record
      CBC : Byte;                               { Control Block Code }
      Tno : TAudioTrack;                        { Track Info }
   end;

var
   TnoInfo : TTnoInfo;

begin
    TnoInfo.CBC:=11;
    TnoInfo.Tno.TrackNo:=Tno;
    CDR_IOCTL_Input(@TnoInfo,SizeOf(TnoInfo));
    AudioTrack:=TnoInfo.Tno;
end;

{==============================================================}

(**************************************************************)
(* CDR_PlayAudio
(*
(* Play a requested audio track from and to the given sector
(*
(* Input:
(*    pStartSec = First sector to play
(*    pPlayLen  = Number of sectors to play
(**************************************************************)

procedure CDR_PlayAudio (pStartSec, pPlayLen: Longint);

type
   TPlayReq = record
      ReqHdr   : TReqHdr;                     { Request Header }
      AddrMode : Byte;                          { Address Mode }
      StartSec : Longint;               { Start Sector to Play }
      PlayLen  : Longint;          { Number of Sectors to Play }
   end;

var
   PlayReq : TPlayReq;

begin
   with PlayReq do
   begin
      ReqHdr.HdrLen:=SizeOf(TPlayReq);
      ReqHdr.SubUnit:=0;
      ReqHdr.CmdCode:=132;
      ReqHdr.Status:=0;
      AddrMode:=1;
      StartSec:=pStartSec;
      PlayLen:=pPlayLen;
    end;
    CDX_SendRequest(PlayReq);
    CDR_Status:=PlayReq.ReqHdr.Status;
end;

{===========================================================================}

(****************************************************************************)
(* RB2HSG
(*
(* Convert a Red Book type value to a HSG type value
(*
(* Input:
(*    RB = Red Book address
(* Output:
(*    A longint containing HSG addrees
(*
(****************************************************************************)

function RB2HSG (RB: TRedBook) : Longint;

var
   HSG : Longint;

begin
   HSG:=RB.Frm;
   HSG:=HSG+Longint(RB.Sec)*75;
   HSG:=HSG+Longint(RB.Min)*75*60;
   RB2HSG:=HSG;
end;

{--------------------------------------------------------------}

begin
   CDR_Status:=0;
   CDX_GetLetterNumber;
   CDX_GetCDROMDriveLetters(CD_Drives);
end.