//---GEM--------------------------------------------------------------------------
//	textures.cpp - GEM texture engine
//
//	version:                                                                       
//		0.1		19/08/99		ACiD                                               
//
//	desc:                                                                        
//		this version handling only .bmp files                                      
//                                                                                 
//	(c)	ZONE51 1999
//--------------------------------------------------------------------------------

#define		STRICT	
#include	"gem.h"

#define SAFE_DELETE(p)  { if(p) { delete (p);     (p)=NULL; } }
#define SAFE_RELEASE(p) { if(p) { (p)->Release(); (p)=NULL; } }

GEM_TEXTURE::~GEM_TEXTURE()
{
	SAFE_DELETE( next );
	SAFE_RELEASE( pddsSurface );
	SAFE_RELEASE( pd3dTexture );
	DeleteObject( hbBitmap );
	SAFE_DELETE( pBuffer );
}


struct	TEXTUREFORMATSEARCH
{
	LPDDPIXELFORMAT			pddpf;

	WORD					dwBPP;
	BOOL					bUsePalette;
	BOOL					bFound;
};


//--------------------------------------------------------------------------------
//name: GetDDFromDevice()
//desc: retrives a DD interface from D3DDevice interface
//--------------------------------------------------------------------------------
LPDIRECTDRAW4 GetDDFromDevice( LPDIRECT3DDEVICE3 pd3dDevice )
{
	LPDIRECTDRAW4        pDD = NULL;
	LPDIRECTDRAWSURFACE4 pddsRender;

    if( pd3dDevice )
	{
	    // Get the current render target
		if( SUCCEEDED( pd3dDevice->GetRenderTarget( &pddsRender ) ) )
		{
		    // Get the DDraw4 interface from the render target
			pddsRender->GetDDInterface( (VOID**)&pDD );
			pddsRender->Release();
		}
	}

	return pDD;
}



//--------------------------------------------------------------------------------
//name: TextureSearch()
//desc: using in pixel format enumeration to find desire format of texture
//--------------------------------------------------------------------------------
static HRESULT CALLBACK TextureSearch( DDPIXELFORMAT* pddpf, LPVOID pparam )
{
    if( NULL==pddpf || NULL==pparam )
        return DDENUMRET_OK;

	TEXTUREFORMATSEARCH* pTFS = (TEXTUREFORMATSEARCH*)pparam;

    // Skip any funky modes
    if( pddpf->dwFlags & (DDPF_LUMINANCE|DDPF_BUMPLUMINANCE|DDPF_BUMPDUDV) )
        return DDENUMRET_OK;

	// Check for palettized formats
	if( pTFS->bUsePalette )
	{
		if( !( pddpf->dwFlags & DDPF_PALETTEINDEXED8 ) )
			return DDENUMRET_OK;

		// Accept the first 8-bit palettized format we get
        memcpy( pTFS->pddpf, pddpf, sizeof(DDPIXELFORMAT) );
		pTFS->bFound = TRUE;
        return DDENUMRET_CANCEL;
    }

	// Else, skip any paletized formats (all modes under 16bpp)
	if( pddpf->dwRGBBitCount < 16 )
		return DDENUMRET_OK;

	// skip any FourCC formats
	if( pddpf->dwFourCC != 0 )
		return DDENUMRET_OK;

	if( pddpf->dwFlags&DDPF_ALPHAPIXELS )
		return DDENUMRET_OK;

    // Check if we found a good match
    if( pddpf->dwRGBBitCount == pTFS->dwBPP )
    {
        memcpy( pTFS->pddpf, pddpf, sizeof(DDPIXELFORMAT) );
		pTFS->bFound = TRUE;
        return DDENUMRET_CANCEL;
    }

    return DDENUMRET_OK;
}

//----------------------------------------------------------------------------
//	main texture engine
//----------------------------------------------------------------------------

static	GEM_TEXTURE*		g_TextureList = NULL;

class GEM_TEXTUREENGINE
{
public:

	GEM_TEXTUREENGINE();
   ~GEM_TEXTUREENGINE();

} g_gteTextureEngine;

GEM_TEXTUREENGINE::GEM_TEXTUREENGINE()
{
}

GEM_TEXTUREENGINE::~GEM_TEXTUREENGINE()
{
	SAFE_DELETE( g_TextureList );
}

//----------------------------------------------------------------------------
//name: FindTexture()
//desc:	look for texture in global texture list and retur its pointer or null 
//		in case that texture doesnt exist
//----------------------------------------------------------------------------
GEM_TEXTURE* FindTexture(TCHAR* name)
{
	for( GEM_TEXTURE* tmp=g_TextureList ; tmp ; tmp=tmp->next )
	{
		if( !lstrcmpi( name, tmp->name ) )
			return tmp;
	}

	return NULL;
}


//----------------------------------------------------------------------------
//name: LoadImageFile()
//desc: loads a image from .bmp file, check a resource first and then try to
//		load image from file
//----------------------------------------------------------------------------
HRESULT LoadImageFile(TCHAR* pname, GEM_TEXTURE* lpgtTexture)
{
	lpgtTexture->hbBitmap=(HBITMAP)LoadImage( NULL, pname, IMAGE_BITMAP, 0, 0, 
											  LR_CREATEDIBSECTION ); 

	if( lpgtTexture->hbBitmap )
		return S_OK;

	lpgtTexture->hbBitmap=(HBITMAP)LoadImage( NULL, pname, IMAGE_BITMAP, 0, 0, 
											  LR_CREATEDIBSECTION|LR_LOADFROMFILE );

	if ( !lpgtTexture->hbBitmap )
		return DDERR_NOTFOUND;

	return S_OK;
}


//----------------------------------------------------------------------------
//name: GEM_LoadTexture()
//desc: this function create a new texture and add it to global list, load a
//		image from file or resourece, so far only .bmp files can be loaded
//		this function do not create any DDSurfaces or D3DTextures, you must
//		use one of update function
//----------------------------------------------------------------------------
HRESULT AddTexture(TCHAR* name, DWORD stage)
{

	if( FindTexture( name ) )
		return S_OK;

	GEM_TEXTURE*	pgtTexture = new GEM_TEXTURE;
	if( !pgtTexture )
		return DDERR_OUTOFMEMORY;

	ZeroMemory( pgtTexture, sizeof(GEM_TEXTURE) );

	pgtTexture->dwStage = stage;
	lstrcpy( pgtTexture->name, name );

	if( FAILED( LoadImageFile( name, pgtTexture ) ) )
	{
		delete pgtTexture;
		return E_FAIL;
	}

	//add texture to list
	pgtTexture->next = g_TextureList;	
	if( g_TextureList )
		g_TextureList->prev=pgtTexture;

	g_TextureList = pgtTexture;

	return S_OK;
}


//----------------------------------------------------------------------------
//name: CopyToSurface()
//desc: copise a bitmap to surface
//----------------------------------------------------------------------------
HRESULT CopyToSurface(LPDIRECTDRAWSURFACE4 lpddsTarget, HBITMAP hbm )
{
	HDC						hdcBitmap;
	HDC						hdcSurface;
	DDSURFACEDESC2			ddsd;
	HRESULT					hr;
	//LPDIRECTDRAW4			pDD4;

	ddsd.dwSize = sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	hr = lpddsTarget->GetSurfaceDesc( &ddsd );
	if( FAILED(hr) )
		return hr;

	hdcBitmap = CreateCompatibleDC( NULL );
	if( !hdcBitmap )
		return E_FAIL;

	if( !SelectObject( hdcBitmap, hbm ) )
		return E_FAIL;

	DWORD	retval;

	if( SUCCEEDED( lpddsTarget->GetDC( &hdcSurface ) ) )
	{
		retval=BitBlt( hdcSurface, 0, 0, ddsd.dwWidth, ddsd.dwHeight, hdcBitmap, 0, 0, SRCCOPY );		
		if ( FAILED( lpddsTarget->ReleaseDC( hdcSurface ) ) )
			return E_FAIL;
	}

	DeleteDC(hdcBitmap);
	if (!retval) 
		return E_FAIL;

	return S_OK;
}


//----------------------------------------------------------------------------
//name: GEM_UpdateTexture()
//desc: create a DDSurface and DDTexture and copy image to surface
//----------------------------------------------------------------------------
HRESULT UpdateTexture(TCHAR* name, LPDIRECT3DDEVICE3 pd3dDevice)
{
	HRESULT					hr;
	GEM_TEXTURE*			lpgtTexture = FindTexture( name );

	if( lpgtTexture==NULL )
		return DDERR_NOTFOUND;

	LPDIRECTDRAW4			lpDD4 = GetDDFromDevice( pd3dDevice );

	if( !lpDD4 )
		return DDERR_NOTFOUND;

	BITMAP					bm;
	HBITMAP					hbBitmap = lpgtTexture->hbBitmap;

	GetObject( hbBitmap, sizeof(BITMAP), &bm );
	DWORD					dwHeight = (DWORD)bm.bmHeight;
	DWORD					dwWidth  = (DWORD)bm.bmWidth;

	DDSURFACEDESC2			ddsd;
	ZeroMemory( &ddsd, sizeof(ddsd) );
	ddsd.dwSize					= sizeof(ddsd);
	ddsd.ddpfPixelFormat.dwSize = sizeof(ddsd.ddpfPixelFormat);

	ddsd.dwFlags		= DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT | 
						  DDSD_CAPS | DDSD_TEXTURESTAGE;
	ddsd.dwHeight		= dwHeight;
	ddsd.dwWidth		= dwWidth;
	ddsd.dwTextureStage = lpgtTexture->dwStage;
	ddsd.ddsCaps.dwCaps = DDSCAPS_TEXTURE ;

	//ajusting dimension of the bitmap to be a pow of 2 
	for( ddsd.dwHeight=1 ; dwHeight>ddsd.dwHeight ; ddsd.dwHeight<<=1 );
	for( ddsd.dwWidth=1  ; dwWidth>ddsd.dwWidth   ; ddsd.dwWidth<<=1  );

	//set a texture dimension to be a square
	if( ddsd.dwHeight>ddsd.dwWidth )
		ddsd.dwWidth=ddsd.dwHeight;
	else
		if( ddsd.dwHeight<ddsd.dwWidth )
			ddsd.dwHeight=ddsd.dwWidth;

	//searching for apropiate pixel format
	TEXTUREFORMATSEARCH			tfs;
	ZeroMemory( &tfs, sizeof(tfs) );

	tfs.pddpf		= &ddsd.ddpfPixelFormat;
	tfs.bUsePalette	= FALSE;
	tfs.bFound		= FALSE;
	tfs.dwBPP		= 16;

	pd3dDevice->EnumTextureFormats( &TextureSearch, &tfs );
	if( !tfs.bFound )
		return DDERR_NOTFOUND;

	hr = lpDD4->CreateSurface( &ddsd, &lpgtTexture->pddsSurface, NULL); 
	if(	FAILED(hr) )
		return hr;

	hr = lpgtTexture->pddsSurface->QueryInterface( IID_IDirect3DTexture2, (LPVOID*)&lpgtTexture->pd3dTexture );
	if( FAILED(hr) )
		return hr;

	return CopyToSurface( lpgtTexture->pddsSurface, lpgtTexture->hbBitmap );
}

HRESULT UpdateAllTextures(LPDIRECT3DDEVICE3 pd3dDevice)
{
	HRESULT				hr;

	for( GEM_TEXTURE* tmp=g_TextureList ; tmp ; tmp=tmp->next )
	{
		hr = UpdateTexture( tmp->name, pd3dDevice );
		if( FAILED(hr) )
			return hr;
	}

	return S_OK;
}

LPDIRECT3DTEXTURE2 GetTexture(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return NULL;

	return lpgtTexture->pd3dTexture;
}

LPDIRECTDRAWSURFACE4 GetSurface(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return NULL;

	return lpgtTexture->pddsSurface;
}

HRESULT RemoveTexture(TCHAR* name)
{
	GEM_TEXTURE*	lpgtTexture = FindTexture( name );

	if( !lpgtTexture )
		return DDERR_NOTFOUND;

	if( lpgtTexture->prev )
		lpgtTexture->prev->next = lpgtTexture->next;

	if( lpgtTexture->next )
		lpgtTexture->next->prev = lpgtTexture->prev;

	lpgtTexture->prev = NULL;
	lpgtTexture->next = NULL;

	SAFE_DELETE( lpgtTexture );
	
	return S_OK;
}













	


	













