#include "gemutil.h"
#include "gem.h"
#include "gemmath.h"
#include <tchar.h>
#include <stdio.h>


BYTE ReadByte(FILE *f)
{
 return fgetc(f);
}

/*
 *  odczytuje slowo
 */
WORD ReadWord(FILE *f)
{
	WORD		tmp;

 fread(&tmp,sizeof(WORD),1,f);
 return tmp;
}

/*
 *  odczytuje podwojne slowo
 */
DWORD ReadDword(FILE *f)
{
	DWORD		tmp;

 fread(&tmp,sizeof(DWORD),1,f);
 return tmp;
}

/*
 *  odczytuje floata
 */
D3DVALUE ReadFloat(FILE *f)
{
	D3DVALUE		tmp;

 fread(&tmp,sizeof(D3DVALUE),1,f);
 return tmp;
}

/*
 *  odczytuje ciag znakow zakonczony 0, alokujac dla niego pamiec
 *  zwraca wskaznik na odczytany string
 */

void ReadASCIIZ(FILE* f, TCHAR* str)
{
	TCHAR		tmp;
	WORD		i = 0;

	while( tmp = ReadByte( f ) )
		str[i++] = tmp;

	str[i] = 0;
}


/*
 *  przezkakuje pojedynczy chunk, kturego identyfikator zostal poprzednio
 *  odczytany
 */
int JumpThroughtChunk(FILE *f)
{
	DWORD		position;

 position=ReadDword(f)-6;
 fseek(f,position,SEEK_CUR);

 if (feof(f))
  return gem_error_bad_file;

 return gem_error_ok;
}

/*
 *  odczytuje z pliku liste wierzcholkow
 */
int ReadVerticesList(FILE *f, GEM_MESH *mesh)
{
	DWORD		size;
	DWORD		pos;
	WORD		number;
	D3DVALUE		x, y, z;

	size = ReadDword(f)-6;
	pos = ftell(f);
	number = ReadWord(f);

	mesh->vertexNbr = number;
	mesh->vertices = new D3DVERTEX[number];

	if( !mesh->vertices )
		return gem_error_memory;

	for( int i=0; (!feof(f)) && i<number; i++ )
	{
		x=ReadFloat(f);
		y=ReadFloat(f);
		z=ReadFloat(f);
		mesh->vertices[i].x=x;
		mesh->vertices[i].y=z;	//swap vector
		mesh->vertices[i].z=y;
	 }


	fseek(f,pos+size,SEEK_SET);

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}

/*
 *  odczytuje liste facow (scianek) obiektu, kazda scianka jest opisana
 *  przez trzy wierzcholki ktore ja twoza, oraz przez material ktory jest
 *  na nia nalozony
 */
int ReadFacesList(FILE *f, GEM_MESH *mesh, GEM_SCENE *scena)
{
	long			size;
	long			pos;
	WORD			number;
	WORD			a, b, c;
	WORD			sign;
	char			matname[80];
	GEM_OBJECT*   	obj;
	GEM_MATERIAL*	mat;	
	WORD			usednbr;
	WORD			allnbr;
	GEM_ELEMENT*	elem;
	
	mesh->elemList = NULL;
	size = ReadDword(f)-6;
	pos = ftell(f);
	number= ReadWord(f);

	mesh->faceNbr = number;
	mesh->faces = new GEM_FACE[number];	
	
	for( int k=0 ; k<number; k++ )
	{
		mesh->faces[k].used = FALSE;
	}

	if( !mesh->faces )
		return gem_error_memory;

	for( int i=0 ; (!feof(f)) && i<number ; i++ )
	{
		a=ReadWord(f);
		b=ReadWord(f);
		c=ReadWord(f);

		mesh->faces[i].indices[0]=a;
		mesh->faces[i].indices[1]=c;
		mesh->faces[i].indices[2]=b;						

		ReadWord(f);
	}

	if (feof(f))
		return gem_error_bad_file;

	allnbr = number;
	usednbr = 0;

	while( (!feof(f)) && ftell(f)<size+pos )
	{
		sign=ReadWord(f);
		if( sign==FACES_MATERIAL_CHUNK )
		{
			ReadDword(f);
			ReadASCIIZ( f, matname );
			obj = FindObject( scena, matname );
			mat = (GEM_MATERIAL*)obj->object;

			number = ReadWord( f );
			usednbr += number;

			elem = new GEM_ELEMENT;
			
			elem->idx = new WORD[number*3];
			elem->number = number*3;
			elem->mat = mat;

			for (i=0; i<number; i++)
			{
				a = ReadWord(f);
				
				elem->idx[i*3]   = mesh->faces[a].indices[0];
				elem->idx[i*3+1] = mesh->faces[a].indices[1];
				elem->idx[i*3+2] = mesh->faces[a].indices[2];

				mesh->faces[a].used = TRUE;
			}

			AddElement( elem, mesh );
		}
		else
			if( JumpThroughtChunk(f)!=gem_error_ok )
				return gem_error_bad_file;
	}

	if( allnbr > usednbr )
	{
		elem = new GEM_ELEMENT;

		elem->number = 3*(allnbr - usednbr);
		elem->idx = new WORD[elem->number];
		elem->mat = NULL;

		WORD index = 0;

		for( int j = 0; j<allnbr; j++)
		{
			if( !mesh->faces[j].used )
			{
				elem->idx[index++] = mesh->faces[j].indices[0];
				elem->idx[index++] = mesh->faces[j].indices[1];
				elem->idx[index++] = mesh->faces[j].indices[2];
			}
		}

		AddElement( elem, mesh );
	}


	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  wczytuje z pliku liste mapowania
 */
int ReadMappingList(FILE *f, GEM_MESH *mesh)
{
	DWORD			size;
	DWORD			pos;
	WORD			number;
	D3DVALUE		u, v;

	size = ReadDword(f)-6;
	pos = ftell(f);
	number = ReadWord(f);

	for( int i=0 ; (!feof(f)) && i<number ; i++ )
	{
		u=ReadFloat(f);
		v=ReadFloat(f);

		mesh->vertices[i].tu=u;
		mesh->vertices[i].tv=v;
	}

	fseek( f, pos+size, SEEK_SET);

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje lokalny uklad wspolzednych i przeksztalca wspolzedne
 *  wierzcholkow obiektu tak aby byly w nim wyrazone (translacja i odpowiedni
 *  obrot poniewaz poczatkowe wspolzedne obiektu sa zapisane wzgledem
 *  globalego ukladu swiata)
 */
int ReadLocalCoords(FILE *f, GEM_MESH *mesh)
{
	DWORD			size;
	DWORD			pos;
	D3DVALUE		a,b,c;

	GEM_MATRIX		mtx=IdentMatrix();

	size = ReadDword(f)-6;
	pos = ftell(f);


	for( int i=1 ; i<=3 ; i++ )
		for( int j=1 ; j<=3 ; j++ )
			mtx(j,i) = ReadFloat(f);

	SwapMatrix( mtx );

	a=ReadFloat(f);
	b=ReadFloat(f);
	c=ReadFloat(f);


	for( i=0 ; i<mesh->vertexNbr ; i++ )
	{
		mesh->vertices[i].x-=a;
		mesh->vertices[i].y-=c;
		mesh->vertices[i].z-=b;

		GEM_VECTOR	vec(mesh->vertices[i].x, mesh->vertices[i].y, mesh->vertices[i].z);
		vec=vec*mtx;

		mesh->vertices[i].x=vec.x;
		mesh->vertices[i].y=vec.y;
		mesh->vertices[i].z=vec.z;
	}

	fseek(f,pos+size,SEEK_SET);

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje caly chunk obiektu
 */
int ReadTrimeshChunk(FILE *f, GEM_OBJECT *object, GEM_SCENE *scena)
{
	long			size;
	long			pos;
	WORD			sign;
	GEM_MESH*		mesh;
	int				err;

	mesh = new GEM_MESH;
	if (!mesh)
		return gem_error_memory;

	object->type = MESH;
	object->object = mesh;

	size=ReadDword(f)-6;
	pos=ftell(f);


	while ((!feof(f))&&ftell(f)<size+pos)
	{
		sign=ReadWord(f);
		switch( sign )
		{
			case VERTICES_CHUNK:
				if( (err=ReadVerticesList( f, mesh ))!=gem_error_ok )
					return err;
			break;

			case FACES_CHUNK:
				if( (err=ReadFacesList( f, mesh, scena ))!=gem_error_ok )
					return err;
			break;

			case MAPPING_COORDS_CHUNK:
				if( (err=ReadMappingList( f, mesh ))!=gem_error_ok )
					return err;
			break;

			case LOCAL_COORDS_CHUNK:
				if( (err=ReadLocalCoords( f, mesh ))!=gem_error_ok )
					return err;
			break;

			default:
				if ( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
			break;

		}
	}

	if (feof(f))
		return gem_error_bad_file;

	CalculateNormals( mesh );

	return gem_error_ok;
}

/*
 *  odczytuje informacje o obiekcia swiatla
 */
int ReadLightChunk(FILE *f, GEM_OBJECT *object)
{
	long			size;
	long			pos;
	D3DVALUE 		a, b, c;
	GEM_LIGHT*		light;


	light = new GEM_LIGHT ;

	if (!light)
		return gem_error_memory;

	object->type = LIGHT;
	object->object = light;

	size = ReadDword(f)-6;
	pos = ftell(f);

	a=ReadFloat(f);
	b=ReadFloat(f);
	c=ReadFloat(f);

	light->position.x=a;
	light->position.y=c;
	light->position.z=b;


	if( pos+size<ftell(f) )
		if( ReadWord(f)==SPOTLIGHT_CHUNK )
		{
			a=ReadFloat(f);
			b=ReadFloat(f);
			c=ReadFloat(f);

			ReadFloat(f);
			ReadFloat(f);
		}

	fseek(f,pos+size,SEEK_SET);

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}

/*
 *  odczytuje chunk kamery
 */
int ReadCameraChunk(FILE *f, GEM_OBJECT *object)
{
	DWORD			size;
	DWORD			pos;
	D3DVALUE 		a,b,c;
	GEM_CAMERA*		camera;

	camera = new GEM_CAMERA;

	if( !camera )
		return gem_error_memory;

	object->type = CAMERA;
	object->object = camera;

	size=ReadDword(f)-6;
	pos=ftell(f);

	a=ReadFloat(f);
	b=ReadFloat(f);
	c=ReadFloat(f);
	camera->position.x=a;
	camera->position.y=c;
	camera->position.z=b;

	a=ReadFloat(f);
	b=ReadFloat(f);
	c=ReadFloat(f);
	camera->target.x=a;
	camera->target.y=c;
	camera->target.z=b;

	a=ReadFloat(f);
	camera->roll=a;
	a=ReadFloat(f);
	camera->fov=LensToFOV(a);

	fseek( f, pos+size, SEEK_SET);

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odchytuje chunk obiektu (w takim chunku moga byc zapisane informacje
 *  o zrodlach swiatla, obiektach 3d (mesh'e) oraz o kamerze)
 */
int ReadObjectChunk(FILE *f, GEM_SCENE *s)
{
	long			size; //rozmiar chunka
	long			pos;
	WORD			sign;
	GEM_OBJECT		*object;
	int				err;

	object = new GEM_OBJECT;
	ZeroMemory( object, sizeof(GEM_OBJECT) );

	if( !object )
		return gem_error_memory;

	size = ReadDword(f)-6;
	pos = ftell(f);
	ReadASCIIZ( f, object->name );

	while( (!feof(f)) && ftell(f)<size+pos )
	{
		sign=ReadWord(f);
		switch( sign ) 
		{
			case TRIMESH_CHUNK:
				if( (err=ReadTrimeshChunk(f, object, s))!=gem_error_ok )
					return err;
			break;

			case LIGHT_CHUNK:
				if( (err=ReadLightChunk(f, object))!=gem_error_ok )
					return err;
			break;

			case CAMERA_CHUNK:
				if( (err=ReadCameraChunk(f, object))!=gem_error_ok )
					return err;
			break;

			default:
				if( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
     break;
    }
  }

 if (feof(f))
  return gem_error_bad_file;

 AddObject( object, s );

 return gem_error_ok;
}


int ReadMapChunk( FILE *f, GEM_MAP *map )
{
	WORD			sign;
	long			size;
	long			pos;	
	
	size = ReadDword(f)-6;
	pos = ftell(f);

	while ((!feof(f))&&ftell(f)<size+pos)
	{
		sign=ReadWord(f);
		if( sign == MAP_FILENAME_CHUNK )
		{
			ReadDword(f);
			ReadASCIIZ( f, map->filename );
			if( AddTexture( map->filename )!=DD_OK )
			{					
				fseek( f, size+pos, SEEK_SET );
				return gem_error_invalid_texture;						
			}
		}
		else
			if( JumpThroughtChunk(f)!=gem_error_ok )
				return gem_error_bad_file;
	}

	if( feof(f) )
		return gem_error_bad_file;
	
	return gem_error_ok;
}


/*
 *  odczytuje informacje o materiale
 */
int ReadMaterialChunk(FILE *f, GEM_SCENE *s)
{
	WORD			sign;
	long			size;
	long			pos;
	GEM_OBJECT*		obj;
	GEM_MATERIAL*	mat;
	int				err;	

	obj = new GEM_OBJECT;
	mat = new GEM_MATERIAL;
	if (!obj||!mat)
		return gem_error_memory;

	ZeroMemory( obj, sizeof(GEM_OBJECT) );

	obj->type = MATERIAL;
	obj->object = mat;

	mat->texture1	= NULL;
	mat->texture2	= NULL;
	mat->bump		= NULL;
	mat->reflection = NULL;
 

	size = ReadDword(f)-6;
	pos = ftell(f);

	while ((!feof(f))&&ftell(f)<size+pos)
	{
		sign=ReadWord(f);
		switch (sign)
		{
			case MATERIAL_NAME_CHUNK:
				ReadDword(f);
				ReadASCIIZ( f, obj->name );
				
				lstrcpy( mat->name, obj->name );
			break;

			case TEXTURE_MAP1_CHUNK:
				mat->texture1 = new GEM_MAP;
				if( !mat->texture1 )
					return gem_error_memory;
				if( (err=ReadMapChunk(f, mat->texture1))!=gem_error_ok )				
					if (err = gem_error_invalid_texture)
					{
						delete mat->texture1;
						mat->texture1 = NULL;
					}
					else
						return err;																	
			break;			

			case TEXTURE_MAP2_CHUNK:
				mat->texture2 = new GEM_MAP;
				if( !mat->texture2 )
					return gem_error_memory;
				if( (err=ReadMapChunk(f, mat->texture2))!=gem_error_ok )
					if (err = gem_error_invalid_texture)
					{
						delete mat->texture2;
						mat->texture2 = NULL;
					}
					else
						return err;
			break;

			default:
				if( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
			break;
		}
	}

	if( feof(f) )
		return gem_error_bad_file;

	AddObject( obj, s );

	return gem_error_ok;
}


/*
 *  odczytuje jeden z glownych chunkow pliku - chunk edytora, jego
 *  subchunkami sa chanki zawierajace inforamcje o obiektach i materialach
 */
int ReadEditorChunk(FILE *f, GEM_SCENE *scene)
{
	WORD			sign;
	long			size;
	long			pos;
	int				err;

	size = ReadDword(f)-6;
	pos = ftell(f);

	while ((!feof(f))&&ftell(f)<size+pos)
	{
		sign=ReadWord(f);
		switch (sign)
		{
			case OBJECT_CHUNK:
				if ((err=ReadObjectChunk(f,scene))!=gem_error_ok)
					return err;
			break;

			case MATERIAL_CHUNK:
				if ((err=ReadMaterialChunk(f,scene))!=gem_error_ok)
					return err;
			break;

			default:
				if (JumpThroughtChunk(f)!=gem_error_ok)
					return gem_error_bad_file;
			break;
		}
	}

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;

}


/*
 *  odczytuje chunk zawierajacy informacje o wersji 3ds'a ktory zostal
 *  uzyty do otrzymania czytanego pliku
 */
int ReadVersionChunk(GEM_SCENE *s, FILE *f)
{

	ReadDword(f);
	s->version = ReadDword(f);

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje nformacje o obiekcie ktorego sciezka jest wlasnie czytana
 *  (potrz nizej)
 */
int ReadObjectInfo( FILE *f, GEM_KEYFRAMER *keyframer )
{
	WORD		parent;

	ReadDword(f);
	ReadASCIIZ( f, keyframer->objName );
	
	ReadWord(f);
	ReadWord(f);
	parent=ReadWord(f);

	keyframer->link=parent;

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  czyta wpolrzedne pivota obiektu, jest to punkt wokol ktorego nastepuje
 *  rotacja obiektu
 */
int ReadPivot(FILE *f, GEM_KEYFRAMER *keyframer)
{
	D3DVALUE		a,b,c;

	ReadDword(f);

	a=ReadFloat(f);
	b=ReadFloat(f);
	c=ReadFloat(f);

	keyframer->pivot.x=a;
	keyframer->pivot.y=c;
	keyframer->pivot.z=b;

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}

/*
 *  odczytuje track obrotu, w 3ds sciezka jest zapisana jako os obrotu, oraz
 *  kat obrotu, w tej funkcji nastepuje konwersja ma kwaternion
 */
int ReadQuaternionTrack(FILE *f, GEM_TRACK* track)
{
	
	DWORD			number;
	WORD			flag;
	D3DVALUE		angle, a, b, c;
	DWORD			frame;
	GEM_KEY*		key;
	GEM_QUATERNION	old(1,0,0,0);

	track->keyList = NULL;
	track->flags = 0;

	ReadDword(f);
	ReadWord(f);
	ReadDword(f);
	ReadDword(f);

	number = ReadDword(f);

	for( DWORD i = 0 ; (!feof(f)) && i<number ; i++ )
	{
		key=new GEM_KEY;

		if (!key)
			return gem_error_memory;

		frame = ReadDword(f);
		key->frame = frame;

		key->tension = 0.0f;
		key->bias = 0.0f;
		key->continuity = 0.0f;
		key->easefrom = 0.0f;
		key->easeto = 0.0f;

		flag=ReadWord(f);
		for( int j=0 ; j<16 ; j++ )
			if( flag&(1<<j) )
				switch (j)
				{
					case 0: key->tension=ReadFloat(f); break;
					case 1: key->continuity=ReadFloat(f); break;
					case 2: key->bias=ReadFloat(f); break;
					case 3: key->easeto=ReadFloat(f); break;
					case 4: key->easefrom=ReadFloat(f); break;
				}

		angle=ReadFloat(f);
		a=ReadFloat(f);
		b=ReadFloat(f);
		c=ReadFloat(f);

		key->_qtr = old*FromAxisAngle( a, c, b, angle );
		old = key->_qtr;

		AddKey(key,track);
	}

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje sciezke wektorowa (moze byc to sciezka pozycji, skalowania lub
 *  koloru)
 */
int ReadVectorTrack(FILE *f, GEM_TRACK *track)
{
	
	DWORD			number;
	WORD			flag;
	D3DVALUE		a, b, c;
	DWORD			frame;
	GEM_KEY*		key;

	track->keyList = NULL;
	track->flags = 0;

	ReadDword(f);
	ReadWord(f);
	ReadDword(f);
	ReadDword(f);

	number = ReadDword(f);

	for( DWORD i = 0 ; (!feof(f))&&i<number ; i++ )
	{
		key=new GEM_KEY;

		if( !key )
			return gem_error_memory;

		frame=ReadDword(f);
		key->frame=frame;

		key->tension = 0.0f;
		key->bias = 0.0f;
		key->continuity = 0.0f;
		key->easefrom = 0.0f;
		key->easeto = 0.0f;

		flag=ReadWord(f);
		for( int j=0 ; j<16 ; j++ )
			if( flag&(1<<j) )
				switch( j )
				{
					case 0: key->tension=ReadFloat(f); break;
					case 1: key->continuity=ReadFloat(f); break;
					case 2: key->bias=ReadFloat(f); break;
					case 3: key->easeto=ReadFloat(f); break;
					case 4: key->easefrom=ReadFloat(f); break;
				}

		a=ReadFloat(f);
		b=ReadFloat(f);
		c=ReadFloat(f);
		key->_vec.x=a;
		key->_vec.y=c;	//wector swap
		key->_vec.z=b;

		AddKey(key,track);
	}

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje jednowymiarowa sciezke (roll, FOV)
 */
int ReadFloatTrack(FILE *f, GEM_TRACK* track)
{
	
	DWORD			number;
	WORD			flag;
	D3DVALUE		a;
	DWORD			frame;
	GEM_KEY*		key;

	track->keyList = NULL;
	track->flags = 0;

	ReadDword(f);
	ReadWord(f);
	ReadDword(f);
	ReadDword(f);

	number=ReadDword(f);

	for( DWORD i = 0 ; (!feof(f))&&i<number ; i++ )
	{
		key = new GEM_KEY;;

		if( !key )
			return gem_error_memory;

		frame=ReadDword(f);
		key->frame=frame;

		key->tension=0;
		key->bias=0;
		key->continuity=0;
		key->easefrom=0;
		key->easeto=0;

		flag=ReadWord(f);
		for( int j = 0 ; j<16 ; j++ )
			if( flag&(1<<j) )			
				switch (j)
				{
					case 0: key->tension=ReadFloat(f); break;
					case 1: key->continuity=ReadFloat(f); break;
					case 2: key->bias=ReadFloat(f); break;
					case 3: key->easeto=ReadFloat(f); break;
					case 4: key->easefrom=ReadFloat(f); break;
				}
			
		a=ReadFloat(f);
		key->_val=a;

		AddKey(key,track);
	}

	if( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje sciezke morfingu, kazdy key zawiera nazwe obiektu do
 *  ktorego ma byc zmorphowany obiekt
 */
int ReadMorphTrack(FILE *f, GEM_TRACK *track, GEM_SCENE *s)
{
	
	DWORD			number;
	WORD			flag;	
	DWORD			frame;
	GEM_KEY*		key;
	TCHAR			objname[80];
	GEM_OBJECT*		obj;

	track->keyList = NULL;
	track->flags = 0;

	ReadDword(f);
	ReadWord(f);
	ReadDword(f);
	ReadDword(f);

	number=ReadDword(f);

	for( DWORD i = 0 ; (!feof(f)) && i<number ; i++ )
	{
		key = new GEM_KEY;

		if( !key )
			return gem_error_memory;

		frame = ReadDword(f);
		key->frame = frame;

		key->tension=0;
		key->bias=0;
		key->continuity=0;
		key->easefrom=0;
		key->easeto=0;

		flag=ReadWord(f);
		for( int j = 0 ; j<16 ; j++ )
			if( flag&(1<<j) )			
				switch (j)
				{
					case 0: key->tension=ReadFloat(f); break;
					case 1: key->continuity=ReadFloat(f); break;
					case 2: key->bias=ReadFloat(f); break;
					case 3: key->easeto=ReadFloat(f); break;
					case 4: key->easefrom=ReadFloat(f); break;
				}
     
		ReadASCIIZ( f, objname );
		obj=FindObject( s, objname );
		if( !obj )
			return gem_error_bad_file;

		key->_obj=(GEM_MESH *)obj->object;
		
		AddKey( key, track );
	}

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}

/*
 *  wczytuje track hida, jest on przydatny przy animomaniu skomplikowanych
 *  scen jezeli wiadomo ze nie ktore obiekty w danej chwili sa niewidoczne
 *  si im atrybut na hide, co zapogiega ich transformacja i rendrowaniu
 */
int ReadHideTrack(FILE *f, GEM_TRACK *track)
{
	
	DWORD			number;
	WORD			flag;	
	DWORD			frame;
	GEM_KEY			*key;	
	int	     		hide=0;

	track->keyList = NULL;
	track->flags = 0; 

	ReadDword(f);
	ReadWord(f);
	ReadDword(f);
	ReadDword(f);

	number=ReadDword(f);

	for( DWORD i = 0 ; (!feof(f))&&i<number ; i++ )
	{
		key = new GEM_KEY;

		if( !key )
			return gem_error_memory;

		frame=ReadDword(f);
		key->frame=frame;

		key->tension=0;
		key->bias=0;
		key->continuity=0;
		key->easefrom=0;
		key->easeto=0;

		flag=ReadWord(f);
		for( int j = 0 ; j<16 ; j++ )
			if( flag&(1<<j) )			
				switch (j)
				{
					case 0: key->tension=ReadFloat(f); break;
					case 1: key->continuity=ReadFloat(f); break;
					case 2: key->bias=ReadFloat(f); break;
					case 3: key->easeto=ReadFloat(f); break;
					case 4: key->easefrom=ReadFloat(f); break;
				}
     
		key->_val=(D3DVALUE)(hide^=1);

		AddKey( key, track );
	}

	if( feof(f) )
		return gem_error_bad_file;

 return gem_error_ok;
}


/*
 *  odczytuje informacje o hierarchi obiektu (identyfikator hierarchi)
 */
int ReadHierarchy(FILE *f, GEM_KEYFRAMER *node)
{
	WORD		hierarchy;

	ReadDword(f);
	hierarchy=ReadWord(f);

	node->hierarchy=hierarchy;

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje pelna informacje o sciezce obiektu 3d
 *  tj. sciezke translacji, obrotu, skalowania i morfingu, a takze
 *  informacje o obiekcie do ktorego odnosi sie ta sciezka, pivot point
 *  oraz hierarchie
 */
int ReadMeshTrackChunk(FILE *f, GEM_KEYFRAMER *keyframer, GEM_SCENE *s)
{

	long			size;
	long			pos;
	WORD			sign;
	GEM_MESH_TRACK* track;
	int				err;

	keyframer->type = MESHTRACK;
	track=new GEM_MESH_TRACK;

	if( !track )
		return gem_error_memory;

	size=ReadDword(f)-6;
	pos=ftell(f);

	track->position=NULL;
	track->rotation=NULL;
	track->scale=NULL;
	track->morph=NULL;
	track->hide=NULL;

	while( (!feof(f))&&ftell(f)<size+pos )
	{
		sign=ReadWord(f);

		switch (sign)
		{
			case OBJECT_INFO_CHUNK:
				if( (err=ReadObjectInfo(f,keyframer))!=gem_error_ok )
					return err;
			break;

			case ROTATION_TRACK_CHUNK:
				track->rotation = new GEM_TRACK;
				if( !track->rotation )
					return gem_error_memory;

				if( (err=ReadQuaternionTrack(f,track->rotation))!=gem_error_ok )
					return err;
				InitQuatSpline( track->rotation );
			break;

			case POSITION_TRACK_CHUNK:
				track->position = new GEM_TRACK;
				if( !track->position )
					return gem_error_memory;

				if( (err=ReadVectorTrack(f, track->position))!=gem_error_ok )
					return err;
				InitVectorSpline(track->position);
			break;

			case SCALE_TRACK_CHUNK:
				track->scale = new GEM_TRACK;
				if( !track->scale )
					return gem_error_memory;

				if( (err=ReadVectorTrack(f, track->scale))!=gem_error_ok )
					return err;
				InitVectorSpline(track->scale);
			break;

			case MORPH_TRACK_CHUNK:
				track->morph = new GEM_TRACK;
				if( !track->morph )
					return gem_error_memory;

				if( (err=ReadMorphTrack(f, track->morph, s))!=gem_error_ok )
					return err;
			break;

			case HIDE_TRACK_CHUNK:
				track->hide = new GEM_TRACK;
				if( !track->hide )
					return gem_error_memory;

				if( (err=ReadHideTrack(f, track->hide))!=gem_error_ok )
					return err;
			break;

			case HIERARCHY_CHUNK:
				if( (err=ReadHierarchy(f, keyframer))!=gem_error_ok )
					return err;
			break;

			case OBJECT_PIVOT_CHUNK:
				if( (err=ReadPivot(f, keyframer))!=gem_error_ok )
					return err;
			break;

			default:
				if( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
			break;
		}
	}

	keyframer->track=track;

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje informacje o ruchu zrodla swiatla
 */
int ReadLightTrackChunk(FILE *f, GEM_KEYFRAMER *keyframer)
{

	long				size;
	long				pos;
	WORD				sign;
	GEM_LIGHT_TRACK*	track;
	int					err;

	keyframer->type=LIGHTTRACK;
	track = new GEM_LIGHT_TRACK;

	if( !track )
		return gem_error_memory;

	size = ReadDword(f)-6;
	pos = ftell(f);

	track->position = NULL;
	track->color = NULL;

	while( (!feof(f))&&ftell(f)<size+pos )
	{
		sign=ReadWord(f);

		switch (sign)
		{
			case OBJECT_INFO_CHUNK:
				if( (err=ReadObjectInfo(f, keyframer))!=gem_error_ok )
					return err;
			break;

			case POSITION_TRACK_CHUNK:
				track->position = new GEM_TRACK;
				if( !track->position )
					return gem_error_memory;

				if( (err=ReadVectorTrack(f,track->position))!=gem_error_ok )
					return err;
				InitVectorSpline(track->position);
			break;

			case HIERARCHY_CHUNK:
				if( (err=ReadHierarchy(f, keyframer))!=gem_error_ok )
					return err;
			break;

			default:
				if( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
			break;
		}

	}

	keyframer->track=track;

	if ( feof(f) )
		return gem_error_bad_file;

	return gem_error_ok;
}



/*
 *  odczytuje informacje o ruchu celu kamery
 */
int ReadCameraTrgTrackChunk(FILE *f, GEM_KEYFRAMER *keyframer)
{

	long				 size;
	long				 pos;
	WORD				 sign;
	GEM_CAMERATRG_TRACK* track;
	int 				 err;


	keyframer->type	= CAMERATRGTRACK;
	track = new GEM_CAMERATRG_TRACK; 

	if (!track)
		return gem_error_memory;

	size = ReadDword(f)-6;
	pos = ftell(f);

	track->position=NULL;

	while( (!feof(f))&&ftell(f)<size+pos )
	{
		sign = ReadWord(f);

		switch (sign)
		{
			case OBJECT_INFO_CHUNK:
				if( (err=ReadObjectInfo(f, keyframer))!=gem_error_ok )
					return err;
			break;

			case POSITION_TRACK_CHUNK:
				track->position = new GEM_TRACK;
				if( !track->position )
					return gem_error_memory;
				
				if( (err=ReadVectorTrack(f,track->position))!=gem_error_ok )
					return err;
				InitVectorSpline(track->position);
			break;

			case HIERARCHY_CHUNK:
				if( (err=ReadHierarchy(f, keyframer))!=gem_error_ok )
					return err;
			break;

			default:
				if (JumpThroughtChunk(f)!=gem_error_ok)
					return gem_error_bad_file;
			break;
		}

	}

	keyframer->track=track;

 if (feof(f))
  return gem_error_bad_file;

 return gem_error_ok;

}


/*
 *  czyta sciezki kamery (pozycje i roll'a)
 */
int ReadCameraTrackChunk(FILE *f, GEM_KEYFRAMER *keyframer)
{

	long				size;
	long				pos;
	WORD				sign;
	GEM_CAMERA_TRACK*	track;
	int					err;


	keyframer->type = CAMERATRACK;
	track = new GEM_CAMERA_TRACK;

	if( !track )
		return gem_error_memory;

	size = ReadDword(f)-6;
	pos = ftell(f);

	track->roll = NULL;
	track->position = NULL;
	track->fov = NULL;

	while( (!feof(f))&&ftell(f)<size+pos )
	{
		sign=ReadWord(f);

		switch (sign)
		{
			case OBJECT_INFO_CHUNK:
				if( (err=ReadObjectInfo(f, keyframer))!=gem_error_ok )
					return err;
			break;

			case POSITION_TRACK_CHUNK:
				track->position = new GEM_TRACK;
				if( !track->position )
					return gem_error_memory;
				
				if( (err=ReadVectorTrack(f, track->position))!=gem_error_ok )
					return err;
				InitVectorSpline(track->position);
			break;

			case ROLL_TRACK_CHUNK:
				track->roll = new GEM_TRACK;
				if( !track->roll )
					return gem_error_memory;

				if( (err=ReadFloatTrack(f, track->roll))!=gem_error_ok )
					return err;
				InitFloatSpline(track->roll);
			break;

			case FOV_TRACK_CHUNK:
				track->fov = new GEM_TRACK;
				if( !track->fov )
					return gem_error_memory;

				if( (err=ReadFloatTrack(f, track->fov))!=gem_error_ok )
					return err;
				InitFloatSpline(track->fov);
			break;


			case HIERARCHY_CHUNK:
				if( (err=ReadHierarchy(f, keyframer))!=gem_error_ok )
					return err;
			break;

			default:
				if (JumpThroughtChunk(f)!=gem_error_ok)
					return gem_error_bad_file;
			break;
		}

	}

	keyframer->track=track;

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  odczytuje chunk klatek czyli informacje od ktorej klatki zaczyna sie
 *  animacja i na ktorej sie konczy
 */
int ReadFramesChunk(FILE *f, GEM_SCENE *scene)
{

	ReadDword(f);
	scene->startFrame = ReadDword(f);
	scene->endFrame = ReadDword(f);

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;

}

/*
 *  odchytuje informacje o wszystkich sciezkach obiektow
 */
int ReadKeyframerChunk(FILE *f, GEM_SCENE *scene)
{
	long			size;
	long			pos;
	WORD			sign;
	GEM_KEYFRAMER*	keyframer;
	int				err;

	size = ReadDword(f)-6;
	pos = ftell(f);

	while( (!feof(f))&&ftell(f)<size+pos )
	{
		sign = ReadWord(f);

		switch (sign)
		{
			case MESH_TRACK_CHUNK:
				keyframer = new GEM_KEYFRAMER;
				if( !keyframer )
					return gem_error_memory;
				if( (err=ReadMeshTrackChunk(f,keyframer,scene))!=gem_error_ok )
					return err;
				AddKeyframer( keyframer, scene );
			break;

			case OMNILIGHT_TRACK_CHUNK:
				keyframer = new GEM_KEYFRAMER;
				if( !keyframer )
					return gem_error_memory;
				if( (err=ReadLightTrackChunk(f,keyframer))!=gem_error_ok )
					return err;
				AddKeyframer( keyframer, scene );
			break;

			case CAMERA_TRACK_CHUNK:
				keyframer = new GEM_KEYFRAMER;
				if( !keyframer )
					return gem_error_memory;
				if( (err=ReadCameraTrackChunk(f,keyframer))!=gem_error_ok )
					return err;
				AddKeyframer( keyframer, scene );
			break;

			case CAMERA_TARGET_TRACK_CHUNK:
				keyframer = new GEM_KEYFRAMER;
				if( !keyframer )
					return gem_error_memory;
				if( (err=ReadCameraTrgTrackChunk(f,keyframer))!=gem_error_ok )
					return err;
				AddKeyframer( keyframer, scene );
			break;

			case FRAMES_CHUNK:
				if( (err=ReadFramesChunk(f,scene))!=gem_error_ok )
					return err;
			break;

			default:
				if (JumpThroughtChunk(f)!=gem_error_ok)
					return gem_error_bad_file;
			break;
		}
	}

	if (feof(f))
		return gem_error_bad_file;

	return gem_error_ok;
}


/*
 *  wczytuje plik 3DS
 */
int Read3DS(TCHAR* name, GEM_SCENE *scene)
{
	WORD			sign;
	FILE			*f;
	int				err;

	if( !(f=fopen(name,"rb")) )
		return gem_error_file_open;

	sign=ReadWord(f);
	if (sign != MAIN_CHUNK)
		return gem_error_bad_file;

	ReadDword(f);
	while (1)
	{
		sign = ReadWord(f);
		if( feof(f) )
			break;

		switch( sign )
		{
			case EDITOR_CHUNK:
				if( (err=ReadEditorChunk(f,scene))!=gem_error_ok )
					return err;
			break;

			case KEYFRAMER_CHUNK:
				if( (err=ReadKeyframerChunk(f,scene))!=gem_error_ok )
					return err;
				PrepareKeyframer(scene);
			break;

			case VERSION_CHUNK:
				if( (err=ReadVersionChunk(scene,f))!=gem_error_ok )
					return err;
			break;

			default:
				if( JumpThroughtChunk(f)!=gem_error_ok )
					return gem_error_bad_file;
			break;
		}
	}

	LinkHierarchy(scene);

	return gem_error_ok;
}
