//---DIRECTGEM----------------------------------------------------------------
//
//----------------------------------------------------------------------------
#include "gemmath.h"
#include <math.h>

D3DVALUE Norm(GEM_QUATERNION& q)
{
	return q.w*q.w + q.x*q.x + q.y*q.y +q.z*q.z;
}


GEM_QUATERNION Normalize(GEM_QUATERNION& q)
{
	D3DVALUE		len = sqrt( q.w*q.w + q.x*q.x + q.y*q.y +q.z*q.z );

	if ( !len )
		return q;

	len = 1/len;
	return GEM_QUATERNION( q.w*len, q.x*len, q.y*len, q.z*len );
}


GEM_QUATERNION Inverse(GEM_QUATERNION& q)
{
	D3DVALUE		norm = q.w*q.w + q.x*q.x + q.y*q.y +q.z*q.z;

	if ( norm<0.0f )
		return GEM_QUATERNION(0,0,0,0);
	
	norm = 1.0f/norm;
	return GEM_QUATERNION( q.w*norm, -q.x*norm, -q.y*norm, -q.z*norm );	
}

	
GEM_QUATERNION UnityInverse(GEM_QUATERNION& q)
{
	return GEM_QUATERNION( q.w, -q.x, -q.y, -q.z );
}

GEM_QUATERNION exp(GEM_QUATERNION& q)
{
	GEM_QUATERNION	tmp;
	D3DVALUE		ang = sqrt( q.x*q.x + q.y*q.y + q.z*q.z );
	D3DVALUE		sn;
	
	tmp.w = cos( ang );
	sn = sin( ang );

	if( sn < 0.001f )
		sn = 1.0f;
	else
		sn = sn/ang;

	tmp.x = q.x*sn;
	tmp.y = q.y*sn;
	tmp.z = q.z*sn;

	return tmp;
}

GEM_QUATERNION ln(GEM_QUATERNION& q)
{
	GEM_QUATERNION	tmp;
	D3DVALUE		ang;
	D3DVALUE		sn;

	tmp.w = 0.0f;

	if( fabs( q.w )<=1.0f )
	{
		ang = acos( q.w );
		sn = sin( ang );

		if( sn>0.001f )
		{
			sn = ang/sn;
			tmp.x = sn*q.x;
			tmp.y = sn*q.y;
			tmp.z = sn*q.z;

			return tmp;
		}
	}

	tmp.x = q.x;
	tmp.y = q.y;
	tmp.z = q.z;

	return tmp;
}


GEM_QUATERNION operator^(GEM_QUATERNION& q, D3DVALUE t)
{
	GEM_QUATERNION	tmp;	
	D3DVALUE		ang;
	D3DVALUE		sn1, sn2;
	D3DVALUE		coeff;

	if ( fabs(q.w)<=1.0f )
	{
		ang = acos( q.w );
		tmp.w = cos( t*ang );
		sn1 = sin ( t*ang );
		sn2 = sin ( ang );

		if ( sn2<0.001f )
			coeff = t;
		else
			coeff = sn1/sn2;

		tmp.x = q.x*coeff;
		tmp.y = q.y*coeff;
		tmp.z = q.z*coeff;

		return tmp;
	}

	return GEM_QUATERNION(0, 0, 0, 0);
}


D3DVALUE Dot(GEM_QUATERNION& p, GEM_QUATERNION& q)
{
	return p.w*q.w + p.x*q.x + p.y*q.y + p.z*q.z;
}

GEM_QUATERNION operator*(GEM_QUATERNION& p, GEM_QUATERNION& q)
{
	GEM_QUATERNION	tmp;

	tmp.w = p.w*q.w - p.x*q.x - p.y*q.y - p.z*q.z;
    tmp.x = p.w*q.x + p.x*q.w + p.y*q.z - p.z*q.y;
    tmp.y = p.w*q.y + p.y*q.w + p.z*q.x - p.x*q.z;
    tmp.z = p.w*q.z + p.z*q.w + p.x*q.y - p.y*q.x;

	return tmp;
}



GEM_QUATERNION operator*(GEM_QUATERNION& p, D3DVALUE a)
{
	return GEM_QUATERNION( a*p.w, a*p.x, a*p.y, a*p.z );
}

GEM_QUATERNION operator*(D3DVALUE a, GEM_QUATERNION& p)
{
	return GEM_QUATERNION( a*p.w, a*p.x, a*p.y, a*p.z );
}

GEM_QUATERNION operator+(GEM_QUATERNION& p, GEM_QUATERNION& q)
{
	return GEM_QUATERNION( p.w+q.w, p.x+q.x, p.y+q.y, p.z+q.z );
}

GEM_QUATERNION operator-(GEM_QUATERNION& p, GEM_QUATERNION& q)
{
	return GEM_QUATERNION( p.w-q.w, p.x-q.x, p.y-q.y, p.z-q.z );
}

GEM_QUATERNION operator-(GEM_QUATERNION& q)
{
	return GEM_QUATERNION( -q.w, -q.x, -q.y, -q.z );
}

GEM_QUATERNION Slerp(GEM_QUATERNION& p, GEM_QUATERNION& q, D3DVALUE t)
{
	D3DVALUE		ang;
	D3DVALUE		sn;
	D3DVALUE		dot;
	D3DVALUE		t1, t2;
	
	dot = Dot( p, q );

	if( dot>1.0f )
		dot = 1.0f;
	
	ang = acos( dot );
	sn = sin( ang );

	if ( sn < 0.001f )
	{
		t1 = 1-t;
		t2 = t;
	}
	else
	{
		sn = 1.0f/sn;
		t1 = sin(ang*(1-t))*sn;
		t2 = sin(ang*t)*sn;
	}

	return t1*p + t2*q;
}


//M_QUATERNION ExtraSpinSlerp(GEM_QUATERNION& p, GEM_QUATERNION& q, D3DVALUE t, D3DVALUE spin);

GEM_QUATERNION Squad(GEM_QUATERNION& p, GEM_QUATERNION& a, GEM_QUATERNION& b, GEM_QUATERNION& q, D3DVALUE t)
{
	return Slerp( Slerp( p, q, t ), Slerp( a, b, t ), 2*t*(t-1) );
}

GEM_QUATERNION FromAxisAngle(D3DVALUE x, D3DVALUE y, D3DVALUE z, D3DVALUE angle)
{
	D3DVALUE		ang = 0.5*angle;
	D3DVALUE		sn = sin( ang );
	GEM_QUATERNION	tmp;

	tmp.w = cos( ang );
	tmp.x = -sn*x;
	tmp.y = -sn*y;
	tmp.z = -sn*z;

	return tmp;
}

GEM_QUATERNION FromAxisAngle(GEM_VECTOR& vAxis, D3DVALUE angle)
{
	D3DVALUE		ang = 0.5*angle;
	D3DVALUE		sn = sin( ang );
	GEM_QUATERNION	tmp;

	tmp.w = cos( ang );
	tmp.x = sn*vAxis.x;
	tmp.y = sn*vAxis.y;
	tmp.z = sn*vAxis.z;

	return tmp;
}

GEM_MATRIX FromQuat(GEM_QUATERNION& q)
{
	D3DVALUE		xx = q.x*q.x; 
	D3DVALUE		yy = q.y*q.y; 
	D3DVALUE		zz = q.z*q.z;
    D3DVALUE		xy = q.x*q.y; 
	D3DVALUE		xz = q.x*q.z; 
	D3DVALUE		yz = q.y*q.z;
    D3DVALUE		wx = q.w*q.x; 
	D3DVALUE		wy = q.w*q.y; 
	D3DVALUE		wz = q.w*q.z;
	GEM_MATRIX		mat = IdentMatrix();
    
    mat(1,1) = 1 - 2 * ( yy + zz ); 
    mat(1,2) =     2 * ( xy - wz );
    mat(1,3) =     2 * ( xz + wy );

    mat(2,1) =     2 * ( xy + wz );
    mat(2,2) = 1 - 2 * ( xx + zz );
    mat(2,3) =     2 * ( yz - wx );

    mat(3,1) =     2 * ( xz - wy );
    mat(3,2) =     2 * ( yz + wx );
    mat(3,3) = 1 - 2 * ( xx + yy );


	return mat;
}

