/***********************************************************************************
 * 4dmath.cpp           -       4 dimensional matrix math
 *
 * Author:              Ron Bakker        (madrigal@gmx.net)
 * Description:         Starfield screensaver
 * Date:                04-03-1998
 ***********************************************************************************/

#include <math.h>
#include <string.h>
#include "glidesystem.h"
#include "4dmath.h"

MATRIX4D                 viewTransformMatrix;

/***********************************************************************************
 * Indentity_4D         - 4D Identity Matrix
 ***********************************************************************************/
LPMATRIX4D Indentity_4D (void)
{
    static MATRIX4D matrix;
    
    matrix[0][0] = 1.0f, matrix[0][1] = 0.0f, matrix[0][2] = 0.0f, matrix[0][3] = 0.0f;
    matrix[1][0] = 0.0f, matrix[1][1] = 1.0f, matrix[1][2] = 0.0f, matrix[1][3] = 0.0f;
    matrix[2][0] = 0.0f, matrix[2][1] = 0.0f, matrix[2][2] = 1.0f, matrix[2][3] = 0.0f;
    matrix[3][0] = 0.0f, matrix[3][1] = 0.0f, matrix[3][2] = 0.0f, matrix[3][3] = 1.0f;
    
    return (LPMATRIX4D) &matrix;
}

/***********************************************************************************
 * RotationX_4D         - Generate a matrix with a rotation around the X-axis (in
 *                        degrees)
 ***********************************************************************************/
LPMATRIX4D RotationX_4D (float degrees)
{
    static MATRIX4D matrix;

    // Calculating degrees to radians
    float c = (float)cos( degrees * DEG_TO_RAD );           // Cosinus
    float s = (float)sin( degrees * DEG_TO_RAD );           // Sinus

    matrix[0][0] = 1.0f, matrix[0][1] = 0.0f, matrix[0][2] = 0.0f, matrix[0][3] = 0.0f;
    matrix[1][0] = 0.0f, matrix[1][1] =    c, matrix[1][2] =    s, matrix[1][3] = 0.0f;
    matrix[2][0] = 0.0f, matrix[2][1] =   -s, matrix[2][2] =    c, matrix[2][3] = 0.0f;
    matrix[3][0] = 0.0f, matrix[3][1] = 0.0f, matrix[3][2] = 0.0f, matrix[3][3] = 1.0f;
    
    return (LPMATRIX4D) &matrix;
}

/***********************************************************************************
 * RotationY_4D         - Generate a matrix with a rotation around the Y-axis (in
 *                        degrees)
 ***********************************************************************************/
LPMATRIX4D RotationY_4D (float degrees)
{
    static MATRIX4D matrix;

    // Calculating degrees to radians
    float c = (float)cos( degrees * DEG_TO_RAD );           // Cosinus
    float s = (float)sin( degrees * DEG_TO_RAD );           // Sinus

    matrix[0][0] =    c, matrix[0][1] = 0.0f, matrix[0][2] =   -s, matrix[0][3] = 0.0f;
    matrix[1][0] = 0.0f, matrix[1][1] = 1.0f, matrix[1][2] = 0.0f, matrix[1][3] = 0.0f;
    matrix[2][0] =    s, matrix[2][1] = 0.0f, matrix[2][2] =    c, matrix[2][3] = 0.0f;
    matrix[3][0] = 0.0f, matrix[3][1] = 0.0f, matrix[3][2] = 0.0f, matrix[3][3] = 1.0f;

    return (LPMATRIX4D) &matrix;
}

/***********************************************************************************
 * RotationZ_4D         - Generate a matrix with a rotation around the Z-axis (in
 *                        degrees)
 ***********************************************************************************/
LPMATRIX4D RotationZ_4D (float degrees)
{
    static MATRIX4D matrix;

    // Calculating degrees to radians
    float c = (float)cos( degrees * DEG_TO_RAD );           // Cosinus
    float s = (float)sin( degrees * DEG_TO_RAD );           // Sinus

    matrix[0][0] =    c, matrix[0][1] =    s, matrix[0][2] = 0.0f, matrix[0][3] = 0.0f;
    matrix[1][0] =   -s, matrix[1][1] =    c, matrix[1][2] = 0.0f, matrix[1][3] = 0.0f;
    matrix[2][0] = 0.0f, matrix[2][1] = 0.0f, matrix[2][2] = 1.0f, matrix[2][3] = 0.0f;
    matrix[3][0] = 0.0f, matrix[3][1] = 0.0f, matrix[3][2] = 0.0f, matrix[3][3] = 1.0f;

    return (LPMATRIX4D) &matrix;
}

/***********************************************************************************
 * Translation_4D         - Generate a matrix that translates over vector (x,y,z)
 ***********************************************************************************/
LPMATRIX4D Translation_4D (float x, float y, float z )
{
    static MATRIX4D matrix;
    
    matrix[0][0] = 1.0f, matrix[0][1] = 0.0f, matrix[0][2] = 0.0f, matrix[0][3] = 0.0f;
    matrix[1][0] = 0.0f, matrix[1][1] = 1.0f, matrix[1][2] = 0.0f, matrix[1][3] = 0.0f;
    matrix[2][0] = 0.0f, matrix[2][1] = 0.0f, matrix[2][2] = 1.0f, matrix[2][3] = 0.0f;
    matrix[3][0] =    x, matrix[3][1] =    y, matrix[3][2] =    z, matrix[3][3] = 1.0f;

    return (LPMATRIX4D) matrix;
}

/***********************************************************************************
 * SetViewTransformMatrix_4D - Sets the active viewing transformation matrix, which
 *                             transforms the object into view-space
 ***********************************************************************************/
void SetViewTransformMatrix_4D (MATRIX4D *matrix)
{
    memcpy (viewTransformMatrix, *matrix, sizeof(MATRIX4D));
}

/***********************************************************************************
 * SetViewTransformMatrix_4D - Sets the active viewing transformation matrix, which
 *                             transforms the object into view-space
 ***********************************************************************************/
void ViewMatrixMult_4D (LPMATRIX4D matrix)
{
    MATRIX4D resultMatrix, tempMatrix;
    int i, j;

    memcpy (tempMatrix, matrix, sizeof(MATRIX4D));
    for( j = 0; j < 4; j++ )
    {
        for( i = 0; i < 4; i++ )
        {
            resultMatrix[j][i] = 
                viewTransformMatrix[j][0] * tempMatrix[0][i] +
                viewTransformMatrix[j][1] * tempMatrix[1][i] +
                viewTransformMatrix[j][2] * tempMatrix[2][i] +
                viewTransformMatrix[j][3] * tempMatrix[3][i];
        }
    }
    memcpy (viewTransformMatrix, resultMatrix, sizeof(MATRIX4D));
}

/***********************************************************************************
 * TransformVerticesToView_4D - Transforms a vertexlist into view-space .. this is
 *                              a sort of camera operation.
 ***********************************************************************************/
void TransformVerticesToView_4D (GrVertex3D *viewVerts, GrVertex3D *orgVerts, unsigned numberOfVerts)
{
    GrVertex3D tmpVerts, v;
    MATRIX4D matrix;
    unsigned i;

    memcpy (matrix, viewTransformMatrix, sizeof(MATRIX4D));
    for(i=0; i<numberOfVerts; i++ )
    {
        v = orgVerts[i];
        tmpVerts = v;
        tmpVerts.x = v.x * matrix[0][0] + v.y * matrix[1][0] + v.z * matrix[2][0] + v.w * matrix[3][0];
        tmpVerts.y = v.x * matrix[0][1] + v.y * matrix[1][1] + v.z * matrix[2][1] + v.w * matrix[3][1];
        tmpVerts.z = v.x * matrix[0][2] + v.y * matrix[1][2] + v.z * matrix[2][2] + v.w * matrix[3][2];
        tmpVerts.w = v.x * matrix[0][3] + v.y * matrix[1][3] + v.z * matrix[2][3] + v.w * matrix[3][3];
        viewVerts[i] = tmpVerts;
    }
}

/***********************************************************************************
 * ProjectVerticesToViewport - Perspective projects the a vertexlist into viewport
 *                             coordinates
 ***********************************************************************************/
#define VP_OFFSET 1.0f
#define VP_SCALE  0.5f

void ProjectVerticesToViewport (GrVertex3D *projVerts, GrVertex3D *viewVerts, unsigned numberOfVerts)
{
    GrVertex3D tmpVerts, v;
    MATRIX4D matrix;
    unsigned i;

    /* simplified perspective proj matrix assume unit clip volume */
    matrix[0][0] = 1.0f, matrix[0][1] = 0.0f, matrix[0][2] = 0.0f, matrix[0][3] = 0.0f;
    matrix[1][0] = 0.0f, matrix[1][1] = 1.0f, matrix[1][2] = 0.0f, matrix[1][3] = 0.0f;
    matrix[2][0] = 0.0f, matrix[2][1] = 0.0f, matrix[2][2] = 1.0f, matrix[2][3] = 1.0f;
    matrix[3][0] = 0.0f, matrix[3][1] = 0.0f, matrix[3][2] = 0.0f, matrix[3][3] = 0.0f;

    for( i = 0; i < numberOfVerts; i++ ) {
        v = viewVerts[i];
        tmpVerts = v;
        tmpVerts.x = v.x * matrix[0][0] + v.y * matrix[1][0] + v.z * matrix[2][0] + v.w * matrix[3][0];
        tmpVerts.y = v.x * matrix[0][1] + v.y * matrix[1][1] + v.z * matrix[2][1] + v.w * matrix[3][1];
        tmpVerts.z = v.x * matrix[0][2] + v.y * matrix[1][2] + v.z * matrix[2][2] + v.w * matrix[3][2];
        tmpVerts.w = v.x * matrix[0][3] + v.y * matrix[1][3] + v.z * matrix[2][3] + v.w * matrix[3][3];
        tmpVerts.x /= tmpVerts.w, tmpVerts.y /= tmpVerts.w, tmpVerts.z /= tmpVerts.w;
        tmpVerts.x += VP_OFFSET, tmpVerts.x *= VP_SCALE;
        tmpVerts.y += VP_OFFSET, tmpVerts.y *= VP_SCALE;
        projVerts[i] = tmpVerts;
    }
}


