/* Copyright 1998 by Scott Franke [druid-]
    sfranke@scf.usc.edu
    http://www-scf.usc.edu/~sfranke/glj

	This is a utility written for "druid-'s Stonehenge" release version 1.00
	Written for Operation 3DFX's Second programming contest.
*/

/* 
=======================================
Wavefront .obj to Hubris model (.hm) converter
	version 0.2 - Win32 console
=======================================

todo:
	loading and saving multiple objects within one file
	convert to java (?)
		or at least windowed app
	add ogl preview of object w/ some texture

By Scott Franke [druid-]
	druid-'s GL Journal
	http://www-scf.usc.edu/~sfranke/glj
*/

#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include <string.h>
#include <malloc.h>

#include "obj2hm.h"

int exit;
model_t newmodel;
model_t oldmodel;

void main(int argc, char *argv[])
{
	char filename[12];
//	model_t newmodel;
	data_t newdata;
//	model_t oldmodel;

	exit = 0;

	cout << "druid-'s .obj to .hm converter v0.1\n";
	cout << "-----------------------------------\n";

//	argv[1] = new char[12];
//	strcpy(argv[1], "frac");

	if(argv[1]=='\0')
	{
		cout << "use: obj2hm filename.obj\n";
		return;
	}

	addExt(argv[1], "obj", filename);

	loadOBJ(filename, &newmodel);
	if(exit)
		return;

	processOBJ(&newmodel, &newdata);

	printspecs(&newmodel);

	addExt(argv[1], "hm\0", filename);

	saveHM(filename, &newmodel, &newdata);

	loadHM(filename, &oldmodel);

	printspecs(&oldmodel);

	comp(&newmodel, &oldmodel);

	freemodel(&newmodel);
	freemodel(&oldmodel);
	return;
}

void comp(model_t *m1, model_t *m2)
{
	int i, j, cnt, mismatch;
	float error, tot_error;
	if(m1->num_vert != m2->num_vert)
	{
		cout << "Diff num_vert\n";
	} else
	{
		cnt = tot_error = 0;
		for(i=0;i < (m1->num_vert) ;i++)
		{
			for(j=0;j<3;j++)
			{
				if(m1->verts[i][j] != m2->verts[i][j])
				{
					error = m2->verts[i][j] - m1->verts[i][j];
					tot_error += error;
					cnt++;
				}
			}
		}
		cout << "Total of " << cnt << "/" << m1->num_vert * 3 << " different vertice coordinates\n";
		if(m1->num_vert)
			cout << "Average error per element: " << tot_error / (m1->num_vert * 3) << "\n";
	}	

	cin.get();

	if(m1->num_norm != m2->num_norm)
	{
		cout << "Diff num_norm\n";
	} else
	{
		cnt = tot_error = 0;
		for(i=0;i < (m1->num_norm) ;i++)
		{
			for(j=0;j<3;j++)
			{
				if(m1->norms[i][j] != m2->norms[i][j])
				{
					error = m2->norms[i][j] - m1->norms[i][j];
					tot_error += error;
					cnt++;
				}
			}
		}
		cout << "Total of " << cnt << "/" << m1->num_norm * 3 << " different normal elements\n";
		if(m1->num_norm != 0)
			cout << "Average error per element: " << tot_error / (m1->num_norm * 3) << "\n";
	}

	cin.get();

	if(m1->num_tex != m2->num_tex)
	{
		cout << "Diff num_tex\n";
	} else
	{
		cnt = tot_error = 0;
		for(i=0;i < (m1->num_tex) ;i++)
		{
			for(j=0;j<2;j++)
			{
				if(m1->tex[i][j] != m2->tex[i][j])
				{
					error = m2->tex[i][j] - m1->tex[i][j];
					if((error > .01) || (error < -.01))
					{
						cout << "tex " << i << "," << j << "\n\t";
						cout << m1->tex[i][j] << " " << m2->tex[i][j] << "\n";
						cin.get();
					}
					tot_error += error;
					cnt++;
				}
			}
		}
		cout << "Total of " << cnt << "/" << m1->num_tex * 2 << " different tex elements\n";
		if(m1->num_tex)
		{
			cout << "Total error: " << tot_error << endl;
			cout << "Average error per element: " << tot_error / (m1->num_tex * 2) << "\n";
		}
	}

	if(m1->num_tri != m2->num_tri)
	{
		cout << "Diff num_tri\n";
	} else
	{
		cnt = mismatch = 0;
		for(i=0;i< m1->num_tri; i++)
		{
			for(j=0;j<3;j++)
			{
				if(m1->tris[i].coord[j] != m2->tris[i].coord[j])
				{
					mismatch++;
					cout << "Mismatch in coord for tri " << i << "," << j << endl;
				}
				if(m1->tris[i].coord[j] > m1->num_vert)
				{
					cout << "Error in coord for m1->tri " << i << "," << j
						 << " = " << m1->tris[i].coord[j] << endl;
					cnt++;
				}
				if(m2->tris[i].coord[j] > m1->num_vert)
				{
					cout << "Error in coord for m2->tri " << i << "," << j
						 << " = " << m2->tris[i].coord[j] << endl;
					cnt++;
				}
				if(m1->tris[i].norm[j] != m2->tris[i].norm[j])
				{
					mismatch++;
					cout << "Mismatch in norm for tri " << i << "," << j << endl;
				}
				if(m1->tris[i].norm[j] > m1->num_norm)
				{
					cout << "Error in norm for m1->tri " << i << "," << j
						 << " = " << m1->tris[i].norm[j] << endl;
					cnt++;
				}
				if(m2->tris[i].norm[j] > m1->num_norm)
				{
					cout << "Error in norm for m2->tri " << i << "," << j
						 << " = " << m2->tris[i].norm[j] << endl;
					cnt++;
				}
				if(m1->tris[i].tex[j] != m2->tris[i].tex[j])
				{
					mismatch++;
					cout << "Mismatch in tex for tri " << i << "," << j << endl;
				}
				if(m1->tris[i].tex[j] > m1->num_tex)
				{
					cout << "Error in tex for m1->tri " << i << "," << j
						 << " = " << m1->tris[i].tex[j] << endl;
					cnt++;
				}
				if(m2->tris[i].tex[j] > m1->num_tex)
				{
					cout << "Error in tex for m2->tri " << i << "," << j
						 << " = " << m2->tris[i].tex[j] << endl;
					cnt++;
				}
			}
		}
		cout << "Total of " << cnt << "/" << m1->num_tri * 9 << " different tri elements\n";
	}

}

void addExt(char *base, char *ext, char *string)
{
	int i;

	for(i=0;i<8;i++)
	{
		if(base[i]!='\0')
			string[i] = base[i];
		else
		{
			string[i++] = '.';
			string[i++] = ext[0];
			string[i++] = ext[1];
			string[i++] = ext[2];
			string[i++] = '\0';
			i=8;
		}
	}
}

void freemodel(model_t *model)
{
	if(model->verts && model->num_vert)
		delete [] model->verts;
	if(model->tris && model->num_tri)
		delete [] model->tris;
	if(model->norms && model->num_norm)
		delete [] model->norms;
	if(model->tex && model->num_tex)
		delete [] model->tex;
}

void printspecs(model_t *model)
{
	cout << "Model specs: \n";
	cout << "-------------\n";
	cout << "Vertices: " << model->num_vert << endl
		 << "Normal data: " << model->num_norm << endl
		 << "Texture data: " << model->num_tex << endl
		 << "Triangles: " << model->num_tri << endl
		 << "Bounding box: (" << model->v_min[0] << ", "
							  << model->v_min[1] << ", "
							  << model->v_min[2] << ")\n"
		 << "              (" << model->v_max[0] << ", "
							  << model->v_max[1] << ", "
							  << model->v_max[2] << ")\n";
	cout << "-------------\n";

}

void processOBJ(model_t *model, data_t *data)
{
	int i, j;
	vect3_t v_scale;
	float newval;

	cout << "Processing data...\n"
		 << "------------------\n";

	model->flags=0;
	if(model->num_vert > 0)
		model->flags += FLAG_VERT;
	if(model->num_norm > 0)
		model->flags += 2;
	if(model->num_tex > 0)
		model->flags += 4;
	if(model->num_tri > 0)
		model->flags += 8;

	for(i=0;i<3;i++)
	{
		model->v_min[i] = model->verts[0][i];
		model->v_max[i] = model->verts[0][i];
	}

	cout << "\tFinding bounding box...\n";
	for(i=0;i<model->num_vert;i++)
	{
		for(j=0;j<3;j++)
		{
			if(model->verts[i][j] > model->v_max[j])
				model->v_max[j] = model->verts[i][j];
			else if(model->verts[i][j] < model->v_min[j])
				model->v_min[j] = model->verts[i][j];
		}
	}

	data->new_vert = new int3_t[model->num_vert];
	data->new_norm = new int3_t[model->num_norm];
	data->new_tex =  new int2_t[model->num_tex];
	
	cout << "\tScaling values...\n"
		 << "\t-----------------\n";
	for(i=0;i<3;i++)
	{
		if(model->v_max[i] == model->v_min[i])
		{
			v_scale[i] = 0;
			continue;
		}
		v_scale[i] = 65500.0f / (model->v_max[i] - model->v_min[i]);
	}

	cout << "\t\tScale factors (" << v_scale[0]
						<< ", " << v_scale[1]
						<< ", " << v_scale[2] << ")\n";

	for(i=0;i<model->num_vert;i++)
		for(j=0;j<3;j++)
		{
			newval = (model->verts[i][j] - model->v_min[j])
									* v_scale[j];
			data->new_vert[i][j] = newval;
				/* ERROR CHECKING */ if((newval < 0) || (newval > 65535))
										cout << "Out of bounds on vertex " << i << " " << j << "\n"
											<< "\tOriginal : " << model->verts[i][j] << "\n"
											<< "\tScaled   : " << newval << "\n";
		}

	for(i=0;i<model->num_norm;i++)
		for(j=0;j<3;j++)
		{
			newval = (model->norms[i][j] + 1) * 32700.0f;
			data->new_norm[i][j] = newval;
				/* ERROR CHECKING */ if((newval < 0) || (newval > 65535))
										cout << "Out of bounds on normal " << i << " " << j << "\n"
											<< "\tOriginal : " << model->norms[i][j] << "\n"
											<< "\tScaled   : " << newval << "\n";
		}

	for(i=0;i<model->num_tex;i++)
		for(j=0;j<2;j++)
		{
			newval = (model->tex[i][j] + 1) * 32700.0f;
			data->new_tex[i][j] = newval;
			/* ERROR CHECKING */ if((newval < 0) || (newval > 65535))
									cout << "Out of bounds on texture " << i << " " << j << "\n"
										 << "\tOriginal : " << model->tex[i][j] << "\n"
										 << "\tScaled   : " << newval << "\n";
		}


		for(i=0;i<model->num_tri;i++)
			for(j=0;j<3;j++)
			{
				model->tris[i].coord[j] -= 1;
				if(model->num_norm > 0)
					model->tris[i].norm[j]  -= 1;
				else
					model->tris[i].norm[j] = 0;
				if(model->num_tex > 0)
					model->tris[i].tex[j]   -= 1;
				else
					model->tris[i].tex[j] = 0;
			}
}

void saveHM(char filename[], model_t *model, data_t *data)
{
	int size, total;

	cout << "Saving " << filename << "\n";
	model->version = 1;

	FILE *file;
	file = fopen(filename, "wb");
	if(!file)
	{
		cout << "Cannot open " << filename << "\n";
		return;
	}

	total = 0;
	size = 0;

		/* HEADER */

	cout << "Writing header information.\tSize = ";
	fwrite(&(model->version), sizeof(unsigned char), 1, file);
	fwrite(&(model->flags), sizeof(unsigned char), 1, file);
		total += sizeof(unsigned char) * 2;
		size  += sizeof(unsigned char) * 2;
	fwrite(model->v_max, sizeof(vect3_t), 1, file);
	fwrite(model->v_min, sizeof(vect3_t), 1, file);
		total += sizeof(vect3_t) * 2;
		size += sizeof(vect3_t) * 2;
	fwrite(&(model->num_vert), sizeof(unsigned short int), 1, file);
	fwrite(&(model->num_tri), sizeof(unsigned short int), 1, file);
	fwrite(&(model->num_norm), sizeof(unsigned short int), 1, file);
	fwrite(&(model->num_tex), sizeof(unsigned short int), 1, file);
		total += sizeof(unsigned short int) * 2;
		size  += sizeof(unsigned short int) * 2;
	cout << size << " bytes\n";

		/* TRIANGLE DATA */
	if(model->num_tri > 0)
	{
		cout << "Writing triangle data.\t\tSize = ";
		fwrite(model->tris, sizeof(tri_t), model->num_tri, file);
			size  = sizeof(tri_t) * model->num_tri;
			total += sizeof(tri_t) * model->num_tri;
		cout << size << " bytes\n";
	}

		/* TEXTURE DATA */
	if(model->num_tex > 0)
	{
		cout << "Writing vertex texture data.\tSize = ";
		fwrite(data->new_tex, sizeof(int2_t), model->num_tex, file);
			size   = sizeof(int2_t) * model->num_tex;
			total += sizeof(int2_t) * model->num_tex;
		cout << size << " bytes\n";
	}

		/* NORMAL DATA */
	if(model->num_norm > 0)
	{
		cout << "Writing vertex normal data.\tSize = ";
		fwrite(data->new_norm, sizeof(int3_t), model->num_norm, file);
			size   = sizeof(int3_t) * model->num_norm;
			total += sizeof(int3_t) * model->num_norm;
		cout << size << " bytes\n";
	}

		/* VERTEX DATA */
	if(model->num_vert > 0)
	{
		cout << "Writing vertex position data.\tSize = ";
		fwrite(data->new_vert, sizeof(int3_t), model->num_vert, file);
			size   = sizeof(int3_t) * model->num_vert;
			total += sizeof(int3_t) * model->num_vert;
		cout << size << " bytes\n";
	}
	

	cout << "\n For a grand total of " << total << " bytes.\n";

	fclose(file);
}

void loadHM(char filename[], model_t *model)
{
	FILE *file;
	vect3_t v_scale;
	float newval;
	data_t data;
	int i, j;

	file = fopen(filename, "rb");
	if(!file)
	{
		cout << "Cannot open " << filename << "\n";
		return;
	}

	fread(&(model->version), sizeof(unsigned char), 1, file);
	fread(&(model->flags), sizeof(unsigned char), 1, file);
	fread(model->v_max, sizeof(vect3_t), 1, file);
	fread(model->v_min, sizeof(vect3_t), 1, file);
	fread(&(model->num_vert), sizeof(unsigned short int), 1, file);
	fread(&(model->num_tri), sizeof(unsigned short int), 1, file);
	fread(&(model->num_norm), sizeof(unsigned short int), 1, file);
	fread(&(model->num_tex), sizeof(unsigned short int), 1, file);

		/* TRIANGLE DATA */
	if(model->num_tri > 0)
	{
		model->tris = new tri_t[model->num_tri];
		fread(model->tris, sizeof(tri_t), model->num_tri, file);
	}

		/* TEXTURE DATA */
	if(model->num_tex > 0)
		{
		data.new_tex = new int2_t[model->num_tex];
		model->tex = new vect2_t[model->num_tex];
		fread(data.new_tex, sizeof(int2_t), model->num_tex, file);
	}

		/* NORMAL DATA */
	if(model->num_norm > 0)
	{
		data.new_norm = new int3_t[model->num_norm];
		model->norms = new vect3_t[model->num_norm];
		fread(data.new_norm, sizeof(int3_t), model->num_norm, file);
	}

		/* VERTEX DATA */
	if(model->num_vert > 0)
	{
		data.new_vert = new int3_t[model->num_vert];
		model->verts = new vect3_t[model->num_vert];
		fread(data.new_vert, sizeof(int3_t), model->num_vert, file);
	}

	for(i=0;i<3;i++)
	{
		if(model->v_max[i] == model->v_min[i])
		{
			v_scale[i] = 0;
			continue;
		}
		v_scale[i] = 65500.0f / (model->v_max[i] - model->v_min[i]);
	}

	cout << "Scaling new data...\n";
	for(i=0;i<model->num_vert;i++)
	{
		for(j=0;j<3;j++)
		{
			newval = data.new_vert[i][j] / v_scale[j] + model->v_min[j];
			model->verts[i][j] = newval;
		}
	}
	for(i=0;i<model->num_norm;i++)
	{
		for(j=0;j<3;j++)
		{
			newval = data.new_norm[i][j] / 32700.0f - 1.0f;
			model->norms[i][j] = newval;
		}
	}
	for(i=0;i<model->num_tex;i++)
	{
		for(j=0;j<2;j++)
		{
			newval = data.new_tex[i][j] / 32700.0f - 1.0f;
			model->tex[i][j] = newval;
		}
	}
}

void loadOBJ(char filename[], model_t *model)
{
	char type, buffer[80];
	int i, cnt;

	cout << "Loading " << filename << "\n";
	ifstream file(filename);
	if(!file)
	{
		cout << "cannot open " << filename << '\n';
		exit = 1;
		return;
	}

	model->verts = new vect3_t[4096];
	model->tris  = new tri_t[4096];
	model->norms = new vect3_t[4096];
	model->tex  = new vect2_t[4096];
	model->num_vert = model->num_tri = model->num_norm = model->num_tex = 0;

	cnt = 0;

	while(!file.eof())
	{
		file.get(type);
		if((type==' ') || (type=='\n'))
			continue;
		switch(type)
		{
		case '#':		/* Commented line */
		case 'g':		/* group info */
			break;
		case 'v':
			file.get(type);
			switch(type) {
			case ' ':
				file>> model->verts[model->num_vert][0]
					>> model->verts[model->num_vert][1]
					>> model->verts[model->num_vert][2];
				model->num_vert++;
				break;
			case 'n':
				file>> model->norms[model->num_norm][0]
					>> model->norms[model->num_norm][1]
					>> model->norms[model->num_norm][2];
				model->num_norm++;
				break;
			case 't':
				file>> model->tex[model->num_tex][0]
					>> model->tex[model->num_tex][1];
				model->num_tex++;
				break;
			default:
				break;
			}
			break;
		case 'f':
			for(i=0;i<3;i++)
			{
				file >> model->tris[model->num_tri].coord[i];
				type=file.peek();
				if(type!='/')
					while((type!=' ') && (type!='\n'))
					{
						file.get(type);
						type = file.peek();
					}
				else /* texture and/or normal info */
				{
					file.get(type);
					type=file.peek();
					if(type!='/') /* texture */
						file >> model->tris[model->num_tri].tex[i];
					file.get(type);
					type = file.peek();
					if(type!='/') /* normal */
						file >> model->tris[model->num_tri].norm[i];
				}
			}
			model->num_tri++;
			break;
		default:
			break;
		}
		file.getline(buffer,80,'\n');
	}

}

void saveOBJ(char filename[], model_t *model)
{
	int i,j;

	cout << "Saving wavefront .obj file\n";
	ofstream file(filename);
	if(!file)
	{
		cout << "cannot open " << filename << " for saving.\n";
		return;
	}
	file << "# Define a landscape\n";
	for(i=0;i<model->num_vert;i++)
		file << "v " << model->verts[i][0]
		     << " " << model->verts[i][1]
			 << " " << model->verts[i][2]
			 << "\n";

	for(i=1;i<model->num_tri;i++)
	{
		file << "f";
		for(j=0;j<3;j++)
		{
			file << " " << model->tris[i].coord[j] << "/";
			if(model->num_tex > 0)
				file << model->tris[i].tex[j];
			file << "/";
			if(model->num_norm > 0)
				file << model->tris[i].norm[j];
		}
	}
}
