/* Copyright 1998 by Scott Franke [druid-]
    sfranke@scf.usc.edu
    http://www-scf.usc.edu/~sfranke/glj

	This is a utility written for "druid-'s Stonehenge" release version 1.00
	Written for Operation 3DFX's Second programming contest.
*/

/* 
=======================================
Fractal Based Landscape Generation Tool
	version 1.2
=======================================

By Scott Franke [druid-]
	druid-'s GL Journal
	http://www-scf.usc.edu/~sfranke/glj
*/

#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include <math.h>

#define TRUE 1;
#define FALSE 0;

long count, i, j;
unsigned char *temp_cp;
short *temp_ip;
long cnt;
unsigned char **LOD;

typedef struct
{
	short scale;
	short lx, lz;
	unsigned char *data;
	short *landT;
	long numT;
	unsigned char *landTex;
	long numTex;

	short numLOD;
	int triangles;
	int lightmaps;
} terrain_t;

terrain_t terrain;

typedef struct
{
	double Xstart, Ystart, Ca, Cb, Zx, Zy, Xtemp, xsize, ysize, result;
	double Xend, Yend;
	int maxcnt;
} frac_t;

frac_t frac;

void saveFrac(void);
void smoothData(void);
void triangulate(void);
void calcFrac(void);
unsigned char *buildLOD(int numV);

main()
{
	int smooth=0;
	char ctemp;

	terrain.triangles = terrain.lightmaps = FALSE;
	frac.Ca = frac.Cb = frac.Xstart = frac.Ystart = frac.Xtemp = 0.0f;

	cout << "druid-'s Fractal Landscape Generator 1.2\n";

	cout << "xstart: ";
	cin >> frac.Xstart;
	cout << "ystart: ";
	cin >> frac.Ystart;
	cout << "x size: ";
	cin >> frac.xsize;
	cout << "y size: ";
	cin >> frac.ysize;

	cout << "x resolution:";
	cin >> terrain.lx;
	cout << "y resolution:";
	cin >> terrain.lz;

	cout << "height scale factor (1/x): ";
	cin >> terrain.scale;

	cout << "Calculate lightmap?\n";
	cin >> ctemp;
	if(ctemp=='y'||ctemp=='Y')
		terrain.lightmaps=TRUE;
	
	cout << "Number of LOD models?\n";
	cin >> terrain.numLOD;

	frac.maxcnt = 256;

	cout << "Xstart= " << frac.Xstart << " " <<
		"Ystart= " << frac.Ystart << " " <<
		"size = "  << frac.xsize   << " " <<
		"maxcnt= " << frac.maxcnt << "\n";
	cout << " res: " << terrain.lx << "\nLOD models: " << terrain.numLOD << endl;

	temp_cp = new unsigned char[(terrain.lx+4) * (terrain.lz+4)];
	calcFrac();
	smoothData();
	delete temp_cp;

	triangulate();

	terrain.landT = new short[cnt+1];
	for(i=0;i<cnt;i++)
		terrain.landT[i] = temp_ip[i];
	terrain.numT = cnt-1;
	delete temp_ip;

	terrain.triangles = TRUE;

	terrain.landTex = new unsigned char[terrain.lx * terrain.lz * 2];
	cnt=0;
	for(i=0;i<terrain.lz;i++)
		for(j=0;j<terrain.lx;j++)
		{
			terrain.landTex[cnt++] = j*256 / terrain.lx;
			terrain.landTex[cnt++] = i*256 / terrain.lz;
		}
	terrain.numTex = cnt;

	if(terrain.numLOD)
	{
		LOD = new unsigned char * [terrain.numLOD];
		for(i = 0; i<terrain.numLOD;i++)
			LOD[i] = buildLOD(20 * (int) pow(i,1.5));
	}
	else LOD = NULL;

	saveFrac();

	cout << "some data for textures: \n";
	cin.get();
	for(int i=0;i<100;i++)
		cout << (int) terrain.landTex[i] << "\t";

	for(i = 0; i<terrain.numLOD;i++)
		delete LOD[i];
	if(LOD)
		delete LOD;

	delete terrain.data;
	delete terrain.landT;
	delete terrain.landTex;

	return(0);
}

void calcFrac(void)
{
	int k;

	cout << "Calculating fractal data\n";
	for(j=0; j<terrain.lz+3; j++)
	{
		for(k=0; k<terrain.lx+3; k++)
		{
			count=0;
			frac.Ca = frac.Xstart + k * frac.xsize/terrain.lx;
			frac.Cb = frac.Ystart + j * frac.ysize/terrain.lz;
			frac.Zx = frac.Zy = 0;
			do {	
				count+=1;
				frac.Xtemp = frac.Zx*frac.Zx - frac.Zy*frac.Zy;
				frac.Zy = 2*frac.Zx*frac.Zy + frac.Cb;
				frac.Zx = frac.Xtemp + frac.Ca;
				frac.result = frac.Zx*frac.Zx + frac.Zy*frac.Zy;
			} while((count<=frac.maxcnt)&&(frac.result<=4.0));
		temp_cp[j*terrain.lx+k] = count;
		}
	}
}

void smoothData(void)
{
	int j,k;

	terrain.data = new unsigned char[terrain.lx * terrain.lz + 5];
	cout << "Smoothing data\n";
	for(j=0; j<terrain.lz; j++)
	{
		for(k=0; k<terrain.lx; k++)
		{
			terrain.data[j*terrain.lx+k] = (temp_cp[(j)*terrain.lx+k] + temp_cp[(j+1)*terrain.lx+k+1] +
								temp_cp[(j+2)*terrain.lx+k] + temp_cp[(j+1)*terrain.lx+k-1] +
								temp_cp[j*terrain.lx+k-1] + temp_cp[j*terrain.lx+k+1] +
								temp_cp[(j+2)*terrain.lx+k-1] + temp_cp[(j+2)*terrain.lx+k+1] +
								temp_cp[(j+1)*terrain.lx+k]) / 9;		}
	}
}

void triangulate(void)
{
	int j;

	cnt = 0;
	j = 0;

	temp_ip = new short[terrain.lx * terrain.lz * 2];

	for(i=0;i<terrain.lz-2;i++)
	{
		for(j=0;j<terrain.lx;j++)
		{
			temp_ip[cnt] = i*terrain.lx + j; cnt++;
			temp_ip[cnt] = i*terrain.lx + j + terrain.lx; cnt++;
		}
		i++;
		j = terrain.lx-1;
		temp_ip[cnt] = i*terrain.lx + j + terrain.lx; cnt++;
		temp_ip[cnt] = i*terrain.lx + j; cnt++;
		for(j=terrain.lx-2;j>=0;j--)
		{
			temp_ip[cnt] = i*terrain.lx + j + terrain.lx; cnt++;
			temp_ip[cnt] = i*terrain.lx + j; cnt++;
		}
	}
}

unsigned char *buildLOD(int numV)
{
	cout << "\nBuilding LOD model\n";

	cout << "Feature not implemented yet\n";
	terrain.numLOD = 0;

	cout << "Done building LOD model\n";

	return NULL;
}

void saveFrac(void)
{
	FILE *fp;
	int i, j;

	fp = fopen("frac.dat", "wb+");
	if(!fp)
	{
		cout << "error opening frac.dat\n";
		return;
	}
/*
	cout << "Saving raw file\n";
			//  Header information
	char version[] = "v1.2";
	fwrite(version, 4 * sizeof(char), 1, fp);
	fwrite(&terrain.lx, sizeof(short), 1, fp);
	fwrite(&terrain.lz, sizeof(short), 1, fp);
	fwrite(&terrain.scale, sizeof(short), 1, fp);
	fwrite(&terrain.numLOD, sizeof(short), 1, fp);
	fwrite(&terrain.triangles, sizeof(int), 1, fp);
	fwrite(&terrain.lightmaps, sizeof(int), 1, fp);
			// Actual Data
	fwrite(terrain.data, terrain.lx * terrain.lz, 1, fp);
		cout << "Size of terrain data: " << (terrain.lx * terrain.lz) << endl;
	fwrite(&terrain.numT, sizeof(long), 1, fp);
	fwrite(terrain.landT, sizeof(short), terrain.numT, fp);
		cout << terrain.numT << " triangle informations.\n";
		cout << "  Size of initial mesh: " << terrain.numT * sizeof(short) << endl;
	fwrite(&terrain.numTex, sizeof(long), 1, fp);
	fwrite(terrain.landTex, sizeof(unsigned char), terrain.numTex, fp);
		cout << "Size of texture information: " << terrain.numTex << endl;
	for(i=0;i<terrain.numLOD;i++)
		fwrite(LOD[i], (unsigned int)(20 * pow(i, 1.5)), 1, fp);
	fclose(fp);
*/
	cout << "Saving wavefront .obj file\n";
	ofstream file("frac.obj");
	if(!file)
	{
		cout << "cannot load frac.obj\n";
		return;
	}
	file << "# Define a landscape\n";
	for(i=0;i<terrain.lz;i++)
		for(j=0;j<terrain.lx;j++)
		{
			file << "v " << (float)j/terrain.lx * (1000.0f*frac.xsize)
			            << " " << (float)(terrain.data[i*terrain.lx + j]) / terrain.scale
						<< " " << (float)i/terrain.lz * (1000.0f*frac.ysize)
						<< "\n";
		}
	for(j=0;j<terrain.lz-1;j++)
	{
		for(i=1;i<terrain.lx;i++)
		{
			file << "f " << i + (j+1)*terrain.lx << 
					" " << i+j*terrain.lx+1 << 
					" " << i+j*terrain.lx << "\n";
			file << "f " << i+j*terrain.lx+1 << 
					" " << i+(j+1)*terrain.lx << 
					" " << i+(j+1)*terrain.lx+1 << "\n";
		}
	}
}
