// Custom DPCM encoder/decoder for PAiN
//
// dake/calodox
// http://www.calodox.org
//
// under the "do what you want with this file" license.


//  Test on sample.raw (8 bits - signed - 22050 hz - mono) with WinZip
// 
//  QUANTIZE_RATIO at 2 - K&K coeffs 
//
//  sample.raw		uncompressed : 53'243 bytes				zipped : 40'481 bytes
//	dpcm_data.raw   uncompressed : 53'243 bytes				zipped : 24'342 bytes
// 
//  ratio : 2.18
// 
//  QUANTIZE RATIO at 2 - Custom coeffs
// 
//  dpcm_data.raw											zipped : 19'289 bytes
//  
//  ratio : 2.76
//
//  use CoolEdit to read them. 'sample_new.raw' is the decompressed output.


#include "stdafx.h"
#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <limits.h>

// plot the deltas distribution - ansi style	
//#define PLOT_DISTRIBUTION			

//  constant quantization - medium quality  (the highest, the worst)
#define QUANTIZE_RATIO	2

signed char* LoadSample(const char* filename, int& file_length)
{
	FILE* f = fopen(filename, "rb");
	if (f == NULL)
		return NULL;

	fseek(f, 0, SEEK_END);
	file_length = ftell(f);
	fseek(f, 0, SEEK_SET);

	signed char *binary = new signed char[file_length];
	fread(binary, sizeof(signed char), file_length, f);

	fclose(f);
	return binary;
}


#ifdef PLOT_DISTRIBUTION
int distribution[256];

void PlotStats()
{
	int maxval = 0,n,i;

	for (n = 0; n < 256; n++)
		if (distribution[n] > maxval)
			maxval = distribution[n];

	for (n = 0; n < 256; n += 2) {
		for (i = 0; i < (distribution[n]*40) / maxval; i++)
			printf("*");
		printf("\n");
	}
}
#endif

//Kolar & Krishnapuram coeffs - less compression, works better on speech and some instruments
//const float predict_coefficients[]={1.16f, -0.22f, 0.33f, -0.24f, -0.07f};

// custom coeffs - more compression
const float predict_coefficients[] = {0.01f, 0.02f, 0.03f, 0.2f, 0.7f};

void DPCM_encode(const char* output_name, signed char* source, int source_length)
{
	signed char *temp_source = new signed char[source_length];
	signed char *temp_output = new signed char[source_length];

	memcpy(temp_output, source, sizeof(signed char)*source_length);
	memcpy(temp_source, source, sizeof(signed char)*source_length);

#ifdef PLOT_DISTRIBUTION
	memset((char*) &distribution,0,256*sizeof(int));
#endif

	for (int current_pos = 4; current_pos < source_length - 1; current_pos++) {
		// predict the next sample
		// a0*previous[0] + a1*previous[1] + a2*previous[2] + a3*previous[3] + a4*previous[4]

		float predicted = 0.0;

		// linear combination
		for (int i = 0; i < 5; i++)
			predicted += temp_source[current_pos - 4 + i]*predict_coefficients[i];

		// clip to avoid overflow
		if (predicted > SCHAR_MAX)
			predicted = SCHAR_MAX;
		if (predicted < SCHAR_MIN)
			predicted = SCHAR_MIN;

		// compute difference between predicted sample and real sample
		signed char delta = (signed char) (temp_source[current_pos + 1]-predicted);

#ifdef PLOT_DISTRIBUTION
		distribution[delta + 127]++;
#endif

		// quantize
		delta >>= QUANTIZE_RATIO;

		// dequantize and find inverse transformed sample
		int newsample = (int)(predicted + (delta << QUANTIZE_RATIO));

		// clip again
		if (predicted > SCHAR_MAX)
			predicted = SCHAR_MAX;
		if (predicted < SCHAR_MIN)
			predicted = SCHAR_MIN;

		// replace the original sample
		temp_source[current_pos + 1] = newsample;

		// send the binary
		temp_output[current_pos + 1] = (signed char) delta;
	}

	FILE* f = fopen(output_name, "wb");
	fwrite(temp_output, source_length, sizeof(signed char), f);
	fclose(f);

#ifdef PLOT_DISTRIBUTION
	PlotStats();
#endif

	delete[] temp_output;
	delete[] temp_source;
}


void DPCM_decode(const char* output_name, signed char* source, int& source_length)
{
	signed char *temp_source = new signed char[source_length];
	memcpy(temp_source, source, sizeof(signed char)*source_length);

	for (int current_pos = 4; current_pos < source_length - 1; current_pos++) {
		float predicted = 0.0;

		// compute the weighted sum
		for (int i = 0; i < 5; i++)
			predicted += temp_source[current_pos - 4 + i]*predict_coefficients[i];

		// clip clip
		if (predicted > SCHAR_MAX)
			predicted = SCHAR_MAX;
		if (predicted < SCHAR_MIN)
			predicted = SCHAR_MIN;

		// dequantize and find the original sample
		predicted += (temp_source[current_pos + 1] << QUANTIZE_RATIO);

		// clip
		if (predicted > SCHAR_MAX)
			predicted = SCHAR_MAX;
		if (predicted < SCHAR_MIN)
			predicted = SCHAR_MIN;

		// replace original signal
		temp_source[current_pos + 1] = (signed char) predicted;
	}

	FILE* f = fopen(output_name, "wb");
	fwrite(temp_source, source_length, sizeof(signed char), f);
	fclose(f);

	delete[] temp_source;
}



int main(int argc, char* argv[])
{
	// Load sample, it is a 8 bits signed format (raw, no header)
	int data_length;

	printf("Read sample.raw\n");

	signed char *data = LoadSample("sample.raw", data_length);
	if (data==NULL) {printf("Couldn't find sample.raw\n"); return 1;}

	printf("Encode and store transformed data into dpcm_data.raw\n");

	DPCM_encode("dpcm_data.raw", data, data_length);

	// load the transformed data in memory and decompress
	int encode_length = 0;
	signed char *encoded = LoadSample("dpcm_data.raw", encode_length);
	if (encoded==NULL) {printf("Couldn't find dpcm_data.raw\n"); return 1;}

	printf("Decode into new_sample.raw\n");
	DPCM_decode("new_sample.raw", encoded, encode_length);

	delete[] data;
	delete[] encoded;

	return 0;
}
