// Dynamic Huffman compressor
//
// Fabian Giesen 24.08.98

#include <stdio.h>
#include <string.h>

#define OUTBUFLEN 1024

typedef unsigned long ulong;

typedef struct hnode {
          int  count, orgval, index;
          struct hnode *child[2], *parent;
        } huffnode;

typedef struct hcode {
          int  code, size;
        } huffcode;

ulong     count[257], totcount;
huffcode  codes[257];
huffnode *hufftree, *huffroot;
int       hufflen, symsused;
int       bitbuf=0, bitsused=0, bl_count[33];
char      outbuf[OUTBUFLEN];
int       outcp=0;

#ifdef __WATCOMC__

extern void gather_statistics_block(char *blk, int sz);
#pragma aux gather_statistics_block=\
            "xor  eax, eax",\
            "lp:",\
            "mov  al, [esi]",\
            "inc  esi",\
            "inc  dword ptr count[eax*4]",\
            "dec  ecx",\
            "jnz  lp",\
            parm [esi][ecx] modify exact [eax ecx esi];

#else

void gather_statistics_block(char *blk, int sz)
{
  for (int i=0; i<sz; i++) count[blk[i]++];
};

#endif

void gather_statistics(FILE *f)
{
  char *buffer;
  int   i, r, s;

  memset(&count, 0, sizeof(count)); totcount=0;

  buffer=new char[65535];

  fseek(f, 0, SEEK_END); s=ftell(f); fseek(f, 0, SEEK_SET);

  do
  {
    if (s>65535) r=65535; else r=s;

    fread(buffer, 1, r, f);
    gather_statistics_block(buffer, r);

    totcount+=r;

    s-=r;
  } while (s);

  delete[] buffer;

  fseek(f, 0, SEEK_SET);

  count[256]=1;
};

void create_huffman_tree()
{
  int       i, j, huffind, mincount, *allowed;
  huffnode *min1, *min2;

  for (i=0, symsused=0; i<257; i++) if (count[i]) symsused++;

  hufflen=(symsused<<1)-1; huffind=0;
  hufftree=new huffnode[hufflen];
  allowed=new int[hufflen];

  for (i=0; i<257; i++)
  {
    if (count[i])
    {
      memset(&hufftree[huffind], 0, sizeof(huffnode));

      allowed[huffind]=1;
      hufftree[huffind].orgval=i;
      hufftree[huffind].index=huffind;
      hufftree[huffind++].count=count[i];
    };
  };

  for (i=huffind; i<hufflen; i++)
  {
    mincount=0x7FFFFFFF; min1=NULL; min2=NULL;

    for (j=0; j<huffind; j++)
    {
      if ((hufftree[j].count<=mincount) && (allowed[j]))
      {
        min1=&hufftree[j]; mincount=hufftree[j].count;
      };
    };

    allowed[min1->index]=0;

    mincount=0x7FFFFFFF;

    for (j=0; j<huffind; j++)
    {
      if ((hufftree[j].count<=mincount) && (allowed[j]))
      {
        min2=&hufftree[j]; mincount=hufftree[j].count;
      };
    };

    allowed[min2->index]=0;

    min1->parent=min2->parent=&hufftree[huffind];

    hufftree[huffind].count=min1->count+min2->count;
    hufftree[huffind].orgval=-1;
    hufftree[huffind].index=huffind;
    hufftree[huffind].child[0]=min1;
    hufftree[huffind].child[1]=min2;
    hufftree[huffind].parent=NULL;
    allowed[huffind]=1;

    huffind++;
  };

  huffroot=&hufftree[huffind-1];
};

void create_huffman_codes()
{
  int       val, i, bits;
  huffnode *n;

  for (i=0; i<256; i++)
  {
    codes[i].code=codes[i].size=-1;
  };

  for (i=0; i<symsused; i++)
  {
    val=bits=0; n=&hufftree[i];

    while (n->parent)
    {
      val<<=1; if (n->parent->child[1]==n) val|=1;

      n=n->parent; bits++;
    };

    codes[hufftree[i].orgval].code=val;
    codes[hufftree[i].orgval].size=bits;
  };
};

void writeb(char b, FILE *out)
{
  outbuf[outcp++]=b;

  if (outcp==OUTBUFLEN)
  {
    fwrite(&outbuf, 1, outcp, out);
    outcp=0;
  };
};

void flushbuf(FILE *out)
{
  if (outcp)
  {
    fwrite(&outbuf, 1, outcp, out);
    outcp=0;
  };
};

void putbits(int val, int bits, FILE *out)
{
  bitbuf+=(val<<bitsused); bitsused+=bits;

  while (bitsused>7)
  {
    writeb(bitbuf & 0xff, out);

    bitbuf>>=8; bitsused-=8;
  };
};

void flushbits(FILE *out)
{
  while (bitsused>7)
  {
    writeb(bitbuf & 0xff, out);

    bitbuf>>=8; bitsused-=8;
  };

  if (bitsused)
  {
    writeb(bitbuf, out); bitbuf=bitsused=0;
  };
};

void puthuffcode(int code, FILE *out)
{
  putbits(codes[code].code, codes[code].size, out);
};

void endhuffstream(FILE *out)
{
  puthuffcode(256, out);
  flushbits(out);
  flushbuf(out);
};

void do_compress(FILE *in, FILE *out)
{
  char *inbuf;
  int   i, r;

  inbuf=new char[65535];

  fwrite(&count, 4, 256, out);

  do
  {
    r=fread(inbuf, 1, 65535, in);

    if (r) for (i=0; i<r; i++) puthuffcode(inbuf[i], out);
  } while (r==65535);

  delete[] inbuf;

  endhuffstream(out);
};

void compress_huffman(char *name1, char *name2)
{
  FILE *inf, *outf;

  inf=fopen(name1, "rb");

  if (inf)
  {
    printf("Opened file \"%s\".\n", name1);

    printf("Gathering statistics...\n");
    gather_statistics(inf);

    printf("Creating huffman tree\n");
    create_huffman_tree();

    printf("Creating code tables\n");
    create_huffman_codes();

    outf=fopen(name2, "wb");

    if (outf)
    {
      printf("Compressing...");
      do_compress(inf, outf);
      printf("Done!\n");

      fclose(outf);
    };

    fclose(inf);
  };
};

int main(int argc, char *argv[])
{
  setbuf(stdout, NULL);

  printf("Huffman coder (c) 1998 F. Giesen\n\n");

  if (argc!=3)
  {
    printf("Syntax: HUFFC <INFILE.EXT> <OUTFILE.EXT>\n");
    return 1;
  };

  compress_huffman(argv[1], argv[2]);

  return 0;
};
