
; Hugi size coding Compo 29 entry
; -------------------------------
; by Tapani Utriainen
;
; Final submission (80h bytes).
;
; It assumed that ax=bx=0000, cx=00ffh and bp=09xx
;

.486                             ; an elegant weapon, for a more civilized time

;
; Constants used (to simplify calculations of offsets).
;

WIDTH equ 25
HEIGHT equ 10
ROWLEN equ 78
CELLDIV equ 02h
TOPLEFTCELL equ (CELLDIV*256*3 - 1)


cseg  segment use16
   assume cs:cseg, ds:cseg, es:cseg, ss:cseg
   org 100h

; +----------------------------------------------------------------------------
; | Read command line parameter
; +----------------------------------------------------------------------------
;
; Reads a numeric seed from ds:5dh (1st command line arg) to dx
; If no arg present, dx is set to initial value of ax (assumed to be 0000).

start:   
; guarantee assumed initial state (for turbo debugger)  ~17 bytes
;  mov   si, 100h
;  xor   ax, ax
;  xor   bx, bx
;  mov   cx, 00ffh
;  mov   dx, ds
;  mov   di, sp
;  mov   bp, 091ch

   mov   si, 5dh                       ; 3  si <-- ofs for first param
@nextchar:
   aad                                 ; 2  al <-- ah*10 + al; ah <-- 00h
   xchg  ax, dx                        ; 1  dx (dl) <-- seed so far
   mov   ah, dl                        ; 2  ah <-- seed so far (ready for aad)
   lodsb                               ; 1  al <-- read next character
   sub   al, 30h                       ; 2  convert al to digit
   jnc   @nextchar                     ; 2  go on until a space is found
                                       ; --------------------------------------
                                       ; 13  command line parsing

; CPU state after parsing command line:
; ax = 00f0, bx = 0000, cx = 00ff, dx = seed (or 0000 if no seed given)
; si = 005x, di = fffe, bp = 09xx, sp = 0000, CF=1

; +----------------------------------------------------------------------------
; | Randomize table and draw maze
; +----------------------------------------------------------------------------
;
; A printable image of maze is created in memory. Useful for debugging too.
;
; Random numbers are stored in an array in memory. The algorithm uses exactly
; 252 random numbers, however the table created here is larger than that.
;
; Registers are used as follows:
;  ax : the current pattern written (alternates between rows, obviously)
;  bx : counts down the row we are writing to
;  cx : counts down the columns we are writing to
;  dx : the current random number
;  di : the position we are writing to
;  si : the position to the random number table created
;  bp : -
;
; Since the recursive part considers any cell containing 02h (in left position) 
; unvisited, there are no boundary checks on x or y. Instead the maze image
; starts with too many rows of "+--+--+--+" pattern to bound the y coordinate
; from above and the maze image ends with many rows of '$' characters for lower
; bound. The x values are bounded by newlines at the end of the rows.
;
; First "row" is therefore 300h (cx*3 + 3) long, and uses the initial cx=00ff
; The initialization procedure also has to set di>~0d60h to avoid the deletion
; of the first wall to influence the maze image. 
;

   mov   di, TOPLEFTCELL - 0300h       ; 3  di <-- where to fill
   mov   bl, 2*HEIGHT+1                ; 2  bx <-- nof rows to fill
   mov   ax, 7c02h                     ; 3  ax <-- pattern to fill with
   add   si, ax                        ; 2  si <-- ofs for random number table
   push  si                            ; 1  store table ofs
@next_row:
   mov   byte ptr [ds:TOPLEFTCELL], al ; 3  draw entrance character
   xor   ax, 7c02h xor 2b2dh           ; 3  flip pattern: 7c02h <--> 2b2dh
   mov   [di], ah                      ; 2  character for leftmost column
   inc   di                            ; 1
@next_column:
   stosb                               ; 1  fill with pattern
   stosw                               ; 1
   imul  dx, 4e35h                     ; 4  generate a random number
   inc   dx                            ; 1
   mov   [si], dh                      ; 2  store random number in the table
   inc   si                            ; 1
   loop  @next_column                  ; 2  loop over the row

   mov   cl, ROWLEN/3-1                ; 2  cells to fill for next row
   mov   word ptr [di], 0a0dh          ; 4  newline -- without destroying ax
   scasw                               ; 1  add di, 2
   dec   bx                            ; 1  count down rows
   jg    @next_row                     ; 2  fill next row

   mov   word ptr [di-ROWLEN-4], si    ; 3  draw exit character (si=7fxxh)
   mov   ch, CELLDIV                   ; 2  fill a bit more and set ofs to maze
   mov   al, '$'                       ; 2  msdos string terminator
   jz    @next_column                  ; 2  jump once when bx=0
                                       ; --------------------------------------
                                       ; 40 init maze, 11 RNG
   
; +----------------------------------------------------------------------------
; | Select starting position
; +----------------------------------------------------------------------------
;
; Sets si to point to the table of random bytes, so they can be read w. lodsb
; Sets cx and bp to x and y coordinates of the starting cell (respectively)
; A constant adress, to compensate for the maze offset is already present in ch
;

   pop   si                            ; 1  si <-- offset to random table

; x coordinate
   lodsb                               ; 1  get random no
   aam   19h                           ; 2  al <-- al mod 25
   mov   cl, al                        ; 2  cl <-- starting x

; y coordinate
   lodsb                               ; 1  get rnd
   aam   0ah                           ; 2  al <-- al mod 10
   cbw                                 ; 1  ax <-- starting y
                                       ; --------------------------------------
                                       ; 10.5 starting position (next one=half)

; CPU state after init maze:
; ax = 09xx, bx = 0000, cx = 02XX, dx = random
; si = 7c04, di = 12xx, bp = 00YY (soon), sp = 0000

; +----------------------------------------------------------------------------
; | Connect function (recursive)
; +----------------------------------------------------------------------------
;
; in:
;  si = offset of next random byte
;  di = previous pos (when entered first time, di is large; wall is no prob.)
;  bp = 00yy - yy is the y coordinate
;  cx = 02xx - xx is the x coordinate
;  ah = 09h when entered first time, ah = 20h when called recursively
;
; The value of ah will be passed to int 21h, causing a printout when entering
; at the top level, and a call to "DOS null function for CP/M compatibility"
; otherwise.
;
; use of other regs:
;  bx = current position
;  dx = loop counter (always >=4 for initial seeds in 0--100)
;  

connect_from:                          ; FOLLOWING BYTE IS MODIFIED RUNTIME!!
   xchg  ax, bp                        ; 1  bp <-- y, ax <-- 09xx

   imul  bx, bp, 2*ROWLEN/3            ; 3  bx <-- y*52 (bx will be curr. pos)
   add   bx, cx                        ; 2  bx <-- maze_ofs/3 + x + y*52 
   imul  bx, bx, 3                     ; 3  bx <-- 3*(maze_ofs/3 + x + y*52)

   cmp   byte ptr [bx], ch             ; 2  is current pos visited?
   jne   @return                       ; 2  if so, return  (ch = 02h)
   
   add   di, bx                        ; 2  di is previous pos; wall pos is
   sar   di, 1                         ; 2  inbetween current and prev pos.
   stosw                               ; 1  delete the wall (ax = 2020h)

;  mov   dx, 4                         ; devel instruction: to speed things up
   lodsb                               ; 1  retrieve a random byte
@loop_directions:
   and   al, 03h                       ; 2  al <-- al mod 4
   mov   di, bx                        ; 2  di <-- current pos
   pusha                               ; 1  save regs

   shl   al, 2                         ; 3  map al onto {41,45,49,4c}, since
   add   al, 41h                       ; 2  41h=inc cx, that is increase x &
   mov   byte ptr [connect_from], al   ; 3  45h=inc bp, 49h=dec cx, 4dh=dec bp

   mov   ax, 2020h                     ; 3  2 spaces, ah<--20h for DOS nullfunc
   stosw                               ; 1  mark current cell as visited

   pop   edi                           ; 2  restore di, pop si off stack
   call  connect_from                  ; 3  recurse
   push  si                            ; 1  new si, to avoid same random no seq
   push  bx                            ; 1  fill up stack for popa

   popa                                ; 1  restore all regs from stack
   inc   ax                            ; 1  increase direction count
   dec   dx                            ; 1  decrease loop counter
   jnz   @loop_directions              ; 2  loop... many times

   mov   dx, TOPLEFTCELL-ROWLEN        ; 3  ofs to maze
   int   21h                           ; 2  print... or call DOS null function
@return:
   ret                                 ; 1
                                       ; ------------------------------------
                                       ; 48.5 DFS, 5 printout
   cseg  ends
   end   start 

