;;;
;;;  Hugi Size Coding Competition 29 - Maze Generator
;;;
;;;  Entry by Stefan <streu@gmx.de>
;;;
;;;  Basic ideas:
;;;  - parse command line using FCB fields (saves us from whitespace handling)
;;;  - represent maze as byte array, FFh means unvisited cell, >=0 means
;;;    visited cell, so we can use 'inc' as 'test-and-set' instruction
;;;    (first version used a bitfield and 'bts' which generates more code
;;;    despite more flexible "addressing" and initialisation).
;;;  - geometric representation, meaning a "wall" cell between two "visited"
;;;    cells.
;;;  - generate whole output array at once, patch over ghost and house, and
;;;    output everything at once.
;;;
;;;  Assumptions (as per general.txt):
;;;    CH = 0, AX = 0, SI = 0100h, CLD
;;;
;;;  Build using:
;;;     tasm /m/la/t entry
;;;     tlink /t entry
;;;
;;;  DOSEMU does not set the FCB field when starting the process.
;;;  Set DOSEMU=1 if you want to run this in DOSEMU. Turbo Debugger may not
;;;  initialize registers as we want them. Set DEBUGGER=1 if you want to run
;;;  this in the debugger. Both of these enlarge the program, of course.
;;;
;;;  Versions:
;;;  18/Oct/2009        239 bytes: first working version
;;;                     237 bytes: negative logic in bitfields
;;;                     233 bytes: swap bp/bx
;;;                     226 bytes: use partial registers where possible
;;;                     220 bytes: more optimisations
;;;  23/Oct/2009        217 bytes: byte-based representation
;;;                     205 bytes: more optimisations
;;;  24/Oct/2009        199 bytes: negative logic, inc instead of bt
;;;                     187 bytes: more optimisations
;;;  27/Oct/2009        178 bytes: more optimisations, executable data :-)
;;;                     177 bytes: eliminate recursion
;;;

        model   tiny
        p386
        codeseg
        org     100h

        ;; Set to one to run in Turbo Debugger, which doesn't set
        ;; all registers according to plan.
        DEBUGGER = 0

        ;; Set to one to run in DOSEMU, which doesn't initialize
        ;; the FCB field in the PSP.
        DOSEMU   = 0

        Rows    = 10
        Cols    = 25

        RowStride = 2*Cols+3

start:
SEED = $
;;;
;;; Initialize the Maze (part 1)
;;;
        mov     di, offset Maze ; BF 00 40, we need the 0 byte
        push    si              ; pushes 0100h, as per general.txt

CheckTab:                       ; we need CheckTab on an address ending in ...4h.
        db      1, RowStride, -1, -RowStride ; add [di],si; dec bx

;;; Initialisation for debugger
IF DEBUGGER
        push    di
        cld
        mov     di, offset OutputBuffer
        mov     cx, 8000h
        mov     al, 10h
        rep stosb
        pop     di
        xor     ax, ax
        mov     si, 100h
ENDIF

;;;
;;; Initialize the Maze (part 2)
;;;
        pop     bx              ; bx = offset SEED

;;;
;;; Read command line
;;;
        mov     si, 5Dh + DOSEMU*(81h-5Dh) ; file name of first FCB
read1:
        aad                     ; AX = 10*AH + AL
        xchg    ah, al          ; AH = AX
read2:
        lodsb
IF DOSEMU
        cmp     al, 32
        je      read2
ENDIF
        sub     al, 48
        jnb     read1           ; now: AH = seed, CF = set

        mov     [bx], ah

;;;
;;; Initialize the Maze (part 3)
;;;
        xor     ax, ax          ; clears CF
init_again:
        mov     cl, RowStride+1 ; first row of guards, plus first guard of next line
        rep stosb

        mov     dx, 2*Rows+1
init_loop:
        dec     ax
        mov     cl, RowStride-2
        rep stosb               ; a row of unvisited cells/walls
        inc     ax
        stosw                   ; last guard of this line plus first of next line
        dec     dx
        jnz     init_loop

        cmc                     ; do it once again to generate last row of guards, and then some
        jc      init_again

;;;
;;; Generate the maze
;;;
        ;; SI = pos
        ;; BP = direction
        ;; swapping the two means we do not have to double Random(Cols)
        ;; in the address computation, as generate will always compute [di+2*bp]
        ;; anyway.
        xchg    ax, si          ; set SI = 0
        mov     cl, Cols
        call    Random          ; Random(Cols)
        add     ax, 2000h+1+1*RowStride
        xchg    bp, ax

        mov     cl, Rows
        call    Random          ; Random(Rows)
        imul    ax, RowStride
        add     bp, ax

;;;
;;; Generate maze, recursive algorithm, unrolled
;;;
generate:
        ;; in:  bp = direction
        ;;      si = pointer
        ;;      dx = nesting level (starts as 0)
        ;; Mark this cell connected
        inc     dx
        lea     di, [si+bp]     ; si = pos+delta
        lea     si, [di+bp]     ; di = pos+2*delta
        inc     byte ptr [si]   ; set ZF if cell was unvisited (-1), mark cell visited (>=0)
        jnz     gen_been_here

        ;; Pick direction, and repeat for following four directions
        mov     cl, 4
        call    Random
        stosb                   ; mark previous cell visited, AL >= 0
gen_loop:
        pusha
          or    al, offset CheckTab - offset start
          xlat
          cbw
          xchg  bp, ax
          jmp   generate
gen_recurse:
        popa

        ;; Next direction
        inc     ax
        loop    gen_loop
gen_been_here:
        ;; Next recursion
        dec     dx
        jnz     gen_recurse

;;;
;;; Convert maze to memory buffer
;;;
        ;; di = output pointer
        ;; si = source pointer
        ;; cx = row counter
        mov     di, offset OutputBuffer
        push    di
        mov     si, offset Maze + RowStride + 1
        mov     cl, 2*Rows+1

output_next_row:
        ;; First line of a pair (ceilings)
        mov     bp, 2D2Bh       ; "+" and "-"
        mov     ax, 202Bh       ; "+" and " "
output_again:

        ;; Convert single line:
        ;; ax = first case, bp = second case
        mov     dl, Cols+1
output_next_col:
        inc     si
        cmp     byte ptr [si], ch
        jge     output_swap1
        xchg    ax, bp
output_swap1:
        stosw
        mov     [di], ah
        jge     output_swap2
        xchg    ax, bp
output_swap2:
        cmpsb
        dec     dx
        jnz     output_next_col

        ;; We have produced two characters too many. Fortunately,
        ;; that's just the CRLF we need.
        mov     word ptr[di-2], 0A0Dh
        loop    output_continue

        mov     ax, 0924h       ; DOS function number + '$'
        stosb
        pop     dx              ; OutputBuffer
        mov     byte ptr [di-(3*(Cols+1)+4)], dl
        mov     byte ptr [OutputBuffer+(3*Cols+3)], dh
        int     21h
        ret

output_continue:
        ;; Next line (walls/rooms)
        mov     bp, 207Ch
        xor     al,   0Bh       ; AH still " "
        jpo     output_again    ; jump if AH=20h

        lodsw                   ; skip guards
        jmp     output_next_row

;;;
;;; Random Number Generator
;;;
random:
        ;; out: ax = Result

        ;; seed = seed * 0x4E35 + 1
        ;; (seed >> 8) % MAX
        imul    ax, [bx], 4E35h
        inc     ax
        mov     [bx], ax
        shr     ax, 8
        idiv    cl
        shr     ax, 8
        ret

        org 027Fh
        ;; Address of output buffer chosen so that DH=ghost, DL=house
        ;; Size is arbitrary; we generate more output than we need.
OutputBuffer:
        db      8192 dup (?)

        org 4000h
        ;;   xxxxxxxxx     x = guard
        ;;   x.......x     . = wall flag
        ;;   x.o.o.o.x     o = visit flag
        ;;   x.......x
        ;;   x.o.o.o.x
        ;;
        ;;   -1  means cell has not been visited
        ;;   >=0 means cell has been visited (or guard)
Maze:
        db      RowStride*(Rows*2+3) dup (?)

        end     start
