unit quatern;

INTERFACE

uses vector;

const
	EPSILON=1.0e-6;
	X=0;
	Y=1;
	Z=2;
	W=3;

type
	PQuat=^TQuat;
	TQuat=record
		x,y,z,w:double;
	end;

procedure QTFromAng(ang,x,y,z:double;var out:TQuat);
procedure QTToAng(var a:TQuat;var angle,x,y,z:double);
procedure QTMake(w,x,y,z:double;var out:TQuat);
procedure QTIdentity(var out:TQuat);
procedure QTZero(var out:TQuat);
procedure QTCopy(var a,out:TQuat);
procedure QTAdd(var a,b,out:TQuat);
procedure QTSub(var a,b,out:TQuat);
procedure QTMul(var a,b,out:TQuat);
procedure QTDiv(var a,b,out:TQuat);
procedure QTSquare(var a,out:TQuat);
procedure QTSqrt(var a,out:TQuat);
function QTlength(var a:TQuat):double;
function QTDot(var a,b:TQuat):double;
function QTDotUnit(var a,b:TQuat):double;
procedure QTScale(var a:TQuat;s:double;var out:TQuat);
procedure QTRescale(var a:TQuat;s:double;var out:TQuat);
function QTEqual(var a,b:TQuat):boolean;
procedure QTNormalize(var a,out:TQuat);
procedure QTInverse(var a,out:TQuat);
procedure QTNegate(var a,out:TQuat);
procedure QTExp(var a,out:TQuat);
procedure QTLog(var a,out:TQuat);
procedure QTLnDif(var a,b,out:TQuat);
procedure QTSlerp(var a,b:TQuat;spin,alpha:double;var out:TQuat);
procedure QTSlerpl(var a,b:TQuat;spin,alpha:double;var out:TQuat);
procedure QTMatrix(var a:TQuat;var mat:TMatrix);
procedure QTInvMatrix(var a:TQuat;var mat:TMatrix);
procedure QTFromMat (var mat:TMatrix;var out:TQuat);

IMPLEMENTATION

uses math;

procedure QTFromAng(ang,x,y,z:double;var out:TQuat);
var
	s,omega:double;

begin
	omega:=ang/2;
	s:=sin(omega);
	out.w:=cos(omega);
	out.x:=x*s;
	out.y:=y*s;
	out.z:=z*s;
end;

procedure QTToAng(var a:TQuat;var angle,x,y,z:double);
var
	q:TQuat;
	s,omega:double;

begin
	QTNormalize(a,q);
	omega:=acos(q.w);
	angle:=omega*2;
	s:=sin(omega);
	if abs(s)>EPSILON then
		begin
			s:=1/s;
			x:=q.x*s;
			y:=q.y*s;
			z:=q.z*s;
		end
	else
		begin
			x:=0;
			y:=0;
			z:=0;
		end;
end;

procedure QTMake(w,x,y,z:double;var out:TQuat);
begin
	out.w:=w;
	out.x:=x;
	out.y:=y;
	out.z:=z;
end;

procedure QTIdentity(var out:TQuat);
begin
	out.w:=1;
	out.x:=0;
	out.y:=0;
	out.z:=0;
end;

procedure QTZero(var out:TQuat);
begin
	out.w:=0;
	out.x:=0;
	out.y:=0;
	out.z:=0;
end;

procedure QTCopy(var a,out:TQuat);
begin
	out.w:=a.w;
	out.x:=a.x;
	out.y:=a.y;
	out.z:=a.z;
end;

procedure QTAdd(var a,b,out:TQuat);
begin
	out.w:=a.w+b.w;
	out.x:=a.x+b.x;
	out.y:=a.y+b.y;
	out.z:=a.z+b.z;
end;

procedure QTSub(var a,b,out:TQuat);
begin
	out.w:=a.w-b.w;
	out.x:=a.x-b.x;
	out.y:=a.y-b.y;
	out.z:=a.z-b.z;
end;

procedure QTMul(var a,b,out:TQuat);
begin
	out.w:=a.w*b.w-a.x*b.x-a.y*b.y-a.z*b.z;
	out.x:=a.w*b.x+a.x*b.w+a.y*b.z-a.z*b.y;
	out.y:=a.w*b.y+a.y*b.w+a.z*b.x-a.x*b.z;
	out.z:=a.w*b.z+a.z*b.w+a.x*b.y-a.y*b.x;
end;

procedure QTDiv(var a,b,out:TQuat);
var
	q,t,s:TQuat;

begin
	QTCopy(b,q);
	q.x:=-q.x;
	q.y:=-q.y;
	q.z:=-q.z;
	QTMul(a,q,t);
	QTMul(q,q,s);
	out.w:=t.w/s.w;
	out.x:=t.x/s.w;
	out.y:=t.y/s.w;
	out.z:=t.z/s.w;
end;

procedure QTSquare(var a,out:TQuat);
var
	s:double;

begin
	s:=2*a.w;
	out.w:=(a.w*a.w-a.x*a.x-a.y*a.y-a.z*a.z);
	out.x:=s*a.x;
	out.y:=s*a.y;
	out.z:=s*a.z;
end;

procedure QTSqrt(var a,out:TQuat);
var
	m,len,_a,_b:double;
	r:TQuat;

begin
	len:=sqrt(sqr(a.w)+sqr(a.x)+sqr(a.y));
	if len<>0 then
		len:=1/len
	else
		len:=1;
	r.w:=a.w*len;
	r.x:=a.x*len;
	r.y:=a.z*len;
	r.z:=0;
	m:=1/sqrt(sqr(r.w)+sqr(r.x));
	_a:=sqrt((1+r.y)*0.5);
	_b:=sqrt((1-r.y)*0.5);
	out.w:=sqrt(len)*_b*r.w*m;
	out.x:=sqrt(len)*_b*r.x*m;
	out.y:=sqrt(len)*_a;
	out.z:=a.z;
end;

function QTlength(var a:TQuat):double;
begin
	QTLength:=sqrt(sqr(a.w)+sqr(a.x)+sqr(a.y));
end;

function QTDot(var a,b:TQuat):double;
var
	len:double;

begin
	len:=1/(QTLength(a)*QTLength(b));
	QTDot:=(a.w*b.w+a.x*b.x+a.y*b.y+a.z*b.z)*len;
end;

function QTDotUnit(var a,b:TQuat):double;
begin
	QTDotUnit:=(a.w*b.w+a.x*b.x+a.y*b.y+a.z*b.z);
end;

procedure QTScale(var a:TQuat;s:double;var out:TQuat);
begin
	out.w:=a.w*s;
	out.x:=a.x*s;
	out.y:=a.y*s;
	out.z:=a.y*s;
end;

procedure QTRescale(var a:TQuat;s:double;var out:TQuat);
var
	len:double;

begin
	len:=QTLength(a);
	if len=0 then
		begin
			QTidentity(out);
			exit;
		end;
	s:=s/len;
	QTScale(a,s,out);
end;

function QTEqual(var a,b:TQuat):boolean;
begin
	QTEqual:=(a.w=b.w) and (a.x=b.x) and (a.y=b.y) and (a.z=b.z);
end;

procedure QTNormalize(var a,out:TQuat);
begin
	QTScale(a,1/QTlength(a),out);
end;

procedure QTInverse(var a,out:TQuat);
var
	mag:double;

begin
	mag:=(a.w*a.w+a.x*a.x+a.y*a.y+a.z*a.z);
	if mag<>1 then
		mag:=1/mag
	else
		mag:=1;
	out.w:=a.w*mag;
	out.x:=a.x*-mag;
	out.y:=a.y*-mag;
	out.z:=a.z*-mag;
end;

procedure QTNegate(var a,out:TQuat);
begin
	QTNormalize(a,out);
	out.x:=-out.x;
	out.y:=-out.y;
	out.z:=-out.z;
end;

procedure QTExp(var a,out:TQuat);
var
	len,len1:double;

begin
	len:=sqrt(sqr(a.x)+sqr(a.y)+sqr(a.z));
	if len>0 then
		len1:=sin(len)/len
	else
		len1:=1;
	out.w:=cos(len);
	out.x:=a.x*len1;
	out.y:=a.y*len1;
	out.z:=a.z*len1;
end;

procedure QTLog(var a,out:TQuat);
var
	len:double;

begin
	len:=sqrt(sqr(a.x)+sqr(a.y)+sqr(a.z));
	if a.w<>0 then
		len:=ArcTan(len/a.w)
	else
		len:=pi/2;
	out.w:=0;
	out.x:=a.x*len;
	out.y:=a.y*len;
	out.z:=a.z*len;
end;

procedure QTLnDif(var a,b,out:TQuat);
var
	inv,dif:TQuat;
	len,len1,s:double;

begin
	QTInverse(a,inv);
	QTMul(inv,b,dif);
	len:=sqrt(sqr(dif.x)+sqr(dif.y)+sqr(dif.z));
	s:=QTDot(a,b);
	if s<>0 then
		len1:=ArcTan(len/s)
	else
		len1:=pi/2;
	if len<>0 then len1:=len1/len;
	out.w:=0;
	out.x:=dif.x*len1;
	out.y:=dif.y*len1;
	out.z:=dif.z*len1;
end;

procedure QTSlerp(var a,b:TQuat;spin,alpha:double;var out:TQuat);
var
	k1,k2,
	angle,AngleSpin,
	SinA,CosA:double;
	flip:LongInt;

begin
	CosA:=QTDotUnit(a,b);
	if cosa<0 then
		begin
			cosa:=-cosa;
			flip:=-1;
		end
	else
		flip:=1;
	if 1-cosa<EPSILON then
		begin
			k1:=1-alpha;
			k2:=alpha;
		end
	else
		begin
			angle:=acos(cosa);
			SinA:=sin(angle);
			AngleSpin:=angle+spin*pi;
			k1:=sin(angle-alpha*AngleSpin)/SinA;
			k2:=sin(alpha*AngleSpin)/SinA;
		end;
	k2:=k2*flip;
	out.x:=k1*a.x+k2*b.x;
	out.y:=k1*a.y+k2*b.y;
	out.z:=k1*a.z+k2*b.z;
	out.w:=k1*a.w+k2*b.w;
end;

procedure QTSlerpl(var a,b:TQuat;spin,alpha:double;var out:TQuat);
var
	k1,k2,
	angle,AngleSpin,
	SinA,CosA:double;
	flip:LongInt;

begin
	CosA:=QTDotUnit(a,b);
	if 1-abs(cosa)<EPSILON then
		begin
			k1:=0-alpha;
			k2:=alpha;
		end
	else
		begin
			angle:=acos(CosA);
			SinA:=sin(angle);
			AngleSpin:=angle+spin*pi;
			k1:=sin(angle-alpha*AngleSpin)/SinA;
			k2:=sin(alpha*AngleSpin)/SinA;
		end;
	out.x:=k1*a.x+k2*b.x;
	out.y:=k1*a.y+k2*b.y;
	out.z:=k1*a.z+k2*b.z;
	out.w:=k1*a.w+k2*b.w;
end;

procedure QTMatrix(var a:TQuat;var mat:TMatrix);
var
	x2,y2,z2,wx,wy,wz,
	xx,xy,xz,yy,yz,zz:double;

begin
	x2:=a.x+a.x; y2:=a.y+a.y; z2:=a.z+a.z;
	wx:=a.w*x2;   wy:=a.w*y2;   wz:=a.w*z2;
	xx:=a.x*x2;   xy:=a.x*y2;   xz:=a.x*z2;
	yy:=a.y*y2;   yz:=a.y*z2;   zz:=a.z*z2;

	mat[X][X]:=1-(yy+zz);
	mat[X][Y]:=xy+wz;
	mat[X][Z]:=xz-wy;
	mat[X][W]:=0;
	mat[Y][X]:=xy-wz;
	mat[Y][Y]:=1-(xx+zz);
	mat[Y][Z]:=yz+wx;
	mat[Y][W]:=0;
	mat[Z][X]:=xz+wy;
	mat[Z][Y]:=yz-wx;
	mat[Z][Z]:=1-(xx+yy);
	mat[Z][W]:=0;
end;

procedure QTInvMatrix(var a:TQuat;var mat:TMatrix);
var
	x2,y2,z2,wx,wy,wz,
	xx,xy,xz,yy,yz,zz:double;

begin
	x2:=a.x+a.x; y2:=a.y+a.y; z2:=a.z+a.z;
	wx:=a.w*x2;   wy:=a.w*y2;   wz:=a.w*z2;
	xx:=a.x*x2;   xy:=a.x*y2;   xz:=a.x*z2;
	yy:=a.y*y2;   yz:=a.y*z2;   zz:=a.z*z2;

	mat[X][X]:=1-(yy+zz);
	mat[X][Y]:=xy-wz;
	mat[X][Z]:=xz+wy;
	mat[X][W]:=0;
	mat[Y][X]:=xy+wz;
	mat[Y][Y]:=1-(xx+zz);
	mat[Y][Z]:=yz-wx;
	mat[Y][W]:=0;
	mat[Z][X]:=xz-wy;
	mat[Z][Y]:=yz+wx;
	mat[Z][Z]:=1-(xx+yy);
	mat[Z][W]:=0;
end;

procedure QTFromMat(var mat:TMatrix;var out:TQuat);
const
	nxt:array[0..2] of integer=(Y,Z,X);

var
	tr,s:double;
	q:array[0..3] of double;
	i,j,k:integer;


begin
	tr:=mat[X][X]+mat[Y][Y]+mat[Z][Z];
	if tr>0 then
		begin
			s:=sqrt(tr+1);
			out.w:=s/2;
			s:=0.5/s;
			out.x:=(mat[Y][Z]-mat[Z][Y])*s;
			out.y:=(mat[Z][X]-mat[X][Z])*s;
			out.z:=(mat[X][Y]-mat[Y][X])*s;
		end
	else
		begin
			i:=X;
			if mat[Y][Y]>mat[X][X] then i:=Y;
			if mat[Z][Z]>mat[i][i] then i:=Z;
			j:=nxt[i];
			k:=nxt[j];
			s:=sqrt((mat[i][i]-(mat[j][j]+mat[k][k]))+1);
			q[i]:=s/2;
			if s<>0 then s:=0.5/s;
			q[W]:=(mat[j][k]-mat[k][j])*s;
			q[j]:=(mat[i][j]+mat[j][i])*s;
			q[k]:=(mat[i][k]+mat[k][i])*s;
			out.w:=q[W];
			out.x:=q[X];
			out.y:=q[Y];
			out.z:=q[Z];
		end;
end;

end.