unit dsp;

INTERFACE

{ ResetDSP returns true if reset was successful}
function ResetDSP:boolean;

{ Write DAC sets the speaker output level }
procedure WriteDAC(level:byte);

{ ReadDAC reads the microphone input level }
function ReadDAC:byte;

{ SpeakerOn connects the DAC to the speaker }
function SpeakerON:byte;

{ SpeakerOff disconnects the DAC from the speaker,
	but does not affect the DAC operation }
function SpeakerOFF:byte;

{ Functions to pause DMA playback }
procedure DMAstop;
procedure DMAcontinue;

{ Playback plays a sample of a given size back at a given frequency using
	DMA channel. The sample must not cross a page boundry }
procedure PlaySmp(var sound;size:word;frequency:word);

{getSB returns TRUE if SB is installed and set port, dma and int.}
function GetSB:boolean;

IMPLEMENTATION

uses CRT,DOS;

const
	interrupcion:byte=5;
	DirectMemoryAccess:byte=1;

	reset:word=$216;
	read:word=$21a;
	write:word=$21c;
	avail:word=$21e;
	base:word=$210;

function GetSB:BOOLEAN;
var
	contador:integer;
	byt,k:byte;
	t:string;
	err:integer;

begin
	repeat
		port[reset]:=1;
		delay(3);
		port[reset]:=0;

		contador:=0;
		repeat
			byt:=port[read];
			inc(contador);
		until (byt=$aa) or (contador=100);

		if byt<>$AA then
			begin
				inc(base,$10);
				inc(reset,$10);
				inc(read,$10);
				inc(write,$10);
				inc(avail,$10);
			end;
	until (byt=$aa) or (base=$270);

	GetSB:=base<$270;

	for k:=1 to EnvCount do
		begin
			t:=EnvStr(k);
			if pos('BLASTER=',t)>0 then
				begin
					val(copy(t,pos('I',t)+1,1),interrupcion,err);
					break;
				end;
		end;
	if pos('BLASTER=',t)>0 then
		val(copy(t,pos('D',t)+1,1),DirectMemoryAccess,err);
end;

function ResetDSP:boolean;
begin
	{ Reset the DSP, and give some nice long delays just to be safe }
	port[reset]:=1;
	delay(10);
	port[reset]:=0;
	delay(10);
	resetDSP:=(port[avail] and $80=$80) and (port[read]=$AA);
end;

procedure WriteDSP(value:byte);ASSEMBLER;
ASM
	MOV			DX,write

@BUCLE:
	IN			AL,DX
	TEST		AL,$80
	JNZ			@BUCLE

	MOV			AL,value
	OUT			DX,AL
END;

procedure WriteDAC(level:byte);
begin
	WriteDSP($10);
	WriteDSP(level);
end;

function ReadDAC:byte;
begin
	WriteDSP($20);

	ASM
		MOV			CX,16384
		MOV			DX,avail

@BUCLE:
		IN			AL,DX
		DEC			CX
		JZ			@EXIT
		TEST		AL,$80
		JNZ			@BUCLE

@EXIT:
		ADD			DX,4
		IN			AL,DX
	END;
end;

function SpeakerON:byte;
begin
	WriteDSP($D1);
end;

function SpeakerOFF:byte;
begin
	WriteDSP($D3);
end;

procedure DMAcontinue;
begin
	WriteDSP($D4);
end;

procedure DMAstop;
begin
	WriteDSP($D0);
end;

procedure PlaySmp(var sound;size:word;frequency:word);
const
	page:array[0..3] of byte=($87,$83,$81,$82);
	address:array[0..3] of byte=(0,2,4,6);
	length:array[0..3] of byte=(1,3,5,7);

var
	desplazamiento:word;
	pagina:byte;

begin
	SpeakerON;
	DMAstop;

	dec(size);
	{ Set up the DMA chip }
	desplazamiento:=seg(sound) shl 4+ofs(sound);
	pagina:=(seg(sound)+ofs(sound) shr 4) shr 12;

	ASM
		XOR			BH,BH
		MOV			DH,BH
		MOV			CL,DirectMemoryAccess

		MOV			AL,4
		ADD			AL,CL
		OUT			$A,AL

		MOV			AL,0
		OUT			$C,AL

		MOV			AL,$48
		ADD			AL,CL
		OUT			$B,AL

		MOV			AL,BYTE(desplazamiento)
		MOV			BL,CL
		MOV			DL,BYTE(address[BX])
		OUT			DX,AL

		MOV			AL,BYTE(desplazamiento+1)
		OUT			DX,AL

		MOV			AL,pagina
		MOV			DL,BYTE(page[BX])
		OUT			DX,AL

		MOV			AL,BYTE(size)
		MOV			DL,BYTE(length[BX])
		OUT			DX,AL

		MOV			AL,BYTE(size+1)
		OUT			DX,AL

		MOV			AL,CL{DirectMemoryAccess}
		OUT			$A,AL
	END;

	WriteDSP($40);
	WriteDSP(256-(1000000 div frequency));

	{ Set the PlaySmp type (8-bit) }
	WriteDSP($14);
	WriteDSP(lo(size));
	WriteDSP(hi(size));
end;

end.