/*
 * $Id: main.cpp 157 2007-11-12 23:11:40Z ehaase $
 *
 * What We Are
 *
 * Copyright (C) 1994 - 2007 Enver Haase
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */



//
// Includes.
//

#include "main.hpp"
#include "audio/audio.h"
#include "timer.hpp"
#include "exceptions.hpp"
#include "scroller_state.hpp"
#include "c64_state.hpp"
#include "keyboard.hpp"



//
// publicly visible (global) variables.
//

/* The handle of the application's main window */
HWND
main_window_handle = NULL;

/* The DirectDraw object */
LPDIRECTDRAW7
lpdd = NULL;

/* The primary surface */
LPDIRECTDRAWSURFACE7
lpddsprimary = NULL;

/* The back-buffer for double-buffering. */
LPDIRECTDRAWSURFACE7
lpddsback = NULL;

/* a pointer to the created dd palette */
LPDIRECTDRAWPALETTE 
lpddpal = NULL;



//
// private variables.
//

/* The state the application is currently in. */
static state *currentState      = NULL;
/* The state the application was in before. */
static state *lastState         = NULL;
/* Are we operational? Or has somebody pressed, say, ALT-TAB? */
bool
DD_SYSTEM_OPERATIONAL = true; /* It's important not to get a RESUME message when application first comes up. */



//
// private method declarations.
//

/* The game loop. */
static int
gameLoop(void);
/* Switches the application's state. */
static void
transitionTo(state *s) throw (GameException);
/**
 * Helper, shows a message box then exits with an error code.
 */
static void
fatalError(std::string error);
/**
 * Helper, shows a message box then exits with an error code.
 */
static void
fatalError(std::string error, HRESULT res);


//
// IMPLEMENTATION.
// 

static void
transitionTo(state *s) throw (GameException)
{
    if (currentState == s)
    {
        return;
    }

    lastState = currentState;

	if (currentState != NULL)
	{
		currentState->exitState();
	}
	currentState = s;
    if (currentState != NULL)
    {
	    currentState->enterState();
    }
}

static void
trans(state *s)
{
    try{
        transitionTo(s);
    }
    catch (GameException ge)
    {        
        if (ge.getResult() == DDERR_SURFACELOST)
        {
            DD_SYSTEM_OPERATIONAL = false; // we've only lost the surface.
            transitionTo(NULL); // this should always work.
        }
        else
        {
            fatalError(ge.getReason()); // something worse.
        }
    }    
}


void
exitShell(void)
{
    ACloseVoices();
    AStopModule();
    ACloseAudio();

    if (lpdd != NULL)
    {
        lpdd->RestoreDisplayMode();
        lpdd->SetCooperativeLevel(main_window_handle, DDSCL_NORMAL);
    }

    // first release the primary surface
    if (lpddsprimary != NULL)
    {
        lpddsprimary->Release();
    }
    lpddsprimary = NULL;
    lpddsback    = NULL; /* automatically released */
    
    // Release the palette.
    if (lpddpal != NULL)
    {
        lpddpal->Release();
        lpddpal = NULL;
    }

    // release the directdraw object
    if (lpdd!=NULL)
    {
        lpdd->Release();
        lpdd = NULL;
    }

    ShowCursor(true);
}

/**
 * Helper, shows a message box then exits with an error code.
 */
static void fatalError(std::string error)
{
    transitionTo(NULL);
    exitShell();
    MessageBoxA(NULL, error.c_str(), "Fatal Error", MB_ICONSTOP);
    exit(1);
}

static void fatalError(std::string error, HRESULT res)
{
    static const int BUF_MAX = 256;
    char buf[BUF_MAX];
    _ltoa_s(res, buf, 16); // HRESULT is really a LONG
    fatalError(error+std::string(" Error: 0x")+std::string(buf)+std::string("."));
}

/**
 * Initializes the audio.
 */
void
init_audio(void)
{
    UINT err = AInitialize();
    if (err != AUDIO_ERROR_NONE)
    {
        const int bufsize = 255;
        char text[bufsize];
        AGetErrorText(err, (LPSTR) &text, bufsize);
        fatalError(std::string(text));
    }
    AUDIOINFO info;
    LPAUDIOMODULE module;

    info.nDeviceId = AUDIO_DEVICE_MAPPER;
    info.wFormat = (AUDIO_FORMAT_16BITS | AUDIO_FORMAT_STEREO);
    info.nSampleRate = 44100;
    err = AOpenAudio(&info);
    if (err != AUDIO_ERROR_NONE)
    {
        const int bufsize = 255;
        char text[bufsize];
        AGetErrorText(err, (LPSTR) &text, bufsize);
        fatalError(std::string(text));
    }

    ALoadModuleFile("WIZARDYSOUNDMOD" /* "sfx//mod.wizardry" */, &module, 0);

    if (module){
        UINT err = AOpenVoices(module->nTracks);
        if (err != AUDIO_ERROR_NONE)
        {
            const int bufsize = 255;
            char text[bufsize];
            AGetErrorText(err, (LPSTR) &text, bufsize);
            fatalError(std::string(text));
        }
        module->nRestart = 0; // Loop module
        err = APlayModule(module);
        if (err != AUDIO_ERROR_NONE)
        {
            const int bufsize = 255;
            char text[bufsize];
            AGetErrorText(err, (LPSTR) &text, bufsize);
            fatalError(std::string(text));
        }
    }
    else
    {
        fatalError("Could not create sound module.");
    }
}


void
initShell(void)
{
    // create object and test for error
    if (DirectDrawCreateEx(NULL, (LPVOID *) &lpdd, IID_IDirectDraw7, NULL) != DD_OK)
    {
        fatalError("Could not create the DirectDraw object.");
    }

    // set cooperation level to exclusive full screen with ctrl-alt-del allowed
    HRESULT coop = ((IDirectDraw7 *)lpdd)->SetCooperativeLevel(main_window_handle, DDSCL_FULLSCREEN|DDSCL_EXCLUSIVE|DDSCL_ALLOWREBOOT);
    if (coop != DD_OK)
    {
        fatalError("Could not set cooperative level.", coop);
    }

    // set the display mode. 
    if (((LPDIRECTDRAW7)lpdd)->SetDisplayMode(SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_DEPTH, SCREEN_REFRESH_HZ , 0) != DD_OK)
    {
        fatalError("Could not set display mode.");
    }

    
    
    // Create the primary surface
    DDSURFACEDESC2          ddsd; // a direct draw surface description struct
    ZeroMem(ddsd);
    ddsd.dwSize            = sizeof(ddsd);
    ddsd.ddsCaps.dwCaps    = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
    ddsd.dwFlags           = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
    ddsd.dwBackBufferCount = 1;
    HRESULT surfres = lpdd->CreateSurface(&ddsd, &lpddsprimary, NULL);
    if (surfres != DD_OK)
    {
        fatalError("Could not create primary surface.", surfres);
    }

    DDSCAPS2 ddscaps;
    ZeroMem(ddscaps);
    ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
    HRESULT gasres = lpddsprimary->GetAttachedSurface(&ddscaps, &lpddsback);
    if (gasres != DD_OK)
    {
        fatalError("Could not create back buffer for double buffering.", gasres);
    }

    // hide the mouse pointer
    ShowCursor(false);

    init_audio();
}




/**
 * Message handling call-back function.
 * WARNING: This is called from another thread or process
 * as it seems.
 * DO NOT CALL ANY FUNCTIONS OF THIS PROJECT HERE;
 * BUT POST MESSAGES TO OUR PROCESS/THREAD INSTEAD.
 */
LRESULT CALLBACK WindowProc(HWND hwnd,
                            UINT msg,
                            WPARAM wparam,
                            LPARAM lparam)
{
    // this is the main message handler of the system
    
    

    // what is the message
    switch(msg)
    {
    case WM_CREATE:
        {
            // we could do initialization stuff here.
            return 0;
        }
    case WM_PAINT:
        {
            // paint structure?!
            PAINTSTRUCT ps;
            // handle to a device context
            HDC         hdc;
            // start painting
            hdc = BeginPaint(hwnd, &ps);
            // end painting
            EndPaint(hwnd, &ps);
            return 0;
        }
    case WM_LBUTTONDOWN:
        {            
            PostMessage(main_window_handle, WM_APP, APPMSG_ADVANCE_STATE, APPMSG_ADVANCE_STATE);
            return 0;
        }
    case WM_SETFOCUS:
        {
            if (!DD_SYSTEM_OPERATIONAL)
            {
                PostMessage(main_window_handle, WM_APP, APPMSG_RESUME, APPMSG_RESUME);
            }
            return 0;
        }
    default:
        {
            break;
        }
    } // end switch

    // process any messages that we didn't take care of
    return (DefWindowProc(hwnd, msg, wparam, lparam));
} // end WinProc




// WINMAIN ////////////////////////////////////////////////

/**
 * Windows operating system entry point.
 */
int WINAPI WinMain( HINSTANCE hinstance,
                    HINSTANCE hprevinstance,
                    LPSTR lpcmdline,
                    int ncmdshow)
{

    WNDCLASS winclass;  // this will hold the class we create
    HWND     hwnd;      // generic window handle    

    // first fill in the window class stucture
    winclass.style = CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
    winclass.lpfnWndProc    = WindowProc;
    winclass.cbClsExtra     = 0;
    winclass.cbWndExtra     = 0;
    winclass.hInstance      = hinstance;
    winclass.hIcon          = LoadIcon(NULL, IDI_APPLICATION);
    winclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    winclass.hbrBackground  = (HBRUSH) GetStockObject(BLACK_BRUSH);
    winclass.lpszMenuName   = NULL;
    winclass.lpszClassName  = /*(LPCWSTR)*/ WINDOW_CLASS_NAME;

    // register the window class
    if (!RegisterClass(&winclass))
    {
        fatalError("Could not register the window class.");
    }

    // create the window, note the use of WS_POPUP
    hwnd = CreateWindow(    WINDOW_CLASS_NAME,    // class
                            WINDOW_TITLE,  // title
                            WS_POPUP | WS_VISIBLE,
                            0, 0,                           // x,y
                            WINDOW_WIDTH,                   // width
                            WINDOW_HEIGHT,                  // height
                            NULL,                           // handle to parent
                            NULL,                           // handle to menu
                            hinstance,                      // instance
                            NULL);                          // creation parms
    if (!hwnd)
    {
        fatalError("Could not create window.");
    }

    // hide the mouse.
    ShowCursor(false);

    // save the window handle and instance.
    main_window_handle = hwnd;	

	// Return with the game loop's result.
	return gameLoop();
	
}

int
gameLoop(void)
{
    // Enter DirectDraw exclusive mode "shell".
    initShell();

	// Initialize    
	state *scrollerState  = NULL;
	state *c64State       = NULL;
    try{
	    
	    scrollerState  = new scroller_state();
	    c64State       = new c64_state();    
    }
    catch (GameException ge)
    {
        fatalError(ge.getReason());
    }

    DD_SYSTEM_OPERATIONAL = true;
    currentState = NULL;
    lastState = scrollerState;
	trans(scrollerState); // if we can't get there now, we can later as scrollerState was the lastState.
	
	// enter main event loop
	MSG  msg;           // generic message
    bool killed = false;
	while(!killed)      // run until killed.
    {
        // while instead of "if": process ALL messages
        // before using processing time for our "game".
#ifdef AUDIO_PAUSE_WORKAROUND
		while /* if */ (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
#else
		while /* if */ ((DD_SYSTEM_OPERATIONAL?
									PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) :
									(GetMessage(&msg, NULL, 0, 0)!=-1) ))		
#endif
        {
            // test if this is a quit
            if (msg.message == WM_QUIT)
            {                
                    killed = true;
                    break;
            }
            else if (msg.message == WM_APP)
            {
                if (msg.wParam == APPMSG_ADVANCE_STATE)
                {
                    if (currentState == scrollerState)
                    {
                        trans(c64State);
                    } else if (currentState == c64State)
                    {
                        killed = true;
                        break;
                    }
                }
                else if (msg.wParam == APPMSG_RESUME)
                {
                    DD_SYSTEM_OPERATIONAL = true;
                    trans(lastState);
                }
            }
            // translate any accelerator keys
            TranslateMessage(&msg);

            // send the message to the window proc
            DispatchMessage(&msg);
        } // end if/while


        // main game processing goes here               
        try{
            if (currentState != NULL)
            {
                currentState->update();
            }
#ifdef AUDIO_PAUSE_WORKAROUND
            else
            {
                // Paused becaused of ALT-TAB or similar.
                // What an ugly thing that even after the last state's
                // APauseModule() we have to keep updating for a while
                // until it's silent.
                AUpdateAudio();
            }
#endif
        }
        catch (GameException ge)
        {                        
            if (ge.getResult() == DDERR_SURFACELOST){
                DD_SYSTEM_OPERATIONAL = false; // we've only lost the surface.
                trans(NULL); // this should always work.
            }
            else
            {
                fatalError(ge.getReason()); // something worse.
            }            
        }
#ifdef AUDIO_PAUSE_WORKAROUND
		if (!DD_SYSTEM_OPERATIONAL)
		{
			Sleep(200);
		}
#endif
    } // end while


    // shutdown game and release all resources
	trans(NULL);    

    //delete pauseState;
	delete scrollerState;
	delete c64State;    
    
    exitShell();

    // return to Windows like this
    if (msg.message == WM_QUIT)
    {
        return((int) msg.wParam);
    }
    else
    {
        return 0;
    }

} // end WinMain

void
clearSurface(void) throw (GameException)
{
    DDSURFACEDESC2          ddsd; // a direct draw surface description struct
    ZeroMem(ddsd);
    ddsd.dwSize = sizeof(ddsd);
	HRESULT res = lpddsback->GetSurfaceDesc(&ddsd);
	if (res == DDERR_SURFACELOST)
    {
        lpddsback->Restore();
        res = lpddsback->GetSurfaceDesc(&ddsd);
    }
	if (res != DD_OK )
    {
        throwException("Did not get back buffer description.", res);
    }
	res = lpddsback->Lock(NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
    if (res != DD_OK)
    {
        throwException("Could not clear back buffer.", res);		
    }
	if (ddsd.lpSurface != NULL)
	{
		memset(ddsd.lpSurface, 0, ddsd.lPitch * ddsd.dwHeight); // TODO: this is fixed 8 bit per pixel, not very nice.
	}
    lpddsback->Unlock(NULL);
}

void
clearSurfaces(void) throw (GameException)
{
    // Clear the backbuffer; and the primary one.
    // Some graphics card drivers don't do it themselves,
    // especially when restoring after ALT-TAB.
	clearSurface();
    lpddsprimary->Flip(NULL, DDFLIP_WAIT);
	clearSurface();
}
