/*
 * $Id: bitmap.cpp 165 2007-11-14 18:51:07Z hangman $
 *
 * What We Are
 *
 * Copyright (C) 1994 - 2007 Enver Haase
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
 
#include "bitmap.hpp"
#include "../io/iofile.h"
#include "../exceptions.hpp"

void Bitmap::throwException(std::string filename, std::string reason) throw (BitmapCreateError)
{
    throw BitmapCreateError(filename+": "+reason);
}

Bitmap::Bitmap(LPDIRECTDRAW7 pdd, std::string &filename) throw (BitmapCreateError)
{
    SIZE size = loadBitmap(filename);
    pdds = createOffscreenSurface(pdd, size.cx, size.cy);
	if (pdds == NULL)
	{
		throwException(filename, "Cannot create off-screen surface.");
	}
    copyToSurface();    
}

/**
 * Opens a bitmap file and loads the data into bitmap
 */
SIZE 
Bitmap::loadBitmap(std::string &filename) throw (BitmapCreateError)
{
    BITMAP_FILE_PTR bitmap = &bitmap_file;

    // open the file if it exists
    if (IOOpenFile((char *) filename.c_str()) != IO_ERROR_NONE)
    {
        throwException(std::string(filename), "Cannot find file.");
    }

    // now load the bitmap file header
    IOReadFile(&bitmap->bitmapfileheader, sizeof(BITMAPFILEHEADER));

    // test if this is a bitmap file
    if (bitmap->bitmapfileheader.bfType != BITMAP_ID)
    {
        // close the file
        IOCloseFile();

        throwException(std::string(filename), "File is not a bitmap.");
    }

    // now we know this is a bitmap, so read in all the sections

    // first the bitmap infoheader

    // now load the bitmap file header
    IOReadFile(&bitmap->bitmapinfoheader, sizeof(BITMAPINFOHEADER));

    // We don't handle compression yet.
    // 0: no compression
    // 1: 8 bit run length encoding
    // 2: 4 bit run length encoding
    // 3: RGB bitmap with mask (BitField)
    // 4: JPeG
    // 5: PNG
    if (bitmap->bitmapinfoheader.biCompression != 0)
    {
        throwException(std::string(filename), "Cannot handle compression.");
    }
    
    if (bitmap->bitmapinfoheader.biSizeImage == 0)
    {
        throwException(std::string(filename), "File specifies 0 image length.");
    }
    
    if (bitmap->bitmapinfoheader.biPlanes != 1)
    {
        throwException(std::string(filename), "Cannot handle multi-planes bitmaps.");
    }

    if (bitmap->bitmapinfoheader.biBitCount != 8  &&
        bitmap->bitmapinfoheader.biBitCount != 16 &&
        bitmap->bitmapinfoheader.biBitCount != 24 )
    {
        throwException(std::string(filename), "Only 8, 16 or 24 bit BMP supported.");
    }

    // now load the color palette if there is one
    if (bitmap->bitmapinfoheader.biBitCount == 8)
    {
        IOReadFile(&bitmap->palette, 256*sizeof(PALETTEENTRY));

        // now set all the flags in the palette correctly and fix the reversed
        // BGR RGBQUAD data format
        for (int index=0; index < 256; index++)
        {
            // reverse the red and blue fields
            int temp_color = bitmap->palette[index].peRed;
            bitmap->palette[index].peRed  = bitmap->palette[index].peBlue;
            bitmap->palette[index].peBlue = temp_color;

            // always set the flags word to this
            bitmap->palette[index].peFlags = PC_NOCOLLAPSE;
        } // end for index
    } // end if

    // now read in the image, if the image is 8 or 16 bit then simply read it
    // but if its 24 bit then read it into a temporary area and then convert
    // it to a 16 bit image

    if (bitmap->bitmapinfoheader.biBitCount==8 || bitmap->bitmapinfoheader.biBitCount==16)
    {
        // allocate the memory for the image
        if (!(bitmap->buffer = (UCHAR *) malloc(bitmap->bitmapinfoheader.biSizeImage)))
        {
            // close the file
            IOCloseFile();
            throw BitmapCreateError("Cannot allocate memory.");
        } // end if

        // now read it in
        IOReadFile(bitmap->buffer, bitmap->bitmapinfoheader.biSizeImage);

    } // end if
    else
    {
        UCHAR *temp_buffer = (UCHAR *) malloc(bitmap->bitmapinfoheader.biSizeImage);
        if (!temp_buffer){
            // close the file
            IOCloseFile();
            throw BitmapCreateError("Cannot allocate temporary memory.");
        } // end if

        // allocate final 16 bit storage buffer
        if (!(bitmap->buffer = (UCHAR *) malloc(2*bitmap->bitmapinfoheader.biWidth*bitmap->bitmapinfoheader.biHeight)))
        {
            // close the file
            IOCloseFile();

            // release working buffer
            free(temp_buffer);

            throw BitmapCreateError("Cannot allocate memory.");
        } // end if

        // now read it in
        IOReadFile(temp_buffer, bitmap->bitmapinfoheader.biSizeImage);

        // now convert each 24 bit RGB value into a 16 bit value
        for (int index=0; index<bitmap->bitmapinfoheader.biWidth*bitmap->bitmapinfoheader.biHeight; index++)
        {
            // extract RGB components (in BGR order), note the scaling
            UCHAR blue  = (temp_buffer[index*3 + 0] >> 3);
            UCHAR green = (temp_buffer[index*3 + 1] >> 3);
            UCHAR red   = (temp_buffer[index*3 + 2] >> 3);

            // build up 16 bit color word
            USHORT color = ((blue%32) + ((green%32) << 5) + ((red%32) << 10));

            // write color to buffer
            ((USHORT *)bitmap->buffer)[index] = color;

        } // end for index

        // finally write out the correct number of bits
        bitmap->bitmapinfoheader.biBitCount=16;

        free (temp_buffer);
    } // end "else // 24 bit Bitmap"

    // Remember width, height
    SIZE retVal;
    retVal.cx = bitmap->bitmapinfoheader.biWidth;
    retVal.cy = bitmap->bitmapinfoheader.biHeight;

    // close the file
    IOCloseFile();

    // flip the bitmap
    Flip_Bitmap(bitmap->buffer,
        bitmap->bitmapinfoheader.biWidth*(bitmap->bitmapinfoheader.biBitCount/8),
        bitmap->bitmapinfoheader.biHeight);

    return retVal;
}

///////////////////////////////////////////////////////////

Bitmap::~Bitmap(){
    BITMAP_FILE_PTR bitmap = &bitmap_file;
    // this function releases all memory associated with "bitmap"
    if (bitmap->buffer)
    {
        // release memory
        free(bitmap->buffer);

        // reset pointer
        bitmap->buffer = NULL;

    } // end if
    
	pdds->Release();

} // end Unload_Bitmap_File

///////////////////////////////////////////////////////////

LPDIRECTDRAWSURFACE7
Bitmap::createOffscreenSurface(LPDIRECTDRAW7 pdd, int width, int height)
{
	DDSURFACEDESC2 ddsd;
	LPDIRECTDRAWSURFACE7 lpdds;

	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize         = sizeof(ddsd);
    ddsd.dwFlags        = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
	ddsd.dwWidth        = width;
	ddsd.dwHeight       = height;
	if (pdd->CreateSurface(&ddsd, &lpdds, NULL) != DD_OK)
	{
        return NULL;
	}
	else
	{
		return lpdds;
	}
}


void
Bitmap::Flip_Bitmap(UCHAR *image, int bytes_per_line, int height)
{
    // this function is used to flip upside down .BMP images

    UCHAR *buffer; // used to perform the image processing

    // allocate the temporary buffer
    buffer = (UCHAR *) malloc(bytes_per_line*height);
    if (!buffer)
    {
        throw BitmapCreateError("Cannot allocate temporary memory.");
    }

    // copy image to work area
    memcpy(buffer, image, bytes_per_line*height);

    // flip vertically
    for (int index=0; index < height; index++)
    {
        memcpy(&image[((height-1) - index)*bytes_per_line], &buffer[index*bytes_per_line], bytes_per_line);
    }

    // release the memory
    free(buffer);
} // end Flip_Bitmap


LPPALETTEENTRY
Bitmap::getPaletteEntries(void)
{
    return bitmap_file.palette;
}

void
Bitmap::copyToSurface()
{
    

    DDSURFACEDESC2 ddsd;
    memset(&ddsd, 0, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
    HRESULT res = pdds->GetSurfaceDesc(&ddsd);
    if (res != DD_OK)
    {
        return;
    }

    // This "Lock" updates the lpSurface field, the pointer to the raw data.
    res = pdds->Lock(NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
    if (res != DD_OK)
    {
        return;
    }
    UCHAR *primary_buffer = (UCHAR *)ddsd.lpSurface;
    // copy each bitmap line into primary buffer
    // taking into consideration non-linear video
    // cards and the memory pitch lPitch

    if (primary_buffer != NULL)
    {
        int width  = getWidth();
        int height = getHeight();
        if (bitmap_file.bitmapinfoheader.biBitCount == 8)
        {
            for (int y=0; y < height; y++){
                // copy the line
                memcpy(&primary_buffer[y*ddsd.lPitch],      // dest address
                    &bitmap_file.buffer[y*width],           // src address
                    width);                                 // bytes to copy
            } // end for y
        }
        else{ // must biBitCount == 16
            for (int y=0; y < height; y++){
                // copy the line
                memcpy(&primary_buffer[y*ddsd.lPitch],      // dest address
                    &bitmap_file.buffer[y*width*2],         // src address
                    width*2);                               // bytes to copy
            } // end for y
        }
    }
    pdds->Unlock(NULL);
}

int
Bitmap::getHeight(void)
{
	DDSURFACEDESC2 ddsd;
    memset(&ddsd, 0, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
	if (pdds->GetSurfaceDesc(&ddsd) == DD_OK)
	{
		return ddsd.dwHeight;
	}
	else
	{
		return 0;
	}
}

int
Bitmap::getWidth(void)
{
    DDSURFACEDESC2 ddsd;
    memset(&ddsd, 0, sizeof(ddsd));
    ddsd.dwSize = sizeof(ddsd);
	if (pdds->GetSurfaceDesc(&ddsd) == DD_OK)
	{
		return ddsd.dwWidth;
	}
	else
	{
		return 0;
	}
}

/**
 *
 */
void
Bitmap::Blit(LPDIRECTDRAWSURFACE7 dest_pdds, RECT destRectangle)
{
    RECT source;
    source.top    = 0;
    source.left   = 0;
    source.right  = this->getWidth();
    source.bottom = this->getHeight();
    this->Blit(dest_pdds, destRectangle, source);
}

/**
 *
 */
void
Bitmap::Blit(LPDIRECTDRAWSURFACE7 dest_pdds, RECT destRectangle, RECT sourceRectangle)
{
    //HRESULT res = dest_pdds->Blt(&destRectangle, this->pdds, &sourceRectangle, DDBLT_WAIT, NULL);
    HRESULT res = dest_pdds->BltFast(destRectangle.left, destRectangle.top, this->pdds, &sourceRectangle, DDBLTFAST_NOCOLORKEY | DDBLTFAST_WAIT);
    if (res == DDERR_SURFACELOST)
    {
        if (this->pdds->Restore() == DD_OK)
        {
            copyToSurface();
            //dest_pdds->Blt(&destRectangle, this->pdds, &sourceRectangle, DDBLT_WAIT, NULL);
            dest_pdds->BltFast(destRectangle.left, destRectangle.top, this->pdds, &sourceRectangle, DDBLTFAST_NOCOLORKEY | DDBLTFAST_WAIT);
        }
    }
    /* TODO. A dependency on exception.hpp should not be here, and the
     *       method does not even declare it would throw an exception.
     */
    else if (res == DDERR_INVALIDRECT)
    {
        const int BUFSIZE = 1024;
        char problem[BUFSIZE];
        sprintf_s(problem, BUFSIZE, "sTop %d, sBottom %d, sLeft %d, sRight %d, dTop %d, dBottom %d, dLeft %d, dRight %d.",
            sourceRectangle.top, sourceRectangle.bottom, sourceRectangle.left, sourceRectangle.right,
            destRectangle.top,   destRectangle.bottom,   destRectangle.left,   destRectangle.right);
        throw GameException("Invalid rectangles? "+std::string(problem), res);
    }
    else if (res != DD_OK)
    {
        throw GameException("Blit problem.", res);
    }
}
