//---------------------------------------------------------------------------
// ps2 gs library
// done by wiRe / NAPALM
// based on tutorial stuff by Tony Saveski, t_saveski@yahoo.com
//---------------------------------------------------------------------------

#include <tamtypes.h>
#include <kernel.h>
#include "gs.h"

//---------------------------------------------------------------------------
#undef _DEBUG


#ifdef _DEBUG
  #include <stdlib.h>
  #include <stdio.h>
  #define dbgmsg(_msg_) {printf _msg_ ;}
#else
  #define dbgmsg(_msg_) {}
#endif

//---------------------------------------------------------------------------
#define MAX_DMA_TRANSFER_SIZE 16384
#define MAX_TBUF_HANDLES 64

//---------------------------------------------------------------------------
struct {
    uint32 tbp, cbp;
    uint16 w, h;
    uint8 psm, cpsm, tcc, valid;
} tbuf_handle[MAX_TBUF_HANDLES];

uint32 tbuf_tbp = 0;

//---------------------------------------------------------------------------
GS_VMODE gs_vmode;

GS_VMODE_T gs_vmode_t[] = {
    {NON_INTERLACED, PAL, FRAME, 256, 256, 10, PSMCT32},
    {NON_INTERLACED, PAL, FRAME, 320, 256, 8, PSMCT32},
    {NON_INTERLACED, PAL, FRAME, 384, 256, 7, PSMCT32},
    {NON_INTERLACED, PAL, FRAME, 512, 256, 5, PSMCT32},
    {NON_INTERLACED, PAL, FRAME, 640, 256, 4, PSMCT32},
    {INTERLACED, PAL, FIELD, 256, 512, 10, PSMCT32},
    {INTERLACED, PAL, FIELD, 320, 512, 8, PSMCT32},
    {INTERLACED, PAL, FIELD, 384, 512, 7, PSMCT32},
    {INTERLACED, PAL, FIELD, 512, 512, 5, PSMCT32},
    {INTERLACED, PAL, FIELD, 640, 512, 4, PSMCT32},

    {NON_INTERLACED, NTSC, FRAME, 256, 224, 10, PSMCT32},
    {NON_INTERLACED, NTSC, FRAME, 320, 224, 8, PSMCT32},
    {NON_INTERLACED, NTSC, FRAME, 384, 224, 7, PSMCT32},
    {NON_INTERLACED, NTSC, FRAME, 512, 224, 5, PSMCT32},
    {NON_INTERLACED, NTSC, FRAME, 640, 224, 4, PSMCT32},
    {INTERLACED, NTSC, FIELD, 256, 448, 10, PSMCT32},
    {INTERLACED, NTSC, FIELD, 320, 448, 8, PSMCT32},
    {INTERLACED, NTSC, FIELD, 384, 448, 7, PSMCT32},
    {INTERLACED, NTSC, FIELD, 512, 448, 5, PSMCT32},
    {INTERLACED, NTSC, FIELD, 640, 448, 4, PSMCT32}
};

GIF_PACKET gs_init_packet[] = {
    _GIF_TAG(11, 1, 0, 0, 0, 1),    GIF_AD,         // 00: gif tag

    // setup regs
    1,                              GS_PRMODECONT,  // 01: use primitive for mode controll flags
    _GS_ZBUF(0,0,1),                GS_ZBUF_1,      // 02: turn off zbuffer
    15,                             GS_TEST_1,      // 03: skip pixels with A=0, DepthTest off
    ((1<<2)|(1<<6)),                GS_ALPHA_1,     // 04: blend by Cv=(Csrc - Cframe)*Asrc >>7 + Cframe
    0,                              GS_PABE,        // 05: not perform ABC in units of pixels
    1,                              GS_COLCLAMP,    // 06: colclamp on
    0,                              GS_DTHE,        // 07: dont perform dithering
    0,                              GS_FOGCOL,      // 08: fog color is black

    // setup screen
    _GS_FRAME(0,0,0,0),             GS_FRAME_1,     // 09: writeable area
    _GS_XYOFFSET(0x8000,0x8000),    GS_XYOFFSET_1,  // 10: upper/left screen corner is located at 32768/32768
    _GS_SCISSOR(0,0,0,0),           GS_SCISSOR_1,   // 11: drawable area

    // clear screen
    _GIF_TAG(4, 1, 0, 0, 0, 1),     GIF_AD,         // 12: gif tag

    GS_PRIM_SPRITE,         GS_PRIM,        // 13: do a filled rectangle
    _GS_RGBAQ(0,0,0,128,0),         GS_RGBAQ,       // 14: some black color
    _GS_XYZ2(0x8000,0x8000,0),      GS_XYZ2,        // 15: XY
    _GS_XYZ2(0x8000,0x8000,0),      GS_XYZ2         // 16: XY
};

GIF_PACKET gs_swap_screen_packet[] = {
    _GIF_TAG(1, 1, 0, 0, 0, 1),     GIF_AD,         // 00: gif tag
    _GS_FRAME(0,0,0,0),             GS_FRAME_1,     // 01: writeable area
};

GIF_PACKET gs_ee2vram_packet[] = {
    _GIF_TAG(4, 1, 0, 0, 0, 1),     GIF_AD,         // 00: gif tag
    (uint64)0,                      GS_BITBLTBUF,   // 01: dest buffer width / 64
    (uint64)0,                      GS_TRXPOS,      // 02: dest_x,dest_y
    (uint64)0,                      GS_TRXREG,      // 03: width,height
    _GS_TRXDIR(GS_XDIR_EE_GS),      GS_TRXDIR,      // 04: direction
    _GIF_TAG(0, 1, 0, 0, 2, 1),     (uint64)0       // 05: nloop = (w*h*bpp+127)/128 = size in qwords
};

GIF_PACKET gs_finish_packet[] = {
    _GIF_TAG(1, 1, 0, 0, 0, 1),     GIF_AD,         // 00: gif tag
    (uint64)0,                      GS_FINISH,      // 01: raise finish event
};

GIF_PACKET gs_texflush_packet[] = {
    _GIF_TAG(1, 1, 0, 0, 0, 1),     GIF_AD,         // 00: gif tag
    (uint64)0,                      GS_TEXFLUSH,    // 01: texflush
};

GIF_PACKET gs_clear_zbuff_packet[] = {
    _GIF_TAG(6, 1, 0, 0, 0, 1),     GIF_AD,     // 00: clear zbuf - gif tag
    (3<<16)|(15<<0),                GS_TEST_1,  // 01: skip pixels with A=0, DepthTest on, pass all pixels
    GS_PRIM_SPRITE,                 GS_PRIM,    // 02:
    _GS_RGBAQ(0,0,0,128,0),         GS_RGBAQ,   // 03:
    _GS_XYZ2(0x8000,0x8000,0),      GS_XYZ2,    // 04:
    _GS_XYZ2(0x8000,0x8000,0),      GS_XYZ2,    // 05:
    (5<<16)|(15<<0),                GS_TEST_1   // 06: skip pixels with A=0, DepthTest on, skip Z<Zbuf
};

GIF_PACKET gs_init_zbuff_packet[] = {
    _GIF_TAG(1, 1, 0, 0, 0, 1),     GIF_AD,     // 00: gif tag
    0,                              GS_ZBUF_1,  // 01: zbp
};

GIF_PACKET gs_draw_sprite_packet[] = {
    _GIF_TAG(7, 1, 0, 0, 0, 1),                         GIF_AD,     // 00: gif tag
    0,                                                  GS_TEX0_1,  // 01: select tbp and cbp
    0,                                                  GS_RGBAQ,   // 02: select tbp and cbp
    GS_PRIM_SPRITE|GS_PRIM_TEXTURE|GS_PRIM_TEXT_UV,     GS_PRIM,    // 03: do a sprite
    _GS_UV(0,0),                                        GS_UV,      // 04: UV
    _GS_XYZ2(0x8000,0x8000,0),                          GS_XYZ2,    // 05: XY
    _GS_UV(0,0),                                        GS_UV,      // 06: UV
    _GS_XYZ2(0x8000,0x8000,0),                          GS_XYZ2     // 07: XY
};



//---------------------------------------------------------------------------
uint8 gs_detect_mode(void)
{
    dbgmsg(("gs_detect_mode: %s\n", (*((char *)0x1FC80000 - 0xAE) == 'E')?"PAL":"NTSC"));

    if(*((char *)0x1FC80000 - 0xAE) == 'E') return(PAL);
    return(NTSC);
}

//---------------------------------------------------------------------------
void gs_init(uint8 int_mode, uint8 ntsc_pal, uint8 field_mode, uint16 width, uint16 height, uint8 magh, uint8 psm)
{
    dbgmsg(("gs_init...\n"));


    // determine bits per pixel according selected pixel storage mode
    switch(psm) {
        case PSMCT32: gs_vmode.bpp = 32; break;
        case PSMCT24: gs_vmode.bpp = 24; break;
        case PSMCT16:
        case PSMCT16S: gs_vmode.bpp = 16; break;

        default:
            psm = PSMCT32;
            gs_vmode.bpp = 32;
    }


    // setup vars
    gs_vmode.int_mode = int_mode;
    gs_vmode.ntsc_pal = ntsc_pal;
    gs_vmode.field_mode = field_mode;
    gs_vmode.width = width;
    gs_vmode.height = height;
    gs_vmode.magh = magh;
    gs_vmode.psm = psm;
    gs_vmode.visible = 0;
    gs_vmode.zbp = 0;


    // calc values
    gs_vmode.fbw = width/64;

    gs_vmode.fbp_1 = 0;
    gs_vmode.dispfb2_1 = _GS_DISPFB(gs_vmode.fbp_1, gs_vmode.fbw, gs_vmode.psm, 0, 0);
    gs_vmode.frame_1 = _GS_FRAME(gs_vmode.fbp_1, gs_vmode.fbw, gs_vmode.psm, 0);

    gs_vmode.fbp_2 = gs_vmode.fbp_1 + (uint32)((((uint64)gs_vmode.width)*((uint64)gs_vmode.height)*((uint64)gs_vmode.bpp) + ((uint64)0xFFFF)) >> 16);
    gs_vmode.dispfb2_2 = _GS_DISPFB(gs_vmode.fbp_2, gs_vmode.fbw, gs_vmode.psm, 0, 0);
    gs_vmode.frame_2 = _GS_FRAME(gs_vmode.fbp_2, gs_vmode.fbw, gs_vmode.psm, 0);

    gs_vmode.tbp = (gs_vmode.fbp_2 + (uint32)((((uint64)gs_vmode.width)*((uint64)gs_vmode.height)*((uint64)gs_vmode.bpp) + ((uint64)0xFFFF)) >> 16)) << 5;


    // init gs and dependencies
    gs_reset_dma();
    gs_reset(gs_vmode.int_mode, gs_vmode.ntsc_pal, gs_vmode.field_mode);

    GS_SET_PMODE(
        0,      // ReadCircuit1 OFF
        1,      // ReadCircuit2 ON
        1,      // Use ALP register for Alpha Blending
        1,      // Alpha Value of ReadCircuit2 for output selection
        0,      // Blend Alpha with the output of ReadCircuit2
        0xFF    // Alpha Value = 1.0
    );

    GS_SET_REGISTER(GS_DISPFB2, gs_vmode.dispfb2_1);

    GS_SET_DISPLAY2(
        656, (int_mode==INTERLACED)?72:36,
        gs_vmode.magh-1, 0,
        (gs_vmode.width*gs_vmode.magh)-1, (gs_vmode.height)-1
    );

    GS_SET_BGCOLOR(0,0,0);


    // send gif packet and do final initialisation stuff
    gs_init_packet[9*2] = (uint64)gs_vmode.frame_1;                                             // 09: FRAME_1
    gs_init_packet[11*2] = _GS_SCISSOR(0,gs_vmode.width-1, 0,gs_vmode.height-1);                // 11: SCISSOR_1
    gs_init_packet[16*2] = _GS_XYZ2((0x800+gs_vmode.width)<<4,(0x800+gs_vmode.height)<<4,0);    // 16: XYZ2

    GIF_SEND(gs_init_packet, sizeof(gs_init_packet)/16)
    gs_wait_gif();
    gs_finish();


    dbgmsg(("int_mode = %u\nntsc_pal = %u\nfield_mode = %u\nwidth = %u\nheight = %u\nmagh = %u\npsm = %u\nbpp = %u\nfbw = %u\n" \
    "fbp_1 = %u\nfbp_2 = %u\nfbp_user = %u\nframe_1 = %u\ndispfb2_1 = %u\nframe_2 = %u\ndispfb2_2 = %u\n\n", \
    gs_vmode.int_mode, gs_vmode.ntsc_pal, gs_vmode.field_mode, gs_vmode.width, gs_vmode.height, gs_vmode.magh, gs_vmode.psm, gs_vmode.bpp, gs_vmode.fbw,
    gs_vmode.fbp_1, gs_vmode.fbp_2, gs_vmode.fbp_user, gs_vmode.frame_1, gs_vmode.dispfb2_1, gs_vmode.frame_2, gs_vmode.dispfb2_2));
}

//---------------------------------------------------------------------------
void gs_init_vmode(GS_ENUM_VMODE vm)
{
    dbgmsg(("gs_init_vmode(%d)\n", vm));

    gs_init(
        gs_vmode_t[vm].int_mode, gs_vmode_t[vm].ntsc_pal, gs_vmode_t[vm].field_mode,
        gs_vmode_t[vm].width, gs_vmode_t[vm].height,
        gs_vmode_t[vm].magh, gs_vmode_t[vm].psm);
}



//---------------------------------------------------------------------------
void gs_clear_screen(void)
{
    dbgmsg(("gs_clear_screen()\n"));
    GIF_SEND(&gs_init_packet[12*2], 5)
}

//---------------------------------------------------------------------------
void gs_swap_screen(void)
{
    if(gs_vmode.visible == 1) {
        dbgmsg(("gs_swap_screen() -> visible=2\n"));

        gs_vmode.visible = 2;
        gs_swap_screen_packet[1*2] = gs_vmode.frame_1;      // write to frame #1
        GIF_SEND(gs_swap_screen_packet, sizeof(gs_swap_screen_packet)/16)
        GS_SET_REGISTER(GS_DISPFB2, gs_vmode.dispfb2_2);    // frame #2 becomes visible
    } else {
        dbgmsg(("gs_swap_screen() -> visible=1\n"));

        gs_vmode.visible = 1;
        gs_swap_screen_packet[1*2] = gs_vmode.frame_2;      // write to frame #2
        GIF_SEND(gs_swap_screen_packet, sizeof(gs_swap_screen_packet)/16)
        GS_SET_REGISTER(GS_DISPFB2, gs_vmode.dispfb2_1);    // frame #1 becomes visible
    }
}

//---------------------------------------------------------------------------
// wait till all pending gs operations have been processed
//---------------------------------------------------------------------------
void gs_finish(void)
{
    dbgmsg(("gs_finish()\n"));

    *GS_CSR = *GS_CSR | (uint64)2;
    GIF_SEND(gs_finish_packet, sizeof(gs_finish_packet)/16)
    while((*GS_CSR & 2) == 0) __asm__("nop");
    *GS_CSR = *GS_CSR & (!2);
}

//---------------------------------------------------------------------------
void gs_set_viewport(uint16 width, uint16 height)
{
    GIF_DECLARE_PACKET(gif_buf, 3)
    int16 dx,dy;
    int16 x0,y0,x1,y1;


    dbgmsg(("gs_set_viewport(%u,%u)\n", width,height));

    dx = (gs_vmode.width - width)/2;
    dy = (gs_vmode.height - height)/2;

    x0 = dx; if(x0<0) x0 = 0;
    y0 = dy; if(y0<0) y0 = 0;
    x1 = dx+width;  if(x1>gs_vmode.width)  x1 = gs_vmode.width;
    y1 = dy+height; if(y1>gs_vmode.height) y1 = gs_vmode.height;

    GIF_BEGIN_PACKET(gif_buf);
    GIF_ADD_TAG(gif_buf, 2, 1, 0, 0, 0);
    GIF_ADD_DATA(gif_buf, GS_XYOFFSET_1, _GS_XYOFFSET((0x800-dx)<<4, (0x800-dy)<<4));
    GIF_ADD_DATA(gif_buf, GS_SCISSOR_1, _GS_SCISSOR(x0,x1-1, y0,y1-1));
    GIF_SEND_PACKET(gif_buf);
}

//---------------------------------------------------------------------------
void gs_texflush(void)
{
    dbgmsg(("gs_texflush()\n"));
    GIF_SEND(gs_texflush_packet, sizeof(gs_texflush_packet)/16)
}

//---------------------------------------------------------------------------
void gs_clear_zbuff(void)
{
    dbgmsg(("gs_clear_zbuff()\n"));
    GIF_SEND(gs_clear_zbuff_packet, sizeof(gs_clear_zbuff_packet)/16)
}

//---------------------------------------------------------------------------
void gs_init_zbuff(uint8 psmz, uint32 z_init)
{
    uint8 bpp;


    dbgmsg(("gs_init_zbuff(%u,%u)\n", psmz,z_init));

    switch(psmz) {
        case PSMZ32: bpp = 32; break;
        case PSMZ24: bpp = 24; break;
        case PSMZ16:
        case PSMZ16S: bpp = 16; break;
        default:
            bpp = 32; psmz = PSMZ32;
    }

    gs_vmode.zbp = gs_vmode.fbp_2 + (uint32)((((uint64)gs_vmode.width)*((uint64)gs_vmode.height)*((uint64)gs_vmode.bpp) + ((uint64)0xFFFF)) >> 16);
    gs_vmode.tbp = (gs_vmode.zbp + (uint32)((((uint64)gs_vmode.width)*((uint64)gs_vmode.height)*((uint64)bpp) + ((uint64)0xFFFF)) >> 16)) << 5;

    gs_init_zbuff_packet[1*2] = _GS_ZBUF(gs_vmode.zbp,psmz,0);
    GIF_SEND(gs_init_zbuff_packet, sizeof(gs_init_zbuff_packet)/16)

    gs_clear_zbuff_packet[4*2] = _GS_XYZ2(0x8000,0x8000,z_init);
    gs_clear_zbuff_packet[5*2] = _GS_XYZ2((0x800+gs_vmode.width)<<4,(0x800+gs_vmode.height)<<4,z_init);
    GIF_SEND(gs_clear_zbuff_packet, sizeof(gs_clear_zbuff_packet)/16)
}



//---------------------------------------------------------------------------
void gs_ee2vram(uint16 tbp, uint16 x, uint16 y, void *img, uint16 w, uint16 h, uint8 psm)
{
    uint8 bpp;
    uint32 size,todo;


    dbgmsg(("gs_ee2vram(%u,%u,%u,0x%X,%u,%u,%u)\n", tbp,x,y,(uint32)img,w,h,psm));

    switch(psm) {
        case PSMCT32: bpp = 32; break;
        case PSMCT24: bpp = 24; break;
        case PSMCT16:
        case PSMCT16S: bpp = 16; break;
        case PSMT8:
        case PSMT8H: bpp = 8; break;
        case PSMT4: bpp = 4; break;
        default: psm = PSMCT32; bpp = 32;
    }

    size = (((uint32)w)*((uint32)h)*((uint32)bpp) + 127)/128;

    gs_ee2vram_packet[1*2] = _GS_BITBLTBUF(0, 0, 0, tbp, gs_vmode.fbw, psm);
    gs_ee2vram_packet[2*2] = _GS_TRXPOS(0, 0, x, y, 0);
    gs_ee2vram_packet[3*2] = _GS_TRXREG(w, h);

    if(size > MAX_DMA_TRANSFER_SIZE) {
        todo = MAX_DMA_TRANSFER_SIZE;
        size -= MAX_DMA_TRANSFER_SIZE;
    } else {
        todo = size;
        size = 0;
    }
    gs_ee2vram_packet[5*2] = _GIF_TAG(todo, 1, 0, 0, 2, 1);

    k_FlushCache(0);
    gs_send_gif((void*)gs_ee2vram_packet, (uint32)sizeof(gs_ee2vram_packet)/16);
    gs_send_gif((void*)img, (uint32)todo);


    while(size > 0) {
        ((uint8*)img) += todo << 4;

        if(size > MAX_DMA_TRANSFER_SIZE) {
            todo = MAX_DMA_TRANSFER_SIZE;
            size -= MAX_DMA_TRANSFER_SIZE;
        } else {
            todo = size;
            size = 0;
        }
        gs_ee2vram_packet[5*2] = _GIF_TAG(todo, 1, 0, 0, 2, 1);

        k_FlushCache(0);
        gs_send_gif((void*)&gs_ee2vram_packet[5*2], 1);
        gs_send_gif((void*)img, (uint32)todo);
    }
}

//---------------------------------------------------------------------------
void gs_draw_sprite(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, uint16 tbp, uint8 psm, uint16 cbp, uint8 cpsm, uint8 tcc, uint8 tfx, uint32 rgba, uint32 flags)
{
    dbgmsg(("gs_draw_sprite(%i,%i,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u,%u)\n", x,y,z,w,h,tx,ty,tw,th,tbp,psm,cbp,cpsm,tcc,tfx,flags));

    gs_draw_sprite_packet[1*2] = _GS_TEX0(tbp, gs_vmode.fbw, psm, 10, 10, tcc, tfx, cbp, cpsm, 0, 0, 1);
    gs_draw_sprite_packet[2*2] = rgba;
    gs_draw_sprite_packet[3*2] = (GS_PRIM_SPRITE|GS_PRIM_TEXTURE|GS_PRIM_TEXT_UV)|flags;
    gs_draw_sprite_packet[4*2] = _GS_UV((tx)<<4, (ty)<<4);
    gs_draw_sprite_packet[5*2] = _GS_XYZ2((0x800+x)<<4, (0x800+y)<<4, z);
    gs_draw_sprite_packet[6*2] = _GS_UV((tx+tw)<<4, (ty+th)<<4);
    gs_draw_sprite_packet[7*2] = _GS_XYZ2((0x800+x+w)<<4, (0x800+y+h)<<4, z);

    GIF_SEND(gs_draw_sprite_packet, sizeof(gs_draw_sprite_packet)/16)
}



//---------------------------------------------------------------------------
// unload all previous loaded textures and close all handles
//---------------------------------------------------------------------------
void gs_tbuf_reset(void)
{
    dbgmsg(("gs_tbuf_reset()\n"));
    memset(tbuf_handle, 0, sizeof(tbuf_handle));
    tbuf_tbp = gs_vmode.tbp;
}

//---------------------------------------------------------------------------
// load texture to next free place
// return value: >=0 -> handle, -1 -> no handle, -2 -> no vram
//---------------------------------------------------------------------------
int16 gs_tbuf_load(void *img, uint16 w, uint16 h, uint8 psm, void *clut, uint8 cpsm, uint8 tcc)
{
    unsigned u;
    uint8 bpp, cbpp;
    uint16 cw=0,ch=0;


    dbgmsg(("gs_tbuf_load(...)\n"));

    for(u=0; 1; u++)
    {
        if(u >= MAX_TBUF_HANDLES) return(-1);
        if(tbuf_handle[u].valid == 0) break;
    }

    switch(psm) {
        case PSMCT32: bpp = 32; clut = NULL; break;
        case PSMCT24: bpp = 24; clut = NULL; break;
        case PSMCT16:
        case PSMCT16S: bpp = 16; clut = NULL; break;
        case PSMT8:
        case PSMT8H: bpp = 8; break;
        case PSMT4: bpp = 4; break;

        default:
            psm = PSMCT32;
            bpp = 32;
            clut = NULL;
    }
    tbuf_handle[u].tbp = tbuf_tbp;
    tbuf_tbp += ( ((uint32)gs_vmode.width)*((uint32)h)*((uint32)32) / ((uint32)bpp) + 63 ) >> 6;
    if(tbuf_tbp >= (4*1024*1024/2/64)) return(-2);

    tbuf_handle[u].w = w;
    tbuf_handle[u].h = h;
    tbuf_handle[u].psm = psm;
    tbuf_handle[u].tcc = tcc;

    if(clut != NULL) {
        if(bpp == 4) {
            cw = 8;
            ch = 2;
        } else {
            cw = 16;
            ch = 16;
        }

        switch(cpsm) {
            case PSMCT32: cbpp = 32; break;
            case PSMCT24: cbpp = 24; break;
            case PSMCT16:
            case PSMCT16S: cbpp = 16; break;

            default:
                cpsm = PSMCT32;
                cbpp = 32;
        }
        tbuf_handle[u].cbp = tbuf_tbp;
        tbuf_tbp += ( ((uint32)gs_vmode.width)*((uint32)ch)*((uint32)32) / ((uint32)cbpp) + 63 ) >> 6;
        if(tbuf_tbp >= (4*1024*1024/2/64)) return(-2);

        tbuf_handle[u].cpsm = cpsm;
    }

    // validate handle and copy data to vram
    tbuf_handle[u].valid = 1;
    gs_ee2vram(tbuf_handle[u].tbp, 0, 0, img, w, h, psm);
    if(clut != NULL) gs_ee2vram(tbuf_handle[u].cbp, 0, 0, clut, cw, ch, cpsm);
    gs_texflush();
    return(u);
}

//---------------------------------------------------------------------------
// draw whole texture using handle given by gs_tbuf_load
// scale_x/scale_y: 0x1000 = 1.0
//---------------------------------------------------------------------------
/*int16 gs_tbuf_draw_texture(int16 x, int16 y, uint32 z, uint16 scale_x, uint16 scale_y, int16 th, uint8 tfx, uint32 flags)
{
    dbgmsg(("gs_tbuf_draw_texture(...)\n"));

    if((th < 0) || (th >= MAX_TBUF_HANDLES) || (tbuf_handle[th].valid != 1)) return(-1);
    gs_draw_sprite(
        x, y, z,
        (tbuf_handle[th].w*scale_x) >> 12, (tbuf_handle[th].h*scale_y) >> 12,
        0, 0, tbuf_handle[th].w, tbuf_handle[th].h,
        tbuf_handle[th].tbp, tbuf_handle[th].psm,
        tbuf_handle[th].cbp, tbuf_handle[th].cpsm,
        tbuf_handle[th].tcc, tfx, flags,
    );
    return(0);
}*/

//---------------------------------------------------------------------------
// draw sprite from texture using handle given by gs_tbuf_load
//---------------------------------------------------------------------------
int16 gs_tbuf_draw_sprite(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, int16 thndl, uint32 flags)
{
    dbgmsg(("gs_tbuf_draw_sprite(...)\n"));

    if((thndl < 0) || (thndl >= MAX_TBUF_HANDLES) || (tbuf_handle[thndl].valid != 1)) return(-1);
    gs_draw_sprite(
        x, y, z, w, h,
        tx, ty, tw /* < tbuf_handle[thndl].w*/, th /* < tbuf_handle[thndl].h*/,
        tbuf_handle[thndl].tbp, tbuf_handle[thndl].psm,
        tbuf_handle[thndl].cbp, tbuf_handle[thndl].cpsm,
        tbuf_handle[thndl].tcc, 1, 0, flags
    );
    return(0);
}

int16 gs_tbuf_draw_sprite_mod(int16 x, int16 y, uint32 z, uint16 w, uint16 h, uint16 tx, uint16 ty, uint16 tw, uint16 th, int16 thndl, uint32 rgba, uint32 flags)
{
    dbgmsg(("gs_tbuf_draw_sprite_mod(...)\n"));

    if((thndl < 0) || (thndl >= MAX_TBUF_HANDLES) || (tbuf_handle[thndl].valid != 1)) return(-1);
    gs_draw_sprite(
        x, y, z, w, h,
        tx, ty, tw /* < tbuf_handle[thndl].w*/, th /* < tbuf_handle[thndl].h*/,
        tbuf_handle[thndl].tbp, tbuf_handle[thndl].psm,
        tbuf_handle[thndl].cbp, tbuf_handle[thndl].cpsm,
        tbuf_handle[thndl].tcc, 0, rgba, flags
    );
    return(0);
}
