//===============================================================================================
// SIMPLE.EXE
// Copyright (c), Firelight Multimedia, 1999,2000.
//
// This example demonstrates some fundamental FMOD usage, including device enumeration, output
// mode selection, user file I/O callbacks, loading and playing samples and a music file, and
// calling some runtime manipulation and information functions.
//===============================================================================================

#include <stdio.h>
#ifdef PLATFORM_WINDOWS
  #include <conio.h>
  #include <windows.h>
#else
  #include "../../api/inc/wincompat.h"
#endif
#include <math.h>

#include "../../api/inc/fmod.h"
#include "../../api/inc/fmod_errors.h"	// optional

unsigned int myopen(const char *name)
{
	return (unsigned int)fopen(name, "rb");
}

void myclose(unsigned int handle)
{
	fclose((FILE *)handle);
}

int myread(void *buffer, int size, unsigned int handle)
{
	return fread(buffer, 1, size, (FILE *)handle);
}

int myseek(unsigned int handle, int pos, signed char mode)
{
	return fseek((FILE *)handle, pos, mode);
}

int mytell(unsigned int handle)
{
	return ftell((FILE *)handle);
}

void ordercallback(FMUSIC_MODULE *mod, unsigned char param)
{
	printf("\nOrder Callback : param %d\n", param);
}

void instcallback(FMUSIC_MODULE *mod, unsigned char param)
{
	printf("\nInst Callback : param %d\n", param);
}


/*
[
	[DESCRIPTION]

	[PARAMETERS]
 
	[RETURN_VALUE]

	[REMARKS]

	[SEE_ALSO]
]
*/
int main()
{
	FMUSIC_MODULE *mod = 0;
	FSOUND_SAMPLE *samp1 = 0, *samp2 = 0, *samp3 = 0;
	int key;
	int driver, i;
 
	if (FSOUND_GetVersion() < FMOD_VERSION)
	{
		printf("Error : You are using the wrong DLL version!  You should be using FMOD %.02f\n", FMOD_VERSION);
		return 1;
	}

	// ==========================================================================================
	// SELECT OUTPUT METHOD
	// ==========================================================================================
	
	printf("---------------------------------------------------------\n");	
	printf("Output Type\n");	
	printf("---------------------------------------------------------\n");
#ifndef PLATFORM_LINUX    
	printf("1 - Direct Sound\n");
	printf("2 - Windows Multimedia Waveout\n");
	printf("3 - A3D\n");

#else  
    printf("1 - Open Sound System (OSS) (Linux, Solaris, freebsd)\n");
    printf("2 - Elightment Sound Daemon (ESD, Linux, more ...)\n");
    printf("3 - Alsa Sound System (Linux)\n");    
#endif
   
    printf("4 - NoSound\n");   
	printf("---------------------------------------------------------\n");	// print driver names
	printf("Press a corresponding number or ESC to quit\n");

   
	do
	{
		key = getch();
	} while (key != 27 && key < '1' && key > '4');

	switch (key)
	{
#ifndef PLATFORM_LINUX	   
		case '1' :	FSOUND_SetOutput(FSOUND_OUTPUT_DSOUND);
					break;
		case '2' :	FSOUND_SetOutput(FSOUND_OUTPUT_WINMM);
					break;
		case '3' :	FSOUND_SetOutput(FSOUND_OUTPUT_A3D);
					break;
#else	   
   	    case '1' :  FSOUND_SetOutput(FSOUND_OUTPUT_OSS);
         	        break;
	    case '2' :  FSOUND_SetOutput(FSOUND_OUTPUT_ESD);
	                break;
	    case '3' :  FSOUND_SetOutput(FSOUND_OUTPUT_ALSA);
	                break;
#endif	    
	    case '4' :  FSOUND_SetOutput(FSOUND_OUTPUT_NOSOUND);
	                break;	   
	   
	    default :	return 1;
	}
	
    // Set custom file callbacks?  This doesnt have to be done, its just here as an example.
	FSOUND_File_SetCallbacks(myopen, myclose, myread, myseek, mytell);

	// ==========================================================================================
	// SELECT DRIVER
	// ==========================================================================================

	// The following list are the drivers for the output method selected above.
	printf("---------------------------------------------------------\n");	
	switch (FSOUND_GetOutput())
	{
		case FSOUND_OUTPUT_NOSOUND:	printf("NoSound"); break;
	   
#ifndef PLATFORM_LINUX	   
		case FSOUND_OUTPUT_WINMM:	printf("Windows Multimedia Waveout"); break;
		case FSOUND_OUTPUT_DSOUND:	printf("Direct Sound"); break;
		case FSOUND_OUTPUT_A3D:		printf("A3D"); break;
#else
     	case FSOUND_OUTPUT_OSS:     printf("Open Sound System"); break;
	    case FSOUND_OUTPUT_ESD:     printf("Enlightment Sound Daemon"); break;
	    case FSOUND_OUTPUT_ALSA:    printf("Alsa"); break;	   
#endif
	   
	};

	printf(" Driver list\n");	
	printf("---------------------------------------------------------\n");	

	for (i=0; i < FSOUND_GetNumDrivers(); i++) 
	{
		printf("%d - %s\n", i+1, FSOUND_GetDriverName(i));	// print driver names
	}
	printf("---------------------------------------------------------\n");	// print driver names
	printf("Press a corresponding number or ESC to quit\n");

	do
	{
		key = getch();
		if (key == 27) 
			return 0;
		driver = key - '1';
	} while (driver < 0 || driver >= FSOUND_GetNumDrivers());

	FSOUND_SetDriver(driver);					// Select sound card (0 = default)
      		
	// ==========================================================================================
	// INITIALIZE
	// ==========================================================================================
	if (!FSOUND_Init(44100, 32, FSOUND_INIT_USEDEFAULTMIDISYNTH))
	{
		printf("Error!\n");
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	// ==========================================================================================
	// LOAD SONG
	// ==========================================================================================

	// The following list are the drivers for the output method selected above.
	printf("---------------------------------------------------------\n");	
	printf(" Select Music Type\n");	
	printf("---------------------------------------------------------\n");	
	printf("1 - MOD\n");
	printf("2 - MIDI (Using Default Software Synth)\n");
	printf("---------------------------------------------------------\n");	// print driver names
	printf("Press a corresponding number or ESC to quit\n");

	do
	{
		key = getch();
	} while (key != 27 && key < '1' && key > '2');

	switch (key)
	{
		case '1' :	mod = FMUSIC_LoadSong("../../media/invtro94.s3m");
					break;
		case '2' :	mod = FMUSIC_LoadSong("../../media/canyon.mid");
					break;
		default :	return 1;
	}
	if (!mod)
	{
		printf("Error!\n");
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	// ==========================================================================================
	// LOAD SAMPLES
	// ==========================================================================================

	// PCM,44,100 Hz, 8 Bit, Mono
	samp1 = FSOUND_Sample_Load(FSOUND_UNMANAGED, "../../media/drumloop.wav", FSOUND_NORMAL, 0);
	if (!samp1)
	{
		printf("Error!\n");
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}
	FSOUND_Sample_SetLoopMode(samp1, FSOUND_LOOP_OFF);

	// PCM,22,050 Hz, 16 Bit, Mono
	samp2 = FSOUND_Sample_Load(FSOUND_UNMANAGED, "../../media/jungle.wav", FSOUND_NORMAL, 0);
	if (!samp2)
	{
		printf("Error!\n");
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	// PCM,22,050 Hz, 8 Bit, Stereo
	samp3 = FSOUND_Sample_Load(FSOUND_UNMANAGED, "../../media/chimes.wav", FSOUND_NORMAL, 0);
	if (!samp3)
	{
		printf("Error!\n");
		printf("%s\n", FMOD_ErrorString(FSOUND_GetError()));
		return 1;
	}

	// ==========================================================================================
	// DISPLAY HELP
	// ==========================================================================================

	printf("FSOUND Output Method : ");
	switch (FSOUND_GetOutput())
	{
		case FSOUND_OUTPUT_NOSOUND:	printf("FSOUND_OUTPUT_NOSOUND\n"); break;
		case FSOUND_OUTPUT_WINMM:	printf("FSOUND_OUTPUT_WINMM\n"); break;
		case FSOUND_OUTPUT_DSOUND:	printf("FSOUND_OUTPUT_DSOUND\n"); break;
		case FSOUND_OUTPUT_A3D:		printf("FSOUND_OUTPUT_A3D\n"); break;
	};

	printf("FSOUND Mixer         : ");
	switch (FSOUND_GetMixer())
	{
		case FSOUND_MIXER_BLENDMODE:	printf("FSOUND_MIXER_BLENDMODE\n"); break;
		case FSOUND_MIXER_MMXP5:		printf("FSOUND_MIXER_MMXP5\n"); break;
		case FSOUND_MIXER_MMXP6:		printf("FSOUND_MIXER_MMXP6\n"); break;
		case FSOUND_MIXER_QUALITY_FPU:	printf("FSOUND_MIXER_QUALITY_FPU\n"); break;
		case FSOUND_MIXER_QUALITY_MMXP5:printf("FSOUND_MIXER_QUALITY_MMXP5\n"); break;
		case FSOUND_MIXER_QUALITY_MMXP6:printf("FSOUND_MIXER_QUALITY_MMXP6\n"); break;
	};
	printf("FSOUND Driver        : %s\n", FSOUND_GetDriverName(FSOUND_GetDriver()));

	printf("=========================================================================\n");
	printf("Press 1       Play 16bit sound at any time\n");
	printf("      2       Play 8bit sound at any time\n");
	printf("      3       Play 16bit STEREO sound at any time\n");
	printf("      <       Rewind mod back 1 order\n");
	printf("      >       FastForward mod forward 1 order\n");
	printf("      SPACE   Pause/unpause music at any time\n");
	printf("      ESC     Quit\n");
	printf("=========================================================================\n");
	printf("Playing \"%s\"...\n", FMUSIC_GetName(mod));

	{
		int count;
		for (count=0; count < FMUSIC_GetNumSamples(mod) && count < 20; count+=2)
		{
			char *a,*b;
			a = FSOUND_Sample_GetName(FMUSIC_GetSample(mod, count));
			b = FSOUND_Sample_GetName(FMUSIC_GetSample(mod, count+1));
			if (!a)
				a = "";
			if (!b)
				b = "";
			printf("%02d %-33s ", count, a);
			printf("%02d %-33s\n", count+1, b);
		}
	}

	// ==========================================================================================
	// START PLAYING!
	// ==========================================================================================
	FMUSIC_SetOrderCallback(mod, ordercallback, 1);
	FMUSIC_SetInstCallback(mod, instcallback, 5);

	FMUSIC_SetMasterVolume(mod, 128);

	FMUSIC_PlaySong(mod);
	do
	{
		key = 0;
		printf("order = %d, row = %d time = %d.%02d channels playing = %d cpu usage = %.02f%%\r", FMUSIC_GetOrder(mod), FMUSIC_GetRow(mod), FMUSIC_GetTime(mod) / 1000, (FMUSIC_GetTime(mod) % 1000) / 10, FSOUND_GetChannelsPlaying(), FSOUND_GetCPUUsage());

		if (kbhit())
		{
			key = getch();


			if (key == ' ') 
            {
				FMUSIC_SetPaused(mod, !FMUSIC_GetPaused(mod));
            }
			if (key == '1') 
            {
            	FSOUND_PlaySound(FSOUND_FREE, samp1);
            }
			if (key == '2') 
			{
				int channel;

				channel = FSOUND_PlaySoundEx(FSOUND_FREE, samp2, NULL, TRUE);
				FSOUND_SetFrequency(channel, 22050);
				FSOUND_SetVolume(channel, 255);
				FSOUND_SetPan(channel, 255);			// pan it all the way to the right
				FSOUND_SetPaused(channel, FALSE);
			}
			if (key == '3') 
			{
				int channel;

				channel = FSOUND_PlaySoundEx(FSOUND_FREE, samp3, NULL, TRUE);
				FSOUND_SetPan(channel, FSOUND_STEREOPAN);
				FSOUND_SetPaused(channel, FALSE);
			}
			if (key == '>') 
            {
				FMUSIC_SetOrder(mod, FMUSIC_GetOrder(mod)+1);
            }
			if (key == '<') 
            {
				FMUSIC_SetOrder(mod, FMUSIC_GetOrder(mod)-1);
            }

			// secret? keys
//			if (key == '4') FSOUND_DSP_SetActive(FSOUND_DSP_GetClearUnit(),			!FSOUND_DSP_GetActive(FSOUND_DSP_GetClearUnit()));
//			if (key == '5') FSOUND_DSP_SetActive(FSOUND_DSP_GetSFXUnit(),			!FSOUND_DSP_GetActive(FSOUND_DSP_GetSFXUnit()));
//			if (key == '6') FSOUND_DSP_SetActive(FSOUND_DSP_GetMusicUnit(),			!FSOUND_DSP_GetActive(FSOUND_DSP_GetMusicUnit()));
//			if (key == '7') FSOUND_DSP_SetActive(FSOUND_DSP_GetClipAndCopyUnit(),	!FSOUND_DSP_GetActive(FSOUND_DSP_GetClipAndCopyUnit()));
		}
		Sleep(10);

	} while (key != 27);

	FMUSIC_StopSong(mod);

	printf("\n");

	// ==========================================================================================
	// CLEANUP AND SHUTDOWN
	// ==========================================================================================

	FSOUND_Sample_Free(samp1);
	FSOUND_Sample_Free(samp2);
	FSOUND_Sample_Free(samp3);
	FMUSIC_FreeSong(mod);

	FSOUND_Close();
   
    return 0;
}


