//-------------------------------------------------------------------------
//
// File:		ImportInterfaceC.h
// Desc:		Import interface class.
// Author:		memon <memon@inside.org>
//
//-------------------------------------------------------------------------
//	Copyright (c) 2000-2002 Moppi Productions. All Rights Reserved.
//  This file is part of Moppi Demopaja SDK. For conditions of 
//  distribution and use, see the accompanying license.txt file.
//  http://moppi.inside.org/demopaja/
//-------------------------------------------------------------------------

#ifndef __DEMOPAJA_IMPORTINTERFACEC_H__
#define __DEMOPAJA_IMPORTINTERFACEC_H__

namespace Import {
	class ImportInterfaceC;
};

#include "FactoryC.h"
#include "ImportableI.h"
#include "ClassIdC.h"
#include "FileListC.h"
#include "FileHandleC.h"
#include "TimeContextC.h"
#include "DeviceContextC.h"
#include "CommonDialogI.h"
#include <string>
#include <vector>

namespace Import {

	//! Import interface class.
	/*!	Import interface class is a tool class for importers.
		It's purpose is to enable the importers to import other
		files during the load. For example, a 3D file importer can
		request the interface to import it's textures.

		This class is implemented by the system.
	*/
	class ImportInterfaceC
	{
	public:
		//! Default constructor (used internally).
		ImportInterfaceC( PajaSystem::DeviceContextC* pContext, PajaSystem::TimeContextC* pTimeContext, FileListC* pFileList, PluginClass::FactoryC* pFactory, Import::FileHandleC* pCaller );
		//! Default destructor (used internally).
		virtual ~ImportInterfaceC();

		//! Imports requested file.
		/*! Request a import for a specified file. File importer filters can be used
			range the number of possible importers which could match the file extension.

			For example if it is known that the file must be image, SUPERCLASS_IMAGE could be
			passed as the rSuperFilter argument.
			
			If a filter (class Id of super class ID) isn't used NULL_CLASSID or NULL_SUPERCLASS
			should be passed as argument.

			Example:
			\code
			// Request import for a image.
			FileHandleC*	pHandle = m_pImpInterface->request_import( szFileName, SUPERCLASS_IMAGE, NULL_CLASSID );
			\endcode
		*/
		virtual FileHandleC*	request_import( const char* szName, const PluginClass::SuperClassIdC& rSuperFilter,
												const PluginClass::ClassIdC& rClassFilter );

		//! Used internally.
		virtual void			initialize_importable( Import::ImportableI* pImp, PajaTypes::uint32 ui32Reason );

		//! Returns pointer to the file list.
		virtual FileListC*		get_filelist();

		//! Adds search directory.
		/*!	When the request import is called, the file is first tried to load from the
			given path. If it's not found, then the file is tried to load from each
			search directory.
		*/
		virtual void			add_search_directory( const char* szDir );

		//! Adds new common dialog to list.
		virtual void						add_common_dialog( PajaSystem::CommonDialogI* pDlg );
		//! Gets a common dialog based on the class ID.
		virtual PajaSystem::CommonDialogI*	get_common_dialog( const PluginClass::ClassIdC& rClassID );

	private:

		struct SearchDirS {
			std::string	m_sDir;
		};

		PluginClass::FactoryC*		m_pFactory;
		FileListC*					m_pFileList;
		Import::FileHandleC*		m_pCaller;
		PajaSystem::DeviceContextC*	m_pDevContext;
		PajaSystem::TimeContextC*	m_pTimeContext;
		std::vector<SearchDirS>		m_rSearchDirs;
		std::vector<PajaSystem::CommonDialogI*>	m_rCommonDialogs;
	};

};

#endif // __DEMOPAJA_IMPORTINTERFACEC_H__