#include "module.h"
#include "pspthreadman_kernel.h"

//#define DEBUG_MODULE_Start
#define DEBUG_MODULE_Load

typedef struct 
{
    CModule * Module;
} MODULE_THREAD_PARAMETERS;

int MODULE_Thread( SceSize args, void * argp );

CModule::CModule( void )
{
}

CModule::CModule( CString name, CString path ) :
    MODULE_GetType( NULL ),
    MODULE_IsLoaded( NULL ),
    MODULE_IsRunning( NULL ),
    MODULE_Update( NULL ),
    MODULE_IsUpdating( NULL ),
    MODULE_UpdateLibrary( NULL ),
    MODULE_Stop( NULL ),
    Type( MODULE_TYPE_Default ),
    Name( name ),
    Path( path ),
    ModuleID( - 1 ),
    ItIsRunning( false )
{
}

CModule::~CModule( void )
{
    if( ItIsRunning )
        Stop();
}

const CString & CModule::GetName( void ) const
{
    return Name;
}

SceUID CModule::GetModuleID( void ) const
{
    return ModuleID;
}

MODULE_TYPE CModule::GetType( void ) const
{
    return Type;
}

bool CModule::IsLoaded( void )
{
    if( MODULE_IsLoaded == NULL )
        return false;

    return (*MODULE_IsLoaded)();
}

bool CModule::IsRunning( void )
{
    if( MODULE_IsRunning == NULL )
        return false;

    return (*MODULE_IsRunning)();
}

bool CModule::Start( void )
{
    MODULE_THREAD_PARAMETERS thread_parameters;

#ifdef DEBUG_MODULE_Start
    pspDebugScreenInit();    
    pspDebugScreenPrintf( "CModule::Start\n" );
    pspDebugScreenPrintf( "    sceKernelCreateThread\n" );
#endif

    ThreadID = sceKernelCreateThread( ( Name + "_thread" ).GetBuffer(), MODULE_Thread, 8, DEFAULT_THREAD_SIZE, 0, NULL );

    if( ThreadID < 0 )
        return false;

    thread_parameters.Module = this;

#ifdef DEBUG_MODULE_Start
    pspDebugScreenPrintf( "    sceKernelStartThread\n" );
#endif

    if( sceKernelStartThread( ThreadID, sizeof( MODULE_THREAD_PARAMETERS ), &thread_parameters ) < 0 )
        return false;

#ifdef DEBUG_MODULE_Start
    pspDebugScreenPrintf( "    MODULE_HELPERS_GetModuleFromID\n" );
#endif

    while( MODULE_HELPERS_IsThreadRunning( ThreadID )
        && !MODULE_HELPERS_IsModuleRunning( ModuleID )
        )
    {
        sceKernelDelayThread( 1000 );
    }

    Module = MODULE_HELPERS_GetModuleFromID( ModuleID );

    if( Module == NULL )
        return false;

#ifdef DEBUG_MODULE_Start
    pspDebugScreenPrintf( "    ImportFunctions\n" );
#endif

    if( !ImportFunctions() )
        return false;

#ifdef DEBUG_MODULE_Start
    pspDebugScreenPrintf( "    IsLoaded\n" );
#endif

    while( !IsRunning() )
    {
        sceKernelDelayThread( 1000 );
    }

    if( !IsLoaded() )
    {
        Stop();

        return false;
    }
    
#ifdef DEBUG_MODULE_Start
    pspDebugScreenPrintf( "-> OK !\n" );
#endif

    ItIsRunning = true;

    return true;
}

bool CModule::Stop( void )
{
    bool result;

    (*MODULE_Stop)();

    while( IsRunning() && MODULE_HELPERS_IsModuleRunning( ModuleID ) )
    {
        sceKernelDelayThread( 1000 );
    }
    
    result = UnLoad();

    if( result )
        ItIsRunning = false;

    return result;
}

void CModule::Update( void )
{
    if( MODULE_Update != NULL
        && MODULE_IsUpdating != NULL
        )
    {
        (*MODULE_Update)();

        while( (*MODULE_IsUpdating)() )
        {
            sceKernelDelayThread( 10 );
        }
    }
}

void CModule::UpdateLibrary( void )
{
    if( MODULE_UpdateLibrary != NULL )
        (*MODULE_UpdateLibrary)();
}

u32 * CModule::GetFunctionPointer( u32 function_id )
{
    return MODULE_HELPERS_FindExportedNidAddr( Module, Name.GetBuffer(), function_id );
}

bool CModule::Load( void )
{
    SceKernelLMOption option;
    SceUID mpid;
    int status;
    CString file_path( Path + Name );

#ifdef DEBUG_MODULE_Load
    pspDebugScreenInit();    
    pspDebugScreenPrintf( "CModule::Load\n" );

    int free_mem = sceKernelMaxFreeMemSize() - 1*1024*1024;

    pspDebugScreenPrintf( "( free memory: %d bytes)\n", free_mem );

    pspDebugScreenPrintf( "    sceKernelLoadModule\n" );
#endif

    //mpid = 1;   // kernel
    mpid = 2;   // user

    memset( &option, 0, sizeof( option ) );
    option.size = sizeof( option );
    option.mpidtext = mpid;
    option.mpiddata = mpid;
    option.position = 0;
    option.access = 1;

    ModuleID = sceKernelLoadModule( file_path.GetBuffer(), 0, NULL );

    if( ModuleID < 0 )
    {
#ifdef DEBUG_MODULE_Load
        pspDebugScreenPrintf( "FAILED: %x", ModuleID );
#endif

        return false;
    }
    
    sceKernelDelayThread( 2000 );

#ifdef DEBUG_MODULE_Load
    pspDebugScreenPrintf( "    sceKernelStartModule\n" );
#endif

    int result;

    if( ( result = sceKernelStartModule( ModuleID, file_path.GetLength() + 1, file_path.GetBuffer(), &status, NULL ) ) < 0 )
    {
#ifdef DEBUG_MODULE_Load
        pspDebugScreenPrintf( "-> FAILED: %x\n", result );
#endif

        return false;
    }        

#ifdef DEBUG_MODULE_Load
    pspDebugScreenPrintf( "-> OK!\n" );
#endif

    return true;
}

bool CModule::UnLoad( void )
{
    int status;

    if( sceKernelStopModule( ModuleID, 0, NULL, &status, NULL ) < 0 )
        return false;

    if( sceKernelUnloadModule( ModuleID ) < 0 )
        return false;

    return true;    
}

bool CModule::ImportFunctions( void )
{
    MODULE_GetType = ( uint32 ( * )( void ) ) GetFunctionPointer( 0xE79AC995 );
    MODULE_IsLoaded = ( bool ( * )( void ) ) GetFunctionPointer( 0x7DB08111 );
    MODULE_IsRunning = ( bool ( * )( void ) ) GetFunctionPointer( 0xC6877897 );
    MODULE_Update = ( void ( * )( void ) ) GetFunctionPointer( 0x075B3966 );
    MODULE_IsUpdating = ( bool ( * )( void ) ) GetFunctionPointer( 0x44BB6F22 );
    MODULE_UpdateLibrary = ( void ( * )( void ) ) GetFunctionPointer( 0x285E3B1B );
    MODULE_Stop = ( void ( * )( void ) ) GetFunctionPointer( 0x9B70164C );

    if( MODULE_GetType == NULL
        || MODULE_IsLoaded == NULL
        || MODULE_IsRunning == NULL
        || MODULE_Stop == NULL
        )
    {
        return false;
    }
    
    Type = (*MODULE_GetType)();

    return true;
}

int MODULE_Thread( SceSize args, void * argp )
{
    CModule * module = ( ( MODULE_THREAD_PARAMETERS * ) argp )->Module;

    if( module->Load() )
    {
        MODULE_HELPERS_WaitForModuleStart( module->GetModuleID() );
        MODULE_HELPERS_WaitForModuleStop( module->GetModuleID() );
    }

    return sceKernelExitDeleteThread( 0 );
}
