/*
  FLOD 6.0 Player
  2016/08/15
  Christian Corti
  Neoart Costa Rica
*/

document.addEventListener("DOMContentLoaded", function() {
"use strict";

  const Time = "00:00";

  /* variables declaration */
  var dropzone = document.querySelector("html");
  var query    = window.matchMedia("(min-width: 1024px)");
  var chans    = document.getElementById("channels");
  var chan4    = chans.querySelector("article");
  var chan32   = chans.querySelector(".pc");
  var curset   = chan4;

  var browse = document.getElementById("browse");
  var play   = document.getElementById("play");
  var pause  = document.getElementById("pause");
  var stop   = document.getElementById("stop");
  var ntsc   = document.getElementById("ntsc");
  var loop   = document.getElementById("loop");
  var record = document.getElementById("record");
  var cia    = document.getElementById("cia");

  var volume    = document.getElementById("volume");
  var ovolume   = volume.nextElementSibling;
  var stereo    = document.getElementById("stereo");
  var ostereo   = stereo.nextElementSibling;
  var position  = document.getElementById("position");
  var oposition = position.nextElementSibling;
  var songs     = document.getElementById("songs");
  var osongs    = songs.nextElementSibling;
  var format    = document.getElementById("format");

  var tracker  = document.getElementById("tracker");
  var packer   = document.getElementById("packer");
  var title    = document.getElementById("title");
  var duration = document.getElementById("duration");
  var filesize = document.getElementById("filesize");

  var before   = window.performance.now();
  var dragging = false;
  var filename = "";
  var playing  = 0;

  var loader = window.neoart.FileLoader();
  var player = loader.player;
  var lflec  = window.neoart.Flectrum(document.getElementById("lmeter"), 40, 70);
  var rflec  = window.neoart.Flectrum(document.getElementById("rmeter"), 40, 70);

  /* initialize */
  chans.removeChild(chan32);
  chan32.removeAttribute("style");

  rflec.channels = Channels.right;
  player.analyser = [lflec, rflec];

  query.addListener(sizeChange);
  sizeChange(query);

  document.addEventListener("flodPlay",   flodPlay);
  document.addEventListener("flodPause",  flodPause);
  document.addEventListener("flodResume", flodPlay);
  document.addEventListener("flodStop",   flodStop);

  /* panels open/close event */
  document.getElementById("nav").addEventListener("click", function(e) {
    var ele = e.target;
    var pan = ele.getAttribute("data-panel");

    if (pan) {
      pan = document.getElementById(pan);

      if (pan.style.height) {
        set();
      } else {
        pan.style.height = pan.firstElementChild.offsetHeight +"px";
        window.setTimeout(set, 50);
      }
    }

    function set() {
      if (pan.className) {
        ele.removeAttribute("class");
        pan.removeAttribute("class");
      } else {
        pan.addEventListener("transitionend", transition);
        pan.className = "closed";
        ele.className = "down";
      }
    }

    function transition(e) {
      pan.classList.add("hidden");
      pan.removeEventListener("transitionend", transition);
    }
  });

  /* Flectrum add/removal on size change */
  function sizeChange(query) {
    if (query.matches) {
      lflec.channels = Channels.left;
      lflec.visual = Visual.basic;
      rflec.append();
    } else {
      lflec.channels = Channels.both;
      lflec.visual = Visual.split;
      rflec.remove();
    }

    var about = document.getElementById("about");
    about.style.height = about.firstElementChild.offsetHeight +"px";
  }

  /* Amiga/PC channels setup */
  chans.addEventListener("click", function(e) {
    var ele = e.target;
    var idx;

    if (ele.tagName.toLowerCase() == "button") {
      idx = ele.getAttribute("data-index");
      player.mute(idx);
      toggleChan(ele, player.isMuted(idx));
    }
  });

  function setupChans() {
    if (player.channels > 4) {
      if (!chans.contains(chan32)) {
        chans.removeChild(chan4);
        chans.appendChild(chan32);
        chans.removeAttribute("style");
      }

      curset = chan32;
    } else {
      if (!chans.contains(chan4)) {
        chans.removeChild(chan32);
        chans.appendChild(chan4);
        chans.removeAttribute("style");
      }

      curset = chan4;
    }

    restoreChans(curset);
  }

  function restoreChans(list) {
    var idx, node;
    list = list.querySelectorAll("button");

    for (node of list) {
      idx = node.getAttribute("data-index");

      if (idx >= player.channels) {
        node.disabled = "true";
        node.innerText = node.id;
      } else {
        node.removeAttribute("disabled");
        toggleChan(node, player.isMuted(idx));
      }
    }
  }

  function toggleChan(ele, value) {
    if (value) {
      ele.innerText = "off";
      ele.className = "down";
    } else {
      ele.innerText = ele.id;
      ele.removeAttribute("class");
    }
  }

  /* drag and drop events */
  dropzone.addEventListener("dragover", function(e) {
    e.stopPropagation();
    e.preventDefault();
  }, false);

  dropzone.addEventListener("drop", function(e) {
    var file = e.dataTransfer.files[e.dataTransfer.files.length - 1];

    if (!file) { return; }
    load(file);

    e.stopPropagation();
    e.preventDefault();
  }, false);

  /* button events */
  browse.addEventListener("change", function(e) {
    load(e.target.files[0]);
  });

  play.addEventListener("click", function(e) {
    player.play();
  });

  pause.addEventListener("click", function(e) {
    player.pause();
  });

  stop.addEventListener("click", function(e) {
    player.stop();
  });

  ntsc.addEventListener("click", function(e) {
    player.ntsc = !player.ntsc;
    toggle(this, player.ntsc);
    duration.innerText = convert(player.duration);
  });

  loop.addEventListener("click", function(e) {
    player.loop = !player.loop;
    toggle(this, player.loop);
  });

  record.addEventListener("click", function(e) {
    player.record = !player.record;
    toggle(this, player.record);
  });

  cia.addEventListener("click", function(e) {
    player.cia = !player.cia;
    toggle(this, player.cia);
    duration.innerText = convert(player.duration);
  });

  /* slider events */
  volume.addEventListener("input", function(e) {
    player.volume = volume.value;
    ovolume.innerText = Number(volume.value).toFixed(1);
  });

  stereo.addEventListener("input", function(e) {
    player.stereoSeparation = stereo.value;
    ostereo.innerText = Number(stereo.value).toFixed(1);
  });

  position.addEventListener("change", function(e) {
    oposition.innerText = convert(player.seek(position.value));
  });

  position.addEventListener("input", function(e) {
    oposition.innerText = convert(position.value);
  });

  position.addEventListener("mousedown", function(e) {
    dragging = true;
  });

  position.addEventListener("mouseup", function(e) {
    dragging = false;
  });

  songs.addEventListener("change", function(e) {
    player.currentSong = songs.value;
    position.max = player.duration;
    duration.innerText = convert(player.duration);

    restoreChans(curset);
  });

  songs.addEventListener("input", function(e) {
    osongs.innerText = songs.value +"/"+ player.totalSongs;
  });

  format.addEventListener("change", function(e) {
    player.version = format.value;
  });

  format.addEventListener("input", function(e) {
    tracker.innerText = player.format(format.value);
  });

  /* flod events */
  function flodPlay() {
    playing = window.requestAnimationFrame(step);

    play.disabled  = true;
    pause.disabled = false;
    stop.disabled  = false;
    pause.removeAttribute("class");
  }

  function flodPause() {
    window.cancelAnimationFrame(playing);

    play.disabled   = false;
    pause.className = "down";
  }

  function flodStop() {
    window.cancelAnimationFrame(playing);

    play.disabled  = false;
    pause.disabled = true;
    stop.disabled  = true;
    pause.removeAttribute("class");

    if (player.record) { save(); }

    position.value = 0;
    oposition.innerText = Time;
  }

  /* helper functions */
  function cleanup() {
    play.disabled  = true;
    pause.disabled = true;
    stop.disabled  = true;

    position.value = 0;
    position.max = 0;

    songs.value = 1;
    songs.max = 1;

    format.value = 0;
    format.max = 0;

    oposition.innerText = Time;
    osongs.innerText = "0/0";

    tracker.innerText  = "";
    packer.innerText   = "";
    title.innerText    = "";
    duration.innerText = Time;
    filesize.innerText = "0/0";
  }

  function convert(time) {
    var sec = Math.round(time / 1000);
    var min = Math.floor(sec / 60);

    sec -= (min * 60);

    var m = min.toString();
    var s = sec.toString();

    if (min < 10) { m = "0"+ m; }
    if (sec < 10) { s = "0"+ s; }

    return m +":"+ s;
  }

  function print() {
    osongs.innerText   = songs.min +"/"+ songs.max;

    tracker.innerText  = player.format();
    packer.innerText   = loader.packer;
    title.innerText    = player.title || filename;
    duration.innerText = convert(player.duration);
    filesize.innerText = loader.unpacked +"/"+ loader.packed;
  }

  function step(now) {
    var delta = now - before;

    if (delta >= 1000) {
      before = 0;

      if (!dragging) {
        position.value = player.position;
        oposition.innerText = convert(player.position);
      }
    }

    playing = window.requestAnimationFrame(step);
  }

  function toggle(ele, value) {
    if (value) {
      ele.className = "down";
      ele.innerText = ele.getAttribute("data-down");
    } else {
      ele.removeAttribute("class");
      ele.innerText = ele.getAttribute("data-up");
    }
  }

  function load(file) {
    var reader = new FileReader();

    reader.onload = function(e) {
      var pos = file.name.lastIndexOf(".");

      if (pos < 0) {
        filename = file.name;
      } else {
        filename = file.name.substr(0, pos);
      }

      if (loader.load(e.target.result)) {
        position.value = 0;
        position.max = player.duration;

        songs.value = 1;
        songs.max = player.totalSongs;

        format.min = player.minver;
        format.max = player.maxver;
        format.value = player.version;

        toggle(cia, player.cia);
        setupChans();

        print();
        player.play();
      } else {
        cleanup();
      }
    }

    reader.readAsArrayBuffer(file);
  }

  /* save wav file */
  function save() {
    var file = player.waveform;
    var a = document.getElementById("save");
    var b = new Blob([file], {type:"application/octet-stream"});

    a.download = filename +".wav";
    a.href = window.URL.createObjectURL(b);
    a.removeAttribute("class");
    a.removeAttribute("tabindex");
    a.addEventListener("click", restore);
  }

  function restore(e) {
    this.removeEventListener("click", restore);
    this.blur();
    this.className = "disabled";
    this.setAttribute("tabindex", -1);
  }
});