#!/usr/bin/env python

# xhm99: An HFE image manager that focuses on the TI 99
#
# Copyright (c) 2016 Ralph Benzinger <xdt99@endlos.net>
#
# This program is part of the TI 99 Cross-Development Tools (xdt99).
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>.


import sys

VERSION = "1.6.0"


### Utility functions

def ordw(word):
    """word ord"""
    return ord(word[0]) << 8 | ord(word[1])


def rordl(word):
    """reverse long ord"""
    return ord(word[3]) << 24 | ord(word[2]) << 16 | ord(word[1]) << 8 | ord(word[0])


def ords(string):
    """string ord"""
    return [ord(c) for c in string]


def chrw(word):
    """word chr"""
    return chr(word >> 8) + chr(word & 0xFF)


def rchrw(word):
    """reverse word chr"""
    return chr(word & 0xFF) + chr(word >> 8)


def chunk(data, size):
    """split into chunks of equal size"""
    return [data[i:i + size] for i in xrange(0, len(data), size)]


def flatten(listoflists):
    """flattens list of lists into list"""
    return [item for list_ in listoflists for item in list_]


def writedata(n, d, m="wb"):
    """write data to file or STDOUT"""
    if n == "-":
        sys.stdout.write(d)
    else:
        with open(n, m) as f:
            f.write(d)


def readdata(n, m="r"):
    """read data from file or STDIN"""
    if n == "-":
        return sys.stdin.read()
    else:
        with open(n, m) as f:
            return f.read()


def crc16(crc, stream):
    """compute CRC-16 code"""
    msb, lsb = crc >> 8, crc & 0xff
    for b in stream:
        x = b ^ msb
        x ^= (x >> 4)
        msb = (lsb ^ (x >> 3) ^ (x << 4)) & 0xff
        lsb = (x ^ (x << 5)) & 0xff
    return [msb, lsb]


class HFEError(Exception):
    pass


class SDFormat:

    sectors = 9
    trackLen = 17 + 9 * 334 + 113  # 3136

    leadin = [0xaa, 0xa8, 0xa8, 0x22] + [0xaa] * (4 * 16)  # fc ff
    lvleadin = 17
    leadout = [0xaa] * (4 * 77) + [0xaa, 0x50] + [0x55] * (2 + 4 * 35)  # cannot decode
    lvleadout = 113

    addressmark = [0xaa, 0x88, 0xa8, 0x2a]  # fe
    vaddressmark = [0xfe]
    lvaddressmark = 1
    datamark = [0xaa, 0x88, 0x28, 0xaa]  # fb
    vdatamark = [0xfb]
    lvdatamark = 1

    pregap = [0x22] * (4 * 6)  # 00
    lvpregap = 6
    gap1 = [0xaa] * (4 * 11) + [0x22] * (4 * 6)  # ff 00
    lvgap1 = 17
    gap2 = [0xaa] * (4 * 45)  # ff
    lvgap2 = 45

    sectorInterleave = (0, 7, 5, 3, 1, 8, 6, 4, 2,  # offset 0
                        6, 4, 2, 0, 7, 5, 3, 1, 8,  # offset 6
                        3, 1, 8, 6, 4, 2, 0, 7, 5)  # offset 3
    sectorInterleaveWTF = (4, 2, 0, 7, 5, 3, 1, 8, 6,
                           1, 8, 6, 4, 2, 0, 7, 5, 3,
                           7, 5, 3, 1, 8, 6, 4, 2, 0)

    # twisted encoded bytes with clock bits
    fmCodes = [
        [0x22, 0x22, 0x22, 0x22], [0x22, 0x22, 0x22, 0xa2], [0x22, 0x22, 0x22, 0x2a],
        [0x22, 0x22, 0x22, 0xaa], [0x22, 0x22, 0xa2, 0x22], [0x22, 0x22, 0xa2, 0xa2],
        [0x22, 0x22, 0xa2, 0x2a], [0x22, 0x22, 0xa2, 0xaa], [0x22, 0x22, 0x2a, 0x22],
        [0x22, 0x22, 0x2a, 0xa2], [0x22, 0x22, 0x2a, 0x2a], [0x22, 0x22, 0x2a, 0xaa],
        [0x22, 0x22, 0xaa, 0x22], [0x22, 0x22, 0xaa, 0xa2], [0x22, 0x22, 0xaa, 0x2a],
        [0x22, 0x22, 0xaa, 0xaa], [0x22, 0xa2, 0x22, 0x22], [0x22, 0xa2, 0x22, 0xa2],
        [0x22, 0xa2, 0x22, 0x2a], [0x22, 0xa2, 0x22, 0xaa], [0x22, 0xa2, 0xa2, 0x22],
        [0x22, 0xa2, 0xa2, 0xa2], [0x22, 0xa2, 0xa2, 0x2a], [0x22, 0xa2, 0xa2, 0xaa],
        [0x22, 0xa2, 0x2a, 0x22], [0x22, 0xa2, 0x2a, 0xa2], [0x22, 0xa2, 0x2a, 0x2a],
        [0x22, 0xa2, 0x2a, 0xaa], [0x22, 0xa2, 0xaa, 0x22], [0x22, 0xa2, 0xaa, 0xa2],
        [0x22, 0xa2, 0xaa, 0x2a], [0x22, 0xa2, 0xaa, 0xaa], [0x22, 0x2a, 0x22, 0x22],
        [0x22, 0x2a, 0x22, 0xa2], [0x22, 0x2a, 0x22, 0x2a], [0x22, 0x2a, 0x22, 0xaa],
        [0x22, 0x2a, 0xa2, 0x22], [0x22, 0x2a, 0xa2, 0xa2], [0x22, 0x2a, 0xa2, 0x2a],
        [0x22, 0x2a, 0xa2, 0xaa], [0x22, 0x2a, 0x2a, 0x22], [0x22, 0x2a, 0x2a, 0xa2],
        [0x22, 0x2a, 0x2a, 0x2a], [0x22, 0x2a, 0x2a, 0xaa], [0x22, 0x2a, 0xaa, 0x22],
        [0x22, 0x2a, 0xaa, 0xa2], [0x22, 0x2a, 0xaa, 0x2a], [0x22, 0x2a, 0xaa, 0xaa],
        [0x22, 0xaa, 0x22, 0x22], [0x22, 0xaa, 0x22, 0xa2], [0x22, 0xaa, 0x22, 0x2a],
        [0x22, 0xaa, 0x22, 0xaa], [0x22, 0xaa, 0xa2, 0x22], [0x22, 0xaa, 0xa2, 0xa2],
        [0x22, 0xaa, 0xa2, 0x2a], [0x22, 0xaa, 0xa2, 0xaa], [0x22, 0xaa, 0x2a, 0x22],
        [0x22, 0xaa, 0x2a, 0xa2], [0x22, 0xaa, 0x2a, 0x2a], [0x22, 0xaa, 0x2a, 0xaa],
        [0x22, 0xaa, 0xaa, 0x22], [0x22, 0xaa, 0xaa, 0xa2], [0x22, 0xaa, 0xaa, 0x2a],
        [0x22, 0xaa, 0xaa, 0xaa], [0xa2, 0x22, 0x22, 0x22], [0xa2, 0x22, 0x22, 0xa2],
        [0xa2, 0x22, 0x22, 0x2a], [0xa2, 0x22, 0x22, 0xaa], [0xa2, 0x22, 0xa2, 0x22],
        [0xa2, 0x22, 0xa2, 0xa2], [0xa2, 0x22, 0xa2, 0x2a], [0xa2, 0x22, 0xa2, 0xaa],
        [0xa2, 0x22, 0x2a, 0x22], [0xa2, 0x22, 0x2a, 0xa2], [0xa2, 0x22, 0x2a, 0x2a],
        [0xa2, 0x22, 0x2a, 0xaa], [0xa2, 0x22, 0xaa, 0x22], [0xa2, 0x22, 0xaa, 0xa2],
        [0xa2, 0x22, 0xaa, 0x2a], [0xa2, 0x22, 0xaa, 0xaa], [0xa2, 0xa2, 0x22, 0x22],
        [0xa2, 0xa2, 0x22, 0xa2], [0xa2, 0xa2, 0x22, 0x2a], [0xa2, 0xa2, 0x22, 0xaa],
        [0xa2, 0xa2, 0xa2, 0x22], [0xa2, 0xa2, 0xa2, 0xa2], [0xa2, 0xa2, 0xa2, 0x2a],
        [0xa2, 0xa2, 0xa2, 0xaa], [0xa2, 0xa2, 0x2a, 0x22], [0xa2, 0xa2, 0x2a, 0xa2],
        [0xa2, 0xa2, 0x2a, 0x2a], [0xa2, 0xa2, 0x2a, 0xaa], [0xa2, 0xa2, 0xaa, 0x22],
        [0xa2, 0xa2, 0xaa, 0xa2], [0xa2, 0xa2, 0xaa, 0x2a], [0xa2, 0xa2, 0xaa, 0xaa],
        [0xa2, 0x2a, 0x22, 0x22], [0xa2, 0x2a, 0x22, 0xa2], [0xa2, 0x2a, 0x22, 0x2a],
        [0xa2, 0x2a, 0x22, 0xaa], [0xa2, 0x2a, 0xa2, 0x22], [0xa2, 0x2a, 0xa2, 0xa2],
        [0xa2, 0x2a, 0xa2, 0x2a], [0xa2, 0x2a, 0xa2, 0xaa], [0xa2, 0x2a, 0x2a, 0x22],
        [0xa2, 0x2a, 0x2a, 0xa2], [0xa2, 0x2a, 0x2a, 0x2a], [0xa2, 0x2a, 0x2a, 0xaa],
        [0xa2, 0x2a, 0xaa, 0x22], [0xa2, 0x2a, 0xaa, 0xa2], [0xa2, 0x2a, 0xaa, 0x2a],
        [0xa2, 0x2a, 0xaa, 0xaa], [0xa2, 0xaa, 0x22, 0x22], [0xa2, 0xaa, 0x22, 0xa2],
        [0xa2, 0xaa, 0x22, 0x2a], [0xa2, 0xaa, 0x22, 0xaa], [0xa2, 0xaa, 0xa2, 0x22],
        [0xa2, 0xaa, 0xa2, 0xa2], [0xa2, 0xaa, 0xa2, 0x2a], [0xa2, 0xaa, 0xa2, 0xaa],
        [0xa2, 0xaa, 0x2a, 0x22], [0xa2, 0xaa, 0x2a, 0xa2], [0xa2, 0xaa, 0x2a, 0x2a],
        [0xa2, 0xaa, 0x2a, 0xaa], [0xa2, 0xaa, 0xaa, 0x22], [0xa2, 0xaa, 0xaa, 0xa2],
        [0xa2, 0xaa, 0xaa, 0x2a], [0xa2, 0xaa, 0xaa, 0xaa], [0x2a, 0x22, 0x22, 0x22],
        [0x2a, 0x22, 0x22, 0xa2], [0x2a, 0x22, 0x22, 0x2a], [0x2a, 0x22, 0x22, 0xaa],
        [0x2a, 0x22, 0xa2, 0x22], [0x2a, 0x22, 0xa2, 0xa2], [0x2a, 0x22, 0xa2, 0x2a],
        [0x2a, 0x22, 0xa2, 0xaa], [0x2a, 0x22, 0x2a, 0x22], [0x2a, 0x22, 0x2a, 0xa2],
        [0x2a, 0x22, 0x2a, 0x2a], [0x2a, 0x22, 0x2a, 0xaa], [0x2a, 0x22, 0xaa, 0x22],
        [0x2a, 0x22, 0xaa, 0xa2], [0x2a, 0x22, 0xaa, 0x2a], [0x2a, 0x22, 0xaa, 0xaa],
        [0x2a, 0xa2, 0x22, 0x22], [0x2a, 0xa2, 0x22, 0xa2], [0x2a, 0xa2, 0x22, 0x2a],
        [0x2a, 0xa2, 0x22, 0xaa], [0x2a, 0xa2, 0xa2, 0x22], [0x2a, 0xa2, 0xa2, 0xa2],
        [0x2a, 0xa2, 0xa2, 0x2a], [0x2a, 0xa2, 0xa2, 0xaa], [0x2a, 0xa2, 0x2a, 0x22],
        [0x2a, 0xa2, 0x2a, 0xa2], [0x2a, 0xa2, 0x2a, 0x2a], [0x2a, 0xa2, 0x2a, 0xaa],
        [0x2a, 0xa2, 0xaa, 0x22], [0x2a, 0xa2, 0xaa, 0xa2], [0x2a, 0xa2, 0xaa, 0x2a],
        [0x2a, 0xa2, 0xaa, 0xaa], [0x2a, 0x2a, 0x22, 0x22], [0x2a, 0x2a, 0x22, 0xa2],
        [0x2a, 0x2a, 0x22, 0x2a], [0x2a, 0x2a, 0x22, 0xaa], [0x2a, 0x2a, 0xa2, 0x22],
        [0x2a, 0x2a, 0xa2, 0xa2], [0x2a, 0x2a, 0xa2, 0x2a], [0x2a, 0x2a, 0xa2, 0xaa],
        [0x2a, 0x2a, 0x2a, 0x22], [0x2a, 0x2a, 0x2a, 0xa2], [0x2a, 0x2a, 0x2a, 0x2a],
        [0x2a, 0x2a, 0x2a, 0xaa], [0x2a, 0x2a, 0xaa, 0x22], [0x2a, 0x2a, 0xaa, 0xa2],
        [0x2a, 0x2a, 0xaa, 0x2a], [0x2a, 0x2a, 0xaa, 0xaa], [0x2a, 0xaa, 0x22, 0x22],
        [0x2a, 0xaa, 0x22, 0xa2], [0x2a, 0xaa, 0x22, 0x2a], [0x2a, 0xaa, 0x22, 0xaa],
        [0x2a, 0xaa, 0xa2, 0x22], [0x2a, 0xaa, 0xa2, 0xa2], [0x2a, 0xaa, 0xa2, 0x2a],
        [0x2a, 0xaa, 0xa2, 0xaa], [0x2a, 0xaa, 0x2a, 0x22], [0x2a, 0xaa, 0x2a, 0xa2],
        [0x2a, 0xaa, 0x2a, 0x2a], [0x2a, 0xaa, 0x2a, 0xaa], [0x2a, 0xaa, 0xaa, 0x22],
        [0x2a, 0xaa, 0xaa, 0xa2], [0x2a, 0xaa, 0xaa, 0x2a], [0x2a, 0xaa, 0xaa, 0xaa],
        [0xaa, 0x22, 0x22, 0x22], [0xaa, 0x22, 0x22, 0xa2], [0xaa, 0x22, 0x22, 0x2a],
        [0xaa, 0x22, 0x22, 0xaa], [0xaa, 0x22, 0xa2, 0x22], [0xaa, 0x22, 0xa2, 0xa2],
        [0xaa, 0x22, 0xa2, 0x2a], [0xaa, 0x22, 0xa2, 0xaa], [0xaa, 0x22, 0x2a, 0x22],
        [0xaa, 0x22, 0x2a, 0xa2], [0xaa, 0x22, 0x2a, 0x2a], [0xaa, 0x22, 0x2a, 0xaa],
        [0xaa, 0x22, 0xaa, 0x22], [0xaa, 0x22, 0xaa, 0xa2], [0xaa, 0x22, 0xaa, 0x2a],
        [0xaa, 0x22, 0xaa, 0xaa], [0xaa, 0xa2, 0x22, 0x22], [0xaa, 0xa2, 0x22, 0xa2],
        [0xaa, 0xa2, 0x22, 0x2a], [0xaa, 0xa2, 0x22, 0xaa], [0xaa, 0xa2, 0xa2, 0x22],
        [0xaa, 0xa2, 0xa2, 0xa2], [0xaa, 0xa2, 0xa2, 0x2a], [0xaa, 0xa2, 0xa2, 0xaa],
        [0xaa, 0xa2, 0x2a, 0x22], [0xaa, 0xa2, 0x2a, 0xa2], [0xaa, 0xa2, 0x2a, 0x2a],
        [0xaa, 0xa2, 0x2a, 0xaa], [0xaa, 0xa2, 0xaa, 0x22], [0xaa, 0xa2, 0xaa, 0xa2],
        [0xaa, 0xa2, 0xaa, 0x2a], [0xaa, 0xa2, 0xaa, 0xaa], [0xaa, 0x2a, 0x22, 0x22],
        [0xaa, 0x2a, 0x22, 0xa2], [0xaa, 0x2a, 0x22, 0x2a], [0xaa, 0x2a, 0x22, 0xaa],
        [0xaa, 0x2a, 0xa2, 0x22], [0xaa, 0x2a, 0xa2, 0xa2], [0xaa, 0x2a, 0xa2, 0x2a],
        [0xaa, 0x2a, 0xa2, 0xaa], [0xaa, 0x2a, 0x2a, 0x22], [0xaa, 0x2a, 0x2a, 0xa2],
        [0xaa, 0x2a, 0x2a, 0x2a], [0xaa, 0x2a, 0x2a, 0xaa], [0xaa, 0x2a, 0xaa, 0x22],
        [0xaa, 0x2a, 0xaa, 0xa2], [0xaa, 0x2a, 0xaa, 0x2a], [0xaa, 0x2a, 0xaa, 0xaa],
        [0xaa, 0xaa, 0x22, 0x22], [0xaa, 0xaa, 0x22, 0xa2], [0xaa, 0xaa, 0x22, 0x2a],
        [0xaa, 0xaa, 0x22, 0xaa], [0xaa, 0xaa, 0xa2, 0x22], [0xaa, 0xaa, 0xa2, 0xa2],
        [0xaa, 0xaa, 0xa2, 0x2a], [0xaa, 0xaa, 0xa2, 0xaa], [0xaa, 0xaa, 0x2a, 0x22],
        [0xaa, 0xaa, 0x2a, 0xa2], [0xaa, 0xaa, 0x2a, 0x2a], [0xaa, 0xaa, 0x2a, 0xaa],
        [0xaa, 0xaa, 0xaa, 0x22], [0xaa, 0xaa, 0xaa, 0xa2], [0xaa, 0xaa, 0xaa, 0x2a],
        [0xaa, 0xaa, 0xaa, 0xaa]       
    ]

    @classmethod
    def decode(cls, stream):
        """decode FM bit stream into bytes"""
        bs = []
        for i in xrange(0, len(stream), 4):
            l = rordl(stream[i:i + 4])
            # bit format:  ABCDEFGH <->  H...G... F...E... D...C... B...A...
            b = (
                (0x01 if l & 0x80000000 else 0) |
                (0x02 if l & 0x08000000 else 0) |
                (0x04 if l & 0x00800000 else 0) |
                (0x08 if l & 0x00080000 else 0) |
                (0x10 if l & 0x00008000 else 0) |
                (0x20 if l & 0x00000800 else 0) |
                (0x40 if l & 0x00000080 else 0) |
                (0x80 if l & 0x00000008 else 0)
                )
            bs.append(b)
        return bs

    @classmethod
    def encode(cls, track):
        """encode SD track into FM bit stream"""
        stream = []
        for b in track:
            stream.extend(cls.fmCodes[b])
        return stream

    @classmethod
    def interleave(cls, side, track, sector, wtf80t):
        return (
            cls.sectorInterleave[(track * cls.sectors + sector) % 27]
            if not wtf80t or side == 0 else
            cls.sectorInterleaveWTF[(track * cls.sectors + sector) % 27]  # WTF?
            if track < 37 else
            cls.sectorInterleave[((track - 37) * cls.sectors + sector) % 27]  # off-series
        )
            
    @classmethod
    def fixclocks(cls, stream):
        """fix clock bits in stream (inline)"""
        pass  # clocks are correct


class DDFormat:

    sectors = 18
    trackLen = 32 + 18 * 342 + 84  # 6272

    leadin = [0x49, 0x2a] * 32  # 4e
    lvleadin = 32
    leadout = [0x49, 0x2a] * 84  # 4e
    lvleadout = 84

    addressmark = [0x22, 0x91, 0x22, 0x91, 0x22, 0x91, 0xaa, 0x2a]
    vaddressmark = [0xa1, 0xa1, 0xa1, 0xfe]
    lvaddressmark = 4
    datamark = [0x22, 0x91, 0x22, 0x91, 0x22, 0x91, 0xaa, 0xa2]
    vdatamark = [0xa1, 0xa1, 0xa1, 0xfb]
    lvdatamark = 4
    
    pregap = [0x55] * (2 * 12)  # 00
    lvpregap = 12
    gap1 = [0x49, 0x2a] * 22 + [0x55] * (2 * 12)  # 4e/00
    lvgap1 = 34
    gap2 = [0x49, 0x2a] * 24  # 4e
    lvgap2 = 24

    sectorInterleave = (0, 11, 4, 15, 8, 1, 12, 5, 16,
                        9, 2, 13, 6, 17, 10, 3, 14, 7)
    
    # computation rule: w = int(bs[7::-1], 2) * 256 + int(bs[:7:-1], 2)
    mfmCodes = [
        [0x55, 0x55], [0x55, 0x95], [0x55, 0x25], [0x55, 0xa5],
        [0x55, 0x49], [0x55, 0x89], [0x55, 0x29], [0x55, 0xa9],
        [0x55, 0x52], [0x55, 0x92], [0x55, 0x22], [0x55, 0xa2],
        [0x55, 0x4a], [0x55, 0x8a], [0x55, 0x2a], [0x55, 0xaa],
        [0x95, 0x54], [0x95, 0x94], [0x95, 0x24], [0x95, 0xa4],
        [0x95, 0x48], [0x95, 0x88], [0x95, 0x28], [0x95, 0xa8],
        [0x95, 0x52], [0x95, 0x92], [0x95, 0x22], [0x95, 0xa2],
        [0x95, 0x4a], [0x95, 0x8a], [0x95, 0x2a], [0x95, 0xaa],
        [0x25, 0x55], [0x25, 0x95], [0x25, 0x25], [0x25, 0xa5],
        [0x25, 0x49], [0x25, 0x89], [0x25, 0x29], [0x25, 0xa9],
        [0x25, 0x52], [0x25, 0x92], [0x25, 0x22], [0x25, 0xa2],
        [0x25, 0x4a], [0x25, 0x8a], [0x25, 0x2a], [0x25, 0xaa],
        [0xa5, 0x54], [0xa5, 0x94], [0xa5, 0x24], [0xa5, 0xa4],
        [0xa5, 0x48], [0xa5, 0x88], [0xa5, 0x28], [0xa5, 0xa8],
        [0xa5, 0x52], [0xa5, 0x92], [0xa5, 0x22], [0xa5, 0xa2],
        [0xa5, 0x4a], [0xa5, 0x8a], [0xa5, 0x2a], [0xa5, 0xaa],
        [0x49, 0x55], [0x49, 0x95], [0x49, 0x25], [0x49, 0xa5],
        [0x49, 0x49], [0x49, 0x89], [0x49, 0x29], [0x49, 0xa9],
        [0x49, 0x52], [0x49, 0x92], [0x49, 0x22], [0x49, 0xa2],
        [0x49, 0x4a], [0x49, 0x8a], [0x49, 0x2a], [0x49, 0xaa],
        [0x89, 0x54], [0x89, 0x94], [0x89, 0x24], [0x89, 0xa4],
        [0x89, 0x48], [0x89, 0x88], [0x89, 0x28], [0x89, 0xa8],
        [0x89, 0x52], [0x89, 0x92], [0x89, 0x22], [0x89, 0xa2],
        [0x89, 0x4a], [0x89, 0x8a], [0x89, 0x2a], [0x89, 0xaa],
        [0x29, 0x55], [0x29, 0x95], [0x29, 0x25], [0x29, 0xa5],
        [0x29, 0x49], [0x29, 0x89], [0x29, 0x29], [0x29, 0xa9],
        [0x29, 0x52], [0x29, 0x92], [0x29, 0x22], [0x29, 0xa2],
        [0x29, 0x4a], [0x29, 0x8a], [0x29, 0x2a], [0x29, 0xaa],
        [0xa9, 0x54], [0xa9, 0x94], [0xa9, 0x24], [0xa9, 0xa4],
        [0xa9, 0x48], [0xa9, 0x88], [0xa9, 0x28], [0xa9, 0xa8],
        [0xa9, 0x52], [0xa9, 0x92], [0xa9, 0x22], [0xa9, 0xa2],
        [0xa9, 0x4a], [0xa9, 0x8a], [0xa9, 0x2a], [0xa9, 0xaa],
        [0x52, 0x55], [0x52, 0x95], [0x52, 0x25], [0x52, 0xa5],
        [0x52, 0x49], [0x52, 0x89], [0x52, 0x29], [0x52, 0xa9],
        [0x52, 0x52], [0x52, 0x92], [0x52, 0x22], [0x52, 0xa2],
        [0x52, 0x4a], [0x52, 0x8a], [0x52, 0x2a], [0x52, 0xaa],
        [0x92, 0x54], [0x92, 0x94], [0x92, 0x24], [0x92, 0xa4],
        [0x92, 0x48], [0x92, 0x88], [0x92, 0x28], [0x92, 0xa8],
        [0x92, 0x52], [0x92, 0x92], [0x92, 0x22], [0x92, 0xa2],
        [0x92, 0x4a], [0x92, 0x8a], [0x92, 0x2a], [0x92, 0xaa],
        [0x22, 0x55], [0x22, 0x95], [0x22, 0x25], [0x22, 0xa5],
        [0x22, 0x49], [0x22, 0x89], [0x22, 0x29], [0x22, 0xa9],
        [0x22, 0x52], [0x22, 0x92], [0x22, 0x22], [0x22, 0xa2],
        [0x22, 0x4a], [0x22, 0x8a], [0x22, 0x2a], [0x22, 0xaa],
        [0xa2, 0x54], [0xa2, 0x94], [0xa2, 0x24], [0xa2, 0xa4],
        [0xa2, 0x48], [0xa2, 0x88], [0xa2, 0x28], [0xa2, 0xa8],
        [0xa2, 0x52], [0xa2, 0x92], [0xa2, 0x22], [0xa2, 0xa2],
        [0xa2, 0x4a], [0xa2, 0x8a], [0xa2, 0x2a], [0xa2, 0xaa],
        [0x4a, 0x55], [0x4a, 0x95], [0x4a, 0x25], [0x4a, 0xa5],
        [0x4a, 0x49], [0x4a, 0x89], [0x4a, 0x29], [0x4a, 0xa9],
        [0x4a, 0x52], [0x4a, 0x92], [0x4a, 0x22], [0x4a, 0xa2],
        [0x4a, 0x4a], [0x4a, 0x8a], [0x4a, 0x2a], [0x4a, 0xaa],
        [0x8a, 0x54], [0x8a, 0x94], [0x8a, 0x24], [0x8a, 0xa4],
        [0x8a, 0x48], [0x8a, 0x88], [0x8a, 0x28], [0x8a, 0xa8],
        [0x8a, 0x52], [0x8a, 0x92], [0x8a, 0x22], [0x8a, 0xa2],
        [0x8a, 0x4a], [0x8a, 0x8a], [0x8a, 0x2a], [0x8a, 0xaa],
        [0x2a, 0x55], [0x2a, 0x95], [0x2a, 0x25], [0x2a, 0xa5],
        [0x2a, 0x49], [0x2a, 0x89], [0x2a, 0x29], [0x2a, 0xa9],
        [0x2a, 0x52], [0x2a, 0x92], [0x2a, 0x22], [0x2a, 0xa2],
        [0x2a, 0x4a], [0x2a, 0x8a], [0x2a, 0x2a], [0x2a, 0xaa],
        [0xaa, 0x54], [0xaa, 0x94], [0xaa, 0x24], [0xaa, 0xa4],
        [0xaa, 0x48], [0xaa, 0x88], [0xaa, 0x28], [0xaa, 0xa8],
        [0xaa, 0x52], [0xaa, 0x92], [0xaa, 0x22], [0xaa, 0xa2],
        [0xaa, 0x4a], [0xaa, 0x8a], [0xaa, 0x2a], [0xaa, 0xaa],
    ]

    @classmethod
    def interleave(cls, side, track, sector, wtf80t):
        return sector * 11 % cls.sectors

    @classmethod
    def decode(cls, stream):
        """decode MFM bit stream into bytes"""
        lookup = {word[0] << 8 | word[1]: i
                  for i, word in enumerate(cls.mfmCodes)}
        bs = []
        for j, i in enumerate(xrange(0, len(stream), 2)):
            w = ordw(stream[i:i + 2])
            if w == 0x2291:  # address mark
                b = 0xa1
            else:
                try:
                    b = lookup[w]
                except KeyError:
                    # NOTE: no such collisions in lookup table!
                    b = lookup[w | 0x0100]  # extra clock bit
            bs.append(b)
        return bs

    @classmethod
    def encode(cls, track):
        """encode SD track into MFM bit stream"""
        stream = []
        for b in track:
            w = cls.mfmCodes[b]
            stream.extend(w)
        return stream

    @classmethod
    def fixclocks(cls, stream):
        """fix clock bits in stream (inline)"""
        for i in xrange(1, len(stream), 2):
            if stream[i] & 0x80:
                stream[i + 1] &= 0xfe
    

class HFEDisk:

    hfeInterfaceMode = 7
    hfeBitRate = 250

    hfeSDEncoding = 2
    hfeDDEncoding = 0
    validEncodings = [0, 2]
    
    def __init__(self, image):
        """create HFE disk from image"""
        self.header = image[0:512]
        self.lut = image[512:1024]
        self.trackdata = image[1024:]

        self.tracks, self.sides, self.encoding, self.ifmode = \
            self.getHFEParams(self.header)
        if (self.encoding != HFEDisk.hfeSDEncoding and
            self.encoding != HFEDisk.hfeDDEncoding):
            raise HFEError("Invalid encoding for TI 99")
        self.dd = self.encoding == HFEDisk.hfeDDEncoding
        if self.ifmode != 7:
            raise HFEError("Invalid mode for TI 99")

    @classmethod
    def getHFEParams(cls, image):
        """checks if image is HFE image"""
        if image[:8] != "HXCPICFE":
            raise HFEError("Not a HFE image")
        return ord(image[9]), ord(image[10]), ord(image[11]), ord(image[16])
        
    def toDiskImage(self):
        """extract sector data from HFE image"""
        fmt = DDFormat if self.dd else SDFormat
        tracks = self.getTracks()
        return self.extractSectors(fmt, tracks)

    def getTracks(self):
        """return list of decoded track data"""
        size = DDFormat.trackLen if self.dd else SDFormat.trackLen
        decode = DDFormat.decode if self.dd else SDFormat.decode
        chunks = chunk(self.trackdata, 256)
        side0 = "".join(chunks[0::2])
        side1 = "".join(chunks[1::2])
        tracks0 = chunk(decode(side0), size)
        tracks1 = chunk(decode(side1), size) if self.sides == 2 else []
        tracks1.reverse()
        return tracks0 + tracks1

    def extractSectors(self, fmt, tracks):
        """extract sector data from list of track data"""
        sectors = []
        if len(tracks) != self.sides * self.tracks:
            raise HFEError("Invalid track count")
        assert len(tracks[0]) == fmt.trackLen
        for j in range(self.sides * self.tracks):
            track = tracks[j]
            h0, h1 = 0, fmt.lvleadin
            #leadin = track[h0:h1]
            trackSectors = {}
            for i in range(fmt.sectors):
                # pregap
                h0, h1 = h1, h1 + fmt.lvpregap
                #pregap = track[h0:h1]
                # ID address mark
                h0, h1 = h1, h1 + fmt.lvaddressmark
                addressMark = track[h0:h1]
                assert addressMark == fmt.vaddressmark
                # sector ID
                h0, h1 = h1, h1 + 6
                #trackId = track[h0]
                #sideId = track[h0 + 1]
                sectorId = track[h0 + 2]
                assert sectorId not in trackSectors
                #sizeId = track[h0 + 3]
                #crc1 = track[h0 + 4:h0 + 6]
                # gap1
                h0, h1 = h1, h1 + fmt.lvgap1
                #gap1 = track[h0:h1]
                # data mark
                h0, h1 = h1, h1 + fmt.lvdatamark
                dataMark = track[h0:h1]
                assert dataMark == fmt.vdatamark
                # sector data
                h0, h1 = h1, h1 + 258
                trackSectors[sectorId] = track[h0:h0 + 256]
                #crc2 = track[h0 + 256:h0 + 258]
                # gap2
                h0, h1 = h1, h1 + fmt.lvgap2
                #gap2 = track[h0:h1]
            # leadout
            h0, h1 = h1, h1 + fmt.lvleadout
            assert h1 == len(track)
            sectors.extend(flatten([trackSectors[k] for k in sorted(trackSectors)]))
        return "".join([chr(b) for b in sectors])

    @classmethod
    def create(cls, image):
        """create HFE disk from disk image"""
        hfe = cls.createImage(image)
        return HFEDisk(hfe)

    @classmethod
    def createImage(cls, image):
        """create HFE image from disk image"""
        tracks = ord(image[0x11])
        sides = ord(image[0x12])
        dd = image[0x13] == "\x02"
        protected = image[0x10] == "P"

        header = cls.createHeader(tracks, sides, dd, protected)
        lut = cls.createLUT(tracks, dd)

        fmt = DDFormat if dd else SDFormat
        side0, side1 = cls.createTracks(tracks, sides, fmt, image)
        dummy = "\x00" * 256 if not side1 else None
        sandwich = "".join([side0[i:i+256] + (dummy or side1[i:i+256])
                            for i in xrange(0, len(side0), 256)])
        assert len(header) == len(lut) == 512
        return header + lut + sandwich

    @classmethod
    def createHeader(cls, tracks, sides, dd, protected):
        """create HFE disk header"""
        info = "HXCPICFE%c%c%c%c%s%s%c\x01%s%c" % (
            0,  # revision
            tracks, sides,
            HFEDisk.hfeDDEncoding if dd else HFEDisk.hfeSDEncoding,
            rchrw(HFEDisk.hfeBitRate),  # bit rate
            rchrw(0),  # RPM (not used)
            HFEDisk.hfeInterfaceMode,
            rchrw(1),  # LUT offset / 512
            0 if protected else 0xff)
        return info + "\xff" * (512 - len(info))

    @classmethod
    def createLUT(cls, tracks, dd):
        """create HFE LUT"""
        lut = "".join([rchrw(0x31 * i + 2) +
                       ("\xc0\x61" if dd else "\xb0\x61")
                       for i in xrange(tracks)])
        return lut + "\xff" * (512 - 4 * tracks)

    @classmethod
    def createTracks(cls, tracks, sides, fmt, sectors):
        """create HFE tracks"""
        trackdata = ([], [])
        for s in xrange(sides):
            for j in xrange(tracks):
                sectordata = []
                for i in xrange(fmt.sectors):
                    trackId = tracks - 1 - j if s else j  # 0 .. 39 39 .. 0
                    sectorId = fmt.interleave(s, j, i, tracks == 80)
                    offset = ((s * tracks + j) * fmt.sectors + sectorId) * 256
                    sector = ords(sectors[offset:offset + 256])
                    addr = [trackId, s, sectorId, 0x01]
                    crc1 = crc16(0xffff, fmt.vaddressmark + addr)
                    crc2 = crc16(0xffff, fmt.vdatamark + sector)
                    sectordata.extend(
                        fmt.pregap +
                        fmt.addressmark +
                        fmt.encode(addr + crc1) +
                        fmt.gap1 +
                        fmt.datamark +
                        fmt.encode(sector + crc2) +
                        fmt.gap2)
                fmt.fixclocks(sectordata)
                track = fmt.leadin + sectordata + fmt.leadout
                trackdata[s].append(track)
        trackdata[1].reverse()
        return ("".join([chr(b) for t in trackdata[0] for b in t]),
                "".join([chr(b) for t in trackdata[1] for b in t]))


# main

def main(argv):
    import os.path
    import argparse
    import glob

    class GlobStore(argparse.Action):
        """argparse globbing for Windows platforms"""

        def __call__(self, parser, namespace, values, option_string=None):
            if os.name == "nt":
                names = [glob.glob(fn) if "*" in fn or "?" in fn else [fn]
                         for fn in values]
                values = [f for n in names for f in n]
            setattr(namespace, self.dest, values)

    args = argparse.ArgumentParser(
        version=VERSION,
        description="xhm99: HFE image and file manipulation tool")
    cmd = args.add_mutually_exclusive_group(required=True)
    # xdm99 delegattion
    cmd.add_argument(
        "filename", type=str, nargs="?",
        help="HFE image filename")
    # conversion
    cmd.add_argument(
        "-T", "--to-hfe", action=GlobStore, dest="tohfe", nargs="+",
        metavar="<file>", help="convert disk images to HFE images")
    cmd.add_argument(
        "-F", "--from-hfe", "--to-dsk", action=GlobStore, dest="fromhfe", nargs="+",
        metavar="<file>", help="convert HFE images to disk images")
    cmd.add_argument(
        "-I", "--hfe-info", action=GlobStore, dest="hfeinfo", nargs="+",
        metavar="<file>", help="show basic information about HFE images")
    cmd.add_argument(
        "--dump", action=GlobStore, dest="dump", nargs="+",
        metavar="<file>", help="dump raw decoded HFE data")
    # general options
    args.add_argument(
        "-o", "--output", dest="output", metavar="<file>",
        help="set output filename")
    opts, other = args.parse_known_args(argv)

    result = []
    try:
        # delegate to xdm99
        if opts.filename:
            import xdm99 as xdm
            try:
                image = readdata(opts.filename, "rb")
                disk = HFEDisk(image).toDiskImage()
            except IOError:
                disk = "\x00"
            if opts.output:
                other += ["-o", opts.filename]
            barename = os.path.splitext(os.path.basename(opts.filename))[0]
            result = xdm.main([barename[:10].upper()] + other, disk)
            if isinstance(result, tuple):  # disk modified?
                dsk = result[0]
                hfe = HFEDisk.createImage(dsk)
                result = (hfe, opts.filename, "wb")
        # HFE/DSK conversion
        for name in opts.fromhfe or []:
            image = readdata(name, "rb")
            dsk = HFEDisk(image).toDiskImage()
            barename = os.path.splitext(os.path.basename(name))[0]
            result.append((dsk, barename + ".dsk", "wb"))
        for name in opts.tohfe or []:
            image = readdata(name, "rb")
            hfe = HFEDisk.createImage(image)
            barename = os.path.splitext(os.path.basename(name))[0]
            result.append((hfe, barename + ".hfe", "wb"))
        for name in opts.dump or []:
            image = readdata(name, "rb")
            hfe = HFEDisk(image)
            tracks = hfe.getTracks()
            data = "".join([chr(b) for b in flatten(tracks)])
            barename = os.path.splitext(os.path.basename(name))[0]
            result.append((data, barename + ".dump", "wb"))
        # image analysis
        for name in opts.hfeinfo or []:
            image = readdata(name, "rb")
            t, s, enc, ifm = HFEDisk.getHFEParams(image)
            sys.stdout.write("Tracks: %d\nSides: %d\n" % (t, s))
            sys.stdout.write("Encoding: %d\nInterface mode: %d\n" % (enc, ifm))
            if (enc not in HFEDisk.validEncodings or
                ifm != HFEDisk.hfeInterfaceMode):
                sys.stdout.write("Not a suitable HFE image for the TI 99\n")
                return 1
    except (IOError, HFEError) as e:
        sys.exit("Error: " + str(e))

    # write result
    if isinstance(result, tuple):
        result = [result]
    for data, name, mode in result:
        outname = opts.output or name
        try:
            writedata(outname, data, mode)
        except IOError as e:
            sys.exit("Error: " + str(e))

    # return status
    return 0


if __name__ == "__main__":
    status = main(sys.argv[1:])
    sys.exit(status)
