/***************************************************************************
 **                                                                        **
 **  Tiny raytracing engine - Aron, 1998.05.12                             **
 **                                                                        **
 ****************************************************************************/

// - sajat fejlecfile-ok ----------------------------------------------------
#include "compile.h"
#include "defs.h"
#include "tinymath.h"
#include "tracer.h"
#include "traceobj.h"

// - implementacio ----------------------------------------------------------
dword objects;
OBJECT object[10];

// scene#1:
PLANE plane[2]= { { { 0.0, 1.0, 0.0}, {  0.0,  0.0,  0.0}, 0.0},
                  { { 1.0, 0.0, 0.0}, { 80.0,  0.0,  0.0}, 0.0} };
SPHERE sphere[7]= { {{   0,   0, 100},  30},
                    {{   0,   0,-100},  30},
                    {{ 100,   0,   0},  30},
                    {{-100,   0,   0},  30},
                    {{   0,   0,   0},  30} };
CYLINDER cylinder[7]= { {{ 80.0, 110.0, 0.0}, { 120.0, 50.0, 0.0}, 15.0},
                        {{ 120.0, 60.0, 0.0}, { 80.0, 0.0, 0.0}, 15.0},
                        {{ 40.0, 0.0, 0.0}, { 40.0, 110.0, 0.0}, 15.0},
                        {{ -40.0, 0.0, 0.0}, { -40.0, 110.0, 0.0}, 15.0},
                        {{ 40.0, 110.0, 0.0}, { -40.0, 0.0, 0.0}, 15.0},
                        {{ -120.0, 95.0, 0.0}, { -120.0, 0.0, 0.0}, 15.0},
                        {{ -160.0, 95.0, 0.0}, { -80.0, 95.0, 0.0}, 15.0}
                        };

void SetupSceneCNT(byte *tbuffer)
{ dword i;

  // CNT felirat osszerakva csovekbol egy texturazott plane-en:
  objects= 8;

  // alap:
  object[0].type= TPLANE;
  object[0].data= &plane[0];
  object[0].color.textureptr= tbuffer + 5*TEXTURE_SIZE;
  object[0].surfacetype= DISTANCE_SHADED | TEXTURED | SHADOWCASTING;

  // 'C':
  object[1].type= TCYLINDER;
  object[1].data= &cylinder[0];
  object[1].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[1].surfacetype= PHONG_SHADED | TEXTURED;
  object[2].type= TCYLINDER;
  object[2].data= &cylinder[1];
  object[2].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[2].surfacetype= PHONG_SHADED | TEXTURED;

  // 'N':
  object[3].type= TCYLINDER;
  object[3].data= &cylinder[2];
  object[3].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[3].surfacetype= PHONG_SHADED | TEXTURED;
  object[4].type= TCYLINDER;
  object[4].data= &cylinder[3];
  object[4].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[4].surfacetype= PHONG_SHADED | TEXTURED;
  object[5].type= TCYLINDER;
  object[5].data= &cylinder[4];
  object[5].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[5].surfacetype= PHONG_SHADED | TEXTURED;

  // 'T':
  object[6].type= TCYLINDER;
  object[6].data= &cylinder[5];
  object[6].color.textureptr= tbuffer + 7*TEXTURE_SIZE;
  object[6].surfacetype= PHONG_SHADED | TEXTURED;
  object[7].type= TCYLINDER;
  object[7].data= &cylinder[6];
  object[7].color.textureptr= tbuffer + 7*TEXTURE_SIZE;
  object[7].surfacetype= PHONG_SHADED | TEXTURED;

  // object-ek inicializalasa:
  for(i=0; i<objects; i++) InitObject[object[i].type](&object[i]);
}

void SetupSceneBlobs(byte *tbuffer)
{ dword i;

  // csak egy probascene:
  objects= 6;

  // egy sik:
  object[0].type= TPLANE;
  object[0].data= &plane[0];
  object[0].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[0].surfacetype= DISTANCE_SHADED | TEXTURED | SHADOWCASTING;

  // sullyesztett blob-ok:
  for(i=0; i<5; i++)
  { object[1 + i].type= TSPHERE;
    object[1 + i].data= &sphere[i];
    object[1 + i].color.textureptr= tbuffer + i*TEXTURE_SIZE;
    object[1 + i].surfacetype= PHONG_SHADED | TEXTURED;
  }

  // object-ek inicializalasa:
  for(i=0; i<objects; i++) InitObject[object[i].type](&object[i]);
}

void MoveSceneBlobs(float t)
{ // blob-ok mozgatasa Lissajous-gorbek menten:
  sphere[0].center.X= 100.0*fsin(4.0*t);
  sphere[0].center.Z= 100.0*fcos(1.5*t);
  sphere[0].center.Y= 0.0;
  sphere[0].radius= 40.0;
  sphere[1].center.X= 100.0*fsin(3.0*t);
  sphere[1].center.Z= 100.0*fsin(4.5*t);
  sphere[1].center.Y= 0.0;
  sphere[1].radius= 40.0;
  sphere[2].center.X= 100.0*fcos(2.0*t);
  sphere[2].center.Z= 100.0*fcos(2.5*t);
  sphere[2].center.Y= 0.0;
  sphere[2].radius= 40.0;
  sphere[3].center.X= 100.0*fcos(3.5*t);
  sphere[3].center.Z= 100.0*fsin(2.5*t);
  sphere[3].center.Y= 0.0;
  sphere[3].radius= 40.0;
  sphere[4].center.X= 100.0*fsin(1.5*t);
  sphere[4].center.Z= 100.0*fcos(3.5*t);
  sphere[4].center.Y= 0.0;
  sphere[4].radius= 40.0;
}

void SetupScenePipeObject(byte *tbuffer)
{ dword i;

  // vmi scene:
  objects= 10;

  // alapsik:
  object[0].type= TPLANE;
  object[0].data= &plane[0];
  object[0].color.textureptr= tbuffer + 5*TEXTURE_SIZE;
  object[0].surfacetype= DISTANCE_SHADED | TEXTURED | SHADOWCASTING;

  // harom henger:
  object[1].type= TCYLINDER;
  cylinder[0].A.X=  100.0; cylinder[0].A.Y= 0.0; cylinder[0].A.Z= 0.0;
  cylinder[0].B.X= -100.0; cylinder[0].B.Y= 0.0; cylinder[0].B.Z= 0.0;
  cylinder[0].radius= 15.0;
  object[1].data= &cylinder[0];
  object[1].color.textureptr= tbuffer;
  object[1].surfacetype= PHONG_SHADED | TEXTURED;

  object[2].type= TCYLINDER;
  cylinder[1].A.X= 0.0; cylinder[1].A.Y=  100.0; cylinder[1].A.Z= 0.0;
  cylinder[1].B.X= 0.0; cylinder[1].B.Y= -100.0; cylinder[1].B.Z= 0.0;
  cylinder[1].radius= 15.0;
  object[2].data= &cylinder[1];
  object[2].color.textureptr= tbuffer;
  object[2].surfacetype= PHONG_SHADED | TEXTURED;

  object[3].type= TCYLINDER;
  cylinder[2].A.X= 0.0; cylinder[2].A.Y= 0.0; cylinder[2].A.Z=  100.0;
  cylinder[2].B.X= 0.0; cylinder[2].B.Y= 0.0; cylinder[2].B.Z= -100.0;
  cylinder[2].radius= 15.0;
  object[3].data= &cylinder[2];
  object[3].color.textureptr= tbuffer;
  object[3].surfacetype= PHONG_SHADED | TEXTURED;

  // golyok a hengerek vegen:
  object[4].type= TSPHERE;
  sphere[0].center.X= 100.0; sphere[0].center.Y= 0.0; sphere[0].center.Z= 0.0;
  sphere[0].radius= 30.0;
  object[4].data= &sphere[0];
  object[4].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[4].surfacetype= PHONG_SHADED | TEXTURED;
  object[5].type= TSPHERE;
  sphere[1].center.X= -100.0; sphere[1].center.Y= 0.0; sphere[1].center.Z= 0.0;
  sphere[1].radius= 30.0;
  object[5].data= &sphere[1];
  object[5].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[5].surfacetype= PHONG_SHADED | TEXTURED;

  object[6].type= TSPHERE;
  sphere[2].center.X= 0.0; sphere[2].center.Y= 100.0; sphere[2].center.Z= 0.0;
  sphere[2].radius= 30.0;
  object[6].data= &sphere[2];
  object[6].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[6].surfacetype= PHONG_SHADED | TEXTURED;

  object[7].type= TSPHERE;
  sphere[4].center.X= 0.0; sphere[4].center.Y= 0.0; sphere[4].center.Z= 100.0;
  sphere[4].radius= 30.0;
  object[7].data= &sphere[4];
  object[7].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[7].surfacetype= PHONG_SHADED | TEXTURED;
  object[8].type= TSPHERE;
  sphere[5].center.X= 0.0; sphere[5].center.Y= 0.0; sphere[5].center.Z= -100.0;
  sphere[5].radius= 30.0;
  object[8].data= &sphere[5];
  object[8].color.textureptr= tbuffer + 4*TEXTURE_SIZE;
  object[8].surfacetype= PHONG_SHADED | TEXTURED;

  // kozepso nagy golyo:
  object[9].type= TSPHERE;
  sphere[6].center.X= 0.0; sphere[6].center.Y= 0.0; sphere[6].center.Z= 0.0;
  sphere[6].radius= 40.0;
  object[9].data= &sphere[6];
  object[9].color.textureptr= tbuffer + 7*TEXTURE_SIZE;
  object[9].surfacetype= PHONG_SHADED | TEXTURED;

  // object-ek inicializalasa:
  for(i=0; i<objects; i++) InitObject[object[i].type](&object[i]);
}

void SetupSceneColumns(byte *tbuffer)
{ dword i;

  // vmi scene:
  objects= 6;

  // egy sik:
  object[0].type= TPLANE;
  object[0].data= &plane[0];
  object[0].color.textureptr= tbuffer + TEXTURE_SIZE;
  object[0].surfacetype= DISTANCE_SHADED | TEXTURED | SHADOWCASTING;

  // golyo kozepen contract-os texturaval:
  object[1].type= TSPHERE;
  sphere[0].center.X= 0.0; sphere[0].center.Y=30; sphere[0].center.Z= 0.0;
  sphere[0].radius= 30.0;
  object[1].data= &sphere[0];
  object[1].color.textureptr= tbuffer + 7*TEXTURE_SIZE;
  object[1].surfacetype= PHONG_SHADED | TEXTURED;

  // oszlopok:
  object[2].type= TCYLINDER;
  cylinder[0].A.X= 100.0; cylinder[0].A.Y= 0.0; cylinder[0].A.Z= 100.0;
  cylinder[0].B.X= 80.0; cylinder[0].B.Y= 150.0; cylinder[0].B.Z= 80.0;
  cylinder[0].radius= 30.0;
  object[2].data= &cylinder[0];
  object[2].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[2].surfacetype= PHONG_SHADED | TEXTURED;

  object[3].type= TCYLINDER;
  cylinder[1].A.X= -100.0; cylinder[1].A.Y= 0.0; cylinder[1].A.Z= 100.0;
  cylinder[1].B.X= -80.0; cylinder[1].B.Y= 150.0; cylinder[1].B.Z= 80.0;
  cylinder[1].radius= 30.0;
  object[3].data= &cylinder[1];
  object[3].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[3].surfacetype= PHONG_SHADED | TEXTURED;

  object[4].type= TCYLINDER;
  cylinder[2].A.X= 100.0; cylinder[2].A.Y= 0.0; cylinder[2].A.Z= -100.0;
  cylinder[2].B.X= 80.0; cylinder[2].B.Y= 150.0; cylinder[2].B.Z= -80.0;
  cylinder[2].radius= 30.0;
  object[4].data= &cylinder[2];
  object[4].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[4].surfacetype= PHONG_SHADED | TEXTURED;

  object[5].type= TCYLINDER;
  cylinder[3].A.X= -100.0; cylinder[3].A.Y= 0.0; cylinder[3].A.Z= -100.0;
  cylinder[3].B.X= -80.0; cylinder[3].B.Y= 150.0; cylinder[3].B.Z= -80.0;
  cylinder[3].radius= 30.0;
  object[5].data= &cylinder[3];
  object[5].color.textureptr= tbuffer + 2*TEXTURE_SIZE;
  object[5].surfacetype= PHONG_SHADED | TEXTURED;

  // object-ek inicializalasa:
  for(i=0; i<objects; i++) InitObject[object[i].type](&object[i]);
}

void TraceScene(byte *destptr,CAMERA *camera,OMNI_LIGHTSOURCE *light)
{ dword i,x,y;
  VECTOR V,U,V1,V2,P0,VV1,VV2;
  RAY ray;
#ifdef TRACER_ENABLE_REFLECTION
  RAY reflectedray;
#endif
  INTERSECTION intersection;

  float R,G,B;
  VECTOR L,H;
  float diffuse,specular;
  byte *textureptr;

  RAY shadowray;
  byte tracingflags;

  // V vektor (nezett_pont - nezo_pont vektora normalva):
  V.X= camera->target.X - camera->position.X;
  V.Y= camera->target.Y - camera->position.Y;
  V.Z= camera->target.Z - camera->position.Z;
  NormalizeVector(&V);

  // U a felfele mutato vektor a cameraroll fuggvenyeben (normalt):
  U.X= fsin(camera->roll); U.Y= fcos(camera->roll); U.Z= 0;

  // V1/V2 a kepsik meroleges vektorai (V1= U X V, V2= V1 X V):
  CrossProductVectors(&V1,&U,&V);
  CrossProductVectors(&V2,&V1,&V);

  // V1 es V2 meretezese (0.625-os aspect ratio):
  V1.X*= camera->FOV/X_RESOLUTION; V1.Y*= camera->FOV/X_RESOLUTION; V1.Z*= camera->FOV/X_RESOLUTION;
  V2.X*= camera->FOV/(Y_RESOLUTION/0.625); V2.Y*= camera->FOV/(Y_RESOLUTION/0.625); V2.Z*= camera->FOV/(Y_RESOLUTION/0.625);

  // P0 a kepernyo bal felso pontja (P0= V - (V1*XRES/2) - (V2*YRES/2)):
  P0.X= V.X - (V1.X*X_RESOLUTION/2) - (V2.X*Y_RESOLUTION/2);
  P0.Y= V.Y - (V1.Y*X_RESOLUTION/2) - (V2.Y*Y_RESOLUTION/2);
  P0.Z= V.Z - (V1.Z*X_RESOLUTION/2) - (V2.Z*Y_RESOLUTION/2);

  // sugarak kezdopontja minden pixelre ugyanaz:
  ray.Pstart.X= camera->position.X;
  ray.Pstart.Y= camera->position.Y;
  ray.Pstart.Z= camera->position.Z;

  // V2= (0,0,0):
  VV2.X= VV2.Y= VV2.Z= 0.0;
  for(y=0; y<(Y_RESOLUTION + 1); y++)
  { // V1= (0,0,0):
    VV1.X= VV1.Y= VV1.Z= 0.0;
    for(x=0; x<X_RESOLUTION; x++)
    { // (x,y) pixel torlese:
      R= G= B= 0.0;

      // sugar kilovese (x,y) kepernyopontban:
      intersection.t= INFINITE_DISTANCE;
      ray.D.X= P0.X + VV1.X + VV2.X;
      ray.D.Y= P0.Y + VV1.Y + VV2.Y;
      ray.D.Z= P0.Z + VV1.Z + VV2.Z;
      NormalizeVector(&ray.D);

      // metszespontkereses:
      for(i=0; i<objects; i++)
       DetectObject[object[i].type](&ray,&object[i],&intersection,EXACT_INTERSECTION);

      // megvilagitasi modell:
      if(intersection.t != INFINITE_DISTANCE)
      { // metszespontbeli L fenyvektor:
        L.X= light->position.X - intersection.point.X;
        L.Y= light->position.Y - intersection.point.Y;
        L.Z= light->position.Z - intersection.point.Z;

        if(intersection.surfacetype & PHONG_SHADED)
        { // a Phong-megvilagitashoz L-re normalva van szuksegunk:
          NormalizeVector(&L);
          tracingflags= 0;

          // metszespontbeli halfway vektor V megforditasaval (H:= (L + V)/2):
          H.X= (L.X - V.X)/2;
          H.Y= (L.Y - V.Y)/2;
          H.Z= (L.Z - V.Z)/2;
          NormalizeVector(&H);

          // ambient komponens:
          if(intersection.surfacetype & TEXTURED)
          { // ambient komponens a texturabol:
            textureptr= intersection.color->textureptr + 4*(dword)(63.0*intersection.V + intersection.U);

            R= 0.2 * *textureptr;
            G= 0.2 * *(textureptr + 1);
            B= 0.2 * *(textureptr + 2);
          }
          else
          { // Phong diffuz komponens a szinbol:
            R= 0.2 * intersection.color->Rd;
            G= 0.2 * intersection.color->Gd;
            B= 0.2 * intersection.color->Bd;
          }

          // diffuz komponens (diffuse:= L * N):
          diffuse= DotProductVectors(&L,&intersection.normal);
          if(diffuse > 0.0)
          { if(intersection.surfacetype & TEXTURED)
            { // vilagositott Phong diffuz komponens a texturabol:
              textureptr= intersection.color->textureptr + 4*(dword)(63.0*intersection.V + intersection.U);

              R+= 1.7*diffuse * *textureptr;
              G+= 1.7*diffuse * *(textureptr + 1);
              B+= 1.7*diffuse * *(textureptr + 2);
            }
            else
            { // Phong diffuz komponens a szinbol:
              R+= diffuse * intersection.color->Rd;
              G+= diffuse * intersection.color->Gd;
              B+= diffuse * intersection.color->Bd;
            }
          }

          // specular komponens (specular:= 255.0 * (H * N)^n):
          specular= 255.0 * fpow(DotProductVectors(&H,&intersection.normal),PHONG_EXPONENT);
          R+= specular;
          G+= specular;
          B+= specular;
        }
        else
        { // a metszespontbeli L fenyvektor hosszanak kiszamitasa:
          diffuse= (L.X*L.X + L.Y*L.Y + L.Z*L.Z)/(250.0*250.0);
          tracingflags|= L_WAS_NOT_NORMALIZED;

          // fenyesseg a fenyforrastol valo tavolsag fuggvenyeben:
          if(diffuse < 1.0)
          { diffuse= 1.0 - diffuse;
            if(intersection.surfacetype & TEXTURED)
            { // vilagositott Phong diffuz komponens a texturabol:
              textureptr= intersection.color->textureptr + 4*(dword)(63.0*intersection.V + intersection.U);

              R= 1.3 * diffuse * *textureptr;
              G= 1.3 * diffuse * *(textureptr + 1);
              B= 1.3 * diffuse * *(textureptr + 2);
            }
            else
            { // Phong diffuz komponens a szinbol:
              R= diffuse * intersection.color->Rd;
              G= diffuse * intersection.color->Gd;
              B= diffuse * intersection.color->Bd;
            }
          }
        }

        // arnyekkepzes:
        if(intersection.surfacetype & SHADOWCASTING)
        { // sugar kilovese a metszespontbol a fenyforras fele:
          intersection.t= INFINITE_DISTANCE;
          shadowray.Pstart.X= intersection.point.X;
          shadowray.Pstart.Y= intersection.point.Y;
          shadowray.Pstart.Z= intersection.point.Z;
          shadowray.D.X= L.X;
          shadowray.D.Y= L.Y;
          shadowray.D.Z= L.Z;
          // csak ha eddig nem normaltuk az L fenyvektort:
          if(tracingflags & L_WAS_NOT_NORMALIZED) NormalizeVector(&shadowray.D);

          // blokkolaskereses az osszes object-re:
          for(i=0; i<objects; i++)
           if(DetectObject[object[i].type](&shadowray,&object[i],&intersection,BOOLEAN_INTERSECTION) == 1)
           { // i. object blokkolta a pixelt:
             R*= 0.5;
             G*= 0.5;
             B*= 0.5;
           }
        }

#ifdef TRACER_ENABLE_REFLECTION
        // visszaverodes:
        else if(intersection.surfacetype & REFLECTIVE)
        { // sugar tukrozese a feluleti normalisra:
          float N_dot_V= DotProductVectors(&ray.D,&intersection.normal);

          intersection.t= INFINITE_DISTANCE;
          reflectedray.Pstart.X= intersection.point.X;
          reflectedray.Pstart.Y= intersection.point.Y;
          reflectedray.Pstart.Z= intersection.point.Z;
          reflectedray.D.X= (2.0*N_dot_V*intersection.normal.X) - ray.D.X;
          reflectedray.D.Y= (2.0*N_dot_V*intersection.normal.Y) - ray.D.Y;
          reflectedray.D.Z= (2.0*N_dot_V*intersection.normal.Z) - ray.D.Z;
          NormalizeVector(&reflectedray.D);

#ifdef TRACER_DEBUG
          cprintf("\n\r RAY ( %f, %f, %f) REFLECTED TO NORMAL ( %f, %f, %f) IS ( %f, %f, %f)",ray.D.X,ray.D.Y,ray.D.Z,intersection.normal.X,intersection.normal.Y,intersection.normal.Z,reflectedray.D.X,reflectedray.D.Y,reflectedray.D.Z);
#endif

          // tobbi object-et vegigteszteljuk:
          for(i=0; i<objects; i++) DetectObject[object[i].type](&reflectedray,&object[i],&intersection);

          if(intersection.t != INFINITE_DISTANCE)
          { // metszespontbeli L fenyvektor:
            L.X= light->position.X - intersection.point.X;
            L.Y= light->position.Y - intersection.point.Y;
            L.Z= light->position.Z - intersection.point.Z;
            NormalizeVector(&L);

            // diffuz komponens (diffuse:= L * N):
            diffuse= DotProductVectors(&L,&intersection.normal);
            if(diffuse > 0.0)
            { R= diffuse * intersection.color->Rd;
              G= diffuse * intersection.color->Gd;
              B= diffuse * intersection.color->Bd;
            }
          }
        }
#endif

        // trace-lt pixel kirakasa BGRa sorrendben:
        if(B > 255.0) *destptr= 255; else *destptr= (byte)B;
        if(G > 255.0) *(destptr + 1)= 255; else *(destptr + 1)= (byte)G;
        if(R > 255.0) *(destptr + 2)= 255; else *(destptr + 2)= (byte)R;
      }

      // VV1= VV1 + V1:
      VV1.X+= V1.X;
      VV1.Y+= V1.Y;
      VV1.Z+= V1.Z;

      // pixelek atugrasa:
      destptr+= (320/X_RESOLUTION)*4;
    }
    // VV2= VV2 + V2:
    VV2.X+= V2.X;
    VV2.Y+= V2.Y;
    VV2.Z+= V2.Z;

    // sorok atugrasa:
    destptr+= ((180/Y_RESOLUTION) - 1)*4*320;
  }
}