#include "fusa_spatialNode.h"
#include <iostream>
#include "fusa_cameraNode.h"

//this file contains code relevant to spatialNode's data.
//see fusa_spatialNodeTree to see implementations relevant to the
//tree data structure of the nodes.

using namespace fusa;

cSpatialNode::cSpatialNode(cScenegraph *ptr_sceneLink)
{

	m_ID=-1;
	mptr_parentID=-1;
	m_render = true;
	m_worldRender = true;

	mptr_parent = 0;
	mptr_scenegraphLink = ptr_sceneLink;
	m_scaleVec.x=1;
	m_scaleVec.y=1;
	m_scaleVec.z=1;

	mptr_worldRenderPasses = 0;

	m_selectSphereRadius = 10.0;
	m_cullingSphereRadius = 10.0;
	m_hasUpdatedTransformWorld = true;
	m_hasUpdatedTransformLocal = true;

}

cMaterial& cSpatialNode::Material(unsigned int pass)
{
    return m_renderPasses[pass];
}

unsigned int cSpatialNode::getNrOfMaterialPasses()const
{
    return m_renderPasses.size();
}

void cSpatialNode::addMaterialPass(const cMaterial& mat)
{
    m_renderPasses.push_back(mat);
}

void cSpatialNode::preUpdate(float t,cSpatialNode *ptr_parent)
{

	if(ptr_parent)
	{
		//the world position of this object is dependant on the rotation of the parent * local position.


		//in this preUpdate, we calculate how much the object is going to move this frame.
		//this is usefull for collision detection/response.
		//to get the frameMovement delta, we subtract the old world position from the new (calculated) world position.
		if(ptr_parent->m_hasUpdatedTransformWorld || this->m_hasUpdatedTransformLocal)
		{
		  cVec3f temp =m_position.getValue(t) + m_pivotPos + ptr_parent->getWorldPosition();
		temp.x*=ptr_parent->m_worldScaleVec.x;
		temp.y*=ptr_parent->m_worldScaleVec.y;
		temp.z*=ptr_parent->m_worldScaleVec.z;
			m_frameMovementDelta = ptr_parent->getWorldRotation() * (temp);
			m_frameMovementDelta -= m_worldPosition;

			m_worldRotation = ptr_parent->getWorldRotation() * m_rotation;


			m_worldScaleVec.x = m_scaleVec.x * ptr_parent->m_worldScaleVec.x;
			m_worldScaleVec.y = m_scaleVec.y * ptr_parent->m_worldScaleVec.y;
			m_worldScaleVec.z = m_scaleVec.z * ptr_parent->m_worldScaleVec.z;
			m_hasUpdatedTransformWorld = true;
		}
		else
		{
			m_hasUpdatedTransformWorld=false;
			//no movement here, no need to update the world transform.
		}


		if(getNrOfMaterialPasses()!=0)
		{
			mptr_worldRenderPasses = &m_renderPasses;
		}
		else
		{
			mptr_worldRenderPasses = ptr_parent->mptr_worldRenderPasses;
		}


		//make render-flag (if node should render or not) go down in the hierarchy.
		if(!ptr_parent->m_worldRender)
		{
			m_worldRender = false;
		}
		else
		{
			m_worldRender = this->m_render;
		}
	}

	else
	{
		if(this->m_hasUpdatedTransformLocal)
		{
			m_frameMovementDelta = m_position.getValue(t) + m_rotation*m_pivotPos - m_worldPosition;
			m_worldRotation=m_rotation;
			m_worldScaleVec = m_scaleVec;
			m_hasUpdatedTransformWorld = true;
		}
		else
		{
			//no need to update here.
			m_hasUpdatedTransformWorld = false;
		}
		m_worldRender = this->m_render;
		mptr_worldRenderPasses = &m_renderPasses;
	}
}

void cSpatialNode::postUpdate(float t)
{
	//make the move.
	if(m_hasUpdatedTransformWorld)
	{
		m_worldPosition = m_worldPosition + m_frameMovementDelta;
		m_worldRotationMat = m_worldRotation.getMatrix4();
	}
	m_frameMovementDelta.x=m_frameMovementDelta.y=m_frameMovementDelta.z=0;
	//this is recalculated every frame.
	m_hasUpdatedTransformLocal=false;


}

cSpatialNode::NodeType cSpatialNode::getNodeType()const
{
	return ntSpatial;
}

int cSpatialNode::getId()const
{
	return m_ID;
}

fusa::cQuat<float>& cSpatialNode::Rotate()
{
	//FIX_ME interfaces here should be setRotation not returning reference
	//this because we are buffering changes in this variable.
	m_hasUpdatedTransformLocal = true;
	return m_rotation;
}
const fusa::cQuat<float>& cSpatialNode::Rotate()const
{
	return m_rotation;
}

void cSpatialNode::setPosition(const fusa::cVec3f& pos)
{
	m_hasUpdatedTransformLocal = true;
	m_position.setValue(pos);
}
const fusa::cVec3f cSpatialNode::getPosition()const
{
	cVec3f pos = m_position.getValue(0);
	return pos;
}

const fusa::cVec3f cSpatialNode::getPosition(float t)const
{
	cVec3f pos = m_position.getValue(t);
	return pos;
}

void cSpatialNode::translate(const cVec3f &trans)
{
	m_hasUpdatedTransformLocal = true;
	m_position.setValue(m_position.getValue(0) + trans);
}


 const cQuat<float>& cSpatialNode::getWorldRotation()const

{
	return m_worldRotation;
}
const cVec3f& cSpatialNode::getWorldPosition()const
{
	return m_worldPosition;
}


cVec3f cSpatialNode::getWorldScaleVec()const
{
	return m_worldScaleVec;
}


std::string & cSpatialNode::Name()
{
    return m_name;
}

int cSpatialNode::getParentId()const
{
    return mptr_parentID;
}

void cSpatialNode::setRenderFlag(bool shouldNodeBeRendered)
{
    m_render = shouldNodeBeRendered;
}

bool cSpatialNode::shouldRender()const
{
    return m_render;
}

bool cSpatialNode::getWorldShouldRender()const
{
    return m_worldRender;
}



cMaterial* cSpatialNode::getWorldRenderStatePass(unsigned int i)const
{
	return &((*mptr_worldRenderPasses)[i]);
}

const std::string& cSpatialNode::Name()const
{
    return m_name;
}

bool cSpatialNode::cullingSphereOutsideCamFrustrum(cCameraNode *ptr_cam)
{
	//m_worldPosition.getValue(0);
	//ptr_cam->getCameraRotationTransform().getMatrix4();
	//m_sphereRadius
	cMatrix4<float> camMat;
	cMatrix4<float> transMatrix;

	transMatrix.toTranslateMatrix(ptr_cam->getCameraPositionTransform());
	camMat = (ptr_cam->getCameraRotationTransform().getMatrix4()*transMatrix);
	camMat =ptr_cam->getProjectionMatrix()*camMat;
	cVec3f pos = this->getWorldPosition() + m_cullingSphereOffset;

	//FIX_ME this can be speed up by radar implementation.
	cVec4f planes[6] =	{(camMat.getRow(0) + camMat.getRow(3)),//left
						(-camMat.getRow(0) + camMat.getRow(3)), //right
						(camMat.getRow(1) + camMat.getRow(3)), //bottom
						(-camMat.getRow(1) + camMat.getRow(3)), //top
						(camMat.getRow(2) + camMat.getRow(3)), //near
						(-camMat.getRow(2) + camMat.getRow(3))}; //far
	cVec3f planeN;
	for(int i=0; i < 6; i++)
	{
		planeN.set(planes[i].x,planes[i].y,planes[i].z);
		float planeD = planeN.magnitude();
		planes[i].w/=planeD;
		planeN/=planeD;
		float dist = planeN.dotProd(pos) + planes[i].w;
		if(dist <= -m_cullingSphereRadius)
			return true;
	}

	return false;
}

bool cSpatialNode::loadNode(std::ifstream &in, bool binForm,int versionFormat)
{
	if(!in.good())
	{
		return false;
	}

	if(binForm)
	{
		if(versionFormat==1)
		{
			char c=in.get();
			while(c!='\0')
			{
				m_name+=(c);
				c=in.get();
			}
			in.read(reinterpret_cast<char*>(&m_ID),sizeof(int));
			in.read(reinterpret_cast<char*>(&mptr_parentID),sizeof(int));
			cVec3f pos;
			in.read(reinterpret_cast<char*> (&pos.x),sizeof(float)*3);
			m_position.setValue(pos);
			in.read(reinterpret_cast<char*> (&m_rotation.VecPart().x),sizeof(float)*3);
			in.read(reinterpret_cast<char*> (&m_rotation.RealPart()),sizeof(float));
			in.read(reinterpret_cast<char*> (&m_scaleVec.x),sizeof(float)*3);
		}
	}
	else
	{
		if(versionFormat==1)
		{
			std::string buff;
			std::getline(in,buff,':');
			std::getline(in,buff);
			m_name = buff;
			std::getline(in,buff,':');
			in>>m_ID;
			std::getline(in,buff,':');
			in>>mptr_parentID;

			in.ignore();
				//FIX_ME jumping over pivot-calculations for now
			std::getline(in,buff);
			std::getline(in,buff);
			std::getline(in,buff);

			cVec3f tmp;
			std::getline(in,buff,':');
			in>>tmp;
			m_position.setValue(tmp);


			std::getline(in,buff,':');
			in>>m_rotation.VecPart()>>m_rotation.RealPart();


			//FIX_ME .. support for xyz,scale or make exporter apply that in model-space
			//to avoid complexification of correct normal calculations?
			std::getline(in,buff,':');
			in>>m_scaleVec;
			in.ignore();
			in.ignore();



			//FIX_ME jumping over animation handling for now.
			std::getline(in,buff);
			std::getline(in,buff);
			std::getline(in,buff);

		}
	}

	return in.good();
}

cSpatialNode::~cSpatialNode()
{

}

