#include <stdint.h>
#include <stdbool.h>
#include "SMSlib.h"

/****************************
 *        Constants         *
 ****************************/
#define SET_PIXEL_FLUSH 0xff
const uint32_t blank_pattern[8] = { 0 };

/****************************
 *    External Functions    *
 ****************************/

/* vdp_access.c */
void read_halfline (uint16_t addr, uint8_t *pattern);
void write_halfline (uint16_t addr, uint8_t *pattern);

/****************************
 *           State          *
 ****************************/
static bool bitmap2_line_buffered = false;

/****************************
 *           Code           *
 ****************************/

/* Use the background palette for the top half of the screen
 * and the sprite palette for the bottom half of the screen. */
void bitmap2_init (void)
{
    for (int y = 0; y < 12; y++)
    {
        for (int x = 0; x < 32; x++)
        {
            uint16_t tile_index = 64 + y * 32 + x;
            SMS_loadTiles (&blank_pattern, tile_index, 32);
            SMS_loadTileMapArea (x,  0 + y, &tile_index, 1, 1);

            tile_index |= 0x0800;
            SMS_loadTileMapArea (x, 12 + y, &tile_index, 1, 1);
        }
    }
    bitmap2_line_buffered = false;
}

void bitmap2_clear (void)
{
    for (int y = 0; y < 12; y++)
    {
        for (int x = 0; x < 32; x++)
        {
            uint16_t tile_index = 64 + y * 32 + x;
            SMS_loadTiles (&blank_pattern, tile_index, 32);
        }
    }
    bitmap2_line_buffered = false;
}

void bitmap2_set_palette(uint8_t *palette)
{
    uint8_t palette_top   [16] = { 0 };
    uint8_t palette_bottom[16] = { 0 };

    for (int colour = 0; colour < 4; colour++)
    {
        for (int otherbits = 0; otherbits < 4; otherbits++)
        {
            palette_top    [(otherbits << 2) | colour] = palette[colour];
            palette_bottom [(colour << 2) | otherbits] = palette[colour];
        }
    }

    SMS_loadBGPalette     (palette_top);
    SMS_loadSpritePalette (palette_bottom);
    SMS_setBackdropColor (0);
}

void bitmap2_set_pixel (uint8_t x, uint8_t y, uint8_t value)
{
    static uint8_t pattern[2];
    static uint16_t stored_addr;

    uint8_t pixel_mask = 0;
    uint16_t line_addr;

    if (y < 96)
    {
        /* Our 'bitmap' image begins 2K into vram */
        line_addr = 2048 + ((y & 0xf8) << 7) + (((x & ~0x07) + (y & 0x07)) << 2);
    }
    else
    {
        /* The bottom half of the screen is offset by two bytes */
        line_addr = 2050 + (((y - 96) & 0xf8) << 7) + (((x & ~0x07) +(y & 0x07)) << 2);
    }

    if (value == SET_PIXEL_FLUSH)
    {
        write_halfline (stored_addr, pattern);
        bitmap2_line_buffered = false;
        return;
    }

    /* If we need to, flush the old line and load the new one */
    if (line_addr != stored_addr)
    {
        /* Flush the previous line */
        if (bitmap2_line_buffered)
        {
            write_halfline (stored_addr, pattern);
        }

        /* Load the new line from VRAM */
        stored_addr = line_addr;
        read_halfline (line_addr, pattern);
        bitmap2_line_buffered = true;
    }

    /* Bits select the pixel */
    pixel_mask = 0x80 >> (x & 0x07);

    /* Bytes select the colour */
    pattern[0] = (pattern[0] & ~pixel_mask) | ((value & 0x01) ? pixel_mask : 0x00);
    pattern[1] = (pattern[1] & ~pixel_mask) | ((value & 0x02) ? pixel_mask : 0x00);
}

/* Set eight pixels at once */
void bitmap2_set_line (uint8_t x, uint8_t y, uint16_t value)
{
    static uint8_t pattern[2];
    static uint16_t stored_addr;

    uint8_t pixel_mask = 0;
    uint16_t line_addr;

    if (y < 96)
    {
        /* Our 'bitmap' image begins 2K into vram */
        line_addr = 2048 + ((y & 0xf8) << 7) + (((x & ~0x07) + (y & 0x07)) << 2);
    }
    else
    {
        /* The bottom half of the screen is offset by two bytes */
        line_addr = 2050 + (((y - 96) & 0xf8) << 7) + (((x & ~0x07) +(y & 0x07)) << 2);
    }

    write_halfline (line_addr, &value);
}
