#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sms.h>

// Map related constants
#define MAX_MAP_W 32
#define MAX_MAP_H 16
#define MAP_LINE_SHIFT 5
#define REFRESH_MASK 0x80
#define REFRESH_UNMASK (~REFRESH_MASK)

// Joypad related constants
#define REPEAT_START_DELAY 3
#define REPEAT_DELAY 0

typedef struct _joy_repeat_rec {
    UBYTE prev_joy;
    UBYTE tmr;
} joy_repeat_rec;

typedef struct _levelmap {
	UBYTE w, h;
	UBYTE data[MAX_MAP_W*MAX_MAP_H];
} levelmap;

extern unsigned UBYTE soko_tileset_apk[];
extern unsigned UBYTE soko_pal[];

UBYTE pal1[] = {0x00, 0x20, 0x08, 0x28, 0x02, 0x22, 0x0A, 0x2A,
				0x15, 0x35, 0x1E, 0x3E, 0x17, 0x37, 0x0F, 0x3F};

extern char players[];
extern char goals[];
extern char rules[];

UBYTE *maps[] = {
"\
M^^^^#####\
M^^^^#   #\
M^^^^#$  #\
M^^###  $##\
M^^#  $ $ #\
M### # ## #^^^######\
M#   # ## #####  ..#\
M# $  $          ..#\
M##### ### #@##  ..#\
M^^^^#     #########\
M^^^^#######",

"\
M############\
M#..  #     ###\
M#..  # $  $  #\
M#..  #$####  #\
M#..    @ ##  #\
M#..  # #  $ ##\
M###### ##$ $ #\
M^^# $  $ $ $ #\
M^^#    #     #\
M^^############",

"\
M^^^^^^^^########\
M^^^^^^^^#     @#\
M^^^^^^^^# $#$ ##\
M^^^^^^^^# $  $#\
M^^^^^^^^##$ $ #\
M######### $ # ###\
M#....  ## $  $  #\
M##...    $  $   #\
M#....  ##########\
M########M",

"\
M^^^^^^^^^^^########\
M^^^^^^^^^^^#  ....#\
M############  ....#\
M#    #  $ $   ....#\
M# $$$#$  $ #  ....#\
M#  $     $ #  ....#\
M# $$ #$ $ $########\
M#  $ #     #\
M## #########\
M#    #    ##\
M#     $   ##\
M#  $$#$$  @#\
M#    #    ##\
M###########",

"\
M^^^^^^^^#####\
M^^^^^^^^#   #####\
M^^^^^^^^# #$##  #\
M^^^^^^^^#     $ #\
M######### ###   #\
M#....  ## $  $###\
M#....    $ $$ ##\
M#....  ##$  $ @#\
M#########  $  ##\
M^^^^^^^^# $ $  #\
M^^^^^^^^### ## #\
M^^^^^^^^^^#    #\
M^^^^^^^^^^######",

"\
M######^^###\
M#..  #^##@##\
M#..  ###   #\
M#..     $$ #\
M#..  # # $ #\
M#..### # $ #\
M#### $ #$  #\
M^^^#  $# $ #\
M^^^# $  $  #\
M^^^#  ##   #\
M^^^#########",

"\
M^^^^^^^#####\
M^#######   ##\
M## # @## $$ #\
M#    $      #\
M#  $  ###   #\
M### #####$###\
M# $  ### ..#\
M# $ $ $ ...#\
M#    ###...#\
M# $$ #^#...#\
M#  ###^#####\
M####",

"\
M^^^^^^^^^^#######\
M^^^^^^^^^^#  ...#\
M^^^^^^#####  ...#\
M^^^^^^#      . .#\
M^^^^^^#  ##  ...#\
M^^^^^^## ##  ...#\
M^^^^^### ########\
M^^^^^# $$$ ##\
M^#####  $ $ #####\
M##   #$ $   #   #\
M#@ $  $    $  $ #\
M###### $$ $ #####\
M^^^^^#      #\
M^^^^^########",

"\
M^###^^#############\
M##@####       #   #\
M# $$   $$  $ $ ...#\
M#  $$$#    $  #...#\
M# $   # $$ $$ #...#\
M###   #  $    #...#\
M#     # $ $ $ #...#\
M#    ###### ###...#\
M## #  #  $ $  #...#\
M#  ## # $$ $ $##..#\
M# ..# #  $      #.#\
M# ..# # $$$ $$$ #.#\
M##### #       # #.#\
M^^^^# ######### #.#\
M^^^^#           #.#\
M^^^^###############",

"\
M^^^^^^^^^^####\
M^^^^^####^#  #\
M^^^### @###$ #\
M^^##      $  #\
M^##  $ $$## ##\
M^#  #$##     #\
M^# # $ $$ # ###\
M^#   $ #  # $ #####\
M####    #  $$ #   #\
M#### ## $         #\
M#.    ###  ########\
M#.. ..#^####\
M#...#.#\
M#.....#\
M#######",

"\
M^^####\
M^^#  ###########\
M^^#    $   $ $ #\
M^^# $# $ #  $  #\
M^^#  $ $  #    #\
M### $# #  #### #\
M#@#$ $ $  ##   #\
M#    $ #$#   # #\
M#   $    $ $ $ #\
M^####  #########\
M^^#      #\
M^^#      #\
M^^#......#\
M^^#......#\
M^^#......#\
M^^########",

"\
M################\
M#              #\
M# # ######     #\
M# #  $ $ $ $#  #\
M# #   $@$   ## ##\
M# #  $ $ $###...#\
M# #   $ $  ##...#\
M# ###$$$ $ ##...#\
M#     # ## ##...#\
M#####   ## ##...#\
M^^^^#####     ###\
M^^^^^^^^#     #\
M^^^^^^^^#######",

"\
M^^^#########\
M^^##   ##  #####\
M###     #  #    ###\
M#  $ #$ #  #  ... #\
M# # $#@$## # #.#. #\
M#  # #$  #    . . #\
M# $    $ # # #.#. #\
M#   ##  ##$ $ . . #\
M# $ #   #  #$#.#. #\
M## $  $   $  $... #\
M^#$ ######    ##  #\
M^#  #^^^^##########\
M^####",

"\
M^^^^^^^#######\
M^#######     #\
M^#     # $@$ #\
M^#$$ #   #########\
M^# ###......##   #\
M^#   $......## # #\
M^# ###......     #\
M##   #### ### #$##\
M#  #$   #  $  # #\
M#  $ $$$  # $## #\
M#   $ $ ###$$ # #\
M#####     $   # #\
M^^^^### ###   # #\
M^^^^^^#     #   #\
M^^^^^^########  #\
M^^^^^^^^^^^^^####",

"\
M^^^^#######\
M^^^#   #  #\
M^^^#  $   #\
M^### #$   ####\
M^#  $  ##$   #\
M^#  # @ $ # $#\
M^#  #      $ ####\
M^## ####$##     #\
M^# $#.....# #   #\
M^#  $..**. $# ###\
M##  #.....#   #\
M#   ### #######\
M# $$  #  #\
M#  #     #\
M######   #\
M^^^^^#####",

"\
M#####\
M#   ##\
M#    #^^####\
M# $  ####  #\
M#  $$ $   $#\
M###@ #$    ##\
M^#  ##  $ $ ##\
M^# $  ## ## .#\
M^#  #$##$  #.#\
M^###   $..##.#\
M^^#    #.*...#\
M^^# $$ #.....#\
M^^#  #########\
M^^#  #\
M^^####",

"\
M^^^##########\
M^^^#..  #   #\
M^^^#..      #\
M^^^#..  #  ####\
M^^#######  #  ##\
M^^#            #\
M^^#  #  ##  #  #\
M#### ##  #### ##\
M#  $  ##### #  #\
M# # $  $  # $  #\
M# @$  $   #   ##\
M#### ## #######\
M^^^#    #\
M^^^######",

"\
M^^^^^###########\
M^^^^^#  .  #   #\
M^^^^^# #.    @ #\
M^##### ##..# ####\
M##  # ..###     ###\
M# $ #...   $ #  $ #\
M#    .. ##  ## ## #\
M####$##$# $ #   # #\
M^^## #    #$ $$ # #\
M^^#  $ # #  # $## #\
M^^#               #\
M^^#  ###########  #\
M^^####^^^^^^^^^####",

"\
M^^######\
M^^#   @####\
M##### $   #\
M#   ##    ####\
M# $ #  ##    #\
M# $ #  ##### #\
M## $  $    # #\
M## $ $ ### # #\
M## #  $  # # #\
M## # #$#   # #\
M## ###   # # ######\
M#  $  #### # #....#\
M#    $    $   ..#.#\
M####$  $# $   ....#\
M#       #  ## ....#\
M###################",

"\
M^^^^##########\
M#####        ####\
M#     #   $  #@ #\
M# #######$####  ###\
M# #    ## #  #$ ..#\
M# # $     #  #  #.#\
M# # $  #     #$ ..#\
M# #  ### ##     #.#\
M# ###  #  #  #$ ..#\
M# #    #  ####  #.#\
M# #$   $  $  #$ ..#\
M#    $ # $ $ #  #.#\
M#### $###    #$ ..#\
M^^^#    $$ ###....#\
M^^^#      ##^######\
M^^^########"
};

UBYTE *select_line(levelmap *level, UBYTE y) {
	return level->data + (y << MAP_LINE_SHIFT);
}

UBYTE *select_cell(levelmap *level, UBYTE x, UBYTE y) {
	return level->data + (y << MAP_LINE_SHIFT) + x;
}

void load_map(levelmap *level, UBYTE *map) {
	UBYTE x, y, *o, *d;

	level->w = 0;
	level->h = 0;
	memset(level->data, 0, sizeof(level->data));

	x = 0;
	y = -1;
	d = level->data;
	for (o = map; *o; o++) {
		if (*o == 'M') {
			x = 0;
			y++;
			d = select_line(level, y);
		} else {
			*d = (*o) | REFRESH_MASK;
			d++;

			x++;
			if (x > level->w) {
				level->w = x;
			}
		}
	}
	level->h = y + 1;
}

void clear_canvas() {
	UBYTE i;
	UWORD line[MAX_MAP_W];

	memset(line, 0, sizeof(line));
	for (i = 0; i != MAX_MAP_H; i++) {
		set_bkg_map(line, 0, i, MAX_MAP_W, 1);
	}
}

void render_map(levelmap *level) {
	UBYTE i, j, *p;
	UWORD tile;

	for (i = 0; i != level->h; i++) {
		p = select_line(level, i);
		for (j = 0; j != level->w; j++) {
			tile = *p;
			if (tile & REFRESH_MASK) {
				tile &= ~REFRESH_MASK;
				*p = tile;
				tile |= BKG_ATTR_2NDTILESET;
				set_bkg_map(&tile, j, i, 1, 1);
			}
			p++;
		}
	}
}

UBYTE *skip_linebreak(UBYTE *p) {
	while (*p != 10) {
	    p++;
	}
	p++;

	return p;
}

UWORD match_charlist(UBYTE *s, UBYTE c) {
    while(*s != '\n') if(*s==c) return(s); else s++;
    return(0);
}

UBYTE check_rule(UBYTE *p, UBYTE *p2) {
	UBYTE *r;

	r = rules;
	while ((*r != 13) && (*r != 10)) {
		if ((*r == (*p & REFRESH_UNMASK)) && (*(r+1) == (*p2 & REFRESH_UNMASK))) {
			r += 2;
			if (*r == '=') {
				r++;
				*p = *r | REFRESH_MASK;
				r++;
				*p2 = *r | REFRESH_MASK;
				return TRUE;
			}
		}

		r = skip_linebreak(r);
	}

	return FALSE;
}

void shove_it(levelmap *level, BYTE x, BYTE y, BYTE xofs, BYTE yofs) {
	UBYTE *p, *p2;

	p = select_cell(level, x, y);
	p2 = select_cell(level, x + xofs, y + yofs);
	check_rule(p, p2);
}

void move_player(levelmap *level, BYTE xofs, BYTE yofs) {
	UBYTE i, j, *p, *p2;
	char ch2;

	for (i = 0; i != level->h; i++) {
		p = select_line(level, i);
		for (j = 0; j != level->w; j++) {
			if ((*p != ' ') && match_charlist(players, *p)) {
				shove_it(level, j + xofs, i + yofs, xofs, yofs);
				shove_it(level, j, i, xofs, yofs);
			}
			p++;
		}
	}
}

UBYTE check_level_end(levelmap *level) {
	UBYTE i, j, *p, ch;

	for (i = 0; i != level->h; i++) {
		p = select_line(level, i);
		for (j = 0; j != level->w; j++) {
			ch = (*p & REFRESH_UNMASK);
			if ((ch != ' ') && match_charlist(goals, ch)) {
				return FALSE;
			}
			p++;
		}
	}

	return TRUE;
}

int read_joy_repeat(joy_repeat_rec *rpt) {
    UBYTE joy = read_joypad1();
    if (joy != rpt->prev_joy) {
       rpt->prev_joy = joy;
       rpt->tmr = REPEAT_START_DELAY;
    } else if (!rpt->tmr) {
       rpt->tmr = REPEAT_DELAY;
    } else {
       joy = 0;
       rpt->tmr--;
    }
    return joy;
}

void wait_joy_release(int mask) {
    wait_vblank_noint();
    while (read_joypad1() & mask) {
        wait_vblank_noint();
    }
}

void load_main_tileset() {
	unsigned char buffer[4096];

	aplib_depack(soko_tileset_apk, buffer);
	load_tiles(buffer, 256, 128, 4);

	load_tiles(standard_font, 0, 255, 1);
}

void main() {
	UBYTE joy;
	joy_repeat_rec joyrec;

	UBYTE levelno;
	levelmap level;

	UBYTE moved;
	UBYTE finished;

	for (;;) {
		set_vdp_reg(VDP_REG_FLAGS1, VDP_REG_FLAGS1_SCREEN);
		load_palette(soko_pal, 0, 16);
		load_palette(pal1, 16, 16);
		load_main_tileset();

		levelno = 0;
		load_map(&level, maps[levelno]);

		for (;;) {
			clear_canvas();
			load_map(&level, maps[levelno]);

			gotoxy(0, 22);
			printf("Level: %d\n", levelno + 1);
			printf("To restart, press both buttons");

			finished = FALSE;
			while(!finished) {
				joy = read_joy_repeat(&joyrec);

				if (joy & JOY_FIREA) {
					if (joy & JOY_UP) {
						if (levelno > 0) {
							levelno--;
							clear_canvas();
							load_map(&level, maps[levelno]);
						}
					}
					if (joy & JOY_DOWN) {
						levelno++;
						clear_canvas();
						load_map(&level, maps[levelno]);
					}
					if (joy & JOY_FIREB) {
						load_map(&level, maps[levelno]);
					}
				} else {
					if (joy & JOY_UP) {
						move_player(&level, 0, -1);
						moved = TRUE;
					}
					if (joy & JOY_DOWN) {
						move_player(&level, 0, 1);
						moved = TRUE;
					}
					if (joy & JOY_LEFT) {
						move_player(&level, -1, 0);
						moved = TRUE;
					}
					if (joy & JOY_RIGHT) {
						move_player(&level, 1, 0);
						moved = TRUE;
					}
				}

				if (moved) {
					finished = check_level_end(&level);
				}
				moved = FALSE;

				wait_vblank_noint();
				render_map(&level);
			}

			levelno++;
		}
	}
}


#asm
._soko_tileset_apk
	BINARY	"soko.apk"
._soko_pal
	BINARY	"soko.pal"
._players
	BINARY	"players.txt"
._goals
	BINARY	"goals.txt"
._rules
	BINARY	"rules.txt"
#endasm
