pico-8 cartridge // http://www.pico-8.com
version 14
__lua__

--****************************************************************************************************
--* Most of the global stuff/
--****************************************************************************************************

--------------------------------
-- Start the music.
--------------------------------
music(0)

--------------------------------
-- Global variables.
--------------------------------
t = 0
t0 = 0 
t1 = 0
tBounce = 0
logo = {x=0,y=0}
partIndex = 0
patternCount = 0
partPatternIndex = 0
partPosition = 0
xa = 0
ya = 0
matrix = {}


--****************************************************************************************************
--* Intro Part
--****************************************************************************************************
function IntroInit()
    -- Clear screen.
    cls()

    --
    StarFieldInit()
end

function IntroUpdate()
    --
    StarFieldUpdate()
end

function IntroDraw()
    --
    StarFieldDraw()

    --
    if (             t < 256) y = 8+160-0.5*t
    if (256 <= t and t < 512) y = 8+32
    if (512 <= t)             y = 8+160+128-0.5*t

    DrawText({
        "    \"block around THE clock\"    ",
        "",
        "               BY               ",
        "        genesis project         ",
    }, 0, y, 12)
end

function DrawText(lines, x, y, color)
    for line in all(lines) do
	    print(line, x, y, color)
        y += 8
    end
end


--****************************************************************************************************
--* Outro
--****************************************************************************************************
function OutroInit()
    -- Clear screen.
    cls()

    --
    StarFieldInit()
end

function OutroUpdate()
    StarFieldUpdate()
end

function OutroDraw()
    --
    StarFieldDraw()

    --
    if (             t < 256) y = -8+160-0.5*t
    if (256 <= t and t < 512) y = -8+32
    if (512 <= t)             y = -8+32

    DrawText({
    "   you have reached the end of  ",
    "    \"block around THE clock\"    ",
    "",
    "               BY               ",
    "        genesis project         ",
    "",
    "          code - papademos      ",
    "         music - stinsen        ",
    "      graphics - illmidus       ",
    }, 0, y, 12)
end


--****************************************************************************************************
--* Cube Walk
--****************************************************************************************************
--------------------------------
-- The model to render.
--------------------------------
local model
function CubeInit()
    model = {
        vertices = {
            {-50, -50, -50, 1},
            {-50, -50,  50, 1},
            {-50,  50, -50, 1},
            {-50,  50,  50, 1},
            { 50, -50, -50, 1},
            { 50, -50,  50, 1},
            { 50,  50, -50, 1},
            { 50,  50,  50, 1}},
        faces = {
            { 1, 2, 4, 3, 1 },
            { 7, 8, 6, 5, 2 },
            { 5, 6, 2, 1, 3 },
            { 3, 4, 8, 7, 4 },
            { 1, 3, 7, 5, 5 },
            { 6, 8, 4, 2, 6 }
        }
    }
end

--------------------------------
-- Update values which will later be used for drawing
--------------------------------
function CubeUpdate()
    -- Timing for this part.
    if (partOffset < 1) then
        cubeOffset = 128-128*partOffset
        xa = 2*beat
        ya = 0.5
        logo.y = 128-min(32, t)
    elseif (partOffset < 2) then
        cubeOffset = 0
        xa = 2*beat
        ya = 0.5
    elseif (partOffset < 6) then
        cubeOffset = 0
        xa = 2*beat
        ya = 0.5 + 0.25*(partOffset-2)
    elseif (partOffset < 7) then
        cubeOffset = -64*(partOffset-6)
        xa = 2*beat
        ya = 0.5 + 0.25*(partOffset-2)
        t0 = t
    elseif (partOffset < 8) then
        logo.y = max(0, 92+t0-t)
    end

    -- Get bottom offset.
    bottom = 50*max(abs(sin(xa)),abs(cos(xa)))
    
    -- Create transformation matrix.
    matrix = MatrixIdentity()
    matrix = MatrixMultiply(matrix, MatrixTranslate(-100, 0, 0))
    matrix = MatrixMultiply(matrix, MatrixRotateX(xa))
    matrix = MatrixMultiply(matrix, MatrixRotateY(ya))
    matrix = MatrixMultiply(matrix, MatrixTranslate(0, 50-bottom, 300))
    matrix = MatrixMultiply(matrix, MatrixPerspectiveFov(0.1, 1, 0.1, 100))

    -- Transform vertices.
    model.transformedVertices = {}
    for i=1,count(model.vertices) do
        v = TransformPosition(matrix, model.vertices[i])
        v[1] = v[1]/v[3]*20 + 64 + cubeOffset
        v[2] = v[2]/v[3]*20 + 64
        model.transformedVertices[i] = v
    end
end

--------------------------------
-- Draw graphics.
--------------------------------
function CubeDraw()
    -- Clear screen.
    rectfill(0, 0, 128, logo.y, 0)
    memcpy(0x6000+64*logo.y, 0, 64*(128-logo.y))

    -- Iterate, and when visible - draw, faces.
	i = 1
	colors = { 1, 1, 6, 6, 13, 13 }
    for face in all(model.faces) do
        -- Get 3 adjacent vertices on the same face.
        v1 = model.transformedVertices[face[1]]
        v2 = model.transformedVertices[face[2]]
        v3 = model.transformedVertices[face[3]]

        -- Calculate the projected face normal for backface culling
        u = {v1[1]-v2[1], v1[2]-v2[2], v1[3]-v2[3]}
        v = {v3[1]-v2[1], v3[2]-v2[2], v3[3]-v2[3]}
        w = Cross(Normalize(u), Normalize(v))

        -- Draw face if visible.
        if w[3] <= 0 then
			--DrawLineFace(face)
			DrawFilledFace(face, colors[i])
        end
		i += 1
    end
end
function DrawFilledFace(face, color)
	-- Get the index of the topmost vertex.
	i0 = 1
	y0 = model.transformedVertices[face[1]][2]
	for i1=2,4 do
		y1 = model.transformedVertices[face[i1]][2]
		if y1 < y0 then
			i0 = i1
			y0 = y1
		end
	end
	y = y0

	-- Get all vertices (starting and ending with the topmost).
	v = {}
	for i=1,5 do
		v[i] = model.transformedVertices[face[i0]]
		i0 = i0%4+1
	end

	-- Calculate all x-positions.
	x = {}
	xi = 1
	for i=1,4 do
		x0 = flr(v[i][1])
		y0 = flr(v[i][2])
		x1 = flr(v[i+1][1])
		y1 = flr(v[i+1][2])
		dx = x1-x0
		dy = y1-y0
        if dy > 0 then
		    for j=0,dy-1 do
                add(x, flr(x0 + j * dx / dy))
		    end
        end
        if dy < 0 then
		    for j=-1,dy,-1 do
			    add(x, flr(x0 + j * dx / dy))
		    end
        end
	end

	-- Draw horizontal lines.
	n = count(x)
    s = 1
	for i=0,n/2-1,s do
		x0 = x[1+i]
		x1 = x[n-i]
		line(x0, y, x1, y, color)
		y += s
	end
end
function DrawLineFace(face)
	edges = {
		{face[1], face[2]},
		{face[2], face[3]},
		{face[3], face[4]},
		{face[4], face[1]}}
	for edge in all(edges) do
		p0 = model.transformedVertices[edge[1]]
		p1 = model.transformedVertices[edge[2]]
		line(p0[1],p0[2],p1[1],p1[2],c)
	end
end


--****************************************************************************************************
--* Twister
--****************************************************************************************************
--------------------------------
-- The model to render.
--------------------------------
function TwisterInit()
    model = {
        vertices = { 
            {-16, 0, 24 },
            { 16, 0, 24 },
            { 16, 0, 16 },
            {-16, 0, 16 },
            {-16, 0, 24-48 },
            { 16, 0, 24-48 },
            { 16, 0, 16-48 },
            {-16, 0, 16-48 }
        },
        edges = { 
            { 1, 2, 12 }, 
            { 2, 3,  1 }, 
            { 3, 4, 12 }, 
            { 4, 1,  1 }, 
        },
        transformedVertices = { }
    }
    twisterText = {
        "                                ",
        "hi everybody, papademos in the  ",
        "building! i'm pulling an all    ",
        "nighter trying to put together  ",
        "this demo. there's nothing      ",
        "quite like the flow during night",
        "coding!                         ",
        "                                ",
        "i really hope you enjoy the     ",
        "magical tune by stinsen and the ",
        "awesome logo drawn by illmidus. ",
        "                                ",
        "              ....              ",
        "                                ",
        "kudos to andreas for organizing ",
        "the pico-8 evenings and the     ",
        "competition at sectra hq.       ",
        "                                ",
        "              ....              ",
        "                                ",
        "          greetings to          ",
        "                                ",
        "  booze design - censor design  ",
        "  cerebral vortex - checkpoint  ",
        "   cosine - desire - fairlight  ",
        "  hemoroids - lamers - mahoney  ",
        "    maniacs of noise - noice    ",
        "   nuance - offence - oxygene   ",
        "     oxyron - padua - plush     ", 
        " razor 1911 - spaceballs - sync ",
        "   titan - triad - trsi - tscc  ",
        "        up rough - vision       ",
        "                                ",
        "             ....               ",
        "                                ",
        "...and of course lots of hugs   ",
        "and kisses to everybody who's   ",
        "participating in the sectra     ",
        "pico-8 demo compo!              ",
    }
end

--------------------------------
-- Update values which will later be used for drawing
--------------------------------
function TwisterUpdate()
    -- Timing for this part.
    xOffset = 0
    if (partOffset < 1) then
        xOffset = 64-64*partOffset
        yScroll = 128
        t0 = t
        logo.y = 0
    elseif (partOffset < 7) then
        yScroll = 128+0.33*(t0-t)
    else
        if (logo.y < 64) logo.y += 1
    end

    -- Twister calculations.
    for y=32,127 do
        local a = 0.012 * t +
            sin(0.001 * t - 0.001*y) *
            sin(0.0011   * t - 0.0011*y)
        model.transformedVertices[y] = { };
        for i=1,count(model.edges) do
            local x = model.vertices[i][1]
            local z = model.vertices[i][3]
            model.transformedVertices[y][i] = {
                x*cos(a) - z * sin(a) + 80 + xOffset,
                0,
                x*sin(a) + z * cos(a)
            }
        end
    end
end

--------------------------------
-- Draw graphics.
--------------------------------
function TwisterDraw()
    -- Clear screen.
    rectfill(0, 32+logo.y, 128, 128, 0)
    DrawText(twisterText, 0, yScroll, 12)

    --
    local vertices = model.transformedVertices
    for y=32+logo.y,127 do
        for edge in all(model.edges) do
            local x0 = vertices[y][edge[1]][1]
            local x1 = vertices[y][edge[2]][1]
            local c = edge[3]
            if (x0 < x1) line(x0,y,x1,y,c)
        end
    end

    -- Draw logo.
    DrawLogo(logo.y)
    DrawLogo(logo.y-32)
    DrawLogo(logo.y-64)
    DrawLogo(160-logo.y)
end

function DrawLogo(y)
    if y < -31 then
    elseif y < 0 then
        memcpy(0x6000, 64*-y, 64*(32+y))
    elseif y < 96 then
        memcpy(0x6000+64*y, 0, 64*32)
    elseif y < 128 then
        memcpy(0x6000+64*y, 0, 64*(128-y))
    end
end 


--****************************************************************************************************
--* StarField
--****************************************************************************************************
function StarFieldInit()
    --------------------------------
    -- The model to render.
    --------------------------------
    model = {
        vertices = { }
    }
    for i=1,100 do
        model.vertices[i] = { rnd(2)-1, rnd(2)-1, rnd(2)-1 }
    end
end

--------------------------------
-- Update values which will later be used for drawing
--------------------------------
function StarFieldUpdate()
    -- Get rotation angles.
    xa = 0.1*sin(0.001*t)
    ya = 0
    za = 0.001*t
    UpdateStarFieldModel()
end
    
function UpdateStarFieldModel()
    -- Modify model.
    for vertex in all(model.vertices) do
        vertex[3] = (vertex[3] + 2 - 0.01) % 2 - 2
    end

    -- Create transformation matrix.
    matrix = MatrixIdentity()
    matrix = MatrixMultiply(matrix, MatrixRotateZ(za))
    matrix = MatrixMultiply(matrix, MatrixRotateX(xa))
    matrix = MatrixMultiply(matrix, MatrixTranslate(0, 0, 0.5))
    matrix = MatrixMultiply(matrix, MatrixPerspectiveFov(0.4, 1, 1, 100))

    -- Transform vertices.
    model.transformedVertices = {}
    for i=1,count(model.vertices) do
        v = TransformPosition(matrix, model.vertices[i])
        v[1] = v[1]/v[3]*128 + 64
        v[2] = v[2]/v[3]*128 + 64
        model.transformedVertices[i] = v
    end
end

--------------------------------
-- Draw graphics.
--------------------------------
function StarFieldDraw()
    -- Clear screen.
    cls()

    -- Draw all vertices
    for vertex in all(model.transformedVertices) do
        local indices = {12, 1}
        if (vertex[3] > 0) then
            local i = flr(min(1, 0.7*vertex[3]) * count(indices) + 0.5)
            pset(vertex[1], vertex[2], indices[i])
        end
    end
end


--****************************************************************************************************
--* Roto Zoom
--****************************************************************************************************
--------------------------------
-- Initialize stuff
--------------------------------
function RotoZoomInit()
    -- Multiply logo
    memcpy(0x800, 0, 0x800)
    memcpy(0x1000, 0, 0x800)
    memcpy(0x1800, 0, 0x800)

    -- Create the model.
    model = {
        hi = {},
        lo = {},
    }
    loModel = {}
    for y=0,127 do
        model.hi[y] = {}
        model.lo[y] = {}
        for x=0,127 do
			-- copy graphics to the model.
			local s = 4 * (x % 2)
			local c = band(shr(peek(64*y + flr(x/2)),s),15)
            model.hi[y][x] = shl(c,4)
            model.lo[y][x] = c
        end
    end

    for y=0,15 do
        for x=0,15 do
            local i = 16*y + x;
            local address = 0x2000 + 128*y + x
            poke(address, i)
        end
    end

    palt(0, false)
end

--------------------------------
-- Update values which will later be used for drawing
--------------------------------
function RotoZoomUpdate()
    tRot = t        
    tLerp = min(partOffset, 1)
end

--------------------------------
-- Draw graphics.
--------------------------------
function RotoZoomDraw()
    rectfill(0, 0, 128, 32, 0)
	RotoZoom(0, 128)
end

function RotoZoom(y0, height)
    local screen = 0x6000 + 64*y0
    local dx = cos(0.001*tRot) * (0.5 + 0.25*tLerp + 0.5*cos(0.006 * tRot))
    local dy = sin(0.001*tRot) * (0.5 + 0.25*tLerp + 0.5*cos(0.006 * tRot))
    local x = (dx) * 64 * cos(0.5*tLerp + 0.0025*tRot)
    local y = (dy) * 64 * sin(0.2*tLerp + 0.0015*tRot)
    local ux = x
    local uy = y
    for y=0,height-1 do
        vx = ux
        vy = uy
		ux += dx
		uy += dy
        for x=0,63 do
			local c = model.lo[band(vx,127)][band(vy,127)] 
			vx -= dy
			vy += dx
			c += model.hi[band(vx,127)][band(vy,127)]
			vx -= dy
			vy += dx
            poke(screen, c)
            screen += 1
        end
    end
end


--****************************************************************************************************
--* Conway's Game of Life
--****************************************************************************************************
--------------------------------
-- Initialize stuff
--------------------------------
function LifeInit()
    -- The model to render.
    model = {
        hi = {},
        lo = {},
    }
    loModel = {}
	srand(13)
    for y=0,63 do
        model.hi[y] = {}
        model.lo[y] = {}
        for x=0,63 do
			local c = flr(rnd(2))
            model.hi[y][x] = c
            model.lo[y][x] = c
        end
    end

    -- Draw logo.
    memcpy(0x6000, 0, 0x800)
end

--------------------------------
-- Update values which will later be used for drawing
--------------------------------
function LifeUpdate()
    -- Timing for this part.
    tRot = 128*partOffset
    if (partOffset < 1) then
        tZoom = 0.05 + 0.95*partOffset
        lifeHeight = 96
    elseif (partOffset < 2) then
        tZoom = 1
    elseif (partOffset < 7) then
        tZoom = 1
    elseif (partOffset < 8) then
        tZoom = 0.05+0.95-(partOffset-7)
    else
    end

    --
    local lifeCells = model.lo
    local sumCells = model.hi

    -- Split the calculations into odd and even frames.
    if t%2 == 0 then
	    -- Set 64 random cells.
	    for u=0,63 do
		    local x = flr(rnd(63))
		    local y = flr(rnd(63))
		    lifeCells[y][x] = 1
	    end

	    -- Calculate horizontal sums.
        for y=0,63 do
		    local a = lifeCells[y][63]
		    local b = lifeCells[y][0]
		    local c = lifeCells[y][1]
		    for x=0,63 do
			    sumCells[y][x] = a+b+c
			    a,b,c = b,c,lifeCells[y][band(x+2,63)]
		    end
	    end
    else 
	    -- Calculate vertical sums and handle cell death/birth.
        for x=0,63 do
		    local a = sumCells[63][x]
		    local b = sumCells[0][x]
		    local c = sumCells[1][x]
		    for y=0,63 do
			    local sum = a+b+c
			    if (sum == 3) lifeCells[y][x] = 1
			    if (sum < 2 or sum > 3 ) lifeCells[y][x] = 0
			    a,b,c = b,c,sumCells[band(y+2,63)][x]
		    end
	    end
    end
end

--------------------------------
-- Draw graphics.
--------------------------------
function LifeDraw()
	LifeRotoZoom(32, lifeHeight)
end

function LifeRotoZoom(y0, height)
    local lifeCells = model.lo
    local screen = 0x6000 + 64*y0
    local dx = tZoom*cos(0.001*tRot) * (0.75 + 0.5*cos(0.006 * tRot))
    local dy = tZoom*sin(0.001*tRot) * (0.75 + 0.5*cos(0.006 * tRot))
    local x = (dx) * 64 * cos(0.5 + 0.0025*tRot)
    local y = (dy) * 64 * sin(0.2 + 0.0015*tRot)
    local u = {x=x,y=y}
    for y=0,height-1 do
		local vx = u.x
        local vy = u.y
		u.x += dx
		u.y += dy
        for x=0,127,2 do
			local c = lifeCells[band(vx,63)][band(vy,63)]
			vx += dy
			vy -= dx
			c += shl(lifeCells[band(vx,63)][band(vy,63)],4)
			vx += dy
			vy -= dx
            poke(screen, 12*c)
            screen += 1
        end
    end
end


--****************************************************************************************************
--* Block around the Clock
--****************************************************************************************************
parts = {
    { IntroInit, IntroUpdate, IntroDraw, 4, 0, "g*p" },
    { CubeInit, CubeUpdate, CubeDraw, 8, 0, "cube walk" },
    { LifeInit, LifeUpdate, LifeDraw, 8, 0, "life" },
    { TwisterInit, TwisterUpdate, TwisterDraw, 8, 0, "twister" },
    { RotoZoomInit, RotoZoomUpdate, RotoZoomDraw, 8, 0, "rotozoom" },
    { OutroInit, OutroUpdate, OutroDraw, 4, 0, "g*p" },
}

-- Calculate partIndex for all parts.
partIndex = 0
for part in all(parts) do
    partIndex += part[4]
    part[5] = partIndex
end
partIndex = 0

function _update60()
    -- Make sure we have the old partIndex for comparison.
    local oldPartIndex = partIndex

    -- Calculate music related timing values.
    local s = stat(26)-1
    musicPosition = stat(25) + s/384.0
    beat = s/384
    rowIndex = flr(s/6)
    beatIndex = rowIndex % 8

    -- Move to the next part when applicable.
    local oldPatternCount = patternCount
    patternCount = flr(musicPosition)
    if patternCount != oldPatternCount then
        local part = parts[1+partIndex]
        if (patternCount == part[5]) partIndex += 1
    end

    -- Calculate partOffset.
    partIndex = min(partIndex, count(parts)-1)
    if (partIndex != oldPartIndex) then 
        t = 0
        partPosition = patternCount
    end
    partOffset = musicPosition - partPosition

    -- Call Init
    if (t==0) parts[1+partIndex][1]()

    -- Increase time.
    t += 1

    -- Call Update
    parts[1+partIndex][2]()
end

function _draw()
    -- Call Draw 
    parts[1+partIndex][3]()

    -- Draw Bouncing Text.
    local text = parts[1+partIndex][6]
    local w = 2+4*#text
    local y0
    local x=2,y 
    local bounce = abs(16*sin(4*musicPosition))
    if partIndex == 1 then
        if partOffset < 7 then
            y0 = 120-min(32, t)
            y = y0 - bounce
        else 
            y0 = max(0, 92+t0-t)
            y = y0 - bounce
            x = t0-t
        end
    elseif partIndex == 5 then
        local range = 128-w
        tBounce = (tBounce+1)%(2*range)
        y0 = 120
        y = y0 - bounce
        x = abs(range - tBounce)
    else
        y0 = 120
        y = y0 - bounce
    end
    rectfill(   x,   y, x+w, y+8, 0)
    rect(       x,   y, x+w, y+8, 8)
    print(text, x+2, 2+y, 8)
end


--****************************************************************************************************
--* Vector functions
--****************************************************************************************************
function TransformPosition(m, p) return {
    m[1][1] * p[1] + m[1][2] * p[2] + m[1][3] * p[3] + m[1][4],
    m[2][1] * p[1] + m[2][2] * p[2] + m[2][3] * p[3] + m[2][4],
    m[3][1] * p[1] + m[3][2] * p[2] + m[3][3] * p[3] + m[3][4],
    m[4][1] * p[1] + m[4][2] * p[3] + m[4][3] * p[3] + m[4][4]}
end
function Length(v) return
    sqrt(v[1]*v[1]+v[2]*v[2]+v[3]*v[3])
end
function Normalize(v)
    l = Length(v)
    return {v[1]/l, v[2]/l, v[3]/l}
end
function Dot(a, b) return
    a[1] * b[1] + a[2] * b[2] + a[3] * b[3]
end
function Cross(a, b) return {
    a[2] * b[3] - a[3] * b[2],
    a[3] * b[1] - a[1] * b[3],
    a[1] * b[2] - a[2] * b[1]}
end
function Cross2D(a, b) return
    a[1] * b[2] - a[2] * b[1]
end


--****************************************************************************************************
--* Matrix functions
--****************************************************************************************************
function MatrixIdentity() return {
    {1,0,0,0},
    {0,1,0,0},
    {0,0,1,0},
    {0,0,0,1}}
end
function MatrixRotateX(a) return {
    {1,0,0,0},
    {0,cos(a),-sin(a),0},
    {0,sin(a),cos(a),0},
    {0,0,0,1}}
end
function MatrixRotateY(a) return {
    {-sin(a),0,cos(a),0,0},
    {0,1,0,0},
    {cos(a),0,sin(a),0},
    {0,0,0,1}}
end
function MatrixRotateZ(a) return {
    {cos(a),-sin(a),0,0},
    {sin(a),cos(a),0,0},
    {0,0,1,0},
    {0,0,0,1}}
end
function MatrixTranslate(x, y, z) return {
    {1,0,0,x},
    {0,1,0,y},
    {0,0,1,z},
    {0,0,0,1}}
end
function MatrixPostTranslate(x, y, z) return {
    {1,0,0,0},
    {0,1,0,0},
    {0,0,1,0},
    {x,y,z,1}}
end
-- uses left handed orientation
function MatrixPerspectiveFov(fov, aspect, znear, zfar)
    yScale = 1.0 / tan(fov * 0.5)
    q = zfar / (zfar - znear)
    return {
        {yScale / aspect, 0, 0, 0},
        {0, yScale, 0, 0},
        {0, 0, q, 1 },
        {0, 0, -q * znear, 0}}
end
function tan(a)
    return sin(a)/cos(a)
end
function MatrixPerspective(left, right, bottom, top, znear, zfar)
    zRange = zfar / (zfar - znear);
    return {
        {2 * znear / (right - left), 0, 0, 0},
        {0, 2 * znear / (top - bottom), 0, 0 },
        {(left + right) / (left - right), (top + bottom) / (bottom - top), zRange, 1 },
        {0, 0, -znear * zRange, 0}}
end
function MatrixScale(x, y, z, w) return {
    {x,0,0,0},
    {0,y,0,0},
    {0,0,z,0},
    {0,0,0,1}}
end
function MatrixMultiply(a, b) return {
    {b[1][1] * a[1][1] + b[1][2] * a[2][1] + b[1][3] * a[3][1] + b[1][4] * a[4][1],
    b[1][1] * a[1][2] + b[1][2] * a[2][2] + b[1][3] * a[3][2] + b[1][4] * a[4][2],
    b[1][1] * a[1][3] + b[1][2] * a[2][3] + b[1][3] * a[3][3] + b[1][4] * a[4][3],
    b[1][1] * a[1][4] + b[1][2] * a[2][4] + b[1][3] * a[3][4] + b[1][4] * a[4][4]},
    {b[2][1] * a[1][1] + b[2][2] * a[2][1] + b[2][3] * a[3][1] + b[2][4] * a[4][1],
    b[2][1] * a[1][2] + b[2][2] * a[2][2] + b[2][3] * a[3][2] + b[2][4] * a[4][2],
    b[2][1] * a[1][3] + b[2][2] * a[2][3] + b[2][3] * a[3][3] + b[2][4] * a[4][3],
    b[2][1] * a[1][4] + b[2][2] * a[2][4] + b[2][3] * a[3][4] + b[2][4] * a[4][4]},
    {b[3][1] * a[1][1] + b[3][2] * a[2][1] + b[3][3] * a[3][1] + b[3][4] * a[4][1],
    b[3][1] * a[1][2] + b[3][2] * a[2][2] + b[3][3] * a[3][2] + b[3][4] * a[4][2],
    b[3][1] * a[1][3] + b[3][2] * a[2][3] + b[3][3] * a[3][3] + b[3][4] * a[4][3],
    b[3][1] * a[1][4] + b[3][2] * a[2][4] + b[3][3] * a[3][4] + b[3][4] * a[4][4]},
    {b[4][1] * a[1][1] + b[4][2] * a[2][1] + b[4][3] * a[3][1] + b[4][4] * a[4][1],
    b[4][1] * a[1][2] + b[4][2] * a[2][2] + b[4][3] * a[3][2] + b[4][4] * a[4][2],
    b[4][1] * a[1][3] + b[4][2] * a[2][3] + b[4][3] * a[3][3] + b[4][4] * a[4][3],
    b[4][1] * a[1][4] + b[4][2] * a[2][4] + b[4][3] * a[3][4] + b[4][4] * a[4][4]}}
end


--****************************************************************************************************
--* End of Code
--****************************************************************************************************
__gfx__
000000015210000000000000000000000000000000000000000000000000100000000000000000000001441100000000012810000000000000115c6661000000
0000001e5225210000000000000000000000010000000000000000000006dd510000000000000113ccbffffb100000012eee2000000015dcccccccc661100000
0000008e522222521000000000000000001d6d510000029420000000005fddddd51000003ccbbbbbbbbbffbb5000002eeeee210001dcccccccccccccc1110000
0000048e5222222225210000000000001dcccddd5100fa7444000000016fdddddddd5101bfbbbbbbbbbbbbbb5100008eeeee220001ccccccccccccccc1111000
0000488e52222222222252100000001dcccccddddd4aa77444200210056fddddddddddd5ffbbbbbbbbbbbbbc5500008eeeee221001ccccccccd5100110111100
0002888e522211222222222000001dccccc2015dd4aaa77444409a420d6fdddd55ddddd5fbbbbbbbcc3100015510005deeee222000cccc50000ddc66c0011100
0018888e5222000012222220001dccccc200100144aaaa744444a7444dffdddd00115dd5fbbb510000153cbf15100022eeee2220005ccc5dccccccccc1001100
0048888e5222002210001220016cccc2001dc51044aaaa7444449744456fdddd10000015fbbb33ccbbbbbbbb05100022eeee2220001dccccccccccccc1100100
0188888e522201222222100001ccc2001dcccd5544aaaa7444444a444166ddddddd51000fbbbbbbbbbbbbbbb51100022eeee22200011ccccccccccccc1110000
0188888e5222012222222220015dc11dcccccd5549aaaa744444494440dfdddddddddd51fbbbbbbbbbbbbbbb51100022eeee22200011ccccccccccccc1111000
0188888e52220122222222200055cdccccc2015544aaaaa4442444444056dddddddddd55fbbbbbcc333bbbbb55000022eeee22200011ccd510015cccc1111100
0188888e52221000122222200055ccccc200100144aaaaa444144444401fdddd0015dd55d3350000001bbbbb55100022eeee222000000001dcccccccc1111100
0188888e52222210000122200055ccc2001dc51044aaaaa444024444401ddddd100001111003ccbbbbbbbbbb55100022eeee22200005ccccccccccccc1111100
0188888e52222dc6fd5122200055c2001dcccd5549aaaaa4440144444015dddddd511000bfbbbbbbbbbbbbbb55100022eeee22200011ccccccccccccc1111100
0188888e5dc6fffff555d5200055c11dcccccd5544aaaaa4440024444005ddddddddd551ffbbbbbbfa7ebbbc55100022eeeedd200011cccb33333cccc1111100
018888e6f6fff6fff55555d10055cdcccccccd5544aaaaa444dcbbd44005ddddddddd555fbbbbfa7777443001510002d666666510011ccfb3333333335111100
018866666f6ff66ff5555555d15cbcccccc2101544aaaa955dcbbbdd4005dddd8888eee5cbba777777744450015dc6666666661111000ffb3333333333335100
0188666666666f66f555555553f553333210000049a9455dcccbbbddd588888888888ee21faaa7a77774444500dc66666666661111101fbb3333333333333100
0142666666d55666f55555553bf55333333351009455ddccccccbbdddd888888888888e229faaaaa77744444106666666666661111111bbc1533333333333000
000166d500001666655555533b653333333333301cccccccccccccdddd888888888888e2299a941aa7744444106666666666c51111110b500001333333333000
0000100000001666f55555533bb53330015333310dcccccd51ccccddd5888888888888822994100a7774444410dc666c55100000111100000005333300153000
0000000001d66666655555533bb333310011333105ccd51000ccccddd5422100018888822991049aa774444410ddc6100000000000110000000c333300000000
000001d666666666655555533bb53331005c333105cc100000ccccddd510000001888882299faaaaaa74444410ddd6000000000000000000000b333300000000
0001666666666666655555533bb3333354cc333105cc510000ccccdddd10000001888882299aaaaaaa74444410ddd6110001000000000000005b333300000000
00016666666d510000011553cbb333333333333105cc552000ccccdddd50000001888882299aaaaaaf95544410ddd611111100000000000000cb333300000000
000166661000000000000013bbb333333333333105ccd55200ccccddddd0000001888882299aa4100000000000ddd611111500000000000005bb333300000000
000166665510000000000003bbb333311333333105ccdd5521ccccddddd0001124888882299aa4410000000000ddd66666666651100000000cbb333300000000
00016666555510000000000cbbb33331053331000dccddd55dccccdddd88888888888882299aa7a777a4444410ddd66666666651111100001bbb333301000000
00016666555555100000000cbbb33331013333101cccccccccccccddd588888888888881199aaa7a7774444410ddd66666666651111100001bbb333300000000
00016666555555510000000cbbb33331005333501dccccccccccccddd1888888888ee400099aaaaaaa74444410ddd66666666651111100001bbb333301000000
00016666355555550000000cbbb333310013333005ccccccccccccdd5188888888ee2000099faaaaaaa4444410ddd66666666c51111100001bbb333301000000
00000011000000000000000001100000000000000011111000000000000000001110000000000001111000000000000000000000000000000011000000000000
__gff__
0000030301811303030301010000010003030101010103030101000000000000030303030301010101030100000000000000000000000303010101000000000020002000010100000000000000000000200020200000000000000000000000000000000000000000000000000000000008000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
__map__
__sfx__
010c00000c04318000003300c3300c0430c33000331000000c04300000000310c0300c04300000000310c0300c04318000003300c3300c0430c33000331000000c04300000000310c0300c04300000000310c030
010c0000184210c3211862118021246251802024720246250c111181212462500000246250c4211842124625184210c3211862118021246251802024720246250c111181112462500000246250c421183210c111
011000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
010c00002233022230223222232224331242301f3301f2321f3221f3221f3211332122320222201d3301d2301d3321d3321d3221d3221f3301f2301d3301d2321d322113211b3301b2301b3201b3221b3221b322
010c00000c5200e5100f52013510185201a5101b5201f510245201f5101b5201a51018520135100f5200e5100c5200e5100f52013510185201a5101b5201f510245201f5101b5201a51018520135100f5200e510
010c00000a5200e5100f52011510165201a5101b5201d510225201d5101b5201a51016520115100f5200e5100a5200e5100f52011510165201a5101b5201d510225201d5101b5201a51016520115100f5200e510
010c00001a3301a2301a3321a3321b3311b2301d3301d2321d3321d332223301d320223101d320223302223022332223322233222332223322233222332223322232222322223222232222322223221b3111a311
010c0000085200c5100e5200f51014520185101a5201b510205201b5101a5201851014520135100f5200e510085200c5100e5200f51014520185101a5201b510205201b5101a5201851014520135100f5200e510
010c00001a3301a2301a3321a3321b3311b2301d3301d2321d33216331183301a2201831016310163301623016330163321633216332163321633216332163321633216322163221632216322163221331016310
010c00000c0431800008330083300c0430833008331000000c0430000008031080300c0430000008031080300c0431800008330083300c0430833008331000000c0430000008031080300c0430a0000a03016030
010c00002633026230263112431127330272202633026232263322632227330272302732027210293302922229322293221b31229312273302723026330262322633226332273302722027322273222732227322
010c0000184210c3211862118021246251802024720246250c111181212462500000246250c421183210c121184210c3211862118021246251802024720246250c111181212462500000246250c4212462524625
010c0000184110c3111861118011246251801118011246250c111181112461500000246250c411183110c111184110c3111861118011246251801118011246250c111181112461500000246250c4112461524615
010c00000c04318000003300c3300c0430c33000331000000c04300000000310c0300c04300000000310c0300c04318000003000c3000c0430c30000301000000c04300000000010c0330c043000000c0330c030
010c0000263302623026332263222433024220263312622026332263322b3202b2202b3222b322263302623026320263222632226322263222632226322263222631226312263122631226312263122631226312
010c0000184110c3111861118011246251801118011246250c111181112461500000246250c411183110c111184110c3111861118011246251801118011246250c0331811124615000000c0330c411183110c111
010c00000c043180000a330163300c043163300a331000000c043000000a031160300c043000000a031160300c0431800008330143300c0431433008331000000c0430000008031140300c043000000803114030
010c00000c04318000033300f3300c0430f33003331000000c04300000030310f0300c04300000030310f0300c0430c00307330133300c04313330073310c0030c04300000070310c0330c043000000c03313030
010c00000a5200e5200f52011520165201a5201b5201d520225201d5201b5201a52016520115200f5200e520085200c5200e5200f52014520185201a5201b520205201b5201a5201852014520135200f5200e520
010c00000a5200f5101152013510165201b5101d5201f510225201f5101d5201b5101652013510115200f510075200c5200e520115201352017520185201a5201f5201a5201852017520135200e5200c5200b520
010c00002633026322263222631224330243102633026332263222632227330273202631227322293302932029322293222932229322293221d322273322732229332293222b3322b3222c3322b3122c3222c322
010c00002c3302c2202c3222c3122b3302b23229330292222b310292322733027322272222a3312b3312b3322b3322b3322b3222b3223721032220302302f2302b2302623024230232301f2201a2201822017220
010c00000c04318000003300c3300c0430c33000331000000c04300000000310c0300c04300000000310c0300c04318000003300c3300c0430c33000331000000c04300000000310c0330c043000000c0330c030
010c00000c04318000013300d3300c0430d33001331000000c04300000010310d0300c04300000010310d0300c0431800007330133300c0431333007331000000c0430000007031130300c043000000703113030
010c0000194210d3211962119021256251902025720256250d111191212562500000256250d42119421256251f421133211f6211f0212b6251f0202b7202b625131111f1112b625000002b625134211f32113111
010c00002a3302b2212b3222b22229330292222c3302c2202c3202c3222c3222c3322c3222c31224320183202432124327243272432723330232202633026220263202632226322263221f3301f3201f3221f322
000c0000243301f33018330243301f33018330243301f33018320243201f32018320243201f32018320243201f31018310243101f31018310243101f31018310243151f31518315243151f31518315243151f315
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
001000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
__music__
00 00014344
00 000b4645
00 00014347
00 0d0b4845
01 00010304
00 000b0605
00 00010307
00 000b0805
00 00010304
00 000b0605
00 00010307
00 160b0805
00 09010a04
00 000b0e04
00 100b1405
00 11011513
00 09010a04
00 000b0e04
00 17181944
02 16011a41
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344
00 41424344

