/*
 *
 * IdleWild 1.0
 * Copyright 2003 Ling Nero.  All Rights Reserved.
 *
 * IdleWild is a demonstration of:
 * 		- Programming Palm OS5 Notifications
 * 		- Saving, manipulating, and restoring the display window
 *
 * IdleWild has little usefulness, but it was fun to code. It's also
 * DOG SLOW, using only standard system calls, not accessing the
 * bitmaps directly, and not optimized in any way.  To rewrite 
 * with direct bit access, grab the BitmapRsrc.c from PalmSource
 * and go from there :-)
 *
 * Distributed under the GNU Public License (GPL).
 */

#include <PalmOS.h>
#include "idleWild.h"

#define ByteSwap16(n)	( ((((unsigned int) n) << 8) & 0xFF00) | \
						  ((((unsigned int) n) >> 8) & 0x00FF) )


#define myDeferredNotifyEvent MYCRID

#define BLOCK_WIDTH 40
#define NUM_STYLES 6
#define NUM_BLOCKS 2

static void Beep(UInt16, UInt16);
static Boolean Dance(Int16, WinHandle);
static Boolean MainFormHandler(EventType *);
static Boolean WrongOSHandler(EventType *);
static Boolean AppHandleEvent(EventType *);
static void SetupNotifications(void);
static void RegisterForNotifications(Boolean, Int16);
static void UnregisterForNotifications(void);
static void SetTrap(void);

typedef enum {
	random=0, fullDesaturate, desaturate, invert, multiply, difference, darken
} effectsType;

typedef enum {
	t5secs=0, t10secs, t15secs, t20secs, t30secs, t40secs, t50secs, t1min, t2mins
} autoOffType;

typedef struct {
	Boolean enable, demo;
	Int16 iteration, autoOff;
	effectsType effect;
	autoOffType autoOffIdx;
} prefType;

static void SetPreferences(prefType *);
static void SetPreferences(prefType *pref)
{
	UInt16 prefSize;

	prefSize = sizeof(prefType);
	PrefSetAppPreferences(MYCRID,1,1,pref,prefSize,false);
	return;
}

static void GetPreferences(prefType *);
static void GetPreferences(prefType *pref)
{
	UInt16 prefSize;

	prefSize = sizeof(prefType);
	if (PrefGetAppPreferences(MYCRID, 1, pref, &prefSize, false) == noPreferenceFound)
	{
		pref->enable = false;
		pref->demo = false;
		pref->iteration = 1;
		pref->effect = random;
		pref->autoOffIdx = t30secs;
		pref->autoOff = 30;
		SetPreferences(pref);
	}
	return;
}


/*
 * Save display
 * Call the pixel-pushing routine
 * Restore display
 */
static Boolean IdleWild(void);
static Boolean IdleWild(void)
{
	WinHandle Save,oldWin;
  	UInt16 error;
	Int16 i;
	RectangleType screen;
	Boolean done;
	prefType pref;

	oldWin=WinSetDrawWindow(WinGetDisplayWindow());				// Set draw window to display
  	WinGetWindowFrameRect(WinGetDisplayWindow(), &screen);
  	Save=WinSaveBits(&screen, &error);							// Save copy of display

	WinPushDrawState();

	WinSetDrawMode(winPaint);

	done=true;
	GetPreferences(&pref);

	// If in demo mode, iterate over all the styles
	if (pref.demo)
	{
		for (i=1; i<=NUM_STYLES; i++)
		{
			if (!Dance(i, Save))
			{
				done=false;
				goto cleanup;
			}
  			WinRestoreBits(Save,screen.topLeft.x, screen.topLeft.y);
  			Save=WinSaveBits(&screen, &error);							// Save copy of display
		}
	}
	else // otherwise, run 'effect' for 'iteration' times
	{
		for (i=0; i<=pref.iteration; i++)
		{
			if (!Dance(pref.effect, Save))
			{
				done=false;
				goto cleanup;
			}
  			WinRestoreBits(Save,screen.topLeft.x, screen.topLeft.y);
  			Save=WinSaveBits(&screen, &error);							// Save copy of display
		}
	}

cleanup:

	WinPopDrawState();											// Restore draw state, display, draw window
  	WinRestoreBits(Save,screen.topLeft.x, screen.topLeft.y);
  	WinSetDrawWindow(oldWin);

	return done;
}


/*
 * The main routine to push display pixels around.  Returns false if interrupted.  If interrupted,
 * it means the users wants to halt auto off.
 * 
 * Styles:
 * 	0 - Desaturate the blocks to grayscale
 * 	1 - Desaturate only the intersection
 * 	2 - Invert the intersection
 * 	3 - Multiply the RGB values of the intersection
 * 	4 - Difference the RGB values of the intersection
 * 	5 - Darken the RGB values of the intersection
 */

Boolean Dance(Int16 style, WinHandle bgH)
{
	Int16 i, i2, i3, j, k, x,y, startX, toX, toY, xEnd, yEnd;
	Int16 blocksDeltaX[NUM_BLOCKS], blocksDeltaY[NUM_BLOCKS];
	Int16 x1Offset, y1Offset, x2Offset, y2Offset, overlapXOffset, overlapYOffset, scaleFactor;
	Coord extentX, extentY;
	RectangleType block;
	RectangleType blocks[NUM_BLOCKS], blocksNow[NUM_BLOCKS], blocksOverlap;
	UInt16 error, oldCoordSys;
	WinHandle blocksH[NUM_BLOCKS], bufferH;
	RGBColorType rgb, prevRgb, rgb1, rgb2;
	UInt32 attr;
	Boolean finished;
	EventType event;

	bufferH = NULL;
	for (i=0; i<NUM_BLOCKS; i++)
		blocksH[i] = NULL;

	WinGetDisplayExtent(&extentX, &extentY);					// Mindful of irregular sizes

	finished=true;

	// Create some random blocks
	for (i=0,startX=extentX; i<NUM_BLOCKS; i++)
	{
		RctSetRectangle(&blocks[i], SysRandom(0) % (extentX-BLOCK_WIDTH),
				(SysRandom(0) % ((extentY>>1)-BLOCK_WIDTH)) + ((extentY>>1)*(i%2)),
				BLOCK_WIDTH, BLOCK_WIDTH);
		RctCopyRectangle(&blocks[i],&blocksNow[i]);

		// Determine direction to travel
		j = SysRandom(0)%3 + 1;
		blocksDeltaX[i] = blocks[i].topLeft.x < extentX>>1 ? j : -j;
		j = SysRandom(0)%3 + 1;
		blocksDeltaY[i] = blocks[i].topLeft.y < extentY>>1 ? j : -j;

		// Find the left most block position
		if (startX>blocks[i].topLeft.x)
			startX = blocks[i].topLeft.x;
	}

	// Save copies of the movable blocks
	for (i=0; i<NUM_BLOCKS; i++)
	{
		blocksH[i] = WinCreateOffscreenWindow(BLOCK_WIDTH, BLOCK_WIDTH, nativeFormat, &error);
		if (!blocksH[i])
			goto cleanup;

		WinCopyRectangle(NULL, blocksH[i], &blocks[i], 0, 0, winPaint);
		WinSetDrawWindow(blocksH[i]);
		// Frame with whatever is the foreground color
		WinDrawLine(0, 0, 0, blocks[i].extent.y-1);
		WinDrawLine(0, 0, blocks[i].extent.x-1, 0);
		WinDrawLine(0, blocks[i].extent.y-1, blocks[i].extent.x-1, blocks[i].extent.y-1);
		WinDrawLine(blocks[i].extent.x-1, 0, blocks[i].extent.x-1, blocks[i].extent.y-1);
		WinSetDrawWindow(WinGetDisplayWindow());

	}

	// Save a copy of the display for double buffering
	RctSetRectangle(&block, 0, 0, extentX, extentY);
	bufferH = WinCreateOffscreenWindow(extentX, extentY, nativeFormat, &error);
	if (!bufferH)
		goto cleanup;
	WinCopyRectangle(NULL, bufferH, &block, 0, 0, winPaint);

	// Want to work in screen native mode so must know the screen density
	WinScreenGetAttribute(winScreenDensity, &attr);
	if (attr == kDensityDouble) scaleFactor = 2;
	else if (attr == kDensityOneAndAHalf) scaleFactor = 1.5;	// it doesn't really work for 1.5, oh well...
	else scaleFactor = 1;

	if (style == random)
		style = SysRandom(0)%NUM_STYLES + 1;
			
	if (style == fullDesaturate) // desaturate the blocks
	{
		xEnd = BLOCK_WIDTH * scaleFactor;
		oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
		WinSetForeColorRGB(NULL, &prevRgb);
		for (i=0; i<NUM_BLOCKS; i++)
		{
			WinSetDrawWindow(blocksH[i]);
			for (j=0; j<xEnd; j++)
				for (k=0; k<xEnd; k++)
				{
					WinGetPixelRGB(j,k,&rgb);
					rgb.r=(76*rgb.r+150*rgb.g+29*rgb.b)/256;
					rgb.g = rgb.b = rgb.r;
					WinSetForeColorRGB(&rgb, NULL);
					WinPaintPixel(j,k);
				}
		}
		WinSetDrawWindow(WinGetDisplayWindow());
		WinSetForeColorRGB(&prevRgb, NULL);
		WinSetCoordinateSystem(oldCoordSys);
	}


	for (i=startX; i<extentX - BLOCK_WIDTH; i++)
	{
		// Move the blocks in the offscreen buffer
		for (i2=0; i2<NUM_BLOCKS; i2++)
		{
			toX = blocksNow[i2].topLeft.x + blocksDeltaX[i2];
			toY = blocksNow[i2].topLeft.y + blocksDeltaY[i2];
			RctSetRectangle(&block, 0, 0, BLOCK_WIDTH, BLOCK_WIDTH);
			if (toX < 0)
			{
				blocksDeltaX[i2] -= toX;
				toX = 0;
			}
			if (toY < 0)
			{
				blocksDeltaY[i2] -= toY;
				toY = 0;
			}
			if (toX+BLOCK_WIDTH > extentX)
			{
				blocksDeltaX[i2] -= (toX+BLOCK_WIDTH - extentX);
				toX = extentX - BLOCK_WIDTH;
			}
			if (toY+BLOCK_WIDTH > extentY)
			{
				blocksDeltaY[i2] -= (toY+BLOCK_WIDTH - extentY);
				toY = extentY - BLOCK_WIDTH;
			}
			WinCopyRectangle(blocksH[i2], bufferH, &block, toX, toY, winPaint);
			blocksNow[i2].topLeft.x = toX;
			blocksNow[i2].topLeft.y = toY;
		}

		switch (style)
		{
			case fullDesaturate: break;
			default:
				// Find intersections
				for (i2=0; i2<NUM_BLOCKS; i2++)
					for (i3=0; i3<i2; i3++)
					{
						RctGetIntersection(&blocksNow[i3], &blocksNow[i2], &blocksOverlap);
						if (blocksOverlap.extent.x != 0 && blocksOverlap.extent.y != 0)
						{

							xEnd = blocksOverlap.extent.x * scaleFactor;
							yEnd = blocksOverlap.extent.y * scaleFactor;

							WinSetDrawMode(winPaint);
							oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
							x1Offset = (blocksOverlap.topLeft.x - blocksNow[i3].topLeft.x)*scaleFactor;
							y1Offset = (blocksOverlap.topLeft.y - blocksNow[i3].topLeft.y)*scaleFactor;
							x2Offset = (blocksOverlap.topLeft.x - blocksNow[i2].topLeft.x)*scaleFactor;
							y2Offset = (blocksOverlap.topLeft.y - blocksNow[i2].topLeft.y)*scaleFactor;
							overlapXOffset = blocksOverlap.topLeft.x*scaleFactor;
							overlapYOffset = blocksOverlap.topLeft.y*scaleFactor;

							// apply operation to the intersection
							for (x=0; x<xEnd; x++)
								for (y=0; y<yEnd; y++)
								{
									WinSetDrawWindow(blocksH[i3]);
									WinGetPixelRGB(x1Offset+x, y1Offset+y, &rgb1);
									WinSetDrawWindow(blocksH[i2]);
									WinGetPixelRGB(x2Offset+x, y2Offset+y, &rgb2);

									switch (style)
									{
										case desaturate : // desaturate
											if (rgb1.r != 255 || rgb1.g != 255 || rgb1.b != 255) // not white
											{
												WinSetDrawWindow(bufferH);
												rgb.r=(76*rgb1.r+150*rgb1.g+29*rgb1.b)>>8;
												rgb.g = rgb.b = rgb.r;
												WinSetForeColorRGB(&rgb, NULL);
												WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											}

											if (rgb2.r != 255 || rgb2.g != 255 || rgb2.b != 255) // not white
											{
												WinSetDrawWindow(bufferH);
												rgb.r=(76*rgb2.r+150*rgb2.g+29*rgb2.b)>>8;
												rgb.g = rgb.b = rgb.r;
												WinSetForeColorRGB(&rgb, NULL);
												WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											}
											break;
										
										case invert: // Invert
											rgb.r = (rgb1.r ^ rgb2.r);
											rgb.g = (rgb1.g ^ rgb2.g);
											rgb.b = (rgb1.b ^ rgb2.b);
											WinSetDrawWindow(bufferH);
											WinSetForeColorRGB(&rgb, NULL);
											WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											break;

										case multiply: // multiply
											rgb.r = (rgb1.r * rgb2.r)>>8;
											rgb.g = (rgb1.g * rgb2.g)>>8;
											rgb.b = (rgb1.b * rgb2.b)>>8;
											WinSetDrawWindow(bufferH);
											WinSetForeColorRGB(&rgb, NULL);
											WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											break;

										case difference: // difference
											rgb.r = abs(rgb1.r - rgb2.r);
											rgb.b = abs(rgb1.b - rgb2.b);
											rgb.g = abs(rgb1.g - rgb2.g);
											WinSetDrawWindow(bufferH);
											WinSetForeColorRGB(&rgb, NULL);
											WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											break;

										case darken:
											if (rgb1.r != 255 || rgb1.g != 255 || rgb1.b != 255) // not white
											{
												WinSetDrawWindow(bufferH);
												rgb.r = (rgb1.r * rgb1.r)>>8;
												rgb.b = (rgb1.b * rgb1.b)>>8;
												rgb.g = (rgb1.g * rgb1.g)>>8;
												WinSetForeColorRGB(&rgb, NULL);
												WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											}

											if (rgb2.r != 255 || rgb2.g != 255 || rgb2.b != 255) // not white
											{
												WinSetDrawWindow(bufferH);
												rgb.r = (rgb2.r * rgb2.r)>>8;
												rgb.b = (rgb2.b * rgb2.b)>>8;
												rgb.g = (rgb2.g * rgb2.g)>>8;
												WinSetForeColorRGB(&rgb, NULL);
												WinPaintPixel(overlapXOffset+x,overlapYOffset+y);
											}
											break;

										default:
											break;
									}
								}
								
							WinSetDrawWindow(WinGetDisplayWindow());
							WinSetCoordinateSystem(oldCoordSys);
						} // if intersect

					}
					break;
					
		} // switch

		// Write to display from offscreen buffer
		for (i2 = 0; i2<NUM_BLOCKS; i2++)
		{

			WinCopyRectangle(bufferH, NULL, &blocksNow[i2], blocksNow[i2].topLeft.x,
					blocksNow[i2].topLeft.y, winPaint);

			// Restore background -- this is such ugly but time is short...
			if (blocksDeltaX[i2] > 0) // Moved to right
			{
				RctSetRectangle(&block, blocksNow[i2].topLeft.x-blocksDeltaX[i2],
							blocksNow[i2].topLeft.y-blocksDeltaY[i2],
							blocksDeltaX[i2], blocksNow[i2].extent.y);  // vertical
				WinCopyRectangle(bgH, NULL, &block, block.topLeft.x, block.topLeft.y, winPaint);
				WinCopyRectangle(bgH, bufferH, &block, block.topLeft.x, block.topLeft.y, winPaint);
				if (blocksDeltaY[i2] < 0)
					block.topLeft.y = blocksNow[i2].topLeft.y+blocksNow[i2].extent.y;
				block.extent.y = abs(blocksDeltaY[i2]);
				block.extent.x = blocksNow[i2].extent.x;	// horizontal
				WinCopyRectangle(bgH, NULL, &block, block.topLeft.x, block.topLeft.y, winPaint);
				WinCopyRectangle(bgH, bufferH, &block, block.topLeft.x, block.topLeft.y, winPaint);
			}
			else
			{ // Moved to left
				RctSetRectangle(&block, blocksNow[i2].topLeft.x+BLOCK_WIDTH,
							blocksNow[i2].topLeft.y-blocksDeltaY[i2],
							-blocksDeltaX[i2], blocksNow[i2].extent.y); // vertical
				WinCopyRectangle(bgH, NULL, &block, block.topLeft.x, block.topLeft.y, winPaint);
				WinCopyRectangle(bgH, bufferH, &block, block.topLeft.x, block.topLeft.y, winPaint);
				block.topLeft.x = blocksNow[i2].topLeft.x-blocksDeltaX[i2];
				if (blocksDeltaY[i2] < 0)
					block.topLeft.y = blocksNow[i2].topLeft.y+blocksNow[i2].extent.y;
				block.extent.y = abs(blocksDeltaY[i2]);
				block.extent.x = blocksNow[i2].extent.x-blocksDeltaX[i2];	// horizontal
				WinCopyRectangle(bgH, NULL, &block, block.topLeft.x, block.topLeft.y, winPaint);
				WinCopyRectangle(bgH, bufferH, &block, block.topLeft.x, block.topLeft.y, winPaint);
			}

			// stop the animation and return when user taps on the screen or press hard key
			EvtGetEvent(&event, 1);
			if (event.eType == penUpEvent || KeyCurrentState())
			{
				if (event.eType == keyDownEvent)
					event.data.keyDown.chr = NULL;
				EvtFlushPenQueue();
				EvtFlushKeyQueue();
				finished = false;
				goto cleanup;
			}

		} // for i2
	} // for i=startX

cleanup:
	// clean up memory
	for (i=0; i<NUM_BLOCKS; i++)
		if (blocksH[i])
			WinDeleteWindow(blocksH[i], false);

	if (bufferH)
		WinDeleteWindow(bufferH, false);
			
	return finished;
}


static Boolean MainFormHandler(EventType *eventP)
{
	Boolean handled;
	FormType *frmP;
	ListType *listP;
	ControlType *popTrig;
	Char *label;
	prefType pref;

	handled=false;

	GetPreferences(&pref);

	switch (eventP->eType)
	{
		case frmOpenEvent:
			frmP = FrmGetActiveForm();
			CtlSetValue(FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,checkboxID_Enable)), pref.enable);
			CtlSetValue(FrmGetObjectPtr(frmP,FrmGetObjectIndex(frmP,checkboxID_Demo)), pref.demo);

			listP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, listID_Iteration));
			LstSetSelection(listP, pref.iteration);
			popTrig = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, popID_Iteration));
			label = (Char *) CtlGetLabel(popTrig);
			StrCopy(label, LstGetSelectionText(listP,LstGetSelection(listP)));
			CtlSetLabel(popTrig, label);

			listP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, listID_Effects));
			LstSetSelection(listP, pref.effect);
			popTrig = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, popID_Effects));
			label = (Char *) CtlGetLabel(popTrig);
			StrCopy(label, LstGetSelectionText(listP,LstGetSelection(listP)));
			CtlSetLabel(popTrig, label);

			listP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, listID_AutoOff));
			LstSetSelection(listP, pref.autoOffIdx);
			popTrig = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, popID_AutoOff));
			label = (Char *) CtlGetLabel(popTrig);
			StrCopy(label, LstGetSelectionText(listP,LstGetSelection(listP)));
			CtlSetLabel(popTrig, label);

			FrmDrawForm(frmP);
			handled = true;
			break;

		case menuEvent:
			switch (eventP->data.menu.itemID)
			{
				case menuItemID_About:
					frmP=FrmInitForm(3000);
					FrmDoDialog(frmP);
					FrmDeleteForm(frmP);
					handled=true;
					break;

				default: break;
			}
			break;

		case frmCloseEvent:
			SetTrap();
			break;

		default: break;
	}
	return handled;
}

static Boolean AppHandleEvent(EventType *eventP)
{
	UInt16 formID;
	FormType *frmP;

	if (eventP->eType == frmLoadEvent)
	{
		formID = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formID);
		FrmSetActiveForm(frmP);

		switch (formID)
		{
			case MainFormID:
				FrmSetEventHandler(frmP,(FormEventHandlerType *)MainFormHandler);
				break;

			case WrongOSFormID:
				FrmSetEventHandler(frmP, (FormEventHandlerType *)WrongOSHandler);
				break;

			default: ErrFatalDisplay("Invalid Form Load Event");
					 break;
		}
		
		return true;
	}
	return false;
}

/*
 * Standard event loop -- everyone's got the same one to bring to the party
 */

static void EventLoop(void)
{
	UInt16 error;
	EventType event;

	do
	{
		EvtGetEvent(&event, evtWaitForever);
		
		if (SysHandleEvent(&event)) continue;
		if (MenuHandleEvent(0,&event,&error)) continue;
		if (AppHandleEvent(&event)) continue;
		FrmDispatchEvent(&event);
	}
	while (event.eType != appStopEvent);
}

static void Beep(UInt16 times,UInt16 type)
{
	UInt16 i;

	for (i=0; i<times; i++)
		SndPlaySystemSound(type);
}

static void StopApplication(void)
{
	Err err;
	UInt32 version;

	FrmCloseAllForms();			// Otherwise you'd get memory leaks

	// Restore DIA if present
	err = FtrGet(pinCreator, pinFtrAPIVersion, &version);
	if (!err && version)
		PINSetInputAreaState(pinInputAreaUser);

	return;
}

static void SetupNotifications(void)
{
	prefType pref;

	GetPreferences(&pref);
	if (pref.enable)
   		RegisterForNotifications(pref.demo, pref.autoOff);
	else UnregisterForNotifications();

	return;
}


/*
 * Patch system using Notifications
 */

static void RegisterForNotifications(Boolean demo, Int16 autoOff)
{
    UInt16 cardNo;
    LocalID dbID;
	UInt32 temp;

   	if (SysCurAppDatabase(&cardNo, &dbID) != 0)
   		return; // shouldn't ever fail, but just in case.

	if (FtrGet(MYCRID,FTR_PROTECT,&temp) != 0) // protect database if not already
	{
		DmDatabaseProtect(cardNo, dbID, true);
		FtrSet(MYCRID,FTR_PROTECT,1);
	}
	
	// On the Clie OS5 Simulator I couldn't get it to work with sysNotifySleep* notifications
	// Since I don't have a real OS5 Clie to test if the Simulator is to blame, I opted to
	// use another way on Clie OS5 units.  A slower method.
	FtrGet(sysFtrCreator, sysFtrNumOEMCompanyID, &temp);
	if (temp == 'sony')
   		SysNotifyRegister(cardNo, dbID, sysNotifyEventDequeuedEvent, NULL, sysNotifyNormalPriority, 0);
	else
	{

   		SysNotifyRegister(cardNo, dbID, sysNotifySleepRequestEvent, NULL, sysNotifyNormalPriority, 0);
  		SysNotifyRegister(cardNo, dbID, sysNotifySleepNotifyEvent, NULL, sysNotifyNormalPriority, 0);
		SysNotifyRegister(cardNo, dbID, myDeferredNotifyEvent, NULL, sysNotifyNormalPriority, 0);
	}

   	SysNotifyRegister(cardNo, dbID, sysNotifyDeleteProtectedEvent, NULL, sysNotifyNormalPriority, 0);
	SysNotifyRegister(cardNo, dbID, sysNotifySyncStartEvent, NULL, sysNotifyNormalPriority, 0);

	if (demo)
		SysSetAutoOffTime(5);
	else SysSetAutoOffTime(autoOff);

	return;

}

static void UnregisterForNotifications(void)
{
    UInt16 cardNo;
    LocalID dbID;
	UInt32 temp;

   	if (SysCurAppDatabase(&cardNo, &dbID) != 0)
   		return; // shouldn't ever fail, but just in case.

	if (FtrGet(MYCRID,FTR_PROTECT,&temp) == 0) // unprotect database if necessary
	{
		DmDatabaseProtect(cardNo, dbID, false);
		FtrUnregister(MYCRID,FTR_PROTECT);
	}

	FtrGet(sysFtrCreator, sysFtrNumOEMCompanyID, &temp);
	if (temp == 'sony')
		SysNotifyUnregister(cardNo, dbID, sysNotifyEventDequeuedEvent, sysNotifyNormalPriority);
	else
	{
		SysNotifyUnregister(cardNo, dbID, sysNotifySleepRequestEvent, sysNotifyNormalPriority);
		SysNotifyUnregister(cardNo, dbID, sysNotifySleepNotifyEvent, sysNotifyNormalPriority);
		SysNotifyUnregister(cardNo, dbID, myDeferredNotifyEvent, sysNotifyNormalPriority);
	}

	SysNotifyUnregister(cardNo, dbID, sysNotifySyncStartEvent, sysNotifyNormalPriority);
	SysNotifyUnregister(cardNo, dbID, sysNotifyDeleteProtectedEvent, sysNotifyNormalPriority);
	
	// Reset AutoOff to system parameter
	SysSetAutoOffTime(PrefGetPreference(prefAutoOffDurationSecs));

	return;
}

static void SetTrap(void)
{
	FormType *frmP;
	prefType pref;
	Int16 autoOff[] = {5, 10, 15, 20, 30, 40, 50, 60, 120};

	frmP = FrmGetActiveForm();

	GetPreferences(&pref);

	pref.enable = CtlGetValue(FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP,checkboxID_Enable)));
	pref.demo = CtlGetValue(FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP,checkboxID_Demo)));
	pref.iteration = LstGetSelection(FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP,listID_Iteration)));
	pref.effect = LstGetSelection(FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP,listID_Effects)));
	pref.autoOffIdx = LstGetSelection(FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP,listID_AutoOff)));

	pref.autoOff=autoOff[pref.autoOffIdx];

	if (pref.enable)
		RegisterForNotifications(pref.demo,pref.autoOff);
	else UnregisterForNotifications();

	SetPreferences(&pref);

	return;
}

static UInt16 StartApplication(void)
{
	UInt32 romVersion;

	// Determine operating environment

	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (sysGetROMVerMajor(romVersion) < 5)
		FrmGotoForm(WrongOSFormID);
	else
		FrmGotoForm(MainFormID);
	return 0;
}

UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
	UInt16 error,secs, cardNo;
	SysNotifyParamType *notify;
	EventType newEvent;
	EventPtr eventP;
    LocalID dbID;
	UInt32 temp;
	SleepEventParamType *sleep;
	SysNotifyParamType notifyParam;

   	if (SysCurAppDatabase(&cardNo, &dbID) != 0)
   		return 0; // shouldn't ever fail, but just in case.

	switch (cmd)
	{
        case sysAppLaunchCmdNormalLaunch: 
			error = StartApplication();
			if (error) return error;
			EventLoop();
			StopApplication();
			break;
			
		case sysAppLaunchCmdNotify:
			notify = (SysNotifyParamType *)cmdPBP;
			notify->handled = false;

			switch (notify->notifyType)
			{
				case sysNotifyEventDequeuedEvent: // Only on Clie OS5 devices :-(
    				if (FtrGet(MYCRID, FTR_SLEEP, &temp) != 0)
					{
						eventP = (EventPtr) (notify->notifyDetailsP);
						if (eventP->eType == ByteSwap16(keyDownEvent)
								&& eventP->data.keyDown.chr == ByteSwap16(vchrAutoOff))
						{
							secs=SysSetAutoOffTime(0);
							FtrSet(MYCRID, FTR_SLEEP, 1);
							if (!IdleWild())
								eventP->data.keyDown.chr = ByteSwap16(vchrResetAutoOff);
							SysSetAutoOffTime(secs);
							FtrUnregister(MYCRID, FTR_SLEEP);
						}
					}
					break;

				case sysNotifySleepRequestEvent:
				//	Beep(1,sndAlarm);
    				if (FtrGet(MYCRID, FTR_SLEEP, &temp) != 0)
					{
						sleep = (SleepEventParamType *) notify->notifyDetailsP;
						if (sleep->reason == sysSleepAutoOff)
						{
							sleep->deferSleep++;
							notifyParam.notifyType = myDeferredNotifyEvent;
							notifyParam.broadcaster = MYCRID;
							notifyParam.notifyDetailsP = NULL;
							notifyParam.handled = false;
							SysNotifyBroadcastDeferred(&notifyParam,0);
						}
					}
					break;

				case sysNotifySleepNotifyEvent:
				//	Beep(2,sndAlarm);
					FtrUnregister(MYCRID, FTR_SLEEP);
					break;

				case myDeferredNotifyEvent:
    				if (FtrGet(MYCRID, FTR_SLEEP, &temp) != 0)
					{
						secs=SysSetAutoOffTime(0);
						FtrSet(MYCRID, FTR_SLEEP, 1);
						if (IdleWild())
						{
							newEvent.eType = keyDownEvent;
							newEvent.data.keyDown.chr = resumeSleepChr;
							newEvent.data.keyDown.modifiers = commandKeyMask;
							EvtAddEventToQueue(&newEvent);
						}
						else FtrUnregister(MYCRID, FTR_SLEEP);
						SysSetAutoOffTime(secs);
					}
					notify->handled=true;
					break;

				case sysNotifySyncStartEvent:
					UnregisterForNotifications();
    				if (SysCurAppDatabase(&cardNo, &dbID) == 0)
						SysNotifyRegister(cardNo, dbID, sysNotifySyncFinishEvent, NULL, sysNotifyNormalPriority, 0);
					break; 

				case sysNotifySyncFinishEvent:
					SetupNotifications();
    				if (SysCurAppDatabase(&cardNo, &dbID) == 0)
						SysNotifyUnregister(cardNo, dbID, sysNotifySyncFinishEvent, sysNotifyNormalPriority);
					break;

				case sysNotifyDeleteProtectedEvent:
					UnregisterForNotifications();
					break;


				default: break;
			}
			break;

		case sysAppLaunchCmdSystemReset:
      			SetupNotifications();
			break;

		default: break;

	}
	return 0;
}

static Boolean WrongOSHandler(EventPtr eventP)
{
	Boolean handled=false;
	FormType *frmP;

	switch (eventP->eType)
	{
		case frmOpenEvent:
			frmP=FrmGetActiveForm();
			FrmDrawForm(frmP);
			handled=true;
			break;

		default:
			break;
	}

	return handled;
}
