#include "main.h"

u8 __gba_multiboot;
u16 bgColor[160];

#define MAX_STRINGS 8
#define STRINGNODES 16
#define GRAVITY int2fp(9)

typedef struct tagVector2
{
	s32 x, y;
} Vector2, *pVector2;

typedef struct tagString
{
	Vector2 node[STRINGNODES];
	s16 hVel[STRINGNODES], vVel[STRINGNODES];
	s16 hAccel[STRINGNODES], vAccel[STRINGNODES];
	s16 mass, stretch, damping;
	u8 nodeCount, oamIdx, objRadius;
} String, *pString;

s32 Vector2Magnitude(Vector2 vector) 
{
	return (sqrt(((vector).x * (vector).x) + ((vector).y * (vector).y)));
}

void DrawLine8(u16 *dest, s32 x1, s32 y1, s32 x2, s32 y2, u16 color)
{
	s32 xd, yd, slope, temp;
	u16 *pPixel;

	color = color | (color << 8);
	xd = int2fp(x2 - x1);
	yd = int2fp(y2 - y1);

	if(abs(xd) <= abs(yd))
	{
		slope = fpDiv(xd, yd);
		if(y2 < y1)
		{
			temp = y1 * 120;	//swap x and y, multiply y by screen 
			y1 = y2 * 120;		//width (in halfwords), and convert x 
			y2 = temp;			//to fixed point all at the same time
			x1 = int2fp(x2);
		}
		else
		{
			y1 *= 120;
			y2 *= 120;
			x1 = int2fp(x1);
		}
		while(y1 < y2)
		{
			x2 = fp2int(x1);
			if(x2 >= 0 && x2 < 240 && y1 >= 0 && y1 < 0x4b00)
			{
				pPixel = &(dest[y1 + (x2 >> 1)]);
				*pPixel = color;
				*(pPixel + 120) = color;
			}
			y1 += 120;
			x1 += slope;
		}
	}
	else
	{
		slope = fpDiv(yd, xd);
		if(x2 < x1)
		{
			temp = x1;
			x1 = x2;
			x2 = temp;
			y1 = int2fp(y2);
		}
		else
		{
			y1 = int2fp(y1);
		}
		while(x1 < x2)
		{
			y2 = fp2int(y1);
			if(x1 >= 0 && x1 < 240 && y2 >= 0 && y2 < 160)
			{
				pPixel = &(dest[(y2 * 120) + (x1 >> 1)]);
				*pPixel = color;
				*(pPixel + 120) = color;
			}
			x1++;
			y1 += slope;
		}
	}
}

void StringInit(pString string)
{
	u16 i;

	for(i = 0; i < string->nodeCount; i++)
	{
		string->node[i].x = int2fp(50 * (string->oamIdx + 1) + i * 5);
		string->node[i].y = i * string->stretch;
	}
	oamShadow[string->oamIdx].attribute0 = 0;
	oamShadow[string->oamIdx].attribute1 = SPRITE_SIZE_16;
}

void StringUpdate(pString string, pString *stringSet)
{
	u16 i;
	Vector2 v1, v2;
	s32 mag1, mag2, ext1, ext2, gravity, lastNode;

	gravity = fpMul(GRAVITY, string->mass);
	lastNode = string->nodeCount - 1;
	for(i = 1; i < lastNode; i++)
	{
		v1.x = string->node[i - 1].x - string->node[i].x;
		v1.y = string->node[i - 1].y - string->node[i].y;
		mag1 = Vector2Magnitude(v1);
		ext1 = mag1 - string->stretch;

		v2.x = string->node[i + 1].x - string->node[i].x;
		v2.y = string->node[i + 1].y - string->node[i].y;
		mag2 = Vector2Magnitude(v2);
		ext2 = mag2 - string->stretch;

		if(mag1 < 16 || mag2 < 16)
		{
			string->hVel[i] = 0;
			string->vVel[i] = 0;
		}

		string->hAccel[i] = fpMul(fpDiv(v1.x, mag1), ext1) + 
			fpMul(fpDiv(v2.x, mag2), ext2);
		string->vAccel[i] = fpMul(fpDiv(v1.y, mag1), ext1) + 
			fpMul(fpDiv(v2.y, mag2), ext2) + gravity;
	}
	v1.x = string->node[lastNode - 1].x - 
		string->node[lastNode].x;
	v1.y = string->node[lastNode - 1].y - 
		string->node[lastNode].y;
	mag1 = Vector2Magnitude(v1);
	ext1 = mag1 - string->stretch;
	string->hAccel[lastNode] = 
		(div(v1.x << FP, mag1) * ext1 >> FP);
	string->vAccel[lastNode] = 
		(div(v1.y << FP, mag1) * ext1 >> FP) + 
		fpMul(GRAVITY, string->mass);

	for(i = 1; i < string->nodeCount; i++)
	{
		string->hVel[i] = fpMul(string->hVel[i], string->damping) + 
			string->hAccel[i];
		string->vVel[i] = fpMul(string->vVel[i], string->damping) + 
			string->vAccel[i];
		string->node[i].x += string->hVel[i] >> 3;
		string->node[i].y += string->vVel[i] >> 3;
	}

	if(fp2int(string->node[lastNode].x) - string->objRadius < 0)
	{
		string->hVel[lastNode] = 0;
		string->node[lastNode].x = int2fp(string->objRadius);
	}
	else if(fp2int(string->node[lastNode].x) + string->objRadius >= 240)
	{
		string->hVel[lastNode] = 0;
		string->node[lastNode].x = int2fp(240 - string->objRadius);
	}
	if(fp2int(string->node[lastNode].y) - string->objRadius < 0)
	{
		string->vVel[lastNode] = 0;
		string->node[lastNode].y = int2fp(string->objRadius);
	}
	else if(fp2int(string->node[lastNode].y) + string->objRadius >= 160)
	{
		string->vVel[lastNode] = 0;
		string->node[lastNode].y = int2fp(160 - string->objRadius);
	}

	for(i = 0; i < MAX_STRINGS; i++)
	{
		if(stringSet[i] && (stringSet[i] != string))
		{
			ext1 = stringSet[i]->nodeCount - 1;	//use ext1 as temp variable

			v1.x = string->node[lastNode].x - 
				stringSet[i]->node[ext1].x;
			v1.y = string->node[lastNode].y - 
				stringSet[i]->node[ext1].y;
			mag1 = v1.x * v1.x + v1.y * v1.y;
			mag2 = int2fp(string->objRadius + stringSet[i]->objRadius);
			mag2 *= mag2;	//saves a sqrt
			if(mag1 < mag2)
			{
				//if collided, back up to previous position and swap velocities
				string->node[lastNode].x -= string->hVel[lastNode] >> 3;
				string->node[lastNode].y -= string->vVel[lastNode] >> 3;
				stringSet[i]->node[ext1].x -= stringSet[i]->hVel[ext1] >> 3;
				stringSet[i]->node[ext1].y -= stringSet[i]->vVel[ext1] >> 3;

				mag1 = string->hVel[lastNode];
				mag2 = string->vVel[lastNode];
				string->hVel[lastNode] = (stringSet[i]->hVel[ext1]);
				string->vVel[lastNode] = (stringSet[i]->vVel[ext1]);
				stringSet[i]->hVel[ext1] = (mag1);
				stringSet[i]->vVel[ext1] = (mag2);
			}
		}
	}
}

void StringRender(pString string, u16 *buf)
{
	u16 i;
	pOamEntry pOam;

	for(i = 1; i < string->nodeCount; i++)
	{
		DrawLine8(buf, fp2int(string->node[i - 1].x), 
			fp2int(string->node[i - 1].y), fp2int(string->node[i].x), 
			fp2int(string->node[i].y), 1);
	}
	pOam = &(oamShadow[string->oamIdx]);
	pOam->attribute1 = 
		(pOam->attribute1 & ~0x1ff) | 
		(((string->node[string->nodeCount - 1].x >> FP) - 
		(string->objRadius)) & 0x1ff);
	pOam->attribute0 = 
		(pOam->attribute0 & ~0xff) | 
		(((string->node[string->nodeCount - 1].y >> FP) - 
		(string->objRadius)) & 0xff);
}

void CalcBgGradient()
{
	u16 i;
	s32 r, g, b, ri, gi, bi;

	r = 0x000a0000;
	g = 0x001b0000;
	b = 0x001f0000;
	ri = 0xfffff334;
	gi = 0xffffe19a;
	bi = 0xffffde67;

	for(i = 0; i < 160; i++)
	{
		r += ri;
		g += gi;
		b += bi;
		bgColor[i] = RGB((u16)(r >> 16), (u16)(g >> 16), (u16)(b >> 16));
	}
}

int AgbMain()
{
	u16 i, *curBuf;
	String string[MAX_STRINGS], *stringSet[MAX_STRINGS];
	OamEntry oamTemp;

	REG_DISPCNT = MODE4 | OBJ_MAP_LINEAR | BG2_ENABLE | OBJ_ENABLE;

	CalcBgGradient();

	ZeroMemory(oamTemp);
	oamTemp.attribute0 = SPRITE_SD;
	for(i = 0; i < 128; i++)
		oamShadow[i] = oamTemp;

	REG_INTERRUPT = (u32)Irq;
	REG_DISPSTAT = HBL_IRQ;
	REG_IE = IRQ_HBL;
	REG_IME = 1;

//generate palettes
	//zero out BG and OBJ palettes
	MemSet((void*)(u16*)0x5000000, 0, 512);	
	//the color for the fishing pole
	PAL_BG[2] = RGB(10, 7, 5);
	//monkey palette
	Dma3((void*)(&PAL_OBJ[1]), (void*)dMonkeyPalette, 
		5, DMA_DEST_INC | DMA_ENABLE);
	//ball palettes
	for(i = 1; i < 5; i++)		//this makes an orangish gradient palette
		PAL_OBJ[i + 16] = RGB(31 - (i + (i >> 1)), 31 - (i << 2), 15 - (i << 1));
	for(i = 1; i < 5; i++)		//and this makes a green one
		PAL_OBJ[i + 32] = RGB(15 - (i << 1) - i, 31 - ((i << 1) + i), 15 - ((i << 1)));

//load sprites
	UnCompRLE(&(OAM_MEM[8192 + 32]), dSpriteMonkey);
	UnCompRLE(&(OAM_MEM[8192 + 256 + 32]), dSpriteBall);

	for(i = 0; i < 3; i++)
	{
		string[i].nodeCount = 8;
		string[i].mass = 32;
		string[i].stretch = 2048;
		string[i].damping = 250;
		string[i].objRadius = 8;
		string[i].oamIdx = i;
		stringSet[i] = &string[i];
		StringInit(&string[i]);
	}
	oamShadow[string[0].oamIdx].attribute2 = 514;
	oamShadow[string[0].oamIdx].attribute1 = SPRITE_SIZE_32;
	string[0].objRadius = 16;
	oamShadow[string[1].oamIdx].attribute2 = 530 | SPRITE_PALETTE(1);
	oamShadow[string[2].oamIdx].attribute2 = 530 | SPRITE_PALETTE(2);

	while(1)
	{
		if(KEY_UP)
			string[0].node[0].y -= int2fp(2);
		if(KEY_DOWN)
			string[0].node[0].y += int2fp(2);
		if(KEY_LEFT)
			string[0].node[0].x -= int2fp(2);
		if(KEY_RIGHT)
			string[0].node[0].x += int2fp(2);

		for(i = 0; i < MAX_STRINGS; i++)
			if(stringSet[i])
				StringUpdate(stringSet[i], stringSet);
		VSync();
		if(REG_DISPCNT & BACK_BUFFER)
		{
			REG_DISPCNT &= ~BACK_BUFFER;
			curBuf = MODE4_BACKBUFFER;
		}
		else
		{
			REG_DISPCNT |= BACK_BUFFER;
			curBuf = MODE4_FRONTBUFFER;
		}
		i = 0;
		Dma3(curBuf, &i, 0x4b00, DMA_DEST_INC | DMA_SRC_FIXED | DMA_ENABLE);
		DrawLine8(curBuf, 120, 80, fp2int(string[0].node[0].x), 
			fp2int(string[0].node[0].y), 2);
		for(i = 0; i < MAX_STRINGS; i++)
			if(stringSet[i])
				StringRender(stringSet[i], curBuf);
		CopyOAM();
	}

	return 0;
}
