// c_virtual_channel_inline.h: implementation of the c_virtual_channel class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#if !defined(AFX_C_VIRTUAL_CHANNEL_INLINE_H__EB69AC62_2087_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_VIRTUAL_CHANNEL_INLINE_H__EB69AC62_2087_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_virtual_channel::c_virtual_channel(p_mixer mixer, p_module module)
{
	m_mixer = mixer ;
	m_module = module ;

	m_second_volume = 0 ;
	m_second_panning = 0 ;
	m_second_pitch = 0 ;
	m_delta_volume_left = 0 ;
	m_delta_volume_right = 0 ;
	m_direction = 1 ;
	m_end_of_pitch_envelope = false ;
	m_end_of_panning_envelope = false ;
	m_end_of_volume_envelope = false ;
	m_ending_sample_on = false ;
	m_fade_component = 0 ;
	m_fade_on = false ;
	m_final_volume_left = 0 ;
	m_final_volume_right = 0 ;
	m_frac_position = 0 ;
	m_in_background = false ;
	m_instrument = -1 ;
	m_new_sample_on = false ;
	m_note_volume = 0 ;
	m_pitch = 0 ;
	m_panning = 0 ;
	m_panning_envelope_tick = 0 ;
	m_panning_envelope_value = 0 ;
	m_pitch_envelope_tick = 0 ;
	m_pitch_envelope_value = 0 ;
	m_playing = false ;
	m_volume_left = 0 ;
	m_volume_right = 0 ;
	m_ramp_length = m_mixer->get_ramp_length() ;
	m_ramp_position = 0 ;
	m_real_channel = -1 ;
	m_sample = -1 ;
	m_sample_data = (const signed short *)NULL ;
	m_coefs = (const p_coefs) NULL ;
	m_step_length = 0 ;
	m_sustain_on ;
	m_use_default_panning = false ;
	m_volume_envelope_tick = 0 ;
	m_volume_envelope_value = 0 ;

}

c_virtual_channel::~c_virtual_channel()
{

}

void c_virtual_channel::note_cut()
{
	m_new_sample_on = false ;
	m_ending_sample_on = true ;
}

void c_virtual_channel::note_fade()
{
	m_fade_on = true ;
}

void c_virtual_channel::note_off()
{
	/* note off */
	p_instrument pins = m_module->get_instrument(m_instrument) ;
	p_sample psmp = m_module->get_sample(m_sample) ;
	p_envelope pvol = pins->get_volume_envelope() ;
	p_envelope ppan = pins->get_panning_envelope() ;
	p_envelope ppit = pins->get_pitch_envelope() ;
	m_sustain_on = false ;
	
	if (psmp->is_sustain_loop_on())
	{
		if (psmp->is_loop_on())
		{
			/*
			double *temp_beginning = psmp->get_pointer_on_loop_beginning() ;
			double *temp_end = psmp->get_pointer_on_loop_end() ;
			if (   ((m_direction == 1) && (m_sample_data > temp_end))
				|| ((m_direction == -1) && (m_sample_data < temp_beginning)))
			{
				m_sample_data = beginning ;
				m_frac_position = 0 ;
				m_direction = 1 ;
			}
			*/
		}
		else
		{
			if (m_volume_envelope_value < 1)
			{
				/* Volume optimisation */
				m_new_sample_on = false ;
				m_ending_sample_on = true ;
			}
		}
	}
	
	if (m_volume_envelope_on)
	{
		if (pvol->is_sustain_loop_on() && pvol->is_loop_on() && (m_volume_envelope_tick > pvol->get_loop_end_tick()))
		{
			m_volume_envelope_tick = pvol->get_loop_beginning_tick() ;
		}
		
		if (pvol->is_loop_on())
		{
			m_fade_on = true ;
		}
	}
	else
	{
		m_fade_on = true ;
	}
	
	if (ppan->is_on())
	{
		if (ppan->is_sustain_loop_on() && ppan->is_loop_on() && (m_panning_envelope_tick > ppan->get_loop_end_tick()))
		{
			m_panning_envelope_tick = ppan->get_loop_beginning_tick() ;
		}
	}
	
	if (ppit->is_on())
	{
		if (ppit->is_sustain_loop_on() && ppit->is_loop_on() && (m_pitch_envelope_tick > ppit->get_loop_end_tick()))
		{
			m_pitch_envelope_tick = ppit->get_loop_beginning_tick() ;
		}
	}
	
}

void c_virtual_channel::new_note(signed long instrument, signed long note)
{
	m_instrument = instrument ;
	p_instrument pins = m_module->get_instrument(m_instrument) ;
	m_sample = pins->get_sample(note) ;
	m_instrument_note = note ;
	p_sample psmp = m_module->get_sample(m_sample) ;
	m_use_linear_interpolation = psmp->use_linear_interpolation() ;
	signed long sample_note = pins->get_note(note) ;

	
	m_current_vibrato_depth = 0.0 ;
	m_vibrato_time = 0 ;
	m_vibrato_position = 0 ;
	m_ramp_position = 0 ;
	m_new_sample_on = true ;
	m_ending_sample_on = false ;
	m_volume_left = 0 ;
	m_delta_volume_left = 0 ;
	m_volume_right = 0 ;
	m_delta_volume_right = 0 ;
	m_second_volume = m_note_volume = psmp->get_default_volume() ;

	signed long panning = pins->get_panning() ;
	if (panning > 64)
	{
		m_use_default_panning = false ;
		m_second_panning = m_panning = 0 ;
	}
	else
	{
		m_use_default_panning = true ;
		m_second_panning = m_panning = panning - 32 ;
	}

	if (m_use_linear_interpolation)
	{
		m_sample_data = psmp->get_sample_data_pointer() ;
	}
	else
	{
		m_coefs = psmp->get_coefs() ;
	}

	m_direction = 1 ;
	m_frac_position = 0 ;
	m_second_pitch = m_pitch = sample_note ;

	p_envelope pvol = pins->get_volume_envelope() ;
	m_volume_envelope_tick = 0 ;
	m_volume_envelope_on = pvol->is_on() ;
	if (m_volume_envelope_on)
	{
		m_volume_envelope_value = pvol->get_tick_value(m_volume_envelope_tick) ;
	}
	else
	{
		m_volume_envelope_value = 64.0 ;
	}
	m_end_of_volume_envelope = false ;

	p_envelope ppan = pins->get_panning_envelope() ;
	m_panning_envelope_tick = 0 ;
	if (ppan->is_on())
	{
		m_panning_envelope_value = ppan->get_tick_value(m_panning_envelope_tick) ;
	}
	else
	{
		m_panning_envelope_value = 0.0 ;
	}
	m_end_of_panning_envelope = false ;

	p_envelope ppit = pins->get_pitch_envelope() ;
	m_pitch_envelope_tick = 0 ;
	if (ppit->is_on())
	{
		m_pitch_envelope_value = ppit->get_tick_value(m_pitch_envelope_tick) ;
	}
	else
	{
		m_pitch_envelope_value = 0.0 ;
	}
#ifdef _DEBUG
assert((m_pitch_envelope_value >= 0) && (m_pitch_envelope_value <= 64.0));
#endif
	m_end_of_pitch_envelope = false ;

	m_sustain_on = true ;
	m_fade_component = 1024 ;
	m_fade_on = false ;

	reset_second_volume() ;
	reset_second_panning() ;
	reset_second_pitch() ;

#if defined _DEBUG
assert((m_channel_panning > -33) && (m_channel_panning < 33));
assert((m_second_panning > -33) && (m_second_panning < 33));
#endif
}


void c_virtual_channel::reset_second_pitch() 
{
	m_second_pitch = m_pitch ;
}

void c_virtual_channel::reset_second_panning() 
{
#if defined _DEBUG
assert((m_channel_panning > -33) && (m_channel_panning < 33));
#endif

	if (m_module->get_instrument(m_instrument)->get_panning_envelope()->is_on())
	{
		m_second_panning = m_panning_envelope_value ;
	}
	else if (m_use_default_panning)
	{
		m_second_panning = m_panning ;
	}
	else
	{
		m_second_panning = m_channel_panning ;
	}

	m_second_panning += get_pitch_panning() ;
	m_second_panning *= m_module->get_panning_separation() / 128.0 ;

#if defined _DEBUG
	assert((m_second_panning > -33) && (m_second_panning < 33));
#endif
}
void c_virtual_channel::reset_second_volume() 
{
	m_second_volume = m_note_volume ;
}

void c_virtual_channel::update_volume()
{
    double sample_volume = m_module->get_sample(m_sample)->get_volume() ;
	double final_volume = m_second_volume
		* (double) sample_volume
		* (double) m_volume_envelope_value
		* (double) m_channel_volume
		* (double) m_fade_component ;

	double main_panning = m_mixer->get_constant_panning() ;
	double volume_left = final_volume * (32 - m_second_panning) * m_mixer->get_constant_volume_left();
	double volume_right = final_volume * (32 + m_second_panning)* m_mixer->get_constant_volume_right(); 

	if (main_panning == 0)
	{
		m_final_volume_left = volume_left ;
		m_final_volume_right = volume_right ; 
	}
	else if (main_panning < 0)
	{
		m_final_volume_left = volume_left * (1 + main_panning) + volume_right * (-main_panning) ;
		m_final_volume_right = volume_right * (1 + main_panning) ;
	}
	else
	{
		m_final_volume_left = volume_left * (1 - main_panning) ;
		m_final_volume_right = volume_right * (1 - main_panning) + volume_left * (main_panning) ;
	}


	m_ramp_length = m_mixer->get_ramp_length() ;

	if (m_new_sample_on)
	{
		m_ramp_position = 0 ;
		m_volume_left = 0.0 ;
		m_volume_right = 0.0 ;
		m_delta_volume_left = m_final_volume_left / m_ramp_length ;
		m_delta_volume_right = m_final_volume_right / m_ramp_length ;
	}
	else if (m_ending_sample_on)
	{
		m_ramp_position = 0 ;
		m_final_volume_left = 0 ;
		m_final_volume_right = 0 ;
		m_delta_volume_left = -(m_volume_left / m_ramp_length) ;
		m_delta_volume_right = -(m_volume_right / m_ramp_length) ;
	}
	else if ((fabs(m_final_volume_left - m_volume_left) > 1e-8)
		|| (fabs(m_final_volume_right - m_volume_right) > 1e-8))
	{
		m_ramp_position = 0 ;
		m_delta_volume_left = ((m_final_volume_left - m_volume_left) / m_ramp_length) ;
		m_delta_volume_right = ((m_final_volume_right - m_volume_right) / m_ramp_length) ;
	}
	else
	{
		m_ramp_position = m_ramp_length ;
		m_volume_left = m_final_volume_left ;
		m_volume_right = m_final_volume_right ;
	}

#if defined _DEBUG
assert((m_final_volume_left >= 0) && (m_final_volume_right >= 0));
assert((m_volume_left >= 0) && (m_volume_right >= 0));
#endif
}


void c_virtual_channel::set_sample_offset(unsigned long sample_offset)
{
	if (m_use_linear_interpolation)
	{
		if ((m_sample_data + sample_offset) < m_module->get_sample(m_sample)->get_pointer_on_end())
		{
			m_sample_data += sample_offset ;
		}
	}
	else
	{
		if ((m_coefs + sample_offset) < m_module->get_sample(m_sample)->get_coefs_end())
		{
			m_coefs += sample_offset ;
		}
	}
}

void c_virtual_channel::set_playing()
{
	m_playing = true ;
}

void c_virtual_channel::clear_playing()
{
	m_playing = false ;
}

bool c_virtual_channel::update_frame(bool frame_0)
{
	/*if ((!m_ending_sample_on) &&(m_in_background) && (m_volume < 1))
	{
		m_new_sample_on = false ;
		m_ending_sample_on = true ;
	} */
	update_vibrato();
	if (m_fade_on)
	{
		m_fade_component -= m_module->get_instrument(m_instrument)->get_fadeout() ;		
		
		if (m_fade_component < 0)
		{ 			
			return true ;
		}
	}

	if (update_volume_envelope())
	{
		return true ;
	}

	update_panning_envelope() ;
	update_pitch_envelope() ;

	return false ;
}

double c_virtual_channel::get_waveform_value(signed long index, signed long waveform)
{
	index &= 0xFF ;

	switch (waveform)
	{
	case 0:
		{
			return 64 * sin(index * m_sin_cst);
		}
	break;
	case 1:
		{
			return -((index << 8) / 510.0) + 64.0 ;
		}
	break;
	case 2:
		{
			return (index < 128)?64.0:0.0;
		}
	break;
	case 3:
		{
			const double factor = 64.0 / (RAND_MAX);
			return factor * rand();
		}
	break;
	default:
		{
			return 0 ;
		}
	break;
	}
}

const double c_virtual_channel::m_sin_cst = 3.14159265358979323 / 128.0 ;

void c_virtual_channel::update_vibrato()
{
	p_sample psmp = m_module->get_sample(m_sample);

	if ((psmp->get_vibrato_depth() == 0) || (psmp->get_vibrato_speed() == 0) || (psmp->get_vibrato_rate() == 0))
	{
		return ;
	}

	if (m_current_vibrato_depth < psmp->get_vibrato_depth())
	{
		m_current_vibrato_depth += psmp->get_vibrato_rate()/256.0 ;
	}

	if ((m_vibrato_time == 0) || (psmp->get_vibrato_waveform() != 3))
	{
		double value = get_waveform_value(m_vibrato_position, psmp->get_vibrato_waveform()) ;
		slide_second_pitch(value * m_current_vibrato_depth/256.0, true) ;
	}

	m_vibrato_time ++ ;
	if (m_vibrato_time >= psmp->get_vibrato_speed()) ; // speed / 4 ???
	{
		m_vibrato_time = 0 ;
	}
	m_vibrato_position = (m_vibrato_position + psmp->get_vibrato_speed()) & 0xFF ; // <=> % 256
}

void c_virtual_channel::init(signed long real_channel)
{
	m_current_vibrato_depth = 0.0 ;
	m_vibrato_time = 0 ;
	m_vibrato_position = 0 ;
	m_delta_volume_left = 0 ;
	m_delta_volume_right = 0 ;
	m_direction = 1 ;
	m_end_of_pitch_envelope = false ;
	m_end_of_panning_envelope = false ;
	m_end_of_volume_envelope = false ;
	m_ending_sample_on = false ;
	m_fade_component = 0 ;
	m_fade_on = false ;
	m_final_volume_left = 0 ;
	m_final_volume_right = 0 ;
	m_frac_position = 0 ;
	m_in_background = false ;
	m_instrument = -1 ;
	m_new_sample_on = false ;
	m_note_volume = 0 ;
	m_panning = 0 ;
	m_panning_envelope_tick = 0 ;
	m_panning_envelope_value = 0 ;
	m_pitch_envelope_tick = 0 ;
	m_pitch_envelope_value = 0 ;
	m_playing = false ;
	m_volume_left = 0 ;
	m_volume_right = 0 ;
	m_ramp_length = m_mixer->get_ramp_length() ;
	m_ramp_position = 0 ;
	m_real_channel = real_channel ;
	m_sample = -1 ;
	m_sample_data = (signed short *)NULL ;
	m_coefs = (p_coefs) NULL ;
	m_step_length = 0 ;
	m_sustain_on ;
	m_use_default_panning = false ;
	m_volume = 0 ;
	m_volume_envelope_tick = 0 ;
	m_volume_envelope_value = 0 ;
}

bool c_virtual_channel::update_volume_envelope()
{
	if (!m_volume_envelope_on)
	{
		return false ;
	}

	p_envelope pvol = m_module->get_instrument(m_instrument)->get_volume_envelope() ;
		
	/* volume envelope */
		
	if (m_volume_envelope_on && !m_end_of_volume_envelope)
	{
		m_volume_envelope_tick++ ;

		if (m_sustain_on && pvol->is_sustain_loop_on())
		{ 
			/* sustain loop */
			if (m_volume_envelope_tick > pvol->get_sustain_loop_end_tick())
			{
				m_volume_envelope_tick = pvol->get_sustain_loop_beginning_tick() ;
			}
		}
		else if (pvol->is_loop_on())
		{ 
			/* normal loop */
			if (m_volume_envelope_tick > pvol->get_loop_end_tick())
			{
				m_volume_envelope_tick = pvol->get_loop_beginning_tick() ;
			}

		}
		else
		{ 
			/* no loop */
			if (m_volume_envelope_tick >= pvol->get_number_of_ticks())
			{ 
				/* end of envelope */
				if (m_volume_envelope_value < 1)
				{
					return true ;
				}
				else
				{
					m_fade_on = true ;
					m_end_of_volume_envelope = true ;
				}
			}
		}

		if (!m_end_of_volume_envelope)
		{
			m_volume_envelope_value = pvol->get_tick_value(m_volume_envelope_tick) ;
		}
	}

	return false ;
}


void c_virtual_channel::update_panning_envelope()
{
	p_envelope ppan = m_module->get_instrument(m_instrument)->get_panning_envelope() ;

	if (ppan->is_on() && !m_end_of_panning_envelope)
	{
		m_panning_envelope_tick++ ;
			
		if (m_sustain_on && ppan->is_sustain_loop_on())
		{ 
			/* sustain loop */
			if (m_panning_envelope_tick > ppan->get_sustain_loop_end_tick())
			{
				m_panning_envelope_tick = ppan->get_sustain_loop_beginning_tick() ;
			}

		}
		else if (ppan->is_loop_on())
		{ 
			/* normal loop */
			if (m_panning_envelope_tick > ppan->get_loop_end_tick())
			{
				m_panning_envelope_tick = ppan->get_loop_beginning_tick() ;
			}
		}
		else
		{ 
			/* no loop */
			if (m_panning_envelope_tick >= ppan->get_number_of_ticks())
			{ 
				/* end of envelope : */
				m_end_of_panning_envelope = true ;
			}
		}

		if (!m_end_of_panning_envelope)
		{
			m_panning_envelope_value = ppan->get_tick_value(m_panning_envelope_tick) ;
		}
	}
}

void c_virtual_channel::set_second_volume(double volume)
{
	m_second_volume = volume ;
}

void c_virtual_channel::slide_second_volume(double slide_value)
{
	double second_volume = m_second_volume ;

	m_second_volume += slide_value ;
	if ((m_second_volume < 0) || (m_second_volume > 64))
	{
		m_second_volume = second_volume ;
	}
}

void c_virtual_channel::slide_second_panning (double slide_value)
{
	double second_panning = m_second_panning ;

	m_second_panning += slide_value ;
	if ((m_second_panning < -32) || (m_second_panning > 32))
	{
		m_second_panning = second_panning ;
	}
#if defined _DEBUG
assert((m_second_panning > -33) && (m_second_panning < 33));
#endif
}

void c_virtual_channel::set_pitch(double pitch)
{
	m_second_pitch = m_pitch = pitch ;
}

double c_virtual_channel::pitch_to_speed(double pitch)
{
	return pow(2, (pitch/12) - 5) ;
}

void c_virtual_channel::slide_second_pitch(double slide_value, bool linear)
{
	double second_pitch = m_second_pitch ;

	if (linear)
	{
		/* Linear slides */
		m_second_pitch += slide_value / 16 ;

		if (m_second_pitch <= 0)
		{
			m_second_pitch = second_pitch ;
		}		
	}
	else
	{
		/* Amiga slides */
		double amiga_value = pitch_to_amiga (m_second_pitch, m_sample) ;
		amiga_value -= (slide_value * m_module->get_sample(m_sample)->get_C5_speed() / 8363.42) ;

		if (amiga_value <= 0)
		{
			m_second_pitch = second_pitch ;
		}
		else
		{
			m_second_pitch = amiga_to_pitch (amiga_value, m_sample) ;
		}
	}
}

double c_virtual_channel::pitch_to_amiga(double pitch, signed long sample)
{
	p_sample psmp = m_module->get_sample(sample) ;

	return (m_amiga_const/pitch_to_speed(pitch)) ;  
}

double c_virtual_channel::amiga_to_pitch(double amiga_value, signed long sample)
{
	p_sample psmp = m_module->get_sample(m_sample) ;

	return speed_to_pitch (m_amiga_const/amiga_value) ;
}

double c_virtual_channel::speed_to_pitch(double speed)
{
	return 12 * (5 + log(speed) / m_log2_const) ;
}

bool c_virtual_channel::ping_pong_mix_v_ramp(double* &buffer, 
									  double* &end, 
									  signed short *loop_beginning, 
									  signed short *loop_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;
	signed long direction = (m_direction + 1) >> 1 ; /* if m_direction == -1 then direction = 0 
											  else direction == 1 */

	while (buffer < end)
	{
		if (direction)
		{
			const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
			local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += local_volume_left * interpolation ;
			*(local_buffer-1) += local_volume_right * interpolation ;
			if (local_sample_data >= loop_end)
			{
				direction = --direction ;
				local_sample_data = loop_end + (loop_end - local_sample_data) ;
			}
			local_ramp_position++ ;
			local_volume_left += local_delta_volume_left ;
			local_volume_right += local_delta_volume_right ;
			if (local_ramp_position == local_ramp_length)
			{
				if (m_ending_sample_on)
				{
					return true ;
				}
				else if (m_new_sample_on)
				{
					m_new_sample_on = false ;
				}
				break ;
			}
		}
		else
		{
			const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data - 1) - (*local_sample_data)) * (float)local_frac_position ;
			local_sample_data -= (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += local_volume_left * interpolation ;
			*(local_buffer-1) += local_volume_right * interpolation ;
			if (local_sample_data <= loop_beginning)
			{
				direction = ++direction ;
				local_sample_data = loop_beginning + (loop_beginning - local_sample_data) ;
			}
			local_ramp_position++ ;
			local_volume_left += local_delta_volume_left ;
			local_volume_right += local_delta_volume_right ;
			if (local_ramp_position == local_ramp_length)
			{
				if (m_ending_sample_on)
				{
					return true ;
				}
				else if (m_new_sample_on)
				{
					m_new_sample_on = false ;
				}
				break ;
			}
		}
	}

	m_direction = (direction << 1) - 1 ;
	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}
	
void c_virtual_channel::ping_pong_mix(double* &buffer, 
									  double* &end, 
									  signed short *loop_beginning, 
									  signed short *loop_end)
{
	const signed long cst_step_length = m_step_length ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left ;
	const double cst_volume_right = m_volume_right ;
	signed long direction = (m_direction + 1) >> 1 ; /* if m_direction == -1 then direction = 0 
											  else direction == 1 */

	while  ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;

		if (direction)
		{
	
			if ((local_sample_data + (cst_step_length >> 8)) < loop_end)
			{
				do
				{
					const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
					local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					i++ ;
				}
				while (i) ;
				
			}
			else
			{				
				do
				{
					const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
					local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					
					if (local_sample_data >= loop_end)                                                                                                                                      \
					{
						direction-- ; /* direction  1 -> 0 */
						local_sample_data = loop_end + (loop_end - local_sample_data) ;
						break ;
					}
					
					i++ ;
				}
				while (i) ;
			}
		}
		else
		{
			if ((local_sample_data - (cst_step_length >> 8)) > loop_beginning)
			{
				do
				{
					const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data - 1) - (*local_sample_data)) * (float)local_frac_position ;
					local_sample_data -= (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					i++ ;
				}
				while (i) ;
			}
			else
			{
				do
				{
					const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data - 1) - (*local_sample_data)) * (float)local_frac_position ;
					local_sample_data -= (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					
					if (local_sample_data <= loop_beginning)
					{
						direction++ ; /* direction  0 -> 1 */
						local_sample_data = loop_beginning + (loop_beginning - local_sample_data) ;
						break ;
					}
					i++ ;
				}
				while (i) ;
			}
		}
	}


	while (local_buffer < end)
	{
		if (direction)
		{
			const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
			local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += cst_volume_left * interpolation ;
			*(local_buffer-1) += cst_volume_right * interpolation ;
			if (local_sample_data >= loop_end)
			{
				direction = --direction ;
				local_sample_data = loop_end + (loop_end - local_sample_data) ;
			}
		}
		else
		{
			const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data - 1) - (*local_sample_data)) * (float)local_frac_position ;
			local_sample_data -= (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += cst_volume_left * interpolation ;
			*(local_buffer-1) += cst_volume_right * interpolation ;
			if (local_sample_data <= loop_beginning)
			{
				direction = ++direction ;
				local_sample_data = loop_beginning + (loop_beginning - local_sample_data) ;
			}
		}
	}

	m_direction = (direction << 1) - 1 ;
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	buffer = local_buffer ;
}

bool c_virtual_channel::forward_mix_v_ramp(double* &buffer, 
									double* &end, 
									signed short *loop_beginning, 
									signed short *loop_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;

	while (local_buffer < end)
	{
		const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
		local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += local_volume_left * interpolation ;
		*(local_buffer-1) += local_volume_right * interpolation ;
		
		if (local_sample_data >= loop_end)
		{
			local_sample_data = loop_beginning + (local_sample_data - loop_end);
		}
		
		local_ramp_position++ ;
		local_volume_left += local_delta_volume_left ;
		local_volume_right += local_delta_volume_right ;
		if (local_ramp_position == local_ramp_length)
		{
			if (m_ending_sample_on)
			{
				return true ;
			}
			else if (m_new_sample_on)
			{
				m_new_sample_on = false ;
			}
			break ;
		}
	}

	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}

void c_virtual_channel::forward_mix(double* &buffer, 
									double* &end, 
									signed short *loop_beginning, 
									signed short *loop_end)
{
	const signed long	cst_step_length = m_step_length ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left ;
	const double cst_volume_right = m_volume_right ;

	while ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;
		if ((local_sample_data + (cst_step_length >> 8)) < loop_end)
		{
			
			do
			{
				const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
				local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				i++ ;
			}
			while (i) ;
			
		}
		else
		{
			do
			{
				const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
				local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				
				if (local_sample_data >= loop_end)                                                                                                                                      \
				{
					local_sample_data = loop_beginning + (local_sample_data - loop_end);
					break ;
				}

				i++ ;
			}
			while (i) ;
		}

	}

	while (local_buffer < end)
	{
		const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
		local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += cst_volume_left * interpolation ;
		*(local_buffer-1) += cst_volume_right * interpolation ;
		
		if (local_sample_data >= loop_end)
		{
			local_sample_data = loop_beginning + (local_sample_data - loop_end);
		}
	}
	
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	buffer = local_buffer ;
}

bool c_virtual_channel::no_loop_mix_v_ramp(double* &buffer, double* &end, signed short *sample_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;

	while (local_buffer < end)
	{
		const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
		local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += local_volume_left * interpolation ;
		*(local_buffer-1) += local_volume_right * interpolation ;
		
		if (local_sample_data >= sample_end)
		{
			return true ;
		}		
		
		local_ramp_position++ ;
		local_volume_left += local_delta_volume_left ;
		local_volume_right += local_delta_volume_right ;
		if (local_ramp_position == local_ramp_length)
		{
			if (m_ending_sample_on)
			{
				return true ;
			}
			else if (m_new_sample_on)
			{
				m_new_sample_on = false ;
			}
			break ;
		}
	}

	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}

bool c_virtual_channel::no_loop_mix(double* &buffer, double* &end, signed short *sample_end)
{
	const signed long	cst_step_length = m_step_length ;
	const signed short *local_sample_data = m_sample_data ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left ;
	const double cst_volume_right = m_volume_right ;
	
	while ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;

		if ((local_sample_data + (cst_step_length >> 8)) < sample_end)
		{
	
			do
			{
				const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
				local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				i++ ;
			}
			while (i) ;
		}
		else
		{

			do
			{
				const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
				local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				
				if (local_sample_data >= sample_end)
				{
					return true;
				}

				i++ ;
			}
			while (i) ;
		}
	}				
	
	while (local_buffer < end)
	{
		const double interpolation = ((*(signed long *)local_sample_data) << 16) + ((signed long)*(local_sample_data + 1) - (*local_sample_data)) * (float)local_frac_position ;
		local_sample_data += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += cst_volume_left * interpolation ;
		*(local_buffer-1) += cst_volume_right * interpolation ;
		
		if (local_sample_data >= sample_end)
		{
			return true ;
		}
	}
	
	m_frac_position = local_frac_position ;
	m_sample_data = local_sample_data ;
	buffer = local_buffer ;
	return false ;
}

bool c_virtual_channel::no_lin_ping_pong_mix_v_ramp(double* &buffer, 
									  double* &end, 
									  p_coefs loop_beginning, 
									  p_coefs loop_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;
	signed long direction = (m_direction + 1) >> 1 ; /* if m_direction == -1 then direction = 0 
											  else direction == 1 */

	while (buffer < end)
	{
		if (direction)
		{
			const double interpolation = 
				local_coefs->a + 
				local_frac_position * (local_coefs->b + 
				local_frac_position * (local_coefs->c + 
				local_frac_position * (local_coefs->d))) ;
			local_coefs += (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += local_volume_left * interpolation ;
			*(local_buffer-1) += local_volume_right * interpolation ;
			if (local_coefs >= loop_end)
			{
				direction = --direction ;
				local_coefs = loop_end + (loop_end - local_coefs) ;
				local_coefs-- ;
			}
			local_ramp_position++ ;
			local_volume_left += local_delta_volume_left ;
			local_volume_right += local_delta_volume_right ;
			if (local_ramp_position == local_ramp_length)
			{
				if (m_ending_sample_on)
				{
					return true ;
				}
				else if (m_new_sample_on)
				{
					m_new_sample_on = false ;
				}
				break ;
			}
		}
		else
		{
			const double interpolation = 
				local_coefs->a + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->b + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->c + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->d))) ;
			local_coefs -= (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += local_volume_left * interpolation ;
			*(local_buffer-1) += local_volume_right * interpolation ;
			if (local_coefs < loop_beginning)
			{
				direction = ++direction ;
				local_coefs++ ;
				local_coefs = loop_beginning + (loop_beginning - local_coefs) ;
			}
			local_ramp_position++ ;
			local_volume_left += local_delta_volume_left ;
			local_volume_right += local_delta_volume_right ;
			if (local_ramp_position == local_ramp_length)
			{
				if (m_ending_sample_on)
				{
					return true ;
				}
				else if (m_new_sample_on)
				{
					m_new_sample_on = false ;
				}
				break ;
			}
		}
	}

	m_direction = (direction << 1) - 1 ;
	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}
	
void c_virtual_channel::no_lin_ping_pong_mix(double* &buffer, 
									  double* &end, 
									  p_coefs loop_beginning, 
									  p_coefs loop_end)
{
	const signed long cst_step_length = m_step_length ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left ;
	const double cst_volume_right = m_volume_right ;
	signed long direction = (m_direction + 1) >> 1 ; /* if m_direction == -1 then direction = 0 
											  else direction == 1 */

	while  ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;

		if (direction)
		{
	
			if ((local_coefs + (cst_step_length >> 8)) < loop_end)
			{
				do
				{
					const double interpolation = 
						local_coefs->a + 
						local_frac_position * (local_coefs->b + 
						local_frac_position * (local_coefs->c + 
						local_frac_position * (local_coefs->d))) ;
					local_coefs += (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					i++ ;
				}
				while (i) ;
				
			}
			else
			{				
				do
				{
					const double interpolation = 
						local_coefs->a + 
						local_frac_position * (local_coefs->b + 
						local_frac_position * (local_coefs->c + 
						local_frac_position * (local_coefs->d))) ;
					local_coefs += (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					
					if (local_coefs >= loop_end)                                                                                                                                      \
					{
						direction-- ; /* direction  1 -> 0 */
						local_coefs = loop_end + (loop_end - local_coefs) ;
						local_coefs-- ;
						break ;
					}
					
					i++ ;
				}
				while (i) ;
			}
		}
		else
		{
			if ((local_coefs - (cst_step_length >> 8)) > loop_beginning)
			{
				do
				{
					const double interpolation = 
						local_coefs->a + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->b + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->c + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->d))) ;
					local_coefs -= (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					i++ ;
				}
				while (i) ;
			}
			else
			{
				do
				{
					const double interpolation = 
						local_coefs->a + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->b + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->c + 
						(local_frac_position ^ 0xFFFF) * (local_coefs->d))) ;
					local_coefs -= (local_frac_position += cst_step_length) >> 16 ;
					local_frac_position &= 0xFFFF ;	
					local_buffer += 2 ;
					*(local_buffer-2) += cst_volume_left * interpolation ;
					*(local_buffer-1) += cst_volume_right * interpolation ;
					
					if (local_coefs < loop_beginning)
					{
						direction++ ; /* direction  0 -> 1 */
						local_coefs++ ;
						local_coefs = loop_beginning + (loop_beginning - local_coefs) ;
						break ;
					}
					i++ ;
				}
				while (i) ;
			}
		}
	}


	while (local_buffer < end)
	{
		if (direction)
		{
			const double interpolation = 
				local_coefs->a + 
				local_frac_position * (local_coefs->b + 
				local_frac_position * (local_coefs->c + 
				local_frac_position * (local_coefs->d))) ;
			local_coefs += (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += cst_volume_left * interpolation ;
			*(local_buffer-1) += cst_volume_right * interpolation ;
			if (local_coefs >= loop_end)
			{
				direction = --direction ;
				local_coefs = loop_end + (loop_end - local_coefs) ;
				local_coefs-- ;
			}
		}
		else
		{
			const double interpolation = 
				local_coefs->a + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->b + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->c + 
				(local_frac_position ^ 0xFFFF) * (local_coefs->d))) ;
			local_coefs -= (local_frac_position += cst_step_length) >> 16 ;
			local_frac_position &= 0xFFFF ;	
			local_buffer += 2 ;
			*(local_buffer-2) += cst_volume_left * interpolation ;
			*(local_buffer-1) += cst_volume_right * interpolation ;
			if (local_coefs < loop_beginning)
			{
				direction = ++direction ;
				local_coefs++ ;
				local_coefs = loop_beginning + (loop_beginning - local_coefs) ;
			}
		}
	}

	m_direction = (direction << 1) - 1 ;
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	buffer = local_buffer ;
}

bool c_virtual_channel::no_lin_forward_mix_v_ramp(double* &buffer, 
									double* &end, 
									p_coefs loop_beginning, 
									p_coefs loop_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;

	while (local_buffer < end)
	{
		const double interpolation = 
			local_coefs->a + 
			local_frac_position * (local_coefs->b + 
			local_frac_position * (local_coefs->c + 
			local_frac_position * (local_coefs->d))) ;
		local_coefs += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += local_volume_left * interpolation ;
		*(local_buffer-1) += local_volume_right * interpolation ;
		
		if (local_coefs >= loop_end)
		{
			local_coefs = loop_beginning + (local_coefs - loop_end);
		}
		
		local_ramp_position++ ;
		local_volume_left += local_delta_volume_left ;
		local_volume_right += local_delta_volume_right ;
		if (local_ramp_position == local_ramp_length)
		{
			if (m_ending_sample_on)
			{
				return true ;
			}
			else if (m_new_sample_on)
			{
				m_new_sample_on = false ;
			}
			break ;
		}
	}

	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}

void c_virtual_channel::no_lin_forward_mix(double* &buffer, 
									double* &end, 
									p_coefs loop_beginning, 
									p_coefs loop_end)
{
	const signed long	cst_step_length = m_step_length ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left ;
	const double cst_volume_right = m_volume_right ;

	while ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;
		if ((local_coefs + (cst_step_length >> 8)) < loop_end)
		{
			
			do
			{
				const double interpolation = 
					local_coefs->a + 
					local_frac_position * (local_coefs->b + 
					local_frac_position * (local_coefs->c + 
					local_frac_position * (local_coefs->d))) ;
				local_coefs += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				i++ ;
			}
			while (i) ;
			
		}
		else
		{
			do
			{
				const double interpolation = 
					local_coefs->a + 
					local_frac_position * (local_coefs->b + 
					local_frac_position * (local_coefs->c + 
					local_frac_position * (local_coefs->d))) ;
				local_coefs += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				
				if (local_coefs >= loop_end)                                                                                                                                      \
				{
					local_coefs = loop_beginning + (local_coefs - loop_end);
					break ;
				}

				i++ ;
			}
			while (i) ;
		}

	}

	while (local_buffer < end)
	{
		const double interpolation = 
			local_coefs->a + 
			local_frac_position * (local_coefs->b + 
			local_frac_position * (local_coefs->c + 
			local_frac_position * (local_coefs->d))) ;
		local_coefs += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += cst_volume_left * interpolation ;
		*(local_buffer-1) += cst_volume_right * interpolation ;
		
		if (local_coefs >= loop_end)
		{
			local_coefs = loop_beginning + (local_coefs - loop_end);
		}
	}
	
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	buffer = local_buffer ;
}

bool c_virtual_channel::no_lin_no_loop_mix_v_ramp(double* &buffer, double* &end, p_coefs coefs_end)
{
	double *local_buffer = buffer ;
	double local_volume_left = m_volume_left ;
	double local_volume_right = m_volume_right ;
	const signed long	cst_step_length = m_step_length ;
	const double local_delta_volume_left = m_delta_volume_left ;
	const double local_delta_volume_right = m_delta_volume_right ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	signed long local_ramp_position = m_ramp_position ;
	const signed long local_ramp_length = m_ramp_length ;

	while (local_buffer < end)
	{
		const double interpolation = 
			local_coefs->a + 
			local_frac_position * (local_coefs->b + 
			local_frac_position * (local_coefs->c + 
			local_frac_position * (local_coefs->d))) ;
		local_coefs += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += local_volume_left * interpolation ;
		*(local_buffer-1) += local_volume_right * interpolation ;
		
		if (local_coefs >= coefs_end)
		{
			return true ;
		}		
		
		local_ramp_position++ ;
		local_volume_left += local_delta_volume_left ;
		local_volume_right += local_delta_volume_right ;
		if (local_ramp_position == local_ramp_length)
		{
			if (m_ending_sample_on)
			{
				return true ;
			}
			else if (m_new_sample_on)
			{
				m_new_sample_on = false ;
			}
			break ;
		}
	}

	m_ramp_position = local_ramp_position ;
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	m_volume_left = local_volume_left ;
	m_volume_right = local_volume_right ;
	buffer = local_buffer ;
	return false ;
}

bool c_virtual_channel::no_lin_no_loop_mix(double* &buffer, double* &end, p_coefs coefs_end)
{
	const signed long	cst_step_length = m_step_length ;
	const coefs *local_coefs = m_coefs ;
	signed long local_frac_position = m_frac_position ;
	double *local_buffer = buffer ;
	const double cst_volume_left = m_volume_left;
	const double cst_volume_right = m_volume_right;
	
	while ((end - local_buffer) >= 512)
	{
		signed long i = -256 ;

		if ((local_coefs + (cst_step_length >> 8)) < coefs_end)
		{
	
			do
			{
				const double interpolation = 
					local_coefs->a + 
					local_frac_position * (local_coefs->b + 
					local_frac_position * (local_coefs->c + 
					local_frac_position * (local_coefs->d))) ;
				local_coefs += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				i++ ;
			}
			while (i) ;
		}
		else
		{

			do
			{
				const double interpolation = 
					local_coefs->a + 
					local_frac_position * (local_coefs->b + 
					local_frac_position * (local_coefs->c + 
					local_frac_position * (local_coefs->d))) ;
				local_coefs += (local_frac_position += cst_step_length) >> 16 ;
				local_frac_position &= 0xFFFF ;	
				local_buffer += 2 ;
				*(local_buffer-2) += cst_volume_left * interpolation ;
				*(local_buffer-1) += cst_volume_right * interpolation ;
				
				if (local_coefs >= coefs_end)
				{
					return true;
				}

				i++ ;
			}
			while (i) ;
		}
	}				
	
	while (local_buffer < end)
	{
		const double interpolation = 
			local_coefs->a + 
			local_frac_position * (local_coefs->b + 
			local_frac_position * (local_coefs->c + 
			local_frac_position * (local_coefs->d))) ;
		local_coefs += (local_frac_position += cst_step_length) >> 16 ;
		local_frac_position &= 0xFFFF ;	
		local_buffer += 2 ;
		*(local_buffer-2) += cst_volume_left * interpolation ;
		*(local_buffer-1) += cst_volume_right * interpolation ;
		
		if (local_coefs >= coefs_end)
		{
			return true ;
		}
	}
	
	m_frac_position = local_frac_position ;
	m_coefs = local_coefs ;
	buffer = local_buffer ;
	return false ;
}

bool c_virtual_channel::fill_buffer(double * buffer, 
									double * end)
{
	p_sample psmp = m_module->get_sample(m_sample) ;
#ifdef _DEBUG
assert((m_pitch_envelope_value >= 0) && (m_pitch_envelope_value <= 64.0));
#endif

	double final_note = m_second_pitch + m_pitch_envelope_value * 0.5 ;
	if ((final_note < 0) || (final_note > 119))
	{
		return false ;
	}
	
	if ((m_ramp_position >= m_ramp_length) && (m_volume_left < 1e-8) && (m_volume_right < 1e-8))
	{
		return false ;
	}
	m_step_length = (signed long)(psmp->get_output_C5_speed() * pitch_to_speed(final_note) * 65536.0);
	
	if (m_use_linear_interpolation)
	{
		if (psmp->is_loop_on() || (m_sustain_on && psmp->is_sustain_loop_on()))
		{
			bool ping_pong ;
			signed short *loop_beginning ;
			signed short *loop_end ;
			
			if (m_sustain_on && psmp->is_sustain_loop_on())
			{
				loop_beginning = psmp->get_pointer_on_sustain_loop_beginning() ;
				loop_end = psmp->get_pointer_on_sustain_loop_end() ;
				ping_pong = psmp->is_sustain_loop_bidi() ;
				if (ping_pong)
				{
					psmp->prepare_for_bidi_sustain_loop();
				}
				else
				{
					psmp->prepare_for_forward_sustain_loop();
				}
			}
			else
			{
				loop_beginning = psmp->get_pointer_on_loop_beginning() ;
				loop_end = psmp->get_pointer_on_loop_end() ;
				ping_pong = psmp->is_loop_bidi() ;
				if (ping_pong)
				{
					psmp->prepare_for_bidi_loop();
				}
				else
				{
					psmp->prepare_for_forward_loop();
				}
			}
			
			if (ping_pong)
			{ 
				/* ping pong loop */
				if ((m_ramp_position < m_ramp_length)
					&& (ping_pong_mix_v_ramp(buffer, end, loop_beginning, loop_end)))
				{
					return true ;
				}
				
				ping_pong_mix(buffer, end, loop_beginning, loop_end) ;
			}
			else 
			{
				/* forward loop */
				if ((m_ramp_position < m_ramp_length) 
					&& (forward_mix_v_ramp(buffer, end, loop_beginning, loop_end)))
				{
					return true ;
				}
				
				forward_mix(buffer, end, loop_beginning, loop_end);
			}
		}
		else 
		{
			/* no loop */
			signed short *sample_end = psmp->get_pointer_on_end() ;
			psmp->prepare_for_no_loop();
			
			if (
				(
				(m_ramp_position < m_ramp_length) 
				&& (no_loop_mix_v_ramp(buffer, end, sample_end))
				)
				|| (no_loop_mix(buffer, end, sample_end))
				)
			{
				return true ;
			}
		}
	}	 
	else
	{
		/* improved interpolation */

		if (psmp->is_loop_on() || (m_sustain_on && psmp->is_sustain_loop_on()))
		{
			p_coefs loop_beginning ;
			p_coefs loop_end ;
			bool ping_pong = psmp->is_sustain_loop_bidi() ;
			
			if (m_sustain_on && psmp->is_sustain_loop_on())
			{
				loop_beginning = psmp->get_coefs_sloop_b() ;
				loop_end = psmp->get_coefs_sloop_e() ;

				if (!ping_pong)
				{
					psmp->prepare_for_forward_sustain_loop();
				}
				else
				{
					psmp->prepare_for_bidi_sustain_loop();
				}
			}
			else
			{
				loop_beginning = psmp->get_coefs_loop_b() ;
				loop_end = psmp->get_coefs_loop_e() ;

				if (!ping_pong)
				{
					psmp->prepare_for_forward_loop();
				}
				else
				{
					psmp->prepare_for_bidi_loop();
				}
			}
			
			if (ping_pong)
			{
				if ((m_ramp_position < m_ramp_length) 
					&& (no_lin_ping_pong_mix_v_ramp(buffer, end, loop_beginning, loop_end)))
				{
					return true ;
				}
				
				no_lin_ping_pong_mix(buffer, end, loop_beginning, loop_end);
			}
			else
			{
				if ((m_ramp_position < m_ramp_length) 
					&& (no_lin_forward_mix_v_ramp(buffer, end, loop_beginning, loop_end)))
				{
					return true ;
				}
				
				no_lin_forward_mix(buffer, end, loop_beginning, loop_end);
			}
		}
		else 
		{
			/* no loop */
			p_coefs sample_end = psmp->get_coefs_end() ;
			psmp->prepare_for_no_loop();
			
			if (
				(
				(m_ramp_position < m_ramp_length) 
				&& (no_lin_no_loop_mix_v_ramp(buffer, end, sample_end))
				)
				|| (no_lin_no_loop_mix(buffer, end, sample_end))
				)
			{
				return true ;
			}
		}
	}

	return false ;
}

void c_virtual_channel::set_volume(double volume, double panning)
{
	m_channel_volume = volume ;
	m_channel_panning = panning ;
	/*m_second_panning = m_channel_panning = panning ;*/
#if defined _DEBUG
assert((m_channel_panning > -33) && (m_channel_panning < 33));
assert((m_second_panning > -33) && (m_second_panning < 33));
#endif
}

void c_virtual_channel::slide_note_volume(signed long slide_value)
{
	m_note_volume += slide_value ;
	if (m_note_volume > 64)
	{
		m_note_volume = 64 ;
	}
	else if (m_note_volume < 0)
	{
		m_note_volume = 0 ;
	}	

	m_second_volume = m_note_volume ;
}

void c_virtual_channel::multiply_note_volume(double mul_value)
{
	m_note_volume *= mul_value ;
	if (m_note_volume > 64)
	{
		m_note_volume = 64 ;
	}
	else if (m_note_volume < 0)
	{
		m_note_volume = 0 ;
	}	

	m_second_volume = m_note_volume ;
}

double c_virtual_channel::get_note_volume()
{
	return m_note_volume ;
}

void c_virtual_channel::set_note_volume(double note_volume)
{
#if defined _DEBUG
assert((note_volume >= 0)&&(note_volume <= 64));
#endif
	m_second_volume = m_note_volume = note_volume ;
}

void c_virtual_channel::set_channel_panning(double panning)
{
	m_channel_panning = panning ;
#if defined _DEBUG
assert((m_channel_panning > -33) && (m_channel_panning < 33));
#endif
}

void c_virtual_channel::reset_volume()
{
	p_instrument pins = m_module->get_instrument(m_instrument) ;
	p_sample psmp = m_module->get_sample(m_sample) ;

	m_second_volume = m_note_volume = psmp->get_default_volume() ;

	/* Should the envelopes be reset too */
	p_envelope pvol = pins->get_volume_envelope() ;
	m_volume_envelope_tick = 0 ;
	if (m_volume_envelope_on)
	{
		m_volume_envelope_value = pvol->get_tick_value(m_volume_envelope_tick) ;
	}
	else
	{
		m_volume_envelope_value = 64.0 ;
	}
	m_end_of_volume_envelope = false ;

	p_envelope ppan = pins->get_panning_envelope() ;
	m_panning_envelope_tick = 0 ;
	if (ppan->is_on())
	{
		m_panning_envelope_value = ppan->get_tick_value(m_panning_envelope_tick) ;
	}
	else
	{
		m_panning_envelope_value = 0.0 ;
	}
	m_end_of_panning_envelope = false ;

	p_envelope ppit = pins->get_pitch_envelope() ;
	m_pitch_envelope_tick = 0 ;
	if (ppit->is_on())
	{
		m_pitch_envelope_value = ppit->get_tick_value(m_pitch_envelope_tick) ;
	}
	else
	{
		m_pitch_envelope_value = 0.0 ;
	}
	m_end_of_pitch_envelope = false ;

}

void c_virtual_channel::update_pitch_envelope()
{
	p_envelope ppit = m_module->get_instrument(m_instrument)->get_pitch_envelope() ;

	if (ppit->is_on() && !m_end_of_pitch_envelope)
	{
		m_pitch_envelope_tick++ ;
			
		if (m_sustain_on && ppit->is_sustain_loop_on())
		{ 
			/* sustain loop */
			if (m_pitch_envelope_tick > ppit->get_sustain_loop_end_tick())
			{
				m_pitch_envelope_tick = ppit->get_sustain_loop_beginning_tick() ;
			}

		}
		else if (ppit->is_loop_on())
		{ 
			/* normal loop */
			if (m_pitch_envelope_tick > ppit->get_loop_end_tick())
			{
				m_pitch_envelope_tick = ppit->get_loop_beginning_tick() ;
			}
		}
		else
		{ 
			/* no loop */
			if (m_pitch_envelope_tick >= ppit->get_number_of_ticks())
			{ 
				/* end of envelope : */
				m_end_of_pitch_envelope = true ;
			}
		}

		if (!m_end_of_pitch_envelope)
		{
			m_pitch_envelope_value = ppit->get_tick_value(m_pitch_envelope_tick) ;
		}
	}
}

double c_virtual_channel::get_pitch_panning()
{
	p_instrument pins = m_module->get_instrument(m_instrument) ;
	signed long pitch_panning_center = pins->get_pitch_panning_center() ;
	signed long pitch_panning_separation = pins->get_pitch_panning_separation() ;
	return (m_pitch - pitch_panning_center) * pitch_panning_separation * 0.125;
}

void c_virtual_channel::set_volume_envelope(bool value)
{
	m_volume_envelope_on = value ;
}

void c_virtual_channel::set_surround(bool value)
{
	m_surround_on = value ;
}

const double c_virtual_channel::m_amiga_const = 428 ; // amiga constant
const double c_virtual_channel::m_log2_const = log(2) ;

#endif // AFX_C_VIRTUAL_CHANNEL_INLINE_H__EB69AC62_2087_11D1_B35E_DCE971BF2962__INCLUDED_