/* loader.c */
/*
     PLAY_ITW.EXE v0.02b : Player for Impulse Tracker modules files
     Copyright (C) 1997  Olivier AUMAGE
     E-mail : Olivier.Aumage@ens-lyon.fr
     Web : http://www.ens-lyon.fr/~oaumage/

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* 
  *                                         *
   This file defines the functions used
   to load IT modules into memory.
  *                                         *
*/


/* header files */
/*==============*/


/* standard header files */
/*-----------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <windows.h>
#include <string.h>

/* project header files */
/*----------------------*/

#include "types.h"
#include "loader.h"



/* functions definitions */
/*=======================*/


static void *memalloc (size_t size)
/* Allocate 'size' bytes of memory by calling 'malloc'.
   If the allocation fail, a message is printed on the
   'stderr' stream and the program exit. */
{
  /* local variables of 'memalloc' */                 
  /*_______________________________*/

  void *return_value ;
  

  /* 'memalloc' body */
  /*_________________*/

/*  printf ("Memory allocation : %d bytes\n", size) ;*/
  return_value = (void *)GlobalAlloc (GMEM_FIXED, (DWORD)size) ;
  if (return_value == (void *) NULL)
    {
/*      (void) fprintf (stderr, "Memory allocation of %ld bytes failed.\nExiting\n", (long) size) ;*/
      exit (1) ;
    }
  return return_value ;
}



int load_instrument (FILE *module_file, p_instrument pinstrument)
/* load a instrument form the 'module_file' file into the
   'instrument' structure pointed by 'pinstrument'.
   Code return : 0     - successful
                 other - failed */
{
   long starting_position ;
   
   starting_position = ftell (module_file) ;

  {
    
    /* Verification of the IT instrument marker 'IMPI' */
    
    char buffer [5] ;
    size_t return_value ;
    
    return_value = fread (buffer, (size_t) 1, (size_t) 4, module_file) ;
    if (return_value == (size_t) -1)
      return -1 ; /* case when fread failed */
    
    if (strncmp ("IMPI", buffer, 4) != 0)
      return -1 ; /* this is not a normal IT instrument */

    /* End of the marker verification */

  }    
  
  /* Now we can assume (at least for now) that we
     have a regular IT instrument */
     
  /* Reading of the instrument parameters */

  (void) fread (pinstrument->dos_file_name, (size_t) 1, (size_t) 12, module_file) ;
  /* instrument dos file name */
  
  (void) fseek (module_file, 1L, SEEK_CUR) ;
  /* jumping over the 1 byte padding */
  
  (void) fread (&(pinstrument->new_note_action), (size_t) 1, (size_t) 1, module_file) ;
  /* new note action field */
  
  (void) fread (&(pinstrument->duplicate_check_type), (size_t) 1, (size_t) 1, module_file) ;
  /* duplicate check type */
  
  (void) fread (&(pinstrument->duplicate_check_action), (size_t) 1, (size_t) 1, module_file) ;
  /* duplicate check action */
  
  (void) fread (&(pinstrument->fadeout), (size_t) 2, (size_t) 1, module_file) ;
  /* fadeout */
  
  (void) fread (&(pinstrument->pitch_pan_separation), (size_t) 1, (size_t) 1, module_file) ;
  /* pitch-pan separation */
  
  (void) fread (&(pinstrument->pitch_pan_center), (size_t) 1, (size_t) 1, module_file) ;
  /* pitch-pan center */
  
  (void) fread (&(pinstrument->global_volume), (size_t) 1, (size_t) 1, module_file) ;
  /* global volume */
  
  (void) fread (&(pinstrument->default_panning), (size_t) 1, (size_t) 1, module_file) ;
  /* default panning */
  
  (void) fread (&(pinstrument->random_volume_variation), (size_t) 1, (size_t) 1, module_file) ;
  /* random volume variation */

  (void) fread (&(pinstrument->random_panning_variation), (size_t) 1, (size_t) 1, module_file) ;
  /* random panning variation */

  (void) fseek (module_file, 0x20L + starting_position, SEEK_SET) ;
  /* moving the file pointer after Traker version and no of samples */

  (void) fread (pinstrument->name, (size_t) 1, (size_t) 26, module_file) ;
  /* Instrument real name */

/*  printf ("Reading instrument : %s \n", pinstrument->name) ;*/

  (void) fseek (module_file, 0x40L + starting_position, SEEK_SET) ;
  /* adjusting file pointer to the beginning of the note-sample / keyboard table */

  {
  
    /* Reading of the note-sample / keyboard table */
    
    unsigned char note ;
    
    for (note = 0 ; note <= 119 ; note ++)
    {
      (void) fread (&(pinstrument->note_sample_keyboard_table [note].note), (size_t) 1, (size_t) 1, module_file) ;
      /* note of corresponding sample */

      (void) fread (&(pinstrument->note_sample_keyboard_table [note].sample), (size_t) 1, (size_t) 1, module_file) ;    
      /* corresponding sample */
    }
   
    /* End of note-sample / keyboard table reading */ 

  }
  
  
  {
  
    /* Reading of instrument envelopes */
    
    {
    
      /* - Volume envelope */
      
      unsigned char flags ;
      
      (void) fread (&flags, (size_t) 1, (size_t) 1, module_file) ;
      /* flags */
      
      if (flags & 1)
        pinstrument->volume_envelope.envelope_on_off = 1 ;
      else
        pinstrument->volume_envelope.envelope_on_off = 0 ;
      /* volume envelope on/off */
         
      if (flags & 2)
        pinstrument->volume_envelope.loop_on_off = 1 ;
      else
        pinstrument->volume_envelope.loop_on_off = 0 ;
      /* volume envelope loop on/off */
    
      if (flags & 4)
        pinstrument->volume_envelope.sustain_loop_on_off = 1 ;
      else
        pinstrument->volume_envelope.sustain_loop_on_off = 0 ;
      /* volume envelope sustain loop on/off */

      (void) fread (&(pinstrument->volume_envelope.number_of_points), (size_t) 1, (size_t) 1, module_file) ;
      /* number of points of volume envelope */

      (void) fread (&(pinstrument->volume_envelope.loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* volume loop begin */

      (void) fread (&(pinstrument->volume_envelope.loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* volume loop end */

      (void) fread (&(pinstrument->volume_envelope.sustain_loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* volume sustain loop begin */

      (void) fread (&(pinstrument->volume_envelope.sustain_loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* volume sustain loop end */

      {

        /* Reading of volume envelope nodes */
        
        unsigned char node ;

        for (node = 0 ; node < 25 ; node ++)
        {
          unsigned char y_value ;

          (void) fread (&y_value, (size_t) 1, (size_t) 1, module_file) ;
          /* Y value */

          pinstrument->volume_envelope.nodes[node].y_value = (signed short) y_value ;

          (void) fread (&(pinstrument->volume_envelope.nodes[node].tick), (size_t) 2, (size_t) 1, module_file) ;
          /* tick */

        }

        /* End of volume envelope nodes reading */

      }

      pinstrument->volume_envelope.last_tick = pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.number_of_points - 1].tick ;
      pinstrument->volume_envelope.loop_begin_tick = pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.loop_begin].tick ;
      pinstrument->volume_envelope.loop_end_tick = pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.loop_end].tick ;
      pinstrument->volume_envelope.sustain_loop_begin_tick = pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.sustain_loop_begin].tick ;
      pinstrument->volume_envelope.sustain_loop_end_tick = pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.sustain_loop_end].tick ;
      if (pinstrument->volume_envelope.envelope_on_off == 1)
      { /* Volume envelope interpolation */

        unsigned char current_node ;

        pinstrument->interpolated_volume_envelope = memalloc (pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.number_of_points - 1].tick + 2) ;

        for (current_node = 0 ; current_node < (pinstrument->volume_envelope.number_of_points - 1) ; current_node ++)
        {
          unsigned short tick ;
          unsigned short first_tick ;
          unsigned short last_tick ;
          unsigned short number_of_ticks ;
          signed short delta ;
          signed short constant ;

          constant = pinstrument->volume_envelope.nodes[current_node].y_value ;
          first_tick = pinstrument->volume_envelope.nodes[current_node].tick ;
          last_tick = pinstrument->volume_envelope.nodes[current_node + 1].tick ;
          number_of_ticks = last_tick - first_tick ;
          delta =   ((signed short) pinstrument->volume_envelope.nodes[current_node + 1].y_value)
                  - ((signed short) pinstrument->volume_envelope.nodes[current_node].y_value) ;
          for (tick = first_tick ; tick < last_tick ; tick++)
          {
            pinstrument->interpolated_volume_envelope[tick] = (unsigned char)(constant + (delta * (signed short)(tick - first_tick)) / number_of_ticks) ;
          }

        } /* End of current_node loop */

        pinstrument->interpolated_volume_envelope[pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.number_of_points - 1].tick] = (unsigned char) pinstrument->volume_envelope.nodes[pinstrument->volume_envelope.number_of_points - 1].y_value ;

      } /* End of volume envelope interpolation */


      /* End of volume envelope reading */

    }

    (void) fseek (module_file, 1L, SEEK_CUR) ;
    /* adjusting file pointer on a word boundary */

    {

      /* - Panning envelope */

      unsigned char flags ;

      (void) fread (&flags, (size_t) 1, (size_t) 1, module_file) ;
      /* flags */

      if (flags & 1)
        pinstrument->panning_envelope.envelope_on_off = 1 ;
      else
        pinstrument->panning_envelope.envelope_on_off = 0 ;
      /* panning envelope on/off */

      if (flags & 2)
        pinstrument->panning_envelope.loop_on_off = 1 ;
      else
        pinstrument->panning_envelope.loop_on_off = 0 ;
      /* panning envelope loop on/off */

      if (flags & 4)
        pinstrument->panning_envelope.sustain_loop_on_off = 1 ;
      else
        pinstrument->panning_envelope.sustain_loop_on_off = 0 ;
      /* panning envelope sustain loop on/off */
        
      (void) fread (&(pinstrument->panning_envelope.number_of_points), (size_t) 1, (size_t) 1, module_file) ;
      /* number of points of panning envelope */
      
      (void) fread (&(pinstrument->panning_envelope.loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* panning loop begin */

      (void) fread (&(pinstrument->panning_envelope.loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* panning loop end */

      (void) fread (&(pinstrument->panning_envelope.sustain_loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* panning sustain loop begin */

      (void) fread (&(pinstrument->panning_envelope.sustain_loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* panning sustain loop end */

      {

        /* Reading of panning envelope nodes */

        unsigned char node ;

        for (node = 0 ; node < 25 ; node ++)
        {
          signed char y_value ;

          (void) fread (&(y_value), (size_t) 1, (size_t) 1, module_file) ;
          /* Y value */

          pinstrument->panning_envelope.nodes[node].y_value = (signed short) y_value ;

          (void) fread (&(pinstrument->panning_envelope.nodes[node].tick), (size_t) 2, (size_t) 1, module_file) ;
          /* tick */

        }

        /* End of panning envelope nodes reading */

      }

      pinstrument->panning_envelope.last_tick = pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.number_of_points - 1].tick ;
      pinstrument->panning_envelope.loop_begin_tick = pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.loop_begin].tick ;
      pinstrument->panning_envelope.loop_end_tick = pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.loop_end].tick ;
      pinstrument->panning_envelope.sustain_loop_begin_tick = pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.sustain_loop_begin].tick ;
      pinstrument->panning_envelope.sustain_loop_end_tick = pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.sustain_loop_end].tick ;

      if (pinstrument->panning_envelope.envelope_on_off == 1)
      { /* Panning envelope interpolation */

        unsigned char current_node ;

        pinstrument->interpolated_panning_envelope = memalloc (pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.number_of_points - 1].tick + 2) ;

        for (current_node = 0 ; current_node < (pinstrument->panning_envelope.number_of_points - 1) ; current_node ++)
        {
          unsigned short tick ;
          unsigned short first_tick ;
          unsigned short last_tick ;
          unsigned short number_of_ticks ;
          signed short delta ;
          signed short constant ;

          constant = pinstrument->panning_envelope.nodes[current_node].y_value ;
          first_tick = pinstrument->panning_envelope.nodes[current_node].tick ;
          last_tick = pinstrument->panning_envelope.nodes[current_node + 1].tick ;
          number_of_ticks = last_tick - first_tick ;
          delta =   ((signed short) pinstrument->panning_envelope.nodes[current_node + 1].y_value)
                  - ((signed short) pinstrument->panning_envelope.nodes[current_node].y_value) ;
          for (tick = first_tick ; tick < last_tick ; tick++)
          {
            pinstrument->interpolated_panning_envelope[tick] = (signed char)(constant + (delta * (signed short)(tick - first_tick)) / number_of_ticks) ;
          }

        } /* End of current_node loop */

        pinstrument->interpolated_panning_envelope[pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.number_of_points - 1].tick] = (signed char)pinstrument->panning_envelope.nodes[pinstrument->panning_envelope.number_of_points - 1].y_value ;

      } /* End of panning envelope interpolation */

      /* End of panning envelope reading */

    }

    (void) fseek (module_file, 1L, SEEK_CUR) ;
    /* adjusting file pointer on a word boundary */

    {

      /* - Pitch envelope */

      unsigned char flags ;

      (void) fread (&flags, (size_t) 1, (size_t) 1, module_file) ;
      /* flags */

      if (flags & 1)
        pinstrument->pitch_envelope.envelope_on_off = 1 ;
      else
        pinstrument->pitch_envelope.envelope_on_off = 0 ;
      /* pitch envelope on/off */

      if (flags & 2)
        pinstrument->pitch_envelope.loop_on_off = 1 ;
      else
        pinstrument->pitch_envelope.loop_on_off = 0 ;
      /* pitch envelope loop on/off */

      if (flags & 4)
        pinstrument->pitch_envelope.sustain_loop_on_off = 1 ;
      else
        pinstrument->pitch_envelope.sustain_loop_on_off = 0 ;
      /* pitch envelope sustain loop on/off */

      (void) fread (&(pinstrument->pitch_envelope.number_of_points), (size_t) 1, (size_t) 1, module_file) ;
      /* number of points of pitch envelope */

      (void) fread (&(pinstrument->pitch_envelope.loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* pitch loop begin */

      (void) fread (&(pinstrument->pitch_envelope.loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* pitch loop end */

      (void) fread (&(pinstrument->pitch_envelope.sustain_loop_begin), (size_t) 1, (size_t) 1, module_file) ;
      /* pitch sustain loop begin */

      (void) fread (&(pinstrument->pitch_envelope.sustain_loop_end), (size_t) 1, (size_t) 1, module_file) ;
      /* pitch sustain loop end */

      {

        /* Reading of pitch envelope nodes */

        unsigned char node ;

        for (node = 0 ; node < 25 ; node ++)
        {
          signed char y_value ;

          (void) fread (&(y_value), (size_t) 1, (size_t) 1, module_file) ;
          /* Y value */

          pinstrument->pitch_envelope.nodes[node].y_value = (signed short) y_value ;

          (void) fread (&(pinstrument->pitch_envelope.nodes[node].tick), (size_t) 2, (size_t) 1, module_file) ;
          /* tick */

        }

        /* End of pitch envelope nodes reading */

      }

      /* End of pitch envelope reading */

    }

    /* End of envelopes reading */

  }

  /* End of instrument reading */

  return 0 ;

}
   


int load_sample (FILE *module_file, p_sample psample)
/* load a sample form the 'module_file' file into the
   'sample' structure pointed by 'psample'.
   Code return : 0     - successful
                 other - failed */
{
  {

    /* Verification of the IT sample marker 'IMPS' */

    char buffer [5] ;
    size_t return_value ;
    
    return_value = fread (buffer, (size_t) 1, (size_t) 4, module_file) ;
    if (return_value == (size_t) -1)
      return -1 ; /* case when fread failed */

    if (strncmp ("IMPS", buffer, 4) != 0)
      return -1 ; /* this is not a normal IT instrument */

    /* End of the marker verification */

  }    

  /* Now we can assume (at least for now) that we
     have a regular IT sample */
     
  /* Reading of the sample parameters */
  
  (void) fread (psample->dos_file_name, (size_t) 1, (size_t) 12, module_file) ;
  /* dos file name */
  
  (void) fseek (module_file, 1L, SEEK_CUR) ;
  /* going to global volume field */

  (void) fread (&(psample->global_volume), (size_t) 1, (size_t) 1, module_file) ;
  /* global volume */
  
  {
  
    /* sample flags reading */
    
    unsigned char flags ;

    (void) fread (&flags, (size_t) 1, (size_t) 1, module_file) ;
    
    if (flags & 2)
      psample->is_16_bits_sample = 1 ;
    else
      psample->is_16_bits_sample = 0 ;
    /* sample quality */
      
    if (flags & 4)
      psample->is_stereo_sample = 1 ;
    else
      psample->is_stereo_sample = 0 ;   
    /* sample type */

    if (flags & 16)
      psample->use_loop = 1 ;
    else
      psample->use_loop = 0 ;
    /* sample loop */
      
    if (flags & 32)
      psample->use_sustain_loop = 1 ;
    else
      psample->use_sustain_loop = 0 ;   
    /* sample sustain loop */
    
    if (flags & 64)
      psample->loop_direction = 1 ;
    else
      psample->loop_direction = 0 ;   
    /* sample loop direction */

    if (flags & 128)
      psample->sustain_loop_direction = 1 ;
    else
      psample->sustain_loop_direction = 0 ;   
    /* sample sustain loop direction */
    
    /* end of flags reading */
    
    }
    
  (void) fread (&(psample->default_volume), (size_t) 1, (size_t) 1, module_file) ;
  /* default volume */

  (void) fread (psample->name, (size_t) 1, (size_t) 26, module_file) ;
  /* sample real name */
/*  printf ("Reading sample : %s \n", psample->name) ;*/

  {

    /* Reading of 'convert' field */

    unsigned short convert ;

    (void) fread (&convert, (size_t) 2, (size_t) 1, module_file) ;

    if (convert & 1)
      psample->is_signed_sample = 1 ;
    else
      psample->is_signed_sample = 0 ;

    /* End of 'convert' field reading */

  }

  (void) fread (&(psample->length), (size_t) 4, (size_t) 1, module_file) ;
  /* sample length */

/*  printf ("  length : %lu\n", psample->length) ;*/

  (void) fread (&(psample->loop_begin), (size_t) 4, (size_t) 1, module_file) ;
  /* loop begin */
/*  printf ("  loop beginning : %lu\n", psample->loop_begin) ;*/

  (void) fread (&(psample->loop_end), (size_t) 4, (size_t) 1, module_file) ;
  /* loop end */
/*  printf ("  loop end : %lu\n", psample->loop_end) ;*/

  (void) fread (&(psample->C5_speed), (size_t) 4, (size_t) 1, module_file) ;
  /* C-5 speed */

  (void) fread (&(psample->sustain_loop_begin), (size_t) 4, (size_t) 1, module_file) ;
  /* sustain loop begin */

  (void) fread (&(psample->sustain_loop_end), (size_t) 4, (size_t) 1, module_file) ;
  /* sustain loop end */

  {

    /* Sample data reading */

    long current_file_position ;

    long sample_data_position ;
        


    (void) fread (&sample_data_position, (size_t) 4, (size_t) 1, module_file) ;
    /* reading sample pointer */

    current_file_position = ftell (module_file) ;
    /* saving the current position of the file pointer */

    (void) fseek (module_file, sample_data_position, SEEK_SET) ;
    /* going to sample data */
    
    psample->sample_data = (double *) memalloc ((size_t) (8 * (psample->length + 256))) ;
    /* allocating memory for a mono sample */


    if (psample->is_16_bits_sample)
    {
      /* 16 bits sample */
      if (psample->is_signed_sample)
      {
        /* 16 bits signed sample */
        unsigned long i ;
        signed short data ;

        for (i = 0 ; i < psample->length ; i ++)
        {
          (void) fread (&data, (size_t) 2, (size_t) 1, module_file) ;
          /* reading a signed 16 bits sample */

          psample->sample_data[i] = (double)data ;
          /* saving this sample with a no conversion */
        }

        /* End : 16 bits signed sample */
      }
      else
      {
        /* 16 bits unsigned sample */
        unsigned long i ;
        unsigned short data ;
        signed long converted_data ;

        for (i = 0 ; i < psample->length ; i ++)
        {
          (void) fread (&data, (size_t) 2, (size_t) 1, module_file) ;
          /* reading a unsigned 16 bits sample */

          converted_data = ((signed long) data) - 32767 ; /* drop the out-of-bound samples */
          if (converted_data > 32767)
          {
            converted_data = 32767 ;
          }

          psample->sample_data[i] = (double) converted_data ;
          /* saving this sample with an unsigned -> signed conversion */
        }

        /* End : 16 bits unsigned sample */
      }
      /* End : 16 bits sample */
    }
    else
    {
      /* 8 bits sample */
      if (psample->is_signed_sample)
      {
        /* 8 bits signed sample */
        unsigned long i ;
        signed char data ;

        for (i = 0 ; i < psample->length ; i ++)
        {
          (void) fread (&data, (size_t) 1, (size_t) 1, module_file) ;
          /* reading a signed 8 bits sample */

          psample->sample_data[i] = (double)((signed long) data) * 256 ;
          /* saving this sample with a 8 bits -> 16 bits conversion */
        }

        /* End : 8 bits signed sample */
      }
      else
      {
        /* 8 bits unsigned sample */
        unsigned long i ;
        unsigned char data ;
        signed long converted_data ;

        for (i = 0 ; i < psample->length ; i ++)
        {
          (void) fread (&data, (size_t) 1, (size_t) 1, module_file) ;
          /* reading a unsigned 8 bits sample */

          converted_data = ((signed long) data) - 127 ; /* drop the out-of-bound samples */
          if (converted_data > 127)
          {
            converted_data = 127 ;
          }

          psample->sample_data[i] = (double)(converted_data * 256) ;
          /* saving this sample with a 8 bits -> 16 bits and unsigned -> signed conversion */
        }

        /* End : 8 bits unsigned sample */
      }
      /* End : 8 bits sample */

	  if ((psample->length >= 16) && !((psample->use_loop) || (psample->use_sustain_loop)))
	  {
		  unsigned long i ;
		  for (i = 0 ; i < 16 ; i++)
		  {
			  psample->sample_data[psample->length - i] *= i / 16.0 ;
		  }
	  }
    }

    fseek (module_file, current_file_position, SEEK_SET) ;
    /* restoring the file pointer after the
       'SamplePointer' field of the sample header */

    /* End of sample data reading */

  }

  (void) fread (&(psample->vibrato_speed), (size_t) 1, (size_t) 1, module_file) ;
  /* vibrato speed */

  (void) fread (&(psample->vibrato_depth), (size_t) 1, (size_t) 1, module_file) ;
  /* vibrato depth */

  (void) fread (&(psample->vibrato_rate), (size_t) 1, (size_t) 1, module_file) ;
  /* vibrato rate */

  (void) fread (&(psample->vibrato_waveform_type), (size_t) 1, (size_t) 1, module_file) ;
  /* vibrato waveform type */

  /* End of sample reading */
  
  return 0 ;

}



int load_pattern (FILE *module_file, p_pattern ppattern)
/* load a pattern form the 'module_file' file into the
   'pattern' structure pointed by 'ppattern'.
   Code return : 0     - successful
                 other - failed */
{


   /* unlike the other IT modules parts, there is not
      a marker for IT pattern header */

   (void) fread (&(ppattern->length), (size_t) 2, (size_t) 1, module_file) ;
   /* pattern length */
/*   printf ("Pattern length : %u\n", (unsigned)ppattern->length) ;*/

   (void) fread (&(ppattern->number_of_rows), (size_t) 2, (size_t) 1, module_file) ;
   /* number of rows in this pattern */
/*   printf ("  number of rows : %u\n", (unsigned)ppattern->number_of_rows) ;*/

   (void) fseek (module_file, 4L, SEEK_CUR) ;
   /* jumping over the 4 bytes padding */

   ppattern->packed_pattern_data = (unsigned char *) memalloc (ppattern->length) ;
   /* allocating 'ppattern->length' bytes for the pattern */

   (void) fread (ppattern->packed_pattern_data, (size_t) 1, (size_t) ppattern->length, module_file) ;

   /* End of pattern reading */

   return 0 ;
}



int load_module (const char *module_file_name, p_module pmodule)
/* load a module from the file 'module_file_name' into the 
   'module' structure pointed by 'pmodule'.
   Code return : 0     - successful
                 other - failed */
{
  
  /* local variables of 'load_module' */                 
  /*__________________________________*/
  
  FILE *module_file ; /* file descriptor pointer of the module file */
  

  /* 'load_module' body */
  /*____________________*/
  
  module_file = fopen (module_file_name, "rb") ; 
  /* Open the module file in binary mode */
  
  if (module_file == (FILE *)NULL)
    return -1 ;
  /* Verification of the validity of the file descriptor.
     A NULL file descriptor indicate that the 'fopen' function
     failed. */
     
  {

    /* Verification of the IT module marker 'IMPM' */

    char buffer [5] ;
    size_t return_value ;

    return_value = fread (buffer, (size_t) 1, (size_t) 4, module_file) ;
    if (return_value == (size_t) -1)
      return -1 ; /* case when fread failed */

    if (strncmp ("IMPM", buffer, 4) != 0)
      return -1 ; /* the file isn't a normal IT file */
      /* it may be a compressed IT file, but I don't
         support them yet */
  }

  /* Now we can assume (at least for now) that we
     have a regular IT module */
/*  printf ("File Marker OK : IMPM\n") ;*/

  /* Reading of the main module parameters */

  (void) fread (pmodule->name, (size_t) 1, (size_t) 26, module_file) ;
  /* module name */
/*  printf ("Module name %s\n", pmodule->name) ; */

  (void) fseek (module_file, 2L, SEEK_CUR) ;
  /* jumping over the 2 bytes padding */

  (void) fread (&(pmodule->number_of_orders), (size_t) 2, (size_t) 1, module_file) ;
  /* number of orders */
/*  printf ("Number of orders : %d\n", (int) pmodule->number_of_orders) ;*/

  (void) fread (&(pmodule->number_of_instruments), (size_t) 2, (size_t) 1, module_file) ;
  /* number of instruments */
/*  printf ("Number of instruments : %d\n", (int) pmodule->number_of_instruments) ;*/

  (void) fread (&(pmodule->number_of_samples), (size_t) 2, (size_t) 1, module_file) ;
  /* number of samples */
/*  printf ("Number of samples : %d\n", (int) pmodule->number_of_samples) ;*/

  (void) fread (&(pmodule->number_of_patterns), (size_t) 2, (size_t) 1, module_file) ;
  /* number of patterns */
/*  printf ("Number of patterns : %d\n", (int) pmodule->number_of_patterns) ;*/

  (void) fseek (module_file, 4L, SEEK_CUR) ;
  /* jumping over the 4 bytes of tracker versions fields */

  {
    /* reading of the module flags */

    unsigned short flags ;

    (void) fread (&flags, (size_t) 2, (size_t) 1, module_file) ;

    if (flags & 1)
      pmodule->is_stereo = 1 ; /* stereo module */
    else
      pmodule->is_stereo = 0 ; /* mono module */

/*    if (pmodule->is_stereo)
    {
      printf ("  Stereo module\n") ;
    }
    else
    {
      printf ("  Mono module\n") ;
    }*/

    if (flags & 4)
      pmodule->use_instruments = 1 ; /* use instruments */
    else
      pmodule->use_instruments = 0 ; /* use samples */

/*    if (pmodule->use_instruments)
    {
      printf ("  use instruments\n") ;
    }
    else
    {
      printf ("  use samples\n") ;
    }*/

    if (flags & 8)
      pmodule->type_of_slides = 1 ; /* use linear slides */
    else
      pmodule->type_of_slides = 0 ; /* use amiga slides */

/*    if (pmodule->type_of_slides)
    {
      printf ("  linear slides\n") ;
    }
    else
    {
      printf ("  Amiga slides\n") ;
    }*/

    if (flags & 16)
      pmodule->old_effects = 1 ; /* use old effects */
    else
      pmodule->old_effects = 0 ; /* use IT effects */


/*    if (pmodule->old_effects)
    {
      printf ("  old effects\n") ;
    }
    else
    {
      printf ("  IT effects\n") ;
    }*/

    /* end of flag reading */
    if (flags & 32)
      pmodule->g_effect = 1 ;
    else
      pmodule->g_effect = 0 ; 

  }

  {
    /* reading of the 'special' field */

    unsigned short special ;

    (void) fread (&special, (size_t) 2, (size_t) 1, module_file) ;

    if (special & 1)
      pmodule->has_a_message = 1 ;
    else
      pmodule->has_a_message = 0 ;

/*    if (pmodule->has_a_message)
    {
      printf ("  has a message\n") ;
    }
    else
    {
      printf ("  has no message\n") ;
    }*/

    /* end of 'special' field reading */
  }

  (void) fread (&(pmodule->global_volume), (size_t) 1, (size_t) 1, module_file) ;
  /* global volume */
/*  printf ("  Global volume : %d\n", (int)pmodule->global_volume) ;*/

  (void) fread (&(pmodule->mixing_volume), (size_t) 1, (size_t) 1, module_file) ;
  /* mixing volume */
/*  printf ("  Mixing volume : %d\n", (int)pmodule->mixing_volume) ;*/

  (void) fread (&(pmodule->initial_speed), (size_t) 1, (size_t) 1, module_file) ;
  /* initial speed */
/*  printf ("  Initial speed : %d\n", (int)pmodule->initial_speed) ;*/

  (void) fread (&(pmodule->initial_tempo), (size_t) 1, (size_t) 1, module_file) ;
  /* initial tempo */
/*  printf ("  Initial tempo : %d\n", (int)pmodule->initial_tempo) ;*/

  (void) fread (&(pmodule->panning_separation), (size_t) 1, (size_t) 1, module_file) ;
  /* panning separation */
/*  printf ("  Panning separation : %d\n", (int)pmodule->panning_separation) ;*/

  (void) fseek (module_file, 1L, SEEK_CUR) ;
  /* jumping over the 1 byte padding */

  if (pmodule->has_a_message)
  {
    /* reading of the module message */

    unsigned short message_length ;

    unsigned long message_offset ;

    (void) fread (&message_length, (size_t) 2, (size_t) 1, module_file) ;
    /* message length */

    (void) fread (&message_offset, (size_t) 4, (size_t) 1, module_file) ;
    /* message offset from the beginning of the file */

    (void) fseek (module_file, message_offset, SEEK_SET) ;
    /* positionning the file pointer to the beginning of the module message */

    pmodule->message = (char *) memalloc ((size_t) (message_length + 1)) ;
    /* allocating memory for the module message */

    (void) fread (pmodule->message, (size_t) 1, (size_t) message_length, module_file) ;
    /* reading the message */
  
    pmodule->message[message_length] = (char) 0 ; 
    /* making sure that the message string is terminated */
  
    (void) fseek (module_file, 0x40L, SEEK_SET) ;
    /* resuming the file pointer */

    /* end of the module message reading */
  }

  else
    (void) fseek (module_file, 10L, SEEK_CUR) ;
    /* jumping over the 10 bytes padding */

  (void) fread (pmodule->channels_pannings, (size_t) 1, (size_t) 64, module_file) ;
  /* reading channels pannings */

  (void) fread (pmodule->channels_volumes, (size_t) 1, (size_t) 64, module_file) ;
  /* reading channels volumes */

  pmodule->orders = (unsigned char *) memalloc ((size_t) pmodule->number_of_orders) ;
  /* allocating memory for orders list */

  (void) fread (pmodule->orders, (size_t) 1, (size_t) pmodule->number_of_orders, module_file) ;
  /* reading orders */

  if (pmodule->use_instruments)
  {
    /* Reading of instruments */

    unsigned short i ;


/*    printf ("Loading instruments ...\n") ;*/

    for (i = 0 ; i < pmodule->number_of_instruments ; i ++)
    {
      long current_file_position ;

      long instrument_offset ;



      (void) fread (&instrument_offset, (size_t) 4, (size_t) 1, module_file) ;
      /* reading the instrument offset */

      current_file_position = ftell (module_file) ;
      /* saving the current file pointer position */

      (void) fseek (module_file, instrument_offset, SEEK_SET) ;
      /* going to the start point of the instrument in the module file */

      pmodule->instruments[i] = (p_instrument) memalloc (sizeof (instrument)) ;
      /* allocating memory for the instrument */

      if (load_instrument (module_file, pmodule->instruments[i]) != 0)
        return -1 ;
      /* loading the instrument */

      (void) fseek (module_file, current_file_position, SEEK_SET) ;
      /* return to previous position (next instrument offset) in the file */

    }

/*    printf ("Instruments loaded \n") ;*/
    /* End of instruments reading */

  }



  {
    /* Reading of samples */

    unsigned short i ;


/*    printf ("Loading samples ...\n");*/

    for (i = 0 ; i < pmodule->number_of_samples ; i ++)
    {
      long current_file_position ;

      long sample_offset ;

/*      printf ("Sample : %d\n", i);*/

/*      printf ("Sample offset\n");*/

      (void) fread (&sample_offset, (size_t) 4, (size_t) 1, module_file) ;
      /* reading the sample offset */

/*      printf ("Saving file pointer\n");*/
      current_file_position = ftell (module_file) ;
      /* saving the current file pointer position */

/*      printf ("Going to sample\n");*/
      (void) fseek (module_file, sample_offset, SEEK_SET) ;
      /* going to the start point of the sample in the module file */

/*      printf ("Allocating memory\n") ;*/
      pmodule->samples[i] = (p_sample) memalloc (sizeof (sample)) ;
      /* allocating memory for the sample */

/*      printf ("Calling load_sample\n") ;*/
      if (load_sample (module_file, pmodule->samples[i]) != 0)
        return -1 ;
      /* loading the sample */

/*      printf ("Returning to previous position\n") ;*/
      (void) fseek (module_file, current_file_position, SEEK_SET) ;
      /* return to previous position (next sample offset) in the file */

    }

/*    printf ("Samples loaded\n" ) ;*/

    /* End of samples reading */

  }



  {
    /* Reading of patterns */

    unsigned short i ;

/*    printf ("Loading patterns ...\n") ;*/


    for (i = 0 ; i < pmodule->number_of_patterns ; i ++)
    {
      long current_file_position ;

      long pattern_offset ;



      (void) fread (&pattern_offset, (size_t) 4, (size_t) 1, module_file) ;
      /* reading the pattern offset */

      current_file_position = ftell (module_file) ;
      /* saving the current file pointer position */

      (void) fseek (module_file, pattern_offset, SEEK_SET) ;
      /* going to the start point of the pattern in the module file */

      pmodule->patterns[i] = (p_pattern) memalloc (sizeof (pattern)) ;
      /* allocating memory for the pattern */

      if (load_pattern (module_file, pmodule->patterns[i]) != 0)
        return -1 ;
      /* loading the pattern */
      
      (void) fseek (module_file, current_file_position, SEEK_SET) ;
      /* return to previous position (next pattern offset) in the file */

    }

/*    printf ("Patterns loaded\n") ;*/

    /* End of patterns reading */

  }

  /* End of module loading function */

  fclose (module_file) ;

  return 0 ;

}

/************/
/* loader.c */
/************/

