COMMENT #

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

The following text is information that I have acquired through my own
dealings with protected mode.  In no way do I claim this information to
be accurate, this is not a technical specification.  If you have any
questions regarding the information presented here, feel free to send me
e-mail at fysx@spartan.pei.edu (I try to answer all mail).

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

How will this information will be useful to you?

The information here may be useful as a guideline for producing your own
protected mode kernel.  You will require outside information and
resources.  I recommend you get the sources to as many DOS extenders as
you can.  Visit the Intel home page; grab their textfile on the 80386
(386intel.txt).  If you ever get really stumped, e-mail me, or someone
else you know, for assistance.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

My experience with protected mode.

On March 19th of 1995, I began writting my own protected mode entry
code.  Like many individuals, I wanted to know how something worked,
rather than simply be content to use it.  So, I decided, no better way
to learn how to code under protected mode, then to write my own
protected mode kernel.  I have been programming under my own kernel
since then.  It is currently 3.5Kb in size, supports RAW/XMS/DPMI entry,
contains functions for handling of files, memory, and system items
(such as interrupts, psp, descriptor allocation, etc), and works with TASM,
MASM, Watcom C, and Borland C.  Presently it is for proprietary use
only, but may be made public if there is sufficient request.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Entering protected mode.

The most familiar modes of entry are RAW, XMS, VCPI, and DPMI.  Each of
these are stereo-types for the behavior of memory managers.  RAW refers
to a state where high memory is not being managed by any device other
then the BIOS.  XMS (eXtended Memory Specification) is a set of
functions so that real mode programs may access memory above 1Mb.
HIMEM.SYS is a common XMS driver.  VCPI (Virtual Control Program
Interface) was created so that programs may gain access to protected
mode while under an EMS (Expanded Memory System) memory manager.  VCPI
was required due to the way EMS allows programs access to memory above
1Mb.  DPMI (DOS Protected Mode Interface) has become the present
standard of protected mode functions.  It was developed so that DOS
programs may have access to protected mode, without any compromise to
system integrity.

Of the four entry modes, each has a unique protected mode entry form.
While RAW/XMS require you to perform all of the protected mode setup and
operation, there are subtle differences regarding how memory above 1Mb
is accessed, and access enabled.  VCPI is a dead standard and will not
be discussed further.  DPMI requires that you ask the DPMI server to go
to protected mode.

DPMI is the simplest form of entry and maintainence, and is also the most
common interface for system routines.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

RAW/XMS Protected Mode.

This section will cover the basics of entering and maintaining a USE32
semi-flat mode of execution.  Only information relating to this function
will be discussed in this document.  If you require information beyond
this scope, I recommend you get some good documents on the Intel
processor, and would hope that you are willing to apply many hours of
your free time to the subject.  Information outside of this scope will
not have a real world application, and would be purely for educational
purposes.  In fact, information relating to RAW/XMS protected mode has
little real world application at the present, with the current surplus
of DOS extenders and the mad rush to multi-tasking operating systems
(such as Win95), but I have decided to include a minimal description of
the operations under this mode for educational purposes of the many who
have asked for this information.  Perhaps someone will write better
protected mode code because they have an understanding of how protected
mode works?


When asked "How do you enter protected mode?" the most common reply I
give is:

mov eax,1
mov cr0,eax

This is correct, although not very informative, and incomplete.  It is
a "first step".

The segment registers (CS, DS, ES, FS, GS, SS) contain a 16-bit visible
portion (selector/segment) and a 64-bit invisible portion (descriptor).
The CPU mode (real/protected/v86/smm) determines how the invisible
portion will be loaded in relation to the visible portion.

A segment descriptor is a structure containing information about a
"segments" size, base address, access rights, etc.  A selector is a
pointer into a descriptor table.

When you load a segment register in protected mode, the visible portion
is loaded with a selector, the invisible portion is loaded with the
descriptor located by the selector.  (When a segment register is loaded
in real/v86 mode, the processor creates the appropriate descriptor
information and fills the invisible portion for you.)

So in addition to enabling the protected mode bit of CR0 (Control
Register 0) we need to create a segment descriptor table.

Two types of segment descriptor tables are available.  A GDT (Global
Descriptor Table) and an LDT (Local Descriptor Table).  Here we will
only deal with the GDT.  A GDTR (Global Descriptor Table Register)
contains the size and location of this table.

After we have created our segment descriptor table, we need to tell the
CPU where our table is located (see the instruction lgdt for additional
information).  We do this by loading the GDTR with the appropriate
structure.  (We can do this in any operating mode, as protected mode is
the only mode that will use this table.)

>From then on the CPU knows where our descriptors are located.  If we
load a segment register (while in protected mode) the CPU will then
fetch the appropriate descriptor.

At this point we have enough knowledge that we can make the jump to
protected mode, and create memory "segments" that our code and data will
have access to.  But, we can't have interrupts.

The PIC also has an equivalent of a real and protected mode (or more
appropriately a DWORD and QWORD mode).  This mode determines the size of
an interrupt descriptor (don't confuse this with a segment descriptor).

The DWORD mode is provided for compatibility with real mode.  The
familiar interrupt table located at 00000000h, with the format
offset:segment, addressable by IntNum*4, is actually a DWORD IDT
(Interrupt Descriptor Table).  Similar to the GDT, the IDT has an IDTR
(Interrupt Descriptor Table Register) which also contains the size and
location of the table.  In real mode the value of the IDTR has a size of
400h (4*256) and a location of 000000000h.  This is important when
switching back to real mode.  Because real mode uses an IDT you can
rellocate the real mode interrupt table.  I will cover why you would
want to do this later.

Protected mode uses a QWORD version of the interrupt descriptor.  This
interrupt descriptor (like the segment descriptor) contains the address
of the interrupt routine, its access/priviledge, type, etc.

After creating a protected mode IDT we tell the CPU where it is
located (same as the GDT), then we may enable interrupts for protected
mode (see lidt).

At this point, we know enough to jump to protected mode, load our
segment registers with a selector that points to a descriptor that maps
all of our USE32 segment, and we have a protected mode interrupt table.
Provided we don't want to call any real mode routines, we've got
everything we need.

Here are some examples.

Say our USE32 segment is defined as

Code32 SEGMENT PARA PUBLIC USE32
Code32 ENDS

(you want to make sure its para aligned, for ease of calculating linear
address) then the code to create a descriptor for that segment is as
follows:

  ; get linear base address of Code32 segment
  mov eax,Code32
  shl eax,4
  mov Code32Addr,eax

  ; fix up Code32 and Data32 descriptors
  mov eax,m Code32addr
  or dword ptr GDT_Code32+2,eax
  or dword ptr GDT_Data32+2,eax

Code32Addr is useful for translating linear addresses into Code32
relative addresses, such as the video frame buffer

  mov edi,000A0000h
  sub edi,Code32Addr

(now EDI can be used to reference the display)


Our global descriptor table is defined as follows:


GDT     LABEL     BYTE
GDT_NULL          db      8 DUP (0)
GDT_Code32        db      0FFh, 0FFh,   0h,   0h,   0h,  9Ah, 0CFh,   0h
GDT_Data32        db      0FFh, 0FFh,   0h,   0h,   0h,  9Ah, 0CFh,   0h
GDT_Flat32        db      0FFh, 0FFh,   0h,   0h,   0h,  92h, 0CFh,   0h
GDT_CodeKernel    db      0FFh, 0FFh,   0h,   0h,   0h,  9Ah,   0h,   0h
GDT_DataKernel    db      0FFh, 0FFh,   0h,   0h,   0h,  92h,   0h,   0h
GDT_Stack32       db      0FFh, 0FFh,   0h,   0h,   0h,  92h,   0h,   0h

*** note *** the first entry must be NULL
(for information on the format of a segment descriptor see 386intel.txt)

the structure for the GDTR is:

; Reference to the Global Descriptor Table
pm_GDT            LABEL   FWORD
                  dw      $-GDT       ; size of GDT
pm_GDTaddr        dd      offset GDT  ; ***note*** add Code32Addr to
                                      ; this, so its value will be
                                      ; linear


Now what makes things prettier, and necessary under DPMI, is a selector
table.

Selectors         LABEL   WORD
Code32Sel         dw      8h
Data32Sel         dw      10h
Flat32Sel         dw      18h
CodeKernelSel     dw      20h
DataKernelSel     dw      28h
Stack16Sel        dw      30h


We now have setup the GDT.  So lets switch to protected mode, then
continue (entry is in a USE16 segment).

  cli               ; dissable interrupts
  mov eax,1         ; set PE bit
  mov cr0,eax       ; enable protected mode
  jmp $+02h         ; reload IPQ
  lgdt pm_GDT       ; load Global Descriptor Table Register

  mov ax,Data32Sel  ; load segment registers
  mov ds,ax
  mov es,ax
  mov fs,ax
  mov gs,ax
  mov ss,ax
  mov esp,offset Stack+STACKSIZE

  ; make a 64-bit return to our USE32 segment
  push dword ptr Code32Sel    ; code selector
  push large offset Enter32   ; entry routine
  db 66h, 0CBh                ; 64-bit return


At this point we're executing in a semi-flat environment, with interrupts
disabled.  CS, DS, ES, FS, GS, SS contain descriptors with 4G limits and
a base of our Code32 segment.
Observe the jmp $+02h to reload the IPQ.  This is absolutely necessary!!


>From here we build an interrupt descriptor table.

; Protected Mode Interrupt Descriptor Table
pmIDT dq 256 dup (0)

; Reference to Protected Mode Interrupt Descriptor Table
pm_IDT            LABEL   FWORD
                  dw      256*8h
pm_IDTaddr        dd      o pmIDT


; Reference to Real Mode Interrupt Descriptor Table upon entry
r0_IDT            LABEL   FWORD
                  dw      256*4h
r0_IDTaddr        dd      0


  ; Create pmIDT table
  ; 256 entries pointing to NULL_ISR
  mov cx,256
  mov edi,offset pmIDT
  mov edx,offset NULL_ISR
  sub edx,Code32addr                  ; make edx relative to Code32
  Make_pmIDT_Table:
    mov ax,dx
    stosw
    mov ax,Code32Sel
    stosw
    mov ax,8E00h
    stosw
    shld eax,edx,16
    stosw
    loop Make_pmIDT_Table

and NULL_ISR is:

NULL_ISR:
  iretd


so we can now load the IDTR  (but first we want to save the original for
return to real mode)

  sidt r0_IDT
  lidt pm_IDT
  sti


There you have it, executing in protected mode.

Now we're going to want some functions for setting/getting the interrupt
routine addresses.

;Ŀ;
; Get Protected mode interrupt descriptor                                   ;
; Entry:  BL - Interrupt Vector                                             ;
; Exit:   AX:EDX - Sel:Ofs of address                                       ;
;;
GetPMInt:
  movzx edi,bl
  shl edi,3
  add edi,pm_IDTaddr
  sub edi,Code32addr
  mov dx,word ptr [edi+6]
  shl edx,16
  mov dx,word ptr [edi]
  mov ax,word ptr [edi+2]
  ret


;Ŀ;
; Set Protected mode interrupt descriptor                                   ;
; Entry:  BL - Interrupt Vector                                             ;
;         AX:EDX - Sel:Ofs of address                                       ;
;;
SetPMInt:
  movzx edi,bl
  shl edi,3
  add edi,pm_IDTaddr
  sub edi,Code32addr
  cli
  mov word ptr [edi],dx
  shr edx,16
  mov word ptr [edi+6],dx
  mov word ptr [edi+2],ax
  sti
  ret



We probably want to be able to go to real mode, to service interrupts,
load files, etc.


To return to real mode:

you want to make sure your executing in a USE16 segment, so likely this
requires you to make another 64-bit return to the kernel segment

  push dword ptr CodeKernelSel    ; code selector
  push large offset rmIntCB       ; entry routine
  retf                            ; 64-bit return


rmIntCB:

  ; calculate real mode SS:ESP pair for protected mode SS:ESP
  mov eax,Code32Addr
  add eax,esp
  mov word ptr t_ESP  ,ax
  mov word ptr t_ESP+2,0
  xor ax,ax
  shr eax,4
  mov m t_SS,ax

  cli                         ; disable interrupts
  mov ss,Stack16Sel           ; load SS with 16 byte stack
  lidt rm_IDT                 ; load real mode IDT
  xor eax,eax                 ; jump to real mode
  mov cr0,eax
  db 0EAh                     ; reload IPQ and load a valid CS for
  dw $+04                     ; a return from any call we may make after
  dw Kernel                   ; this point

  mov ax,Code32               ; load real mode segment registers
  mov ds,ax
  mov es,ax
  mov fs,ax
  mov gs,ax
  mov ss,t_SS
  mov esp,t_ESP

  sti                         ; enable interrupts


  t_SS dw 0                   ; temporary holders for real mode
  t_ESP dd 0



That should get you safely back to real mode.
Of course, for interrupt redirection, or function calling across modes,
you probably want to do some code to preserve registers and flags.  You
should be able to determine the return to protected mode; follow
those steps in reverse.

If you noticed, I used rm_IDT instead of r0_IDT.  For processing real
mode interrupts, I temporarily relocate the real mode Interrupt table.
When an interrupt occurs, I will get control, from there I can determine
if it should carry through to the real mode handler, or back to
protected mode (in the case of an IRQ).  This is only necessary if you
will be doing processing of real mode interrupts, in real time, while
you have a protected mode ISR that must be serviced on each IRQ.
An alternative to this method is to code your ISR's in real mode, but
that sort of defeats the purpose of programming in protected mode.  In
any event, you will want to get interrupt calling working first, and
then attempt to get protected mode call backs to work.

I believe that covers the fundamentals of RAW/XMS entry and exit.  It's
not much of a kernel yet.  You will want to add functions for file and
memory handling.  File handling will require a buffer in low memory, and
a heap manager will require asking the BIOS or an XMS driver where
memory is available and how much there is.  In addition to this, you may
want to support some common DPMI functions, such as descriptor
modification and allocation.



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

DPMI Entry.

If you just finished reading about RAW/XMS entry, don't get too fearful.
DPMI is the easiest form of protected mode entry.

The method I will cover requires a few similar steps to RAW/XMS, although
they aren't necessary because of the variety of functions provided by
DPMI.

First item on the list is to rellocate your GDT as you did in RAW/XMS.

Next you want to detect whether a DPMI server is present.  If it isn't
you can't do a DPMI entry.

Then you ask DPMI to go to protected mode.  Here DPMI converts your
real mode segments into protected mode selectors, switches to protected
mode, and reloads the segment registers with selector equivalents
(nothing will appear different then before the call).

At this point, you just allocate N desciptors (one for each entry in
your GDT), then copy your GDT into the descriptors DPMI returned.  Now,
jump to your USE32 segment, and begin executing the main program.

To terminate DPMI you use the standard DOS exit:

  mov ah,4Ch
  int 21h

All interrupts will be setup upon entry.  Because of the differing
implementations of DPMI, some interrupts will be redirected to real
mode, while others are not.  Some functions will be extended (such as
file) other times, not.  I would advise that ANY system function, that
requires a segment:offset pair (such as Function 9h Int 21h), be
redirected to the real mode function using the DPMI simulate real mode
interrupt (Function 0300h) or the DPMI call real mode procedure with far
return frame (Function 0301h).

You may want to write your kernel routines in a way such that if DPMI
is present, DPMI will perform your request, and if DPMI is not, it will
be performed by the kernel.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Writting a small protected mode kernel.

I am assuming that because this document will be focused around the
demo scene, you are interested in writting a small kernel for 4Kb or
64Kb intros.  I feel the need to write an entire kernel, unnecessary
under those circumstances.  The future of demo competitions will likely
allow the extender to be separate from the presentation executable.  If
the competition allows this, I would advise that you use a publically
available protected mode extender and stub loader, or a DPMI only
protected mode kernel.

If you are writting a protected mode executable for a stub loader, it
will be up to you to create the necessary header.  You may need to
refer to the loaders documentation.

If you choose to write a DPMI only kernel, this is a fairly trivial
task.  DPMI takes care of everything for you.  You simply need to detect
the pressence of a DPMI server, and then asks its permission to go to
protected mode.  From there you have full DPMI functionality (I'm sure
most of you have become accustomed to using the DPMI function sets).

I have included the soure for a small (224 bytes) DPMI only kernel.  It's
very barebones, and you may possibly get it smaller still.  To test the
kernel you will need a copy of TASM and TLINK (something else may work).
Simply compile and link this textfile.  The executable should be 841
bytes (512 bytes for the .EXE header (this can be shrunk to 32 bytes),
224 bytes for the protected mode kernel, and the rest for the background
routine (by karl/nooon I think?)).  You will also need a DPMI server
(Win95 or CWSDPMI will do fine).  If the code causes an exception upon
start up, try increasing the size of Stack16 segment; some DPMI
servers require a larger stack to perform their initialization.

TASM /m2 KERNEL.ASM
TLINK /3 KERNEL.OBJ, KERNEL.EXE

If you have any questions regarding this code, contact me.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Pre-emptive multi-tasking or multi-threading.

This section is not meant for multi-tasking of two programs, it is meant
for multi-threading two routines internal to your own program.

Why would you want to do that?
Say you wrote an editor that saves out a large file.  When the user
selects save, rather then making him/her wait for it's completion, you
spawn another thread that will save in the background.
Or, you want to decompress samples will the user is at the sound card
selection screen to an invitro.
Or, you want to upload samples to the GUS while displaying a flashy
effect.


The concept is simple.  You enter the ISR for IRQ0 from one address, and
return on a different address.

You need a structure for each thread that contains ALL registers:


STRUC dThread
  FPU_State   db 108  dup (?)
  Gen_Regs    dd 8    dup (?)
  Seg_Regs    dd 4    dup (?)
  reg_ESP     dd ?
  reg_SS      dd ?
  reg_EFLAGS  dd ?
  reg_CS      dd ?
  reg_EIP     dd ?
ENDS

Upon entry to the ISR, you save every register on the stack:

pushad
push ds es fs gs ss

then you determine the address of the current thread's structure, and
copy the registers from the stack to their appropriate addresses.
Remember to use ESP immediately after the interupt.  Then use fsave
[ebx.FPU_State] to save the FPU state.

Now, increment the thread structure pointer to the next thread
structure.  Load the stack up with the appropriate registers, frstor
restores the FPU state.

and

pop ss gs fs es ds
popad
iretd

as easy as that, you've switched threads.



now, to set the tasks in motion, your going to want to initialize CS,
EIP, ESP, and SS.  Then hook IRQ0.  Once you get that going, you can get
fancier.

#



;ͻ;
; A DPMI only protected mode kernel.                                        ;
; Copyright (c) 1997 by Maxwell Sayles.  All rights reserved.               ;
;ͼ;

  LOCALS
  JUMPS
  .386p

; A few macros to make type casting easier
q EQU qword ptr
d EQU dword ptr
w EQU word ptr
b EQU byte ptr
o EQU offset
m EQU small
l EQU large
s EQU short

; defines
STACK32SIZE       EQU (16384)     ; must be < 64Kb
GDTENTRIES        EQU (2)


;ͻ;
; All code for initializing and entering Protected Mode                     ;
;ͼ;
Kernel    SEGMENT PARA PUBLIC USE16
          ASSUME CS:Kernel, DS:Code32

;Ŀ;
; ********** DOS Entrance ************************************************  ;
;;
Entry16:

;*****************************************************************************;
  cld                       ; Clear direction flag

  ; Set DS to Code32 segment
  push w Code32
  pop ds

  ; Resize PSP
  mov ax,es
  mov bx,ss
  inc bx
  mov ah,4Ah
  int 21h

  ; Test for DPMI (and 386)
  mov ax,1687h
  int 2Fh
  test ax,ax
  jnz DPMI_ErrorExit

  ; Test for 32-bit Protected Mode support
  test bx,1
  jz DPMI_ErrorExit

  ; Save DPMI mode-switch entry point
  mov m w DPMI_ModeSwitch  ,di
  mov m w DPMI_ModeSwitch+2,es

  ; Allocate memory for use by DPMI if necessary
  test si,si
  jz EnterDPMI_ProtMode
  mov bx, si
  mov ah, 48h
  int 21h
  jc DPMI_ErrorExit
  mov es, ax

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; calculate linear addressess and build GDT

  ; get linear base address of Code32 segment
  mov eax,Code32
  shl eax,4
  mov m Code32addr,eax

  ; fix up Code32 and Data32 descriptors
  or m d GDT_Code32+2,eax
  or m d GDT_Data32+2,eax


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Enter protected mode as a 32-bit program
  EnterDPMI_ProtMode:
  mov ax,1
  call DPMI_ModeSwitch
  jc DPMI_ErrorExit


;*****************************************************************************;
  ; In 16-bit protected mode.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Allocate and set descriptors
  push ds
  pop es
  mov edi,o GDT
  mov esi,o Selectors
  mov dx,GDTENTRIES
  SelLoop:
    ; Allocate LDT descriptor
    xor ax,ax
    mov cx,1
    int 31h
    ; Save selector
    mov [esi],ax
    add esi,2
    ; Set descriptor
    mov bx,ax
    mov ax,000Ch
    or b [edi+5],01100000b    ; set CPL 3
    int 31h
    add edi,8
    dec dx
    jnz SelLoop


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Jump to USE32 segment
  push m d Code32Sel
  push l o Enter32
  db 66h, 0CBh


;Ŀ;
; Error Exit.                                                               ;
; Prints Error Message, and returns to DOS with error code 0FFh             ;
;;
DPMI_ErrorExit:
  mov dx,m o NoDPMI_MSG
  mov ah,9h
  int 21h
  mov ax,4CFFh
  int 21h


Kernel    ENDS
;ͻ;
; FMPI segment ends.                                                        ;
;ͼ;


;ͻ;
; Contains code and data for 32-bit Protected Mode.                         ;
;ͼ;
Code32    SEGMENT PARA PUBLIC USE32
          ASSUME CS:Code32, DS:Code32

;Ŀ;
; Error Messages                                                            ;
;;
NoDPMI_MSG LABEL
db 'DPMI Error$'


;Ŀ;
; Global Descriptor Table                                                   ;
;;
GDT     LABEL     BYTE
GDT_Code32        db      0FFh, 0FFh,   0h,   0h,   0h,  9Ah, 0CFh,   0h
GDT_Data32        db      0FFh, 0FFh,   0h,   0h,   0h,  92h, 0CFh,   0h



;Ŀ;
; Initialize system registers and variables then jump to Start32            ;
;;
Enter32:
  mov ax,m CS:Data32Sel
  mov ds,ax
  mov es,ax
  mov ss,ax
  mov esp,o Stack32+STACK32SIZE


;Ŀ;
; Your program begins here.                                                 ;
; You have complete access to all DPMI functions.                           ;
; You are running in a USE32 segment under a semi-flat memory model:        ;
;   CS = DS = ES = SS = Linear address of Code32 Segment                    ;
; You probably want to jump to a different object to keep things tidy.      ;
;;

  ; set 320x200x8bpp
  mov ax,13h
  int 10h

  ; set 16 color gray-scale palette
  mov	 al, 16
  @@10:
    mov dx,3C8h
    out dx,al
    inc edx
    out dx,al
    out dx,al
    out dx,al
    dec al
    jnz @@10

  ; draw fire (routine by karl/nooon (i think?))
  mov edi,61336                 ; where the fire generator begins
  mov ecx,80*865                ; fire height
  mov dh,cl

  mov esi,000A0000h
  sub esi,m Code32Addr          ; address of frame buffer relative to Code32

  @@20:
    shr b [esi+edi-320*2],4
    mov ah,dh
    sar ah,4
    inc ah
    mov al,[esi+edi-320]
    add al,[esi+edi-319]
    rcr al,1
    add al,ah
    rol dh,cl
    mov [esi+edi],al
    inc di
    xor dh,al
    loop @@20

  ; wait for keypress
  xor eax,eax
  int 16h

  ; set 80x50 textmode
  mov ax,03h
  int 10h
  mov ax,1112h
  xor bl,bl
  int 10h


;Ŀ;
; Return to Real Mode and terminate                                         ;
;;
Exit32:
  mov ah,4Ch
  int 21h


;Ŀ;
; Variables                                                                 ;
;;
DPMI_ModeSwitch   dd      ?       ; DPMI mode switch routine entry address
Code32addr        dd      ?       ; Linear address of Code32 segment


;Ŀ;
; Selector Values                                                           ;
; (Must follow same order as descriptors)                                   ;
;;
Selectors         LABEL   WORD
Code32Sel         dw      ?
Data32Sel         dw      ?


;Ŀ;
; 32-bit stack                                                              ;
;;
Stack32 db STACK32SIZE dup (?)


Code32    ENDS
;ͻ;
; 32-bit segment ends.                                                      ;
;ͼ;


;ͻ;
; Temporary entry stack.                                                    ;
;                                                                           ;
; Windows 95 requires a 16 byte stack.                                      ;
; CWSDPMI requires a 32 byte stack.                                         ;
; QDPMI requires a 96 byte stack.                                           ;
;                                                                           ;
;ͼ;
Stack16   SEGMENT PARA PUBLIC STACK 'STACK'
  db 96 dup (?)
Stack16   ENDS
;ͻ;
; Stack ends.                                                               ;
;ͼ;


  END     Entry16