{
 256xfade - a program to crossfade between 2 or more 256-color pictures.
 Written by Trixter / Hornet (Jim Leonard; trixter@mcs.com).
 Based on code by Jussi Lahdenniemi / Virtual Visions.  8/7/95

 256 to 256 crossfading is technically impossible, so you have to cheat.
 How, that depends... :)  You can try the crossfading with a raster.

 Let's take two 320x200 pictures with 256 (different) colors as an example.

 1. Set the video mode to 320x400.

 2. Load the first picture into video memory so that the pixels are set
    with raster... like this:

    x x x x
     x x x x
    x x x x
     x x x x

    So every other pixel in picture is "halved".
    Use video page 0.

 3. Load the second picture into the opposite raster, like this:

     x x x x
    x x x x
     x x x x
    x x x x

    Use video page 1.

 4. Set palette correctly for the first picture, and set page 0 as visual.
 5. Wait for vertical retrace.
 6. Set palette to totally black and page 1 as visual.
 7. Wait for vertical retrace.

 8. Repeat from number four until you want to start fading.

 9. a=0
 10. Set palette for first picture faded (100-a)% towards black, vis.page 0
 11. Vertical retrace.
 12. Palette for second picture faded a% towards the correct palette, vis.pg 1
 13. Vertical retrace.
 14. Increase a until it reaches 100.

 15. You got it!

 This method has one very bad drawback: your picture's colors are dimmed
 to 25% from the original (in a good monitor; the worse the monitor, the
 better the effect ;) because half of the time you're flipping pages,
 you're displaying a color pixel, then a totally black one, then color,
 then black, etc.  With bright colors or white, the flicker is extremely
 bad.  So, how to combat that?

 Instead of swapping between a color and a black pixel, swap between a
 color and a "shadow" pixel that tries to resemble the color one.
 There's not enough palette entries to make a color "shadow" picture (it
 would take anywhere from 48-64 palette entries, which steals away from
 the 256 colors in the picture), but since our eyes are more sensitive
 to light, we can create a 16-element grayscale picture and still have
 240 colors left over for the picture.  Not a bad compromise.

 HOWEVER, this adds a third problem:  When flipping between a color and
 a gray pixel, the colors are now *lightened* by about 25%.  The effect
 looks like all the colors are "washed out".  So, to combat this, we
 pre-process all the pictures in a batch image conversion program like
 Paint Shop Pro, raising the color saturation by about 75%.

 Then, all looks good.  :)
}

{$M 16384,0,393216} {384K heap}
{$f+,r-,q-,N-,E-,G+,A+}
{$l vrs.obj}

{---------------------------Definition section----------------------------}

const
  maxfiles=10;
  sizeofpic=320*200;

Type
  BigPalette=Array [0..2048-1] Of Byte; {holds both palettes}
  TPalette=Array[0..(256*3)-1] Of Byte;
  PPalette=^TPalette;
  filearray=array[0..maxfiles] of string[12];
  buftype=Array[0..sizeofpic-1] Of Byte;

Var
  vrDoBefore,vrDoAfter:Pointer;         {Pointers to procedures that execute just before and after vertical retrace}
  vrDone:Boolean;                       {indicates whether or not we're done with vertical retrace}
  globalFrame:LongInt;                  {External variable used by vertical-retrace code}
  VideoPage:Word;                       {Indicator of what video page we're on (segment, actually)}
  pal,rpal:BigPalette;                  {Storage spaces of Palettes}
  loop,VidPageIdx,j,j2:Integer;         {Temporary / loop variables}
  tp:^buftype;                          {buffer to hold the picture}
  pictures:filearray;                   {holds the names of the files to display}
  NumPictures:byte;                     {Number of pictures we have to display}

{--------------------Vertical-retrace interrupt section--------------------}

Procedure vrInit; External;
{initialize vertical-retrace interrupt}

Procedure vrClose; External;
{close down vertical-retrace interrupt}

Procedure nowBef;
{This is the procedure that gets called just before the vertical retrace
begins.  We wait for vertical retrace, then change the current video page.}
Begin
  Asm                                   {wait for vert retrace}
    mov DX,3dah

    @@1:
    In AL,DX
    Test AL,1
    jnz @@1

    @@2:
    In AL,DX
    Test AL,1
    jz @@1
  End;
  portw[$3D4]:=VideoPage Or $c;         {change the visible video page}
  portw[$3D4]:=$D;
  VideoPage:=VideoPage XOr $8000;       {update the video page segment variable}
End;

Procedure nowAft;
{This is the procedure that gets called just after the vertical retrace
 begins.  We change the video page, update our video page segment
 variable, then slam out the palette to the DAC.}
Begin
  Asm
    mov SI,Offset pal
    mov AX,[VideoPage]
    XOr AX,$8000
    ShR AX,5
    add SI,AX
    mov DX,3c8h
    sub AL,AL
    out DX,AL
    Inc DX
    sub CX,CX
    @@1:
    mov AL,[ES:SI]
    out DX,AL
    mov AL,[ES:SI+1]
    out DX,AL
    mov AL,[ES:SI+2]
    out DX,AL
    add SI,3
    Dec CL
    jnz @@1
  End;
  vrDone:=True;                         {Tell our vertical-retrace code we're
                                         done screwing around.}
End;

{--------------------"Helper" procedure/functions--------------------}

Procedure TextMode; assembler;
asm
  mov  ax,03h
  int  10h
end;

Procedure Pause;assembler;
ASM
  MOV AH, 10h
  INT 16h
END;

{---------------------------Picture Loading--------------------------}

Procedure loadpic(Page:Byte;PictureName:String);
{Loads a picture from disk and sets it up in memory.}

Var f:File;
  w,w2,w3,w4:Integer;

  Function GetGrayColor(D:Byte):Byte;
  {Returns the palette index of the closest gray equivalent of the color given
   in "D".}
  Var
    r,g,b:Word;                         {Red, Green, and Blue elements of a color}
  Begin
    r:=rpal[Page*1024+D*3];             {"Grayness" (luminance) is not determined}
    g:=rpal[Page*1024+D*3+1];           {simply by mixing the colors together;}
    b:=rpal[Page*1024+D*3+2];           {the human eye is more sensitive to}
    r:=(30*r+59*g+11*b) Div 100;        {red and green colors than blue.  Hence,}
    r:=r Div 4;                         {we give red and green elements more weight.}
    GetGrayColor:=240+r;
  End;

Begin
  GetMem(tp,64000);                     {Allocate the buffer for the picture}

  Assign(f,picturename);
  Reset(f,1);
  Seek(f,10);
  BlockRead(f,rpal[page*1024],240*3);
  Seek(f,778);
  BlockRead(f,tp^,64000);
  Close(f);

  {write out the picture in our "checkerboard" pattern}
  w4:=Page*80;
  For w:=0 To 3 Do Begin
    portw[$3c4]:=2+((1 ShL w)*256);
    For w2:=0 To 199 Do Begin
      For w3:=0 To 79 Do Begin
        mem[$a000:Page*32768+w2*160+w3+w4]:=tp^[w2*320+w3*4+w];
        mem[$a000:(1-Page)*32768+w2*160+w3+w4]:=GetGrayColor(tp^[w2*320+w3*4+w]);
      End;
    End;
    w4:=w4 XOr 80;
  End;
  FreeMem(tp,64000);                    {Free up the buffer for the picture}
End;

{-------------------------Palette/Picture mixing-----------------------}

Procedure mix(dest,dest2,src:PPalette;level:Byte);
{Mixes (blends) the color palette and grayscale palette.  Accepts a "level"
 parameter to determine how much to blend.}
Var
  i:Integer;
Begin
  For i:=0 To (240*3)-1 Do
    dest^[i]:=Byte(Word(Word(src^[i])*level) ShR 6);
  For i:=(240*3) To (256*3)-1 Do
    dest2^[i]:=Byte(Word(Word(src^[i])*level) ShR 6);
End;

{-----------------------Initialization Section----------------------}

Procedure Init320x400;
{initializes the video mode}
begin
  Asm
    mov AX,13h
    Int 10h                             {Sets up the video mode 320x200...}
    mov DX,3dah                         {...and promptly unchains it.  This}
    In  AL,DX                           {is so we can change the video}
    mov DX,3c0h                         {resolution later to 320x400, and}
    mov AL,31h                          {also enables us to change the}
    out DX,AL                           {address in memory to start displaying}
    mov AL,240                          {graphics.  This is commonly known}
    out DX,AL                           {as ModeX.}
  End;
  portw[$3c4]:=$604;
  port[$3D4]:=7;
  port[$3D5]:=port[$3D5] And $ef;
  portw[$3D4]:=$0011;
  portw[$3D4]:=$4009;
  portw[$3D4]:=$0014;
  portw[$3D4]:=$e317;                   {...done.}
end;

Procedure InitVars;
{initializes our variables}
begin
  FillChar(mem[$a000:0],65535,240);     {clear video memory}
  FillChar(pal,2048,0);                 {clear palette variable}
  FillChar(rpal,2048,0);                {clear palette variable}
  For loop:=0 To 15 Do Begin               {initialize palette variables}
    rpal[240*3+loop*3]:=loop*4+3;
    rpal[240*3+loop*3+1]:=loop*4+3;
    rpal[240*3+loop*3+2]:=loop*4+3;
    rpal[1024+240*3+loop*3]:=loop*4+3;
    rpal[1024+240*3+loop*3+1]:=loop*4+3;
    rpal[1024+240*3+loop*3+2]:=loop*4+3;
  End;
  VideoPage:=0;                         {set current video page segment to 0}
end;

Procedure InitFiles;

begin
  pictures[0]:='256pic1.scx';
  pictures[1]:='256pic2.scx';
  numpictures:=2;
end;

Procedure CloseDown;
{Shuts down any video modes, interrupt routines, etc.}
begin
  textmode;
  halt(0);
end;

Procedure FadePix;
{this does the 256-color crossfading sequence from complete start
to finish.}
begin
  vrInit;                               {initialize vertical retrace interrupt routines}
  vrDoBefore:=@nowBef;                  {point to the procedure to execute before vert. retr.}
  vrDoAfter:=@nowAft;                   {point to the procedure to execute after vert. retr.}
  VidPageIdx:=0;
  {MAIN DISPLAY LOOP}
  For loop:=0 To NumPictures-1 Do Begin     {for every picture on the command line,}
    FillChar(pal[VidPageIdx*1024],240*3,0); {initialize palette variable}
    loadPic(VidPageIdx,pictures[loop]);{load the picture, suppling video page index}

    For j2:=0 To 64-1 Do Begin          {for 64 counts,}
      vrDone:=False;                    {set the "done with vert. retr." var to false}
      While Not vrDone Do;              {wait until vertical retrace intr. is done}
      mix(@pal[VidPageIdx*1024],@pal[(1-VidPageIdx)*1024],@rpal[VidPageIdx*1024],j2+1);  {bring picture 0 up}
      VidPageIdx:=VidPageIdx XOr 1;     {switch pages}
      mix(@pal[VidPageIdx*1024],@pal[(1-VidPageIdx)*1024],@rpal[VidPageIdx*1024],64-j2); {bring picture 1 down}
      VidPageIdx:=VidPageIdx XOr 1;     {switch pages}
    End;
    {
    This is where you put your pausing routine.  You can do practically
    whatever you want to, as the vert. retr. routine will keep on
    flipping pages and updating the palettes.  The only exception is GUS
    playing, because the VRT code takes too much CPU time making sure
    that everything happens correctly.  (Sound Blaster playing, however,
    works just fine.)
    }
    pause;

    VidPageIdx:=VidPageIdx XOr 1;       {switch pages}
  End;

  VidPageIdx:=VidPageIdx XOr 1;         {switch pages}
  For j:=0 To 64-1 Do Begin             {Let's fade down the last picture...}
    vrDone:=False;                      {wait until...}
    While Not vrDone Do;                {...the vert. retr. int. is done}
    mix(@pal[VidPageIdx*1024],@pal[(1-VidPageIdx)*1024],@rpal[VidPageIdx*1024],64-j); {fade down palette}
  End;
  vrClose;                              {close vert. retr. int. system}
end;

{------------------------------MAIN PROGRAM------------------------------}

Begin
  InitFiles;
  Init320x400;
  InitVars;

  FadePix;

  CloseDown;
End.
