/*
This is the VESA 1.2 Tutorial by Telemachos of Peroxide ported to
C by 7/Rationals.

This is not a line for line translation of the original but it does
implement all of the most important points about using VESA 1.2. I
left out a few small things but added a couple of new things which
interested me.

I was up and coding VESA graphics at 1600x1200 in half a day even
after I had to port Telemachos' code to C so thanks to him for
writing the tutorial; it helped a lot. Any problems with this code
are of course mine. If you need to contact me use:

luitje@acm.org
*/
#include "vesa12.h"

extern struct sVESAInfo VESAInfo;
extern struct sVESAModeInfo VESAModeInfo;

void DisplayVESAInfo();
void DisplayModeInfo();
void DemoAMode(unsigned);
void DemoScrolling();

main()
{

	printf("Peroxide Programming Tips #5, C version\n");

	if(!VESAGetInfo()) {
		printf("VESA driver not found\n");
		exit(1);
	}
	IntroText();
	DisplayVESAInfo();
	DisplayModeInfo();
	DemoAMode(0x100);
	DemoAMode(0x101);
	DemoAMode(0x103);
	DemoAMode(0x105);
	DemoScrolling();
}

void DisplayVESAInfo()
{
	int i;
	unsigned char OEMInfo[100];

	printf("VESA Version Info: ");
	for( i=0; i<4; ++i)
	  printf("%c",VESAInfo.Signature[i]);

	printf(" %d.%d\n",VESAInfo.VersionMaj,VESAInfo.VersionMin);

	for( i=0; i<sizeof(OEMInfo); ++i)
		OEMInfo[i] = *(VESAInfo.OEMstringPtr++);
	printf("OEM string: %s\n",OEMInfo);
//	printf("OEM string: %s\n",VESAInfo.OEMstringPtr); // Does MSC not support far pointers here?

	printf("Total Memory: %d Banks, %ld Bytes, %ld MB\n",
		VESAInfo.TotalMemory,
		(long)VESAInfo.TotalMemory*65536,
		(long)VESAInfo.TotalMemory*65536/1048576);

	printf("DAC widths supported: ");
	for( i=6; i<9; ++i) {
		if(VESASetDACWidth(i) == i)
			printf(" %d",i);
	}
	printf("\n");
	printf("Press any key to continue");

	getch();
}

void DisplayModeInfo()
{
	unsigned mode;
	char string[80];

	printf("Modes supported:\n");

	while((mode = *(VESAInfo.VideoModePtr++)) != 0xffff) {
		VESAGetModeInfo(mode);
		printf("%03X:  %c %c %c %4d-%4d-%2d",
			mode,
			(VESAModeInfo.ModeAttributes.Color?'C':'M'),
			(VESAModeInfo.ModeAttributes.Graphics?'G':'T'),
			(VESAModeInfo.ModeAttributes.BIOSSupport?'B':' '),
			VESAModeInfo.XResolution,
			VESAModeInfo.YResolution,
			VESAModeInfo.BitsPerPixel);
		printf(" Gran: %2d Kb BpL: %2d #Pages %2d Memory model %d\n",
			VESAModeInfo.WinGranularity,
			VESAModeInfo.BytesPerScanLine,
			VESAModeInfo.NumberOfImagePages,
			VESAModeInfo.MemoryModel);

		if(VESAModeInfo.WinAAttributes & VWA_Supported) {
			printf("     Win A: %c %c at %X ",(VESAModeInfo.WinAAttributes&VWA_Read?'R':' '),
				(VESAModeInfo.WinAAttributes&VWA_Write?'W':' '),
				VESAModeInfo.WinASegment);
		}
		if(VESAModeInfo.WinBAttributes & VWA_Supported) {
			printf("Win B: %c %c at %X ",(VESAModeInfo.WinBAttributes&VWA_Read?'R':' '),
				(VESAModeInfo.WinBAttributes&VWA_Write?'W':' '),
				VESAModeInfo.WinBSegment);
		}
		if(VESAModeInfo.MemoryModel == 6) {
			printf(" Red %d(%d) Green %d(%d) Blue %d(%d) Resrvd %d DAC %s",
			VESAModeInfo.RedMaskSize,VESAModeInfo.RedFieldPosition,
			VESAModeInfo.GreenMaskSize,VESAModeInfo.GreenFieldPosition,
			VESAModeInfo.BlueMaskSize,VESAModeInfo.BlueFieldPosition,
			VESAModeInfo.RsvdMaskSize,
			(VESAModeInfo.DirectColorModeInfo&1?"Fix":"Var"));
		}

		printf("\n");
	}
	printf("Press any key to continue");
	getch();
}

void DemoAMode(unsigned mode)
{
	int i,j;
	char string[80];

	if(!VESASetMode(mode)) {
		SetVGAMode(3);
		printf("Your card doesn't support mode %03X\n",mode);
		getch();
		return;
	}

	VESAGetModeInfo(mode);
	VESAInitVars();

	for(i=0; i<10000; ++i) {
		VESAPutPixel(
		(int)((long) VESAModeInfo.XResolution * rand() / 32768),
		(int)((long) VESAModeInfo.YResolution * rand() / 32768),
		(int)((long) 256 * rand() / 32768));
	}

// This part only works if the mode has BIOS support.

	sprintf(string,"This is mode %03X: %d X %d X %d",
		mode,VESAModeInfo.XResolution,VESAModeInfo.YResolution,
		VESAModeInfo.BitsPerPixel);
	WriteXY(5,2,string,50);
	WriteXY(5,4,"Press any key for next mode...",50);

	printf("Press any key to continue");

	getch();
	SetVGAMode(3);
}


void delay(unsigned n)		// delay n clock ticks
{
	unsigned long far *cbios=(unsigned long far *) 0x0040006C;
	unsigned long t;

	t = *cbios + n;
	do {
	} while(*cbios < t);
}

void DemoScrolling()
{
	int i, j, mode=0x101, counter;
	char string[80];

	if(!VESASetMode(mode)) {
		SetVGAMode(3);
		printf("Your card doesn't support mode %03X\n",mode);
		getch();
		return;
	}

	VESAGetModeInfo(mode);
	VESAInitVars();

	VESASetLogicalScanLine(1024);
	VESAModeInfo.XResolution = 1024;
	VESASetScreenPosition(0,0);

	for(i=0; i<10000; ++i) {
		VESAPutPixel(
		(int)((long) VESAModeInfo.XResolution * rand() / 32768),
		(int)((long) VESAModeInfo.YResolution * rand() * 2 / 32768),
		(int)((long) 256 * rand() / 32768));
	}

	for( counter = 0; counter<=150;++counter) {
		VESASetScreenPosition(counter*2,0);
		delay(1);
	}
	for( counter = 150; counter>=0;--counter) {
		VESASetScreenPosition(counter*2,0);
		delay(1);
	}
	for( counter = 0; counter<=150;++counter) {
		VESASetScreenPosition(counter*2,counter*2);
		delay(1);
	}
	for( counter = 150; counter<=VESAModeInfo.YResolution;++counter) {
		VESASetScreenPosition(300,counter*2);
		delay(1);
	}

	SetVGAMode(3);
}

IntroText()
{
printf("      ****************************************************************\n");
printf("      *                                                              *\n");
printf("      *                      SVGA using VESA 1.2                     *\n");
printf("      *                        by : Telemachos                       *\n");
printf("      *                                                              *\n");
printf("      ****************************************************************\n\n");
printf("      Hiya! \n");
printf("      Welcome to the Peroxide Programming Tips #5\n");
printf("      This one is on coding the SVGA card using VESA 1.2\n");
printf("      This code *SHOULD* work on every graphic card that\n");
printf("      supports VESA 1.2 (which is about every card now a days)\n");
printf("                                                                      \n");
printf("      This program will :\n");
printf("         1) Display information about your SVGA card\n");
printf("         2) Display available modes on your card..\n");
printf("         3) Cycle through the most common SVGA modes (256 colors)\n");
printf("         4) Show you how to hardware scroll using the VESA BIOS\n\n");
printf("         Hit any key to get started....\n\n");
getch();

}
