/* fastmem.c - speed oriented memory allocation/deallocation
**
** Copyright (C) 1995  Jonathan Paul Griffiths.  All rights reserved.
**
** You may do anything with this code you wish EXCEPT sell it. You may sell
** any software you create using this code,  but you MUST NOT charge for
** the code itself.  Charging a distribution fee for this code is also
** FORBIDDEN.
*/
#include <string.h>
#include "fastmem.h"

#ifndef NULL
 #define NULL 0L
#endif

/* Bit operations from 'snippets', public domain by Bob Stout - cheers Bob! */
#define BOOL(x) (!(!(x)))
#define BitSet(arg,posn) (arg) = ((arg) | (1 << (posn)))
#define BitClr(arg,posn) (arg) = ((arg) & ~(1 << (posn)))
#define BitTst(arg,posn) BOOL((arg) & (1 << (posn)))

/* fast muls, divs and mods are my own sadistic invention */
#define FAST_32_MUL(x) ((x)<<5)
#define FAST_32_DIV(x) ((x)>>5)
#define FAST_32_MOD(x) ((x) & 0x0000001F)

/* promote a number to a multiple of 32 (upwards) */
#define PROMOTE(x) FAST_32_MUL(FAST_32_DIV((x)+31))

/* private routines */
void FMmark_mem_as_free(A_32_BIT *flags,unsigned int ptr);
void FMmark_mem_as_used(A_32_BIT *flags,unsigned int ptr);
unsigned int  FMget_next_free_block_number(A_32_BIT *flags);



fast_mem *fastmem_init(unsigned int max_items,size_t block_size)
{
 fast_mem *temp_mem;

 /* allocate fast mem header record */
 if((temp_mem=(fast_mem *)malloc(sizeof(fast_mem)))==NULL){
    exit(0);
 }

 /* allocate fast mem heap */
 if((temp_mem->heap=(void *)malloc(block_size*max_items))==NULL){
    exit(0);
 }

 /* allocate fast mem occupancy flags */
 if((temp_mem->occupancy_flags=(A_32_BIT *)malloc(sizeof(A_32_BIT)*FAST_32_DIV(PROMOTE(max_items))))==NULL){
    exit(0);
 }

 /* set occupancy details */
 temp_mem->blocks_remaining = max_items;
 temp_mem->max_blocks = max_items;
 temp_mem->block_size = block_size;

 /* zero occupancy flags */
 memset(temp_mem->occupancy_flags,0,(sizeof(A_32_BIT)*FAST_32_DIV(PROMOTE(max_items))));

 /* return the pointer */
 return temp_mem;
}


fast_mem *fastmem_destroy(fast_mem *fmem)
{
 free(fmem->occupancy_flags);
 free(fmem->heap);
 free(fmem);

 return (fast_mem *)NULL;
}


void fastmem_clear(fast_mem *fmem)
{
 /* clear occupancy flags */
 memset(fmem->occupancy_flags,0,sizeof(A_32_BIT)*FAST_32_DIV(PROMOTE(fmem->max_blocks)));

 /* reset usage info */
 fmem->blocks_remaining = fmem->max_blocks;
}


void *fastmem_allocate(fast_mem *fmem)
{
 unsigned int free_block;

 if(fmem->blocks_remaining){
    unsigned char *temp; /* do this so we don't dereference a void ptr. */

    --fmem->blocks_remaining;

    free_block  = FMget_next_free_block_number(fmem->occupancy_flags);
    FMmark_mem_as_used(fmem->occupancy_flags,free_block);

    temp = (unsigned char *)fmem->heap;

    return (void *) &(temp[free_block*fmem->block_size]);
 }
 else{
     return (void *)NULL;
 }
}


void *fastmem_deallocate(fast_mem *fmem,void *ptr)
{
 unsigned int freed_block;

 /* Note: this only works because void ptrs and ints are the same size (32) */
 freed_block  = ((unsigned int)ptr-(unsigned int)fmem->heap)/(unsigned int)fmem->block_size;

 if(freed_block < fmem->max_blocks){
    ++fmem->blocks_remaining;
    FMmark_mem_as_free(fmem->occupancy_flags,freed_block);

    return (void *)NULL;
 }
 else{
    return ptr;
 }
}


unsigned int fastmem_blocks_remaining(fast_mem *fmem)
{
 return fmem->blocks_remaining;
}




/****************************** PRIVATE  ************************************/

unsigned int FMget_next_free_block_number(A_32_BIT *flags)
{
 unsigned int blocks = 0;
 unsigned int bits = 0;

 /* skip any full blocks */
 while(*flags == 0xffffffff){
       ++flags;
       ++blocks;
 }

 /* flags points to an area with some free blocks */
 while((BitTst(*flags,bits))==1){
       ++bits;
 }

 return FAST_32_MUL(blocks)+bits;
}


void FMmark_mem_as_used(A_32_BIT *flags,unsigned int ptr)
{
 BitSet((flags[FAST_32_DIV(ptr)]),(FAST_32_MOD(ptr)));
}


void FMmark_mem_as_free(A_32_BIT *flags,unsigned int ptr)
{
 BitClr((flags[FAST_32_DIV(ptr)]),(FAST_32_MOD(ptr)));
}




