/* lzo_util.c -- utilities for the the LZO library

   This file is part of the LZO real-time data compression library.

   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   The LZO library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the LZO library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation, Inc.,
   675 Mass Ave, Cambridge, MA 02139, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include <stdio.h>

#include "lzo_conf.h"
#include "lzo_util.h"



/***********************************************************************
// slow but portable <string.h> stuff, only used in assertions
************************************************************************/

int lzo_memcmp(const lzo_voidp s1, const lzo_voidp s2, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memcmp(s1,s2,len);
#else
	const lzo_byte *p1 = (const lzo_byte *) s1;
	const lzo_byte *p2 = (const lzo_byte *) s2;
	int d;

	if (len > 0) do
	{
		d = *p1 - *p2;
		if (d != 0)
			return d;
		p1++;
		p2++;
	}
	while (--len > 0);
	return 0;
#endif
}


lzo_voidp lzo_memcpy(lzo_voidp dest, const lzo_voidp src, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memcpy(dest,src,len);
#else
	lzo_byte *p1 = (lzo_byte *) dest;
	const lzo_byte *p2 = (const lzo_byte *) src;

	if (len > 0) do
		*p1++ = *p2++;
	while (--len > 0);
	return dest;
#endif
}


lzo_voidp lzo_memmove(lzo_voidp dest, const lzo_voidp src, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memmove(dest,src,len);
#else
	lzo_byte *p1 = (lzo_byte *) dest;
	const lzo_byte *p2 = (const lzo_byte *) src;

	if (len <= 0 || p1 == p2)
		return dest;

	if (p1 < p2)
	{
		do
			*p1++ = *p2++;
		while (--len > 0);
	}
	else
	{
		p1 += len;
		p2 += len;
		do
			*--p1 = *--p2;
		while (--len > 0);
	}
	return dest;
#endif
}


lzo_voidp lzo_memset(lzo_voidp s, int c, lzo_uint len)
{
#if (LZO_UINT_MAX <= UINT_MAX)
	return memset(s,c,len);
#else
	lzo_byte *p = (lzo_byte *) s;

	if (len > 0) do
		*p++ = LZO_BYTE(c);
	while (--len > 0);
	return s;
#endif
}


/***********************************************************************
// adler32 checksum - adapted from free code by Mark Adler
// http://quest.jpl.nasa.gov/zlib
************************************************************************/

#define BASE 65521u /* largest prime smaller than 65536 */
#define NMAX 5552
/* NMAX is the largest n such that 255n(n+1)/2 + (n+1)(BASE-1) <= 2^32-1 */

#define DO1(buf,i)  {s1 += buf[i]; s2 += s1;}
#define DO2(buf,i)  DO1(buf,i); DO1(buf,i+1);
#define DO4(buf,i)  DO2(buf,i); DO2(buf,i+2);
#define DO8(buf,i)  DO4(buf,i); DO4(buf,i+4);
#define DO16(buf)   DO8(buf,0); DO8(buf,8);

lzo_uint lzo_adler32(lzo_uint adler, const lzo_byte *buf, lzo_uint len)
{
	lzo_uint s1 = adler & 0xffff;
	lzo_uint s2 = (adler >> 16) & 0xffff;
	int k;

	if (buf == NULL)
		return 1;

	while (len > 0)
	{
		k = len < NMAX ? (int) len : NMAX;
		len -= k;
		while (k >= 16) 
		{
			DO16(buf);
			buf += 16;
			k -= 16;
		}
		if (k != 0) do 
		{
			s1 += *buf++;
			s2 += s1;
        } while (--k);
		s1 %= BASE;
		s2 %= BASE;
	}
	return (s2 << 16) | s1;
}

#undef BASE
#undef NMAX
#undef DO1
#undef DO2
#undef DO4
#undef DO8
#undef DO16


/***********************************************************************
// Runtime check of the assumptions about memory model, byte order and
// other low-level constructs.
************************************************************************/

__inline__ int lzo_assert(int expr)
{
	return (expr) ? 1 : 0;
}


int _lzo_config_check(void)
{
	int r = 1;
	int i;
	lzo_uint adler;
	char wrkmem[8 * sizeof(lzo_byte *)];
	const lzo_bytepp const dict = (const lzo_bytepp) wrkmem;

	r &= lzo_assert(sizeof(lzo_uint) >= 4);
	r &= lzo_assert(sizeof(lzo_uint) >= sizeof(unsigned));

	r &= lzo_assert(sizeof(lzo_ptrdiff_t) >= 4);
	r &= lzo_assert(sizeof(lzo_ptrdiff_t) >= sizeof(ptrdiff_t));

	r &= lzo_assert(sizeof(lzo_voidp) >= 4);
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_byte *));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_voidpp));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(lzo_bytepp));
	r &= lzo_assert(sizeof(lzo_voidp) == sizeof(dict[0]));

	r &= lzo_assert(sizeof(lzo_ptr_t) >= 4);
	r &= lzo_assert(sizeof(lzo_ptr_t) >= sizeof(lzo_voidp));
	r &= lzo_assert(sizeof(lzo_ptr_t) >= sizeof(ptrdiff_t));
	r &= lzo_assert(sizeof(lzo_ptr_t) >= sizeof(lzo_ptrdiff_t));

	/* assert the signedness of our integral types */
	r &= lzo_assert((lzo_uint)     (1ul << (8*sizeof(lzo_uint)-1))      > 0);
	r &= lzo_assert((lzo_ptrdiff_t)(1ul << (8*sizeof(lzo_ptrdiff_t)-1)) < 0);
	r &= lzo_assert((lzo_ptr_t)    (1ul << (8*sizeof(lzo_ptr_t)-1))     > 0);

#if defined(LZO_UNALIGNED_OK)
	r &= lzo_assert(sizeof(lzo_uint) == 4);
	r &= lzo_assert(sizeof(short) == 2);
	if (r == 1)
	{
		unsigned char x[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };
		lzo_uint a[4];
		unsigned short b[4];

		for (i = 0; i < 4; i++)
		{
			a[i] = * (lzo_uint *) &x[i];
			b[i] = * (unsigned short *) &x[i];
		}

#  if (LZO_BYTE_ORDER == LZO_LITTLE_ENDIAN)
		r &= lzo_assert(a[0] == 0x03020100L);
		r &= lzo_assert(a[1] == 0x04030201L);
		r &= lzo_assert(a[2] == 0x05040302L);
		r &= lzo_assert(a[3] == 0x06050403L);
		r &= lzo_assert(b[0] == 0x0100);
		r &= lzo_assert(b[1] == 0x0201);
		r &= lzo_assert(b[2] == 0x0302);
		r &= lzo_assert(b[3] == 0x0403);
#  elif (LZO_BYTE_ORDER == LZO_BIG_ENDIAN)
		r &= lzo_assert(a[0] == 0x00010203L);
		r &= lzo_assert(a[1] == 0x01020304L);
		r &= lzo_assert(a[2] == 0x02030405L);
		r &= lzo_assert(a[3] == 0x03040506L);
		r &= lzo_assert(b[0] == 0x0001);
		r &= lzo_assert(b[1] == 0x0102);
		r &= lzo_assert(b[2] == 0x0203);
		r &= lzo_assert(b[3] == 0x0304);
#  else
#    error
#  endif
	}
#endif

	/* sanity check of the memory model */
	if (r == 1)
	{
		for (i = 0; i < 8; i++)
			r &= lzo_assert((lzo_voidp) (&dict[i]) ==
			                (lzo_voidp) (&wrkmem[i * sizeof(lzo_byte *)]));
	}
	/* assert NULL == 0 */
	if (r == 1)
	{
		BZERO8_PTR(dict,8);
		for (i = 0; i < 8; i++)
			r &= lzo_assert(dict[i] == NULL);
	}

	/* check the lzo_adler32() function */
	if (r == 1)
	{
		adler = lzo_adler32(0, NULL, 0);
		adler = lzo_adler32(adler, __lzo_copyright, 138);
		r &= lzo_assert(adler == 0xb06d2f7cL);
	}

	return r == 1 ? LZO_E_OK : LZO_E_ERROR;
}


/***********************************************************************
//
************************************************************************/

/* If you use the LZO library in a product, you *must* keep this
 * copyright string in the executable of your product.
 */

const lzo_byte __lzo_copyright[] = 
	"\n\n\n"
	"LZO real-time data compression library.\n"
	"Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer\n"
	"<markus.oberhumer@jk.uni-linz.ac.at>\n"
	"\n"
	"LZO version: v" LZO_VERSION_STRING ", " LZO_VERSION_DATE "\n"
	"LZO build date: " __DATE__ " " __TIME__ "\n"
	"\n\n\n";


unsigned lzo_version(void)
{
	return LZO_VERSION;
}

const char *lzo_version_string(void)
{
	return LZO_VERSION_STRING;
}


/***********************************************************************
//
************************************************************************/

int lzo_init(void)
{
	int r;

	r = _lzo_config_check();
	if (r != LZO_E_OK)
		return r;

	return LZO_E_OK;
}


/*
vi:ts=4
*/
